MODULE Query EXPORTS Main;

(* This program accepts commands to create or open ndbm databases and to store, fetch or delete elements. A sorted list of keys in the database may also be printed. *)

IMPORT Stdio, Dbase, Wr, Rd, Lex, Text, Thread;

(* Exceptions are not caught and will stop the program execution. *)

<*FATAL Wr.Failure*>
<*FATAL Rd.Failure*>
<*FATAL Thread.Alerted*>

(* Read all the keys in the database into an array. If the array
   is too small, double its size. *)

PROCEDURE PrintDbase(db: Dbase.T) =
  VAR
    key, tmp: TEXT;
    cursor: CARDINAL := 0;
  BEGIN
    key := Dbase.FirstKey(db);
    WHILE key # NIL DO
      IF cursor > LAST(keyArray^) THEN
        oldKeyArray := keyArray;
        keyArray := NEW(REF ARRAY OF TEXT,2 * NUMBER(oldKeyArray^));
        SUBARRAY(keyArray^,0,NUMBER(oldKeyArray^)) := oldKeyArray^;
      END;
      keyArray[cursor] := key;
      INC(cursor);
      key := Dbase.NextKey(db);
    END;

    (* Use an inefficient sorting algorithm *)

    FOR i := cursor - 1 TO 0 BY -1 DO
      FOR j := 0 TO i - 1 DO
        IF Text.Compare(keyArray[j], keyArray[i]) = 1 THEN
          tmp := keyArray[i];
          keyArray[i] := keyArray[j];
          keyArray[j] := tmp;
        END;
      END;
    END;

    (* Print all the keys *)

    FOR i := 0 TO cursor - 1 DO
      Wr.PutText(Stdio.stdout,keyArray[i] & "\n");
    END;
  END PrintDbase;

(* The array for printing the keys is kept from one invocation to the next instead of starting from scratch each time the PrintDbase procedure is called. *)

VAR
  keyArray := NEW(REF ARRAY OF TEXT,8);
  oldKeyArray : REF ARRAY OF TEXT;
  db: Dbase.T;
  command, name, key, content: TEXT;

BEGIN

  (* Commands are read. For each possible command, the appropriate few lines are executed. *)

  LOOP
    Wr.PutText(Stdio.stdout,"Enter command\n");
    Wr.Flush(Stdio.stdout);

    Lex.Skip(Stdio.stdin);
    command := Lex.Scan(Stdio.stdin);

    IF Text.Equal(command,"exit") THEN EXIT;

    ELSIF Text.Equal(command,"create") THEN
      Lex.Skip(Stdio.stdin);
      name := Lex.Scan(Stdio.stdin);
      db := Dbase.Create(name);

    ELSIF Text.Equal(command,"open") THEN
      Lex.Skip(Stdio.stdin);
      name := Lex.Scan(Stdio.stdin);
      db := Dbase.Open(name);

    ELSIF Text.Equal(command,"close") THEN
      Dbase.Close(db);

    ELSIF Text.Equal(command,"fetch") THEN
      Lex.Skip(Stdio.stdin);
      key := Lex.Scan(Stdio.stdin);
      Wr.PutText(Stdio.stdout,Dbase.Fetch(db,key) & "\n");

    ELSIF Text.Equal(command,"store") THEN
      Lex.Skip(Stdio.stdin);
      key := Lex.Scan(Stdio.stdin);
      Lex.Skip(Stdio.stdin);
      content := Lex.Scan(Stdio.stdin);
      Dbase.Store(db,key,content);

    ELSIF Text.Equal(command,"delete") THEN
      Lex.Skip(Stdio.stdin);
      key := Lex.Scan(Stdio.stdin);
      Dbase.Delete(db,key);

    ELSIF Text.Equal(command,"print") THEN
      PrintDbase(db);

    ELSE
      Wr.PutText(Stdio.stdout,"Incorrect command\n");
    END;
  END;
END Query.

