;*****************************************************************************************
;
;       PROGRAM:                DD_SYNTH.ASM (SOURCE FILE)
;                               DD_SYNTH.DAT (OPTIONS & FREQUENCY SETTINGS ETC.)
;
;       ASSEMBLER:              MPASMWIN v02.70
;
;       AUTHOR:                 Steven Jones         EMAIL stevejones@picknowl.com.au
;
;
;       REVISION HISTORY:
;
;       VERSION:        DATE            COMMENTS
;
;       1.00            NOV 09 2000     First release.
;
;       1.10            JAN 20 2001     Mod to offset frequency setup screen
;                                       to allow negative offsets.
;
;       1.11            MAR 12 2001     Mods to use AD9851. But not the AD9850.
;                                       Requested by Vladimir Gerganov.
;                                       (internal x 6 mult for ref clock)
;                                       We enable the 6 x multiplier, and then use this
;                                       multiplied frequency as the DDS_CLOCK_FREQ.
;                                       ie 26MHz xtal = 156MHz DDS_CLOCK_FREQ.
;                                       changes made :- DDS_CLOCK_FREQ range now
;                                       30 to 190MHz, MAX_DDS_FREQ now 90MHz, bit W32
;                                       of the DDS control word now a 1. (enable 6 x ref)
;
;       1.20                            Mods to use either AD9850 or AD9851.
;                                       (AD9851 can use 1 x refclk or 6 x refclk)
;                                       Not released, for development only.
;                                       See implementation note 1 below.
;
;       1.30            OCT 20 2001     An option to include a REPEATER TX offset
;                                       function. Requested by Cristi Simion - YO3FLR.
;                                       When the new RPT button is pressed an extra
;                                       -600 KHz or +600 KHz offset is added to the
;                                       TX frequency.
;                                       The default REPEATER TX OFFSET is 600 KHz, but
;                                       it can be changed in the RPT OFFSET setup screen.
;                                       See implementation note 8 below.
;                                       Fixed the bug that could infrequently cause the
;                                       display to "lockup".
;                                       Add the option to assemble the program for either
;                                       the PIC 16F84 or 16F628. See the first few lines
;                                       of the DD_SYNTH.DAT file.
;
;*****************************************************************************************

VERSION_NUMBER          EQU     130             ; Software version. (no decimal point)

                LIST N=0, R=DEC, F=INHX8M       ; INCLUDE THE FILE THAT CONTAINS THE
                ORG     H'2100'                 ; PROGRAM OPTIONS, INITIAL PLL & DISPLAY
                INCLUDE "DD_SYNTH.DAT"          ; FREQUENCY RANGES INTO EEPROM.

;*****************************************************************************************
;
;       GENERAL COMMENTS
;
;       This program assumes you are using a PIC16F84 or a PIC16F628 with a 4MHz crystal.
;
;       This program is based on the various versions of SIG_GEN,
;       by Curtis W. Preuss, WB2V. And modified and updated by among others :-
;       Bruce Stough,    - AA0ED
;       Craig Johnson,   - AA0ZZ
;       George Heron,    - N2APB
;
;       The transceiver functions, TX offset & Receive Incremental Tuning (RIT)
;       are based on modifications by Mark Kilmier -VK5EME to SIG_GEN version 3a, which
;       Mark has named SIGGEN2.
;
;       The aim of this project was to include as many of the following functions
;       suggested by Mark, as possible.
;
;       1) Use the original PCB, modified to gain access to RA2..RA4 pins. Using
;          diodes to provide the number of switch inputs required.
;       2) DDS frequency output, 0Hz to 40MHz.
;       3) TX/RX switching. By grounding RA4 to go to TX mode.
;       4) Frequency readout can be offset by 1Hz to 11GHz.
;       5) No requirement for band switching, but able to program upper and lower limits.
;       6) Memorise the last frequency on power up.
;       7) Use low cost mechanical rotary encoder instead of the optical type.
;       8) Variable rate tuning not required.
;       9) By grounding RA2, (PIC pin 1) then turning the rotary encoder, be able to
;          set the step size required. eg 100Hz, 1KHz, 10KHz etc.
;      10) Use 16x2 LCD with or without R/W pin. (use delays rather than busy checks)
;      11) Software fine tuning of the XTAL oscillator frequency still required.
;      12) RIT available in RX mode by grounding, via diodes, both RA3 & RA4.
;      13) Still require the option of programming the software for different XTAL
;          oscillators. eg 66MHz, 100MHz, 125MHz.
;
;
;       Implementation notes.
;
;       1) Use the include file DD_SYNTH.DAT to select which DDS chip the program is
;          assembled for. Once the PIC is programmed, You can verify which DDS option was
;          selected by noting the last digit of the version number displayed on the LCD.
;          ver x.xO for AD9850.
;          ver x.x1 for AD9851, 6 x REFCLK disabled.
;          ver x.x2 for AD9851, 6 x REFCLK enabled. ( DDS_CLOCK_FREQ = 6 x Xtal freq)
;
;          Changes to the DDS chip option can only be made by editing the DD_SYNTH.DAT
;          file, there is no setting screen.
;          See the first few lines of the DD_SYNTH.DAT file.
;       2) Uses the include file DD_SYNTH.DAT to set the following EEPROM defaults.
;
;          DDS_CLOCK_FREQ. range 30 MHz to 130 MHz for AD9850.
;                                30 MHz to 190 MHz for AD9851.
;
;          MAX_DDS_FREQ.   range 0 MHz to  60 MHz for AD9850. (Usually 1/3 DDS CLOCK FREQ)
;                                0 MHz to  90 MHz for AD9851. (or LP filter cutoff freq. )
;          RX_DDS_FREQ.
;          MIN_RX_DDS_FREQ.
;          MAX_RX_DDS_FREQ.
;          MAX_RIT_OFFSET.
;          TX_OFFSET_FREQ.              (Offset from RX frequency)
;          OFFSET_FREQ.     20 GHz max. (Added to TX/RX frequency before display on LCD)
;          MULTIPLIER.         1 to 32. (Frequency display is multiplied by N. Limit N )
;                                       (so result is not to much greater than 68 GHz. )
;
;          (note. during assembly, no checks are made to verify the settings are valid)
;
;          The frequency displayed on the LCD, is calculated as shown below.
;          RX = MULTIPLIER x (RX_DDS_FREQ + OFFSET_FREQ)
;          TX = MULTIPLIER x (RX_DDS_FREQ + OFFSET_FREQ + TX_OFFSET_FREQ)
;
;          The frequency programmed into the DDS, is calculated as shown below.
;          RX = ABS(RX_DDS_FREQ)
;          TX = ABS(RX_DDS_FREQ + TX_OFFSET_FREQ)
;
;          RX_DDS_FREQ, MIN_RX_DDS_FREQ, MAX_RX_DDS_FREQ, TX_OFFSET_FREQ & OFFSET_FREQ
;          may be positive or negative values.
;          (as long as the resulting display frequency is positive)
;
;          Below are examples of using offsets to produce different display frequencies.
;          In each example the DDS output freqency range is 30 MHz to 40 MHz.
;
;          MIN_RX_DDS_FREQ  MAX_RX_DDS_FREQ  OFFSET_FREQ     Frequency display range.
;
;           30 MHZ           40 MHz           100 MHz   =    130 MHz to 140 MHz. \ Note 1
;           30 MHZ           40 MHz           -10 MHz   =     20 MHz to  30 MHz. /
;          -40 MHZ          -30 MHz           170 MHz   =    140 MHz to 130 MHz. \ Note 2
;          -40 MHZ          -30 MHz            60 MHz   =     30 MHz to  20 MHz. /
;
;          Note 1.   CW rotation of the rotary encoder increase both the DDS O/P frequency
;                    and display frequency.
;          Note 2.   CW rotation of the rotary encoder decrease the DDS O/P frequency but
;                    increases the display frequency.
;
;       3) Once the PIC is programmed, all of the above defaults can be modified by
;          entering the setup screens. Changes can be made using the rotary encoder
;          and saved to EEPROM. All setup screens check the changes to make sure they
;          are valid. eg The TX offset cannot be set to a value that when added to the
;          RX frequency, results in a frequency outside the range of the DDS.
;
;       4) When first turned on, the software name and version number is displayed.
;
;       5) After 1 second the main screen is displayed.
;          The frequency displayed is the same as the frequency displayed on the main
;          screen before the power was turned off.
;          If the RA4 (TX) pin is pulled low "TX" is displayed, and the DDS is set to 
;          produce the TX frequency, (RX frequency + TX offset)
;          Otherwise RX is displayed, and the DDS is set to produce the RX frequency.
;          Any changes to the frequency are saved to EEPROM 2 seconds after the rotary
;          encoder stops moving.
;          If RA3 & RA4 via diodes (the RIT button) is briefly pulled low, RIT is
;          displayed on the LCD. The TX frequency cannot be changed. But the RIT frequency
;          can be changed as long as it does not go to far away from the RX frequency.
;          (the range is RX freq +/-MAX_RIT_OFFSET) Pressing the RIT button again, removes
;          the RIT display and the RX frequency reverts to its pre RIT value.
;
;       6) To enter the setup screens, pull RA3 (CAL button) low, while turning on the
;          power. After the version number is displayed the first setup screen, for
;          calibrating the DDS CLOCK freq, is displayed.
;          It sets the DDS to produce 10MHz, and displays the DDS calibration constant.
;          This can be modified to set the boards O/P frequency to exactly 10 MHz.
;          (using a frequency counter)
;          The DDS calibration constant is equal to 2^56/(DDS CLOCK freq). So if you
;          change to a different XTAL freq you can change the constant as shown below.
;
;          For  50.00 MHz,  2^56/50.00 MHz = 1441151880
;               66.00 MHz,  2^56/66.00 MHz = 1091781727
;               66.66 MHz,  2^56/66.66 MHz = 1080972007
;              100.00 MHz,  2^56/100.0 MHz =  720575940
;              120.00 MHz,  2^56/120.0 MHz =  600479950
;
;          This screen will continue to be displayed until the CAL button is pressed
;          briefly. This will save the changes to EEPROM and display the next settings
;          screen. (MAX DDS FREQ)
;          Again, pressing the CAL button will save any changes made on this screen to
;          EEPROM and display the next settings screen. etc.
;          On all setup screens except the first one, if the buttons and rotary encoder
;          are left idle for 10 seconds, any changes made on that screen will be ignored
;          and you will be returned to the main display.
;          Once in the main screen (not RIT) pressing the CAL button for 1 sec will
;          take you to the 3rd setup screen. (the first two screens rarely need changing)
;
;          (note, the CAL button does not function while the TX lead is opperated due to
;          the way the RIT button is wired. While the TX lead is opperated the software
;          is unable to determine the difference between the CAL and RIT buttons, The
;          software gives priority to the RIT function.)
;
;       7) If RA2 (adjust step size button) is pulled low, a cursor is displayed under one
;          digit of the frequency display. This can be changed using the rotary encoder.
;          If for example it is under the 1 KHz digit, when the (adjust step size button)
;          is released, rotating the encoder will change the frequency in 1 KHz steps.
;          There is also a small amount of variable rate tuning. (the faster the encoder
;          is rotated, the greater the step size) Due to 4mS software debouncing, if the
;          encoder is rotated to fast, no change in frequency will occur.
;          Any changes to the step size, while in the main screen, are saved to EEPROM.
;
;       8) Optionaly include a REPEATER TX offset function for use with FM.
;          When the new button RPT (connected via diodes to RA2 and RA3) is
;          pressed briefly -RPT is displayed on the LCD and an extra -600 KHz offset is
;          added to the TX frequency. Briefly pressing RPT again will remove the offset.
;          Pressing RPT for 1 sec will display +RPT and an extra +600 KHz offset is
;          added to the TX frequency. Briefly pressing RPT again will remove the offset.
;          When using the RPT offset, the software checks that the TX frequency remains
;          within the range of the DDS chip. If the TX frequency goes out of the DDS
;          range the RPT offset is removed and the RPT display is cleared.
;          The default REPEATER TX OFFSET is 600 KHz, but it can be changed in the
;          RPT OFFSET setup screen. Range = 0 to 10 MHz.
;
;          The option to include the REPEATER TX offset function can only be made by
;          editing the DD_SYNTH.DAT file.
;          See the first few lines of the DD_SYNTH.DAT file.
;          If the REPEATER TX offset function is included on the PIC16F84, the checks
;          made in the MULTIPLIER setup screen are removed to save code space. So be sure
;          to check that you do not set a large MULTIPLIER and OFFSET FREQUENCY as a
;          display frequency greater than 99 GHz will not fit on the LCD display.
;
;       9) Uses a 16x2 LCD with or without R/W pin. (use delays rather than busy checks)
;          If the LCD has a R/W pin tie it to 0V.
;
;      10) To enable all of the required functions to be programmed into the limited code
;          space, any code that was used more than once or twice was changed to a
;          subroutine. And wherever a subroutine ended with a CALL and RETURN, eg
;               NOP
;               CALL    XXXXXX
;               RETURN
;
;          The code was changed to a GOTO.
;               NOP
;               GOTO    XXXXXX                  ; Return via "GOTO".
;
;          I know this is poor programming practice, but it did save a lot of code space.
;          Each of these modifications as identified by the [Return via "GOTO".] comment.
;
;*****************************************************************************************



;*****************************************************************************************
;               General definitions
;*****************************************************************************************

#DEFINE         ENCODER_A       PORTA,0         ; Rotary encoder switch.
#DEFINE         ENCODER_B       PORTA,1         ; Rotary encoder switch.
#DEFINE         SWITCH_A        PORTA,2         ; External switch array . (active low)
#DEFINE         SWITCH_B        PORTA,3         ;                 "
#DEFINE         SWITCH_C        PORTA,4         ;                 "

#DEFINE         LCD_RS          PORTB,1         ; LCD Reg select. 0 = instruct 1 = data
#DEFINE         UNUSED          PORTB,2         ; Unused pic pin.
#DEFINE         LCD_E           PORTB,3         ; LCD Enable.     0 = disable  1 = enable
;                               PORTB,4..7      ; LCD 4 bit data bus.
#DEFINE         DDS_LOAD        PORTB,0         ; AD9850/51 update pin.
#DEFINE         DDS_CLOCK       PORTB,5         ; AD9850/51 write clock pin.
#DEFINE         DDS_DATA        PORTB,7         ; AD9850/51 serial data input pin.

#DEFINE         ENC_A_DEBOUNCED FLAGS,0         ; Encoder A state after debouncing.
#DEFINE         ENC_B_DEBOUNCED FLAGS,1         ; Encoder B state after debouncing.
#DEFINE         OLD_A_DEBOUNCED FLAGS,2         ; Old encoder a state after debouncing.
#DEFINE         LEADING_FLAG    FLAGS,3         ; Leading zero indicator.
#DEFINE         NEGATIVE        FLAGS,4         ; Number to display is negative.
#DEFINE         NEGATIVE_RX     FLAGS,5         ; Min RX DDS Freq is negative.
#DEFINE         TIMEOUT         FLAGS,6         ; If set, go back to main display.
#DEFINE         EEPROM_UPDATE   FLAGS,7         ; If set, update the EEPROM.

#DEFINE         CAL_SCRN        FLAG2,0         ; Set if displaying 10 MHz CAL screen.
#DEFINE         RIT_SCRN        FLAG2,1         ; Set if displaying RIT screen.
#DEFINE         PRESSED         FLAG2,2         ; Result flag from button pressed call.
#DEFINE         WRITE_FLAG      FLAG2,3         ; Set if we want to do an EEPROM write.
#DEFINE         RPT_FLAG        FLAG2,4         ; Set if RPT offset is positive.
#DEFINE         SW_A_FLAG       FLAG2,5         ; Set if step size button is pressed.

;*****************************************************************************************
;               General equates
;*****************************************************************************************

RESETVECTOR     EQU             H'00'           ; PIC Reset vector.
INTVECTOR       EQU             H'04'           ; PIC Interrupt vector.
TRUE            EQU             H'FF'
FALSE           EQU             H'00'
AD9850          EQU             H'00'
AD9851          EQU             H'01'
YES             EQU             H'01'
NO              EQU             H'00'

MIN_ON          EQU              6              ; Min valid button pressed time. (6 x 8mS)
MAX_ON          EQU             70              ; Max valid button pressed time.(70 x 8mS)
TIMEOUT_NUM     EQU              5              ; Display timeout. 10 sec (5 x 2.1 sec)

;*****************************************************************************************
;               RAM variables
;*****************************************************************************************

                CBLOCK  RAMSTART

                ENC_HISTORY                     ; Last 4 samples of encoder state.
                COUNT_0                         ; Counter for delay routine.
                COUNT_1                         ; General counter variables.

                COUNT_2                         ; ** All variables from here to the end **
                COUNT_3                         ;    of RAM are cleared at power up.
                COUNT_4                         ;
                TEMP_1                          ; Temporary storage.
                TEMP_2                          ;        "
                TEMP_3                          ;        "
                ENC_COUNTER                     ; 1 to 38 = CW. -1 to -38 = CCW.
                INT_COUNT                       ; 8 to 1. Decremented every 1.024 mS.
                SW_B_CNT                        ; CAL switch pressed counter. (x 8mS)
                SW_B_OLD                        ; State of old switch count.
                SW_C_CNT                        ; TX switch pressed counter.  (x 8mS)
                SW_BC_CNT                       ; RIT switch pressed counter. (x 8mS)
                SW_BC_OLD                       ; State of old switch count.
                TIMEOUT_L                       ; 16 bit display timeout down counter,
                TIMEOUT_H                       ; return to main display when = 0.
                EEPROM_WRITE                    ; EEPROM write update down counter.
                FLAGS                           ; 8 misc flags.
                FLAG2                           ;       "

                FREQUENCY:5                     ; 40 bit frequency.
                STEP_NUM                        ; LCD cursor pos & step size num.(0-7)
                                                ; (Do not separate frequency & STEP_SIZE)
                STEP_RATE                       ;
                DDS_OLD:5                       ; 40 bit previous DDS freq.

                ARG1_10                         ; MSD.    11 byte ASCII buffer.
                ARG1_9                          ; MSD. \
                ARG1_8                          ;       |
                ARG1_7                          ;       |
                ARG1_6                          ;       |
                ARG1_5                          ;       | 80 bit maths buffer.
                ARG1_4                          ;       |
                ARG1_3                          ;       |
                ARG1_2                          ;       |
                ARG1_1                          ;       |
                ARG1_0                          ; LSD. /

                ARG2_4                          ; MSD. \
                ARG2_3                          ;       |
                ARG2_2                          ;       | 40 bit maths buffer.
                ARG2_1                          ;       |
                ARG2_0                          ; LSD. /

                ARG3_4                          ; MSD. \
                ARG3_3                          ;       |
                ARG3_2                          ;       | 40 bit maths buffer.
                ARG3_1                          ;       |
                ARG3_0                          ; LSD. /

                ARG4_4                          ; MSD. \
                ARG4_3                          ;       |
                ARG4_2                          ;       | 40 bit maths buffer.
                ARG4_1                          ;       |
                ARG4_0                          ; LSD. /

                ENDC

        IF REPEATER == TRUE

                CBLOCK                          ; Ram for RPT function.

                SW_AB_CNT                       ; RPT switch pressed counter. (x 8mS)
                SW_AB_OLD                       ; State of old switch count.

                ENDC
        ENDIF

                CBLOCK  RAM_INT

                SAVE_STATUS                     ; FOR INTERUPT SERVIVE ROUTINE.
                SAVE_W_REG                      ;             "

                ENDC

;*****************************************************************************************
;
;               Interrupt service routine
;
;               Service the TMR0 interrupt (EVERY 1.024 mS)
;
;               EVERY 1.024 mS.
;               Debounce the rotary encoder by sampling the encoder state every mS.
;               If 4 consecutive checks of the pins are at the same state, that state
;               is considered to be valid. (debounced)
;               Determine the direction of rotation by looking at the debounced
;               Encoder B pin on the rising edge of the debounced encoder A pin.
;               Encoder B pin low = Clockwise, high = Counter-Clockwise.
;               On each determination of the direction, set the ENC_COUNTER variable
;               to the current state of the STEP_RATE variable.
;               (+ STEP_RATE for CW, - STEP_RATE for CCW)
;               Then set the STEP_RATE variable to 38.
;               This variable is then decremented toward 1 every mS. Therefore the step
;               rate increases from 1, for slow rotation, up to 30 for fast rotation.
;               (The variable would have decremented to at least 30 before)
;               (the next rising edge of the debounced encoder A pin.     )
;               If any switches are pressed, or if the rotary encoder has been moved,
;               reset the display timeout down counter.
;               If the rotary encoder has been moved, reset the EEPROM write counter.
;
;               EVERY 8.192 mS.
;               Increment the switch pressed counters if the corresponding PIC pin
;               is pulled low, else set the switch counter back to 0.
;               Increment the 8 bit EEPROM write counter. When it wraps back to 0, set
;               the EEPROM write flag. (Indicates when to update the EEPROM, ie 2 seconds
;               after stopping the rotary encoder write any changes to EEPROM)
;               Decrement the 16 bit display counter. when it reaches 0, set the
;               TIMEOUT flag.
;               (If the rotary encoder or buttons have not been changed for about 10)
;               (seconds, the flag indicates that its time to return from the setup )
;               (screens, back to the main display.                                 )
;
;*****************************************************************************************

                ORG     INTVECTOR

;-----------------------------------------------------------------------------------------
;
;               The following is processed every 1.024 mS.
;
;-----------------------------------------------------------------------------------------

                MOVWF   SAVE_W_REG              ; Save W reg.
                SWAPF   STATUS,W
                MOVWF   SAVE_STATUS             ; Save STATUS reg.
                BCF     RP0                     ; Make sure we are addressing bank 0.

;-----------------------------------------------------------------------------------------
;
;               Keep track of the last 4 samples of the rotary encoder. (debouncing)
;
;-----------------------------------------------------------------------------------------

                RLF     ENC_HISTORY,F           ; Make room in the shift register for
                RLF     ENC_HISTORY,F           ; the two new encoder samples.
                MOVLW   B'11111100'
                ANDWF   ENC_HISTORY,F

                MOVLW   B'00000011'
                ANDWF   PORTA,W                 ; Get the current encoder state.
                IORWF   ENC_HISTORY,F           ; Put the new bits into the shift reg.

                MOVLW   B'01010101'
                ANDWF   ENC_HISTORY,W           ; Mask the last 4 encoder A samples.
                BTFSC   ZERO                    ; Were all 4 samples low ?
                BCF     ENC_A_DEBOUNCED         ; Y. Enc A is truly low !

                XORLW   B'01010101'             ; Mask the last 4 encoder A samples.
                BTFSC   ZERO                    ; Were all 4 samples high ?
                BSF     ENC_A_DEBOUNCED         ; Y. Enc A is truly high !

                MOVLW   B'10101010'
                ANDWF   ENC_HISTORY,W           ; Mask the last 4 encoder B samples.
                BTFSC   ZERO                    ; Were all 4 samples low ?
                BCF     ENC_B_DEBOUNCED         ; Y. Enc B is truly low !

                XORLW   B'10101010'             ; Mask the last 4 encoder B samples.
                BTFSC   ZERO                    ; Were all 4 samples high ?
                BSF     ENC_B_DEBOUNCED         ; Y. Enc B is truly high !

;-----------------------------------------------------------------------------------------
;
;               Determine if the rotary encoder has been moved.
;
;-----------------------------------------------------------------------------------------

                BTFSC   OLD_A_DEBOUNCED         ; Was old Enc A low ? (looking for +edge)
                GOTO    DECODE_DONE             ; N. Done.
                BTFSS   ENC_A_DEBOUNCED         ; Is current encoder A high ?
                GOTO    DECODE_DONE             ; N. Done.

                COMF    STEP_RATE,W             ; We have found a + edge.
                ADDLW   1                       ; Load W with -STEP_RATE. (CW rotation)
                BTFSC   ENC_B_DEBOUNCED         ; Is encoder B high ?
                MOVF    STEP_RATE,W             ; N. Load W to +STEP_RATE. (CCW rotation)
                MOVWF   ENC_COUNTER             ; ENC_COUNTER has moved by +/-STEP_RATE.

                MOVLW   38                      ; Set the step rate to 38. (dec every mS)
                MOVWF   STEP_RATE               ; (Max rate = 38 - (2 x 4mS debounce)=30)
                CLRF    EEPROM_WRITE            ; Reset the EEPROM write counter.
      
;-----------------------------------------------------------------------------------------
;               If rotary encoder has moved or CAL or Step size is pressed,
;               reset the display timeout counter.
;-----------------------------------------------------------------------------------------

SET_TIMEOUT:    MOVLW   TIMEOUT_NUM             ; Reset the display timeout down counter.
                MOVWF   TIMEOUT_H
                CLRF    TIMEOUT_L               ; Load timeout with 10 sec.
                GOTO    DEBOUNCE_END

DECODE_DONE:    BTFSC   SWITCH_A                ; Is the step size button,
                BTFSS   SWITCH_B                ; or CAL button pressed ?
                GOTO    SET_TIMEOUT             ; Y. Reset the display timeout counter.

DEBOUNCE_END:   BCF     OLD_A_DEBOUNCED         ; Clear the old debounced encoder bit.
                BTFSC   ENC_A_DEBOUNCED         ; Is the current debounced bit set.
                BSF     OLD_A_DEBOUNCED         ; Y. Save it for next time.

                DECFSZ  STEP_RATE,W             ; Dec the step rate counter,
                MOVWF   STEP_RATE               ; Do not dec below 1.

                DECFSZ  INT_COUNT,F             ; Dec the interrupt counter. Equal to 0 ?
                GOTO    RS_INT_STATUS           ; N. Exit the interrupt service routine.
                BSF     INT_COUNT,3             ; Y. Set the count back to 8. Continue
                                                ;    with the rest of the int routine.

;-----------------------------------------------------------------------------------------
;
;               The following is processed every 8.192 mS.
;
;-----------------------------------------------------------------------------------------

;-----------------------------------------------------------------------------------------
;               Debounce the RPT button. (Switch A & B)
;-----------------------------------------------------------------------------------------

                BCF     SW_A_FLAG               ; Clear the STEP button flag.

       IF REPEATER == TRUE

                INCFSZ  SW_AB_CNT,W             ; Inc the RPT button count.
                MOVWF   SW_AB_CNT               ; But dont inc past 255.
                MOVF    PORTA,W
                ANDLW   B'00001100'
                BTFSC   ZERO                    ; Are switch A & B pins low,
                GOTO    SW_TX                   ; Y. Dont update STEP, CAL & RIT buttons.
                CLRF    SW_AB_CNT               ; N. RPT not pressed, clear the counter.
        ENDIF
;-----------------------------------------------------------------------------------------
;               Check the state of the STEP SIZE button. (Switch A)
;-----------------------------------------------------------------------------------------

                BTFSS   SWITCH_A                ; If STEP button is pressed
                BSF     SW_A_FLAG               ; set a flag.

;-----------------------------------------------------------------------------------------
;               Debounce the RIT button.
;-----------------------------------------------------------------------------------------

                INCFSZ  SW_BC_CNT,W             ; Inc the RIT button count.
                MOVWF   SW_BC_CNT               ; But dont inc past 255.
                MOVF    PORTA,W
                ANDLW   B'00011000'
                BTFSC   ZERO                    ; Are switch B & C pins low,
                GOTO    SW_END                  ; Y. Dont update CAL and TX buttons.
                CLRF    SW_BC_CNT               ; N. Clear the count.

;-----------------------------------------------------------------------------------------
;               Debounce the CAL button.
;-----------------------------------------------------------------------------------------

                INCFSZ  SW_B_CNT,W              ; Inc the CAL button count.
                MOVWF   SW_B_CNT                ; But dont inc past 255.
                BTFSC   SWITCH_B                ; Is the switch B pin low ?
                CLRF    SW_B_CNT                ; N. Clear the count.

;-----------------------------------------------------------------------------------------
;               Debounce the TX button.
;-----------------------------------------------------------------------------------------

SW_TX:          INCFSZ  SW_C_CNT,W              ; Inc the TX button count.
                MOVWF   SW_C_CNT                ; But dont inc past 255.
                BTFSC   SWITCH_C                ; Is the switch C pin low ?
                CLRF    SW_C_CNT                ; N. Clear the count.
SW_END:

;-----------------------------------------------------------------------------------------
;               Increment the EEPROM update counter.
;-----------------------------------------------------------------------------------------

                INCF    EEPROM_WRITE,F          ; Inc the counter.
                BTFSC   ZERO                    ; Count = 0 ?
                BSF     EEPROM_UPDATE           ; Y. Set the flag. (update the EEPROM)

;-----------------------------------------------------------------------------------------
;               Decrement the Display timeout counter.
;-----------------------------------------------------------------------------------------

                DECF    TIMEOUT_L,F             ; Dec the timeout counter.
                BTFSC   ZERO
                DECF    TIMEOUT_H,F             ; Is it = 0 ?
                BTFSC   ZERO
                BSF     TIMEOUT                 ; Y. Set the timeout flag.

;-----------------------------------------------------------------------------------------
;
;               Return from interrupt.
;
;-----------------------------------------------------------------------------------------

RS_INT_STATUS:  SWAPF   SAVE_STATUS,W
                MOVWF   STATUS                  ; Restore STATUS reg.
                SWAPF   SAVE_W_REG,F
                SWAPF   SAVE_W_REG,W            ; Restore W reg without changing STATUS.
                BCF     T0IF                    ; Clear TMR0 interrupt flag.
                RETFIE

;*****************************************************************************************
;               End of interrupt service routine
;*****************************************************************************************


;*****************************************************************************************
;
;       Text table      (RETLW 'X')
;
;               Placed in page 0, so we dont have to bother with page bits.
;
;               Each string terminates with a space with the MSB set, ( ' ' + H'80' )
;               this is displayed as a space. Strings with ( "" ) terminate at the end
;               of the next string.
;
;*****************************************************************************************

TABLE:          ADDWF   PCL,F                   ; Jump to character pointed to in W reg.


;               LABLE           STRING                  TERMINATION
TABLE_START:

 TEXT_STRING    RIT_OFF:  ,     "RIT "                  , ""
 TEXT_STRING    OFFSET:   ,     "OFFSET"                , ' ' + H'80'

        IF REPEATER == TRUE
 TEXT_STRING    RPT:      ,     "RPT"                   , ' ' + H'80'
        ENDIF

 TEXT_STRING    TX:       ,     "TX"                    , ' ' + H'80'
 TEXT_STRING    MIN_RX:   ,     "MIN "                  , ""
 TEXT_STRING    RX:       ,     "RX"                    , ' ' + H'80'
 TEXT_STRING    MAX:      ,     "MAX"                   , ' ' + H'80'
 TEXT_STRING    DDS_FREQ: ,     "DDS "                  , ""
 TEXT_STRING    FREQ:     ,     "FREQ"                  , ' ' + H'80'
 TEXT_STRING    MULTIP:   ,     "MULTIPLIER"            , ' ' + H'80'
 TEXT_STRING    CAL:      ,     "10"                    , ""
 TEXT_STRING    MHZ:      ,     " MHz CAL"              , ' ' + H'80'
 FILE_STRING    VER:      ,     FILE_VERSION            , ' ' + H'80'
TABLE_END:

;*****************************************************************************************
;
;               Initialise the hardware.
;
;*****************************************************************************************

                ORG     RESETVECTOR

                BSF     RP0                     ; Select bank 1 for TRIS reg access.
                MOVLW   B'00000001'             ; Set prescaler to TMR0.
                MOVWF   OPTION_REG              ; TMR0 rate = 1:4 of instruction clock.
                GOTO    CONTINUE                ; Jump over interrupt routines
                                                ; and text table.

;*****************************************************************************************
;
;               The interupt service routines and text table will be placed here.
;
;*****************************************************************************************

                ORG     TABLE_END

CONTINUE:       CLRF    TRISB                   ; RB0..RB7 = Output.
                BCF     RP0                     ; Return to bank 0 for port access.

 IF PIC == 628
                MOVLW   7                       ; Disable 16F628 comparitors,
                MOVWF   CMCON                   ; RA0-4 are digital I/O.
 ENDIF

                MOVLW   B'10100000'             ; Enable TMR0 overflow interrupt.
                MOVWF   INTCON                  ; Sets interrupt rate to 1.024 mS.

                CLRF    PORTB                   ; Set all pins to LCD & DDS low.

                MOVLW   100
                CALL    MS_WAIT                 ; Wait until the LCD has powered up.

                MOVLW   COUNT_2                 ; Point to start of RAM we want to clear.
                MOVWF   FSR
                MOVLW   SAVE_STATUS - COUNT_2   ; Number of bytes of RAM to clear.
                CALL    CLEAR_BYTES             ; Clear the RAM.

                CALL    INIT_LCD                ; Initialise the LCD display.
                CALL    SEND_DDS_WORD           ; Initialise the DDS chip.

                MOVLW   VER
                CALL    LCD_TEXT                ; Display version info.
VER_LOOP:       BTFSS   EEPROM_WRITE,7          ; Wait 1 sec. (until EEPROM_WRITE > 127)
                GOTO    VER_LOOP

                BTFSS   SWITCH_B                ; Is the CAL button pressed.
                GOTO    CAL_10MHZ               ; Y. Display the calibration screens.

;*****************************************************************************************
;
;       Main display screen.
;
;               Get the frequency and step size from EEPROM.
;               Display RX or TX based on the state of the TX lead.
;               Continuously check the rotary encoder for movement, make sure the
;               frequency stays within set limits. If TX is operated, add the TX offset.
;               Then update the DDS chip if required. Add the frequency offset and
;               multiply the result by MULTIPLIER then display the result on the LCD.
;               If the step size button is pressed, allow the user to alter the step size.
;               If the frequency or step size have been changed update the EEPROM
;               2 seconds after the rotary encoder stops moving.
;               If the CAL button is pressed, display the setup screens.
;               If the RIT button is pressed, display the RIT screen.
;
;*****************************************************************************************

MAIN_DISP:      CALL    LCD_CLEAR               ; Clear the display.
                CALL    EE_TO_RAM               ; Get frequency & step size from EEPROM.

MAIN_LOOP:      CALL    DISP_TX_RX              ; Display TX or RX on the LCD.
                CALL    UP_DOWN_RX              ; Inc or dec RX freq & adjust step size.

                BTFSC   EEPROM_UPDATE           ; Is the EEPROM update flag set?
                CALL    RAM_TO_EE               ; Y. Copy freq and step size to EEPROM.

                CALL    CHK_RIT_BRIEF
                BTFSC   PRESSED                 ; RIT button pressed briefly ?
                GOTO    RIT_DISP                ; Y. Goto the RIT display.

 IF REPEATER == TRUE

                BCF     RPT_FLAG                ; Clear the RPT flag.
                CALL    CHK_RPT_BRIEF
                BTFSC   PRESSED                 ; RPT button pressed briefly ?
                GOTO    NEG_RPT_DISP            ; Y. Goto -RPT display.(-600KHz TX Offset)

                BTFSC   SW_AB_CNT,7             ; RPT button pressed for 1 sec ?
                GOTO    POS_RPT_DISP            ; Y. Goto +RPT display.(+600KHz TX Offset)
 ENDIF

                CALL    CHK_TX
                BTFSC   PRESSED                 ; TX button pressed ?
                CALL    ADD_TX_OFFSET           ; Y. Add the TX offset.

                CALL    FREQ_TO_LCD             ; Update DDS, add offset and display it.

                BTFSS   SW_B_CNT,7              ; CAL button pressed for 1 sec ?
                GOTO    MAIN_LOOP               ; N. Continue updating the display.
                CALL    RAM_TO_EE               ; Y. Copy freq and step size to EEPROM.
                GOTO    SETUP_SCRNS             ;    Goto setup screens.

;*****************************************************************************************
;
;       RPT display screen. Repeater TX offset. ( default 600KHz ) 
;
;               Display RX or TX based on the state of the TX lead.
;
;               Display -RPT on the LCD if the RPT button was pressed briefly.
;               (-600 KHz TX offset)
;
;               Display +RPT on the LCD if the RPT button was pressed for 1 sec.
;               (+600 KHz TX offset)
;
;               Continuously check the rotary encoder for movement, make sure the
;               frequency stays within set limits.
;
;               If TX is operated, add the TX offset & the +600KHz or -600 KHz RPT offset.
;               Then update the DDS chip if required. Add the frequency offset and
;               multiply the result by MULTIPLIER then display the result on the LCD.
;               If the step size button is pressed, allow the user to alter the step size.
;               If the frequency or step size have been changed update the EEPROM
;               2 seconds after the rotary encoder stops moving.
;               If the CAL button is pressed, do nothing.
;               If the RIT button is pressed, do nothing.
;               If the RPT button is pressed, return to the main display.
;
;*****************************************************************************************
 IF REPEATER == TRUE

POS_RPT_DISP:   BSF     RPT_FLAG                ; Use a positive 600KHz offset.

NEG_RPT_DISP:   
RPT_LOOP:       MOVLW   H'8C'
                CALL    LCD_CMD                 ; Position the cursor, line 1 pos 13.
                MOVLW   "+"
                BTFSS   RPT_FLAG
                MOVLW   "-"
                CALL    LCD_CHR                 ; Display '+' or '-'.
                MOVLW   RPT
                CALL    DISP_TEXT               ; Display 'RPT'. Then display RX or TX.

                CALL    UP_DOWN_RX              ; Inc or dec RX freq & adjust step size.

                BTFSC   EEPROM_UPDATE           ; Is the EEPROM update flag set?
                CALL    RAM_TO_EE               ; Y. Copy freq and step size to EEPROM.

                CALL    CHK_TX
                BTFSS   PRESSED                 ; TX button pressed ?
                GOTO    DISP_RPT_RX

                MOVLW   EE_RPT_OFFSET           ; Y. Get the offset from EEPROM.
                CALL    EE_TO_ARG3              ;

                BTFSS   RPT_FLAG
                CALL    NEG_ARG3                ;    If RPT flag is clear use -600 KHz.
                CALL    ADD                     ;    Add the 600 KHz offset to RX freq.

                CALL    ADD_TX_OFFSET           ;    Add the TX offset.


                CALL    PUSH_ARG2               ; Save the TX frequency.
                CALL    PULL_ARG3               ; Copy the TX freq to ARG3.
                BTFSC   FREQUENCY,7             ; Is the RX freq sign bit set ?
                CALL    NEG_ARG3                ; Y. Negate the TX freq.
                BTFSC   ARG3_4,7                ; Is the TX freq < 0 ?
                GOTO    MAIN_DISP               ; Y. Dont allow the use of RPT.
                MOVLW   EE_MAX_DDS_FREQ         ; Get the MAX DDS FREQ from EEPROM.
                CALL    EE_TO_ARG2              ; 
                CALL    CMP                     ; Is the TX freq > MAX DDS FREQ ?
                BTFSS   CARRY                   ;
                GOTO    MAIN_DISP               ; Y. Dont allow the use of RPT.
                CALL    PULL_ARG2               ; Recover the TX frequency.


DISP_RPT_RX:    CALL    FREQ_TO_LCD             ; Update DDS, add offset and display it.

                CALL    CHK_RPT_BRIEF
                BTFSS   PRESSED                 ; RPT button pressed briefly ?
                GOTO    RPT_LOOP                ; N. Continue updating the RPT display.
                GOTO    MAIN_DISP               ; Y. Return to the main display.
 ENDIF

;*****************************************************************************************
;
;       RIT display screen.
;
;               Display RIT on the LCD.
;               Display RX or TX based on the state of the TX lead.
;               If TX is operated, add the TX offset to the RX frequency saved in EEPROM.
;               (not the RIT frequency)
;               If TX is not operated, continuously check the rotary encoder for
;               movement, make sure the RIT frequency stays within the RIT limits.
;               (RX frequency +/- RIT limit)
;               Then update the DDS chip if required. Add the frequency offset and
;               multiply the result by MULTIPLIER then display the result on the LCD.
;               If the step size button is pressed, allow the user to alter the step size.
;               Do not save any changes to EEPROM.
;               If the CAL button is pressed, do nothing.
;               If the RIT button is pressed, return to the main display.
;               (the RIT frequency is discarded upon return)
;
;*****************************************************************************************

RIT_DISP:       CALL    RAM_TO_EE               ; Copy frequency and step size to EEPROM.

                CLRF    STEP_NUM                ; Use the minimum step size, 1 Hz.
                BSF     RIT_SCRN                ; Set the RIT screen flag.

RIT_LOOP:       CALL    DISP_RIT                ; Display 'RIT' on line 1 pos 14, Also
                                                ; display TX or RX on the LCD.
                CALL    CHK_TX
                BTFSS   PRESSED                 ; TX button pressed ?
                GOTO    RIT_RX

                MOVLW   EE_RX_DDS_FREQ
                CALL    EE_TO_ARG2              ; Y. Get the RX DDS freq from EEPROM.

                CALL    ADD_TX_OFFSET           ;    Add the TX offset.

                CLRF    ENC_COUNTER             ;    Reset the encoder position.
                GOTO    RIT_DDS_UPDATE          ;    (dont use position info)

RIT_RX:         CALL    UP_DOWN_RX              ; N. Inc, dec RX freq & adjust step size.
                CALL    LIMIT_RIT               ;    Must be within the RIT range.

RIT_DDS_UPDATE: CALL    FREQ_TO_LCD             ; Update DDS, add offset and display it.

                CALL    CHK_RIT_BRIEF
                BTFSS   PRESSED                 ; RIT button pressed briefly ?
                GOTO    RIT_LOOP                ; N. Continue updating the RIT display.
                BCF     RIT_SCRN                ; Y. Clear the RIT screen flag.
                GOTO    MAIN_DISP               ;    Return to the main display.

;*****************************************************************************************
;
;       Calibration screen :- Adjust calibration DDS_CAL to give a DDS O/P of 10 MHz.
;                             (DDS_CAL) = 2^56 / (XTAL_FREQ)
;
;               Range = (EE_CAL_MIN) to (EE_CAL_MAX).
;
;*****************************************************************************************

CAL_10MHZ:      BSF     CAL_SCRN                ; Set the CAL screen flag.
                BSF     SW_B_CNT,7              ; Indicate CAL button pressed a long time.

                CALL    LCD_CLEAR               ; Clear the display.
                MOVLW   H'83'
                CALL    LCD_CMD                 ; Position the cursor, line 1 pos 4.
                MOVLW   CAL
                CALL    LCD_TEXT                ; Display '10 MHz CAL'.

                MOVLW   EE_DDS_CAL              ; Get the variable we want to adjust
                CALL    EE_TO_ARG2              ; from EEPROM.

DDS_CAL:        CALL    UP_DOWN                 ; Chk rotary encoder & adjust step size.
                                                ; (inc or dec ARG2)
                MOVLW   EE_CAL_MIN              ; Get the lower limit from EEPROM.
                CALL    EE_TO_ARG3

                CALL    MAXIMUM                 ; Make sure we dont go below that limit.

                MOVLW   EE_CAL_MAX              ; Get the upper limit from EEPROM.
                CALL    EE_TO_ARG3

                CALL    MIN_DISP_CAL            ; Do the following three calls.
                                                ; CALL MINIMUM       Make sure we dont
                                                ;                    go above that limit.
                                                ; CALL DISPLAY_FREQ  Display the variable.
                                                ; CALL CHK_CAL_BRIEF

;                CALL    MINIMUM                 ; Make sure we dont go above that limit.

;                CALL    DISPLAY_FREQ            ; Display the variable.

                CALL    TEN_MEG                 ; Set ARG3 to 10 MHz.

                CALL    CHK_DDS                 ; Update DDS if there were any changes.

;                CALL    CHK_CAL_BRIEF
                BTFSS   PRESSED                 ; CAL button pressed briefly ?
                GOTO    DDS_CAL                 ; N. Continue updating the display.

                MOVLW   EE_DDS_CAL
                CALL    ARG2_TO_EE              ; Y. Save the CAL constant to EEPROM.
                BCF     CAL_SCRN                ;    Clear the CAL screen flag.

;*****************************************************************************************
;
;       Setup screen :- Maximum DDS frequency. (MAX_DDS_FREQ)
;
;               Range = 0 MHz to (60 or 90 MHz)
;
;*****************************************************************************************

                CALL    LCD_HOME                ; Position the cursor, line 1 pos 1.
                MOVLW   MAX
                CALL    LCD_TEXT                ; Display 'MAX '.
                MOVLW   DDS_FREQ
                CALL    LCD_TEXT                ; Display 'DDS FREQ '.

                MOVLW   EE_MAX_DDS_FREQ         ; Get the variable we want to adjust
                CALL    EE_TO_ARG2              ; from EEPROM.

                BCF     TIMEOUT                 ; Clear the display timeout flag.
MAX_DDS:        BTFSC   TIMEOUT                 ; Has the display timeout been reached ?
                GOTO    MAIN_DISP               ; Y. Return to the main display.

                CALL    UP_DOWN                 ; Chk rotary encoder & adjust step size.
                                                ; (inc or dec ARG2)
                CALL    CLEAR_ARG3              ; lower limit is 0.

                CALL    MAXIMUM                 ; Make sure we dont go below that limit.

                CALL    TEN_MEG                 ; Set ARG3 to 10 MHz. (10^7)
                MOVLW   DDS_MAX
                CALL    ARG3_TIMES_W            ; Set ARG3 to 60 or 90 MHz. (60/90 x 10^6)

                CALL    MIN_DISP_CAL            ; Do the following three calls.
                                                ; CALL MINIMUM       Make sure we dont
                                                ;                    go above that limit.
                                                ; CALL DISPLAY_FREQ  Display the variable.
                                                ; CALL CHK_CAL_BRIEF

                BTFSS   PRESSED                 ; CAL button pressed briefly ?
                GOTO    MAX_DDS                 ; N. Continue updating the display.

                MOVLW   EE_MAX_DDS_FREQ
                CALL    ARG2_TO_EE              ; Y. Save the maximum DDS freq to EEPROM.

;*****************************************************************************************
;
;       Setup screen :- Minimum RX DDS frequency. (MIN_RX_DDS_FREQ)
;
;               Range = -(MAX_DDS_FREQ) to (MAX_DDS_FREQ)
;
;*****************************************************************************************

SETUP_SCRNS:    CALL    LCD_HOME                ; Position the cursor, line 1 pos 1.
                MOVLW   MIN_RX
                CALL    LCD_TEXT                ; Display 'MIN RX '.
                MOVLW   DDS_FREQ
                CALL    LCD_TEXT                ; Display 'DDS FREQ'.

                MOVLW   EE_MIN_RX_DDS_FREQ      ; Get the variable we want to adjust
                CALL    EE_TO_ARG2              ; from EEPROM.

                BCF     TIMEOUT                 ; Clear the display timeout flag.
MIN_RX_DDS:     BTFSC   TIMEOUT                 ; Has the display timeout been reached ?
                GOTO    MAIN_DISP               ; Y. Return to the main display.

                CALL    UP_DOWN                 ; Chk rotary encoder & adjust step size.
                                                ; (inc or dec ARG2)

                MOVLW   EE_MAX_DDS_FREQ         ; Get the upper limit from EEPROM.
                CALL    EE_TO_ARG3

                CALL    MINIMUM                 ; Make sure we dont go above that limit.

                CALL    NEG_ARG3                ; Set the limit to negative.

                CALL    MAXIMUM                 ; Make sure we dont go below that limit.

                CALL    DISP_CAL                ; Do the following two calls.
                                                ; CALL DISPLAY_FREQ  Display the variable.
                                                ; CALL CHK_CAL_BRIEF

                BTFSS   PRESSED                 ; CAL button pressed briefly ?
                GOTO    MIN_RX_DDS              ; N. Continue updating the display.

                MOVLW   EE_MIN_RX_DDS_FREQ
                CALL    ARG2_TO_EE              ; Y. Save the minimum RX freq to EEPROM.

                BCF     NEGATIVE_RX
                BTFSC   ARG2_4,7                ; Was the result negative ?
                BSF     NEGATIVE_RX             ; Y. Save it for future use.

;*****************************************************************************************
;
;       Setup screen :- Maximum RX DDS frequency. (MAX_RX_DDS_FREQ)
;
;               If (MIN_RX_DDS_FREQ) is positive.
;               Range = (MIN_RX_DDS_FREQ) to (MAX_DDS_FREQ)
;
;               If (MIN_RX_DDS_FREQ) is negative,
;               Range = (MIN_RX_DDS_FREQ) to 0
;
;*****************************************************************************************

                CALL    LCD_HOME                ; Position the cursor, line 1 pos 1.
                MOVLW   MAX
                CALL    LCD_TEXT                ; Display 'MAX '.

                MOVLW   EE_MAX_RX_DDS_FREQ      ; Get the variable we want to adjust
                CALL    EE_TO_ARG2              ; from EEPROM.

MAX_RX_DDS:     BTFSC   TIMEOUT                 ; Has the display timeout been reached ?
                GOTO    MAIN_DISP               ; Y. Return to the main display.

                CALL    UP_DOWN                 ; Chk rotary encoder & adjust step size.
                                                ; (inc or dec ARG2)
                MOVLW   EE_MIN_RX_DDS_FREQ      ; Get the lower limit from EEPROM.
                CALL    EE_TO_ARG3

                CALL    MAXIMUM                 ; Make sure we dont go below that limit.

                MOVLW   EE_MAX_DDS_FREQ         ; Prepare the upper limit.
                CALL    EE_TO_ARG3

                BTFSC   NEGATIVE_RX             ; Was the lower RX limit negative ?
                CALL    CLEAR_ARG3              ; Y. Change the upper limit to 0.

                CALL    MIN_DISP_CAL            ; Do the following three calls.
                                                ; CALL MINIMUM       Make sure we dont
                                                ;                    go above that limit.
                                                ; CALL DISPLAY_FREQ  Display the variable.
                                                ; CALL CHK_CAL_BRIEF

                BTFSS   PRESSED                 ; CAL button pressed briefly ?
                GOTO    MAX_RX_DDS              ; N. Continue updating the display.

                MOVLW   EE_MAX_RX_DDS_FREQ
                CALL    ARG2_TO_EE              ; Y. Save the maximum RX freq to EEPROM.

;*****************************************************************************************
;
;       Setup screen :- Maximum RIT offset frequency. (MAX_RIT_OFFSET)
;
;               Range = 0 to (MAX_DDS_FREQ)
;
;*****************************************************************************************

                MOVLW   H'84'                   ; Position the cursor, line 1 pos 5.
                CALL    LCD_CMD
                MOVLW   RIT_OFF
                CALL    LCD_TEXT                ; Display 'RIT OFFSET'.

                MOVLW   EE_MAX_RIT_OFFSET       ; Get the variable we want to adjust
                CALL    EE_TO_ARG2              ; from EEPROM.

RIT_OFFSET:     BTFSC   TIMEOUT                 ; Has the display timeout been reached ?
                GOTO    MAIN_DISP               ; Y. Return to the main display.

                CALL    UP_DOWN                 ; Chk rotary encoder & adjust step size.
                                                ; (inc or dec ARG2)
                CALL    CLEAR_ARG3              ; Lower limit of 0.

                CALL    MAXIMUM                 ; Make sure we dont go below that limit.

                MOVLW   EE_MAX_DDS_FREQ         ; Get the upper limit.
                CALL    EE_TO_ARG3

                CALL    MIN_DISP_CAL            ; Do the following three calls.
                                                ; CALL MINIMUM       Make sure we dont
                                                ;                    go above that limit.
                                                ; CALL DISPLAY_FREQ  Display the variable.
                                                ; CALL CHK_CAL_BRIEF

                BTFSS   PRESSED                 ; CAL button pressed briefly ?
                GOTO    RIT_OFFSET              ; N. Continue updating the display.

                MOVLW   EE_MAX_RIT_OFFSET
                CALL    ARG2_TO_EE              ; Y. Save the maximum RX freq to EEPROM.

;*****************************************************************************************
;
;       Setup screen :- TX offset frequency. (TX_OFFSET_FREQ)
;
;               If (MIN_RX_DDS_FREQ) is positive,
;               Range = 0-(MIN_RX_DDS_FREQ) to (MAX_DDS_FREQ)-(MAX_RX_DDS_FREQ)
;
;               If (MIN_RX_DDS_FREQ) is negative,
;               Range = -(MAX_DDS_FREQ)-(MIN_RX_DDS_FREQ) to 0-(MAX_RX_DDS_FREQ)
;
;*****************************************************************************************

                CALL    LCD_HOME                ; Position the cursor, line 1 pos 1.
                MOVLW   TX
                CALL    LCD_TEXT                ; Display 'TX '.
                MOVLW   OFFSET
                CALL    LCD_TEXT                ; Display 'OFFSET '.
                MOVLW   FREQ
                CALL    LCD_TEXT                ; Display 'FREQ '.

                MOVLW   EE_TX_OFFSET_FREQ       ; Get the variable we want to adjust
                CALL    EE_TO_ARG2              ; from EEPROM.

TX_OFFSET:      BTFSC   TIMEOUT                 ; Has the display timeout been reached ?
                GOTO    MAIN_DISP               ; Y. Return to the main display.

                CALL    UP_DOWN                 ; Chk rotary encoder & adjust step size.
                                                ; (inc or dec ARG2)
                CALL    PUSH_ARG2               ; Save the variable we are adjusting.

                MOVLW   EE_MAX_DDS_FREQ         ; Prepare the lower limit.
                CALL    EE_TO_ARG2

                CALL    NEG_ARG2                ; Make the it negative.

                BTFSS   NEGATIVE_RX             ; Was the lower RX limit negative ?
                CALL    CLEAR_ARG2              ; N. Change the limit to 0.

                MOVLW   EE_MIN_RX_DDS_FREQ      ; Get the lower limit from EEPROM.
                CALL    EE_TO_ARG3

                CALL    SUB                     ; Calculate the lower limit.

                CALL    PULL_ARG3               ; Recover the variable we are adjusting.

                CALL    MAXIMUM                 ; Make sure we dont go below the limit.

                CALL    PUSH_ARG2               ; Save the variable we are adjusting.

                MOVLW   EE_MAX_DDS_FREQ         ; Prepare the upper limit.
                CALL    EE_TO_ARG2

                BTFSC   NEGATIVE_RX             ; Was the lower RX limit negative ?
                CALL    CLEAR_ARG2              ; Y. Change the limit to 0.

                MOVLW   EE_MAX_RX_DDS_FREQ      ; Get the upper limit from EEPROM.
                CALL    EE_TO_ARG3

                CALL    SUB                     ; Calculate the upper limit.

                CALL    PULL_ARG3               ; Recover the variable we are adjusting.

                CALL    MIN_DISP_CAL            ; Do the following three calls.
                                                ; CALL MINIMUM       Make sure we dont
                                                ;                    go above that limit.
                                                ; CALL DISPLAY_FREQ  Display the variable.
                                                ; CALL CHK_CAL_BRIEF

                BTFSS   PRESSED                 ; CAL button pressed briefly ?
                GOTO    TX_OFFSET               ; N. Continue updating the display.

                MOVLW   EE_TX_OFFSET_FREQ
                CALL    ARG2_TO_EE              ; Y. Save the TX offset freq to EEPROM.

;*****************************************************************************************
;
;       Setup screen :- RPT OFFSET frequency.
;
;               Range = 0 MHz to 10 MHz
;
;*****************************************************************************************
 IF REPEATER == TRUE

                CALL    LCD_CLEAR               ; Clear LCD & set cursor, line 1 pos 1.
                MOVLW   RPT
                CALL    LCD_TEXT                ; Display 'RPT '.
                MOVLW   OFFSET
                CALL    LCD_TEXT                ; Display 'OFFSET '.

                MOVLW   EE_RPT_OFFSET           ; Get the variable we want to adjust
                CALL    EE_TO_ARG2              ; from EEPROM.

                BCF     TIMEOUT                 ; Clear the display timeout flag.
RPT_OFF:        BTFSC   TIMEOUT                 ; Has the display timeout been reached ?
                GOTO    MAIN_DISP               ; Y. Return to the main display.

                CALL    UP_DOWN                 ; Chk rotary encoder & adjust step size.
                                                ; (inc or dec ARG2)
                CALL    CLEAR_ARG3              ; lower limit is 0.

                CALL    MAXIMUM                 ; Make sure we dont go below that limit.

                CALL    TEN_MEG                 ; Set ARG3 to 10 MHz.

                CALL    MIN_DISP_CAL            ; Do the following three calls.
                                                ; CALL MINIMUM       Make sure we dont
                                                ;                    go above that limit.
                                                ; CALL DISPLAY_FREQ  Display the variable.
                                                ; CALL CHK_CAL_BRIEF

                BTFSS   PRESSED                 ; CAL button pressed briefly ?
                GOTO    RPT_OFF                 ; N. Continue updating the display.

                MOVLW   EE_RPT_OFFSET
                CALL    ARG2_TO_EE              ; Y. Save the maximum DDS freq to EEPROM.
 ENDIF

;*****************************************************************************************
;
;       Setup screen :- Offset frequency. (OFFSET_FREQ)
;
;               Range = -xx MHz to 20 GHz.
;               (allow negative offset, resulting display must always be positive)
;               Lower limit = -(MINIMUM(MIN_RX_DDS_FREQ+TX_OFFSET_FREQ),(MIN_RX_DDS_FREQ)
;
;*****************************************************************************************

                CALL    LCD_CLEAR               ; Clear LCD & set cursor, line 1 pos 1.
                MOVLW   OFFSET
                CALL    LCD_TEXT                ; Display 'OFFSET '.
                MOVLW   FREQ
                CALL    LCD_TEXT                ; Display 'FREQ '.

                MOVLW   EE_OFFSET_FREQ          ; Get the variable we want to adjust
                CALL    EE_TO_ARG2              ; from EEPROM.

OFFSET_FREQU:   BTFSC   TIMEOUT                 ; Has the display timeout been reached ?
                GOTO    MAIN_DISP               ; Y. Return to the main display.

                CALL    UP_DOWN                 ; Chk rotary encoder & adjust step size.
                                                ; (inc or dec ARG2)
                CALL    PUSH_ARG2               ; Save the variable we are adjusting.

                MOVLW   EE_TX_OFFSET_FREQ       ; Prepare the lower limit.
                CALL    EE_TO_ARG2

                MOVLW   EE_MIN_RX_DDS_FREQ      ; Prepare the lower limit.
                CALL    EE_TO_ARG3

                CALL    ADD                     ; Calculate the lower limit.
                CALL    MINIMUM
                CALL    NEG_ARG2

                CALL    PULL_ARG3               ; Recover the variable we are adjusting.

                CALL    MAXIMUM                 ; Make sure we dont go below the limit.

                MOVLW   10                      ; Calculate the lower limit.
                CALL    TEN_TO_POWER_W          ; Set ARG3 to 10 GHz. (10 x 10^9)
                MOVLW   2
                CALL    ARG3_TIMES_W            ; Set ARG3 to 20 GHz. (20 x 10^9)

                CALL    MIN_DISP_CAL            ; Do the following three calls.
                                                ; CALL MINIMUM       Make sure we dont
                                                ;                    go above that limit.
                                                ; CALL DISPLAY_FREQ  Display the variable.
                                                ; CALL CHK_CAL_BRIEF

                BTFSS   PRESSED                 ; CAL button pressed briefly ?
                GOTO    OFFSET_FREQU            ; N. Continue updating the display.

                MOVLW   EE_OFFSET_FREQ
                CALL    ARG2_TO_EE              ; Y. Save the offset freq to EEPROM.

;*****************************************************************************************
;
;       Setup screen :- Frequency display multiplier. (DISPLAY_MULT)
;
;               Range = 1 to 32 (as long as any display on LCD is not to far above 68 GHz)
;                       note that the 68 GHz check is not included if the REPEATER TX
;                       OFFSET function is included.
;
;*****************************************************************************************

                CLRF    STEP_NUM                ; Use the minimum step size. (1)

 IF PIC == 628 || REPEATER == FALSE

                CLRF    COUNT_4                 ; COUNT_4 will hold the upper limit.

                CALL    CLEAR_ARG2

                MOVLW   EE_OFFSET_FREQ          ; Get the offset freq from EEPROM.
                CALL    EE_TO_ARG3
                BTFSC   ARG3_4,7                ; Is the offset negative ?
                CALL    NEG_ARG3                ; Y. Make it positive.


MULT_DISP_LOOP: INCF    COUNT_4,F
                BTFSC   COUNT_4,5               ; Is the upper limit = 32 ?
                GOTO    GOT_LIMIT               ; Y. Done.
                CALL    ADD                     ; ARG2 = ARG2 + offset freq.
                BTFSS   ARG2_4,4                ; Is the result > approx 68 GHz.
                GOTO    MULT_DISP_LOOP          ; Y. Done.

 ENDIF

                                                ; COUNT_4 = upper limit.
GOT_LIMIT:      CALL    LCD_CLEAR               ; Clear LCD & set cursor, line 1 pos 1.
                MOVLW   MULTIP
                CALL    LCD_TEXT                ; Display 'MULTIPLIER'.

                CALL    CLEAR_ARG2              ; Move the mult byte from EEPROM to ARG2.
                CALL    READ_MULT
                MOVWF   ARG2_0

MULT_DISP:      BTFSC   TIMEOUT                 ; Has the display timeout been reached ?
                GOTO    MAIN_DISP               ; Y. Return to the main display.

                CALL    UP_DOWN_A               ; Chk rotary encoder. (inc or dec ARG2)

                CALL    CLEAR_ARG3
                INCF    ARG3_0,F                ; Lower limit is 1.

                CALL    MAXIMUM                 ; Make sure we dont go below that limit.

 IF PIC == 628 || REPEATER == FALSE

                MOVF    COUNT_4,W
 ELSE
                MOVLW   32
 ENDIF
                MOVWF   ARG3_0                  ; Get the upper limit.

                CALL    MINIMUM                 ; Make sure we dont go above that limit.

                CALL    BIN_TO_DEC              ; Convert the number to ASCII decimal.

                MOVLW   H'8E'                   ; Position the cursor, line 1 pos 15.
                CALL    LCD_CMD

                MOVF    ARG1_1,W                ; Display the number in ASCII form.
                CALL    LCD_CHR                 ; 10's digit.
                MOVF    ARG1_0,W
                CALL    LCD_CHR                 ; 1's digit.

                CALL    CHK_CAL_BRIEF
                BTFSS   PRESSED                 ; CAL button pressed briefly ?
                GOTO    MULT_DISP               ; N. Continue updating the display.
                CALL    WRITE_MULT              ; Y. Save the multiplier to EEPROM.
                GOTO    MAIN_DISP               ;    Return to the main display.

;*****************************************************************************************
;               Start of subroutines
;*****************************************************************************************


;*****************************************************************************************
;
;      NAME:    ADD_TX_OFFSET
;
;   PURPOSE:    Add the TX offset (from EEPROM) to ARG2.
;
;     INPUT:    ARG2 = frequency.
;
;    OUTPUT:    ARG2 = frequency + TX offset.
;               (ARG3 destroyed)
;
;*****************************************************************************************

ADD_TX_OFFSET:  MOVLW   EE_TX_OFFSET_FREQ       ; Get the TX offset.
                CALL    EE_TO_ARG3              ; Fall through to ADD,
                                                ; and return via ADD.

;*****************************************************************************************
;
;     NAMES:    ADD                     ARG2 = ARG2 + ARG3.     (all routines are 40 bit)
;               SUB                     ARG2 = ARG2 - ARG3.
;               NEG_ARG2                ARG2 = -ARG2.
;               NEG_ARG3                ARG3 = -ARG3.
;               CLEAR_ARG1              ARG1 = 0.
;               CLEAR_ARG2              ARG2 = 0.
;               CLEAR_ARG3              ARG3 = 0.
;
;*****************************************************************************************

ADD:            MOVLW   ARG1_9                  ; ARG1 = ARG3.
                CALL    COPY_FROM_ARG3
ADDER:          CALL    ADD_ARG1_ARG2           ; ARG1 = ARG1 + ARG2 = ARG3 + ARG2.
ARG1_TO_ARG2:   MOVLW   ARG1_9                  ;
                GOTO    COPY_TO_ARG2            ; ARG2 = ARG2 + ARG3.

SUB:            CALL    NEG_ARG3_ARG1           ; ARG1 = -ARG3.
                GOTO    ADDER

NEG_ARG3:       CALL    NEG_ARG3_ARG1
ARG1_TO_ARG3:   MOVLW   ARG1_9
                GOTO    COPY_TO_ARG3            ; ARG3 = -ARG3.

NEG_ARG2:       MOVLW   ARG1_9                  ; COPY ARG2 TO ARG1.
                CALL    COPY_FROM_ARG2
                CALL    NEG_ARG1
                GOTO    ARG1_TO_ARG2            ; ARG2 = -ARG2.

NEG_ARG3_ARG1:  MOVLW   ARG1_9                  ; ARG1 = ARG3.
                CALL    COPY_FROM_ARG3
NEG_ARG1:       COMF    ARG1_9,F                ;   (complement ARG1)
                COMF    ARG1_8,F
                COMF    ARG1_7,F
                COMF    ARG1_6,F
                COMF    ARG1_5,F
                INCFSZ  ARG1_5,F                ;   (add 1 to ARG1)
                RETURN
                INCFSZ  ARG1_6,F
                RETURN
                INCFSZ  ARG1_7,F
                RETURN
                INCFSZ  ARG1_8,F
                RETURN
                INCF    ARG1_9,F
                RETURN                          ; ARG1 = -ARG3.


ADD_ARG1_ARG2:  MOVF    ARG2_0,W                ; ARG1_9..5 = ARG1_9..5 + ARG2_4..0
                ADDWF   ARG1_5,F                ; Add the 1st (LSD) bytes.
                MOVF    ARG2_1,W
                BTFSC   CARRY
                INCFSZ  ARG2_1,W                ; If there was a carry, inc the next byte.

                ADDWF   ARG1_6,F                ; Add the 2nd bytes.
                MOVF    ARG2_2,W
                BTFSC   CARRY
                INCFSZ  ARG2_2,W                ; If there was a carry, inc the next byte.

                ADDWF   ARG1_7,F                ; Add the 3rd bytes.
                MOVF    ARG2_3,W
                BTFSC   CARRY
                INCFSZ  ARG2_3,W                ; If there was a carry, inc the next byte.

                ADDWF   ARG1_8,F                ; Add the 4th bytes.
                MOVF    ARG2_4,W
                BTFSC   CARRY
                INCFSZ  ARG2_4,W                ; If there was a carry, inc the next byte.

                ADDWF   ARG1_9,F                ; Add the 5th bytes.
                RETURN



CLEAR_ARG1:     CALL    SET_ARG1                ; Set FSR = ARG1_10, COUNT_1 = 11.
                GOTO    CLEAR_LOOP

CLEAR_ARG2:     MOVLW   ARG2_4
                GOTO    CLEAR

CLEAR_ARG3:     MOVLW   ARG3_4
CLEAR:          MOVWF   FSR
                MOVLW   5
CLEAR_BYTES:    MOVWF   COUNT_1                 ; Save the count.
CLEAR_LOOP:     CLRF    INDF                    ; Clear the RAM byte.
                INCF    FSR,F                   ; Inc the RAM address pointer.
                DECFSZ  COUNT_1,F               ; Dec the count.
                GOTO    CLEAR_LOOP              ; Loop until all bytes cleared.
                RETURN

SET_ARG1:       MOVLW   ARG1_10                 ; Set FSR = ARG1_10, COUNT_1 = 11.
                MOVWF   FSR
                MOVLW   11
                MOVWF   COUNT_1
                RETURN

;*****************************************************************************************
;
;      NAME:    CMP
;
;   PURPOSE:    40 bit compare.  ARG2 - ARG3
;
;     INPUT:    40 bit argment ARG2.
;               40 bit argment ARG3.
;
;    OUTPUT:    None.
;               Carry flag clear (borrow) if result was negative.
;               Zero flag set if equal.
;               (ARG2 & ARG3 unchanged)
;
;*****************************************************************************************

CMP:            CALL    OFFSET_ARGS             ; Add a large offset to ARG2 & ARG3.
                                                ; (makes sure we are dealing )
                                                ; (with positive numbers only)

                CALL    NEG_ARG3_ARG1           ; ARG1 = -ARG3
                CALL    ADD_ARG1_ARG2           ; ARG1 = ARG2 - ARG3

OFFSET_ARGS:    MOVLW   B'10000000'             ; Remove the offset from ARG2 and ARG3.
                XORWF   ARG2_4,F
                XORWF   ARG3_4,F
                MOVF    ARG1_9,W                ; See if the result was 0.
                IORWF   ARG1_8,W                ;
                IORWF   ARG1_7,W                ;
                IORWF   ARG1_6,W                ;
                IORWF   ARG1_5,W                ;
                RETURN                          ; (sets zero flag if result was 0)

;*****************************************************************************************
;
;      NAME:    MINIMUM
;
;   PURPOSE:    40 bit minimum. Return the smallest of two argments.
;
;       INPUT:  40 bit argment in ARG2.
;               40 bit argment in ARG3.
;
;       OUTPUT: The smallest argment is returned in ARG2.
;               (ARG3 unchanged)
;
;*****************************************************************************************

MINIMUM:        CALL    CMP
                BTFSC   CARRY                   ; Is ARG2 > ARG3 ?
                CALL    ARG3_TO_ARG2            ; Y. Set ARG2 the same as ARG3.
                RETURN                          ; N. No change required.

;*****************************************************************************************
;
;      NAME:    MAXIMUM
;
;   PURPOSE:    40 bit maximum. Return the largest of two argments.
;
;       INPUT:  40 bit argment in ARG2.
;               40 bit argment in ARG3.
;
;       OUTPUT: The largest argment is returned in ARG2.
;               (ARG3 unchanged)
;
;*****************************************************************************************

MAXIMUM:        CALL    CMP
                BTFSS   CARRY                   ; Is ARG2 > ARG3 ?
                CALL    ARG3_TO_ARG2            ; N. Set ARG2 the same as ARG3.
                RETURN                          ; Y. No change required.

;*****************************************************************************************
;
;      NAME:    UP_DOWN_RX
;
;   PURPOSE:    If the rotary encoder has been moved, increase or decrease the
;               FREQUENCY variable by step size. Make sure it stays within limit.
;               Also see if the user wants to change the step size.
;
;     INPUT:    None. (freq in the FREQUENCY variable)
;
;    OUTPUT:    ARG2 = frequency +- N. (result also placed back in FREQUENCY variable)
;
;*****************************************************************************************

UP_DOWN_RX:     MOVLW   FREQUENCY
                CALL    COPY_TO_ARG2            ; Get the FREQUENCY variable.

                CALL    UP_DOWN                 ; Chk rotary encoder & adjust step size.
                                                ; (inc or dec ARG2)
                MOVLW   EE_MIN_RX_DDS_FREQ
                CALL    EE_TO_ARG3              ; Get the lower limit.

                CALL    MAXIMUM                 ; Make sure we dont go below that limit.

                MOVLW   EE_MAX_RX_DDS_FREQ
                CALL    EE_TO_ARG3              ; Get the upper limit.

MIN_TO_FREQ:    CALL    MINIMUM                 ; Make sure we dont go above that limit.

                MOVLW   FREQUENCY               ; Save result to the FREQUENCY variable.
                GOTO    COPY_FROM_ARG2          ; Return via "GOTO".

;*****************************************************************************************
;
;      NAME:    LIMIT_RIT
;
;   PURPOSE:    When in the RIT display, make sure the RIT frequency (FREQUENCY variable)
;               stays within the limit set by EE_MAX_RIT_OFFSET.
;
;     INPUT:    None. (freq in the FREQUENCY variable)
;
;    OUTPUT:    ARG2 = frequency. (result also placed back in FREQUENCY variable)
;
;*****************************************************************************************

LIMIT_RIT:      MOVLW   EE_RX_DDS_FREQ
                CALL    EE_TO_ARG2              ; Prepare the upper limit.

                MOVLW   EE_MAX_RIT_OFFSET
                CALL    EE_TO_ARG3              ; Prepare the upper limit.

                CALL    ADD                     ; ARG2 = RX DDS freq + max RIT offset.

                CALL    PUSH_ARG2               ; Save the result. (upper limit)

                CALL    SUB                     ; ARG2 = RX DDS freq
                CALL    SUB                     ; ARG2 = RX DDS freq - max RIT offset.
                                                ;      = (lower limit)
                MOVLW   FREQUENCY               ;
                CALL    COPY_TO_ARG3            ; Get the FREQUENCY variable.

                CALL    MAXIMUM                 ; Make sure we dont go below the limit.

                CALL    PULL_ARG3               ; Get the upper limit.

                                                ; Make sure we dont go above that limit.
                GOTO    MIN_TO_FREQ             ; Save result to the FREQUENCY variable.
                                                ; Return via "GOTO".

;*****************************************************************************************
;
;      NAME:    UP_DOWN
;
;   PURPOSE:    If the rotary encoder has been moved, inc or dec the variable in ARG2.
;               Also see if the user wants to change the step size.
;
;     INPUT:    ARG2.
;
;    OUTPUT:    ARG2 = ARG2 +- N.
;
;*****************************************************************************************

UP_DOWN:        CALL    CHK_STEP_SIZE           ; Does the user want to change step size.
UP_DOWN_A:      MOVF    ENC_COUNTER,W           ; Any change to the rotary encoder ?
                BTFSC   ZERO                    ; N. Just return.
                RETURN

                MOVWF   COUNT_2                 ; Save the encoder count.
                CLRF    ENC_COUNTER             ; Reset the encoder position count.

                MOVF    STEP_NUM,W              ; Get the step number.
                CALL    TEN_TO_POWER_W          ; Convert to step size, place it in ARG3.

                BTFSS   COUNT_2,7               ; Is the encoder count negative ?
                GOTO    STEP_UP

                CALL    NEG_ARG3                ; Y. Make step size negative.
                COMF    COUNT_2,F               ;    Make encoder count positive.
                INCF    COUNT_2,F

STEP_UP:        CALL    ADD                     ; Inc ARG2 by the step size.
                DECFSZ  COUNT_2,F               ; Dec the loop count.
                GOTO    STEP_UP                 ; Inc ARG2 until count = 0.
                RETURN

;*****************************************************************************************
;
;      NAME:    TEN_TO_POWER_W
;                            (W)
;   PURPOSE:    Calculate 10^
;
;     INPUT:    W = Power.
;                         (W)
;    OUTPUT:    ARG3 = 10^
;
;*****************************************************************************************

TEN_MEG:        MOVLW   7
TEN_TO_POWER_W: ADDLW   1
                MOVWF   COUNT_3                 ; Save the power + 1.
                CALL    CLEAR_ARG3
                INCF    ARG3_0,F                ; Set ARG3 to 1.
                GOTO    END_TEST
POWER_LOOP:     MOVLW   10
                CALL    ARG3_TIMES_W
END_TEST:       DECFSZ  COUNT_3,F               ; Keep mult by ten until counter = 0.
                GOTO    POWER_LOOP
                RETURN

;*****************************************************************************************
;
;      NAME:    ARG3_TIMES_W
;
;   PURPOSE:    Calculate ARG3 x W
;
;     INPUT:    W = Multiplier.
;
;    OUTPUT:    ARG3 = ARG3 x W.
;
;*****************************************************************************************

ARG3_TIMES_W:   MOVWF   COUNT_0                 ; Save multiplier.
                CALL    PUSH_ARG2               ; Save ARG2.
                CALL    CLEAR_ARG2              ;
                MOVF    COUNT_0,W               ; Get multiplier.
                MOVWF   ARG2_0                  ; Place it in ARG2.
                CALL    MULTIPLY                ; Do the multiply.
                CALL    PULL_ARG2               ; Recover ARG2.
                MOVLW   ARG1_4                  ; Return the result in ARG3.
                GOTO    COPY_TO_ARG3            ; Return via "GOTO".

;*****************************************************************************************
;
;      NAME:    CHK_STEP_SIZE
;
;   PURPOSE:    If the user wants to change the step size, enable the cursor on the
;               frequency display, and set the STEP_NUM variable.
;
;     INPUT:    None.
;
;    OUTPUT:    None.
;
;*****************************************************************************************

CHK_STEP_SIZE:  BTFSS   SW_A_FLAG               ; Adjust step size button pressed ?
                RETURN                          ; N. All done.
                MOVLW   B'00001110'             ; Turn on the cursor.
                CALL    LCD_CMD
CHK_STEP_LOOP:  MOVLW   6
                BTFSC   CAL_SCRN                ;  (if we are in the CAL 10 MHz screen)
                MOVLW   8                       ;  (change the cursor position.       )
                SUBWF   STEP_NUM,W              ; Is cursor before the decimal point ?
                BTFSS   CAL_SCRN                ;  (if we are in the CAL 10 MHz screen)
                BTFSC   CARRY                   ;  (ignore the decimal point.         )
                ADDLW   1                       ; Y. Shift cursor left 1 digit.
                SUBLW   H'C5'                   ; Subtract it from the 1Hz char pos.
                CALL    LCD_CMD                 ; Position the cursor on freq display.

                MOVF    ENC_COUNTER,W           ; Get rotary encoder position.
                CLRF    ENC_COUNTER             ; Reset the encoder position.
                SUBWF   STEP_NUM,F              ; Take it from the step position.
                BTFSC   STEP_NUM,7              ; Is the result < 0 ?
                CLRF    STEP_NUM                ; Y. Limit it to 0.
                MOVLW   7
                SUBWF   STEP_NUM,W
                MOVLW   7
                BTFSC   CARRY                   ; Was the result > 7 ?
                MOVWF   STEP_NUM                ; Y. Limit it to 7.

                BTFSC   SW_A_FLAG               ; Adjust step size button still pressed ?
                GOTO    CHK_STEP_LOOP           ; Y. Keep checking rotary encoder.
                GOTO    LCD_NORM                ; N. Turn off the cursor.
                                                ;    Return via "GOTO".

;*****************************************************************************************
;
;     NAMES:    COPY_TO_ARG2            Copy 5 bytes pointed to by W to ARG2.
;               COPY_TO_ARG3            Copy 5 bytes pointed to by W to ARG3.
;               COPY_FROM_ARG2          Copy ARG2 to the address pointed to by W.
;               COPY_FROM_ARG3          Copy ARG3 to the address pointed to by W.
;               ARG2_TO_ARG3            Copy ARG2 to ARG3.
;               ARG3_TO_ARG2            Copy ARG3 to ARG2.
;               PUSH_ARG2               Copy ARG2 to TEMP ARG.
;               PUSH_ARG3               Copy ARG3 to TEMP ARG.
;               PULL_ARG2               Copy TEMP ARG to ARG2.
;               PULL_ARG3               copy TEMP ARG to ARG3.
;
;   PURPOSE:    Move 5 bytes of RAM from source address to dest address.
;               (starting at source and dest, then source+1 dest+1, etc.)
;
;               RAM usage       TEMP_1 = destination address.
;                               TEMP_2 = source address.
;                               TEMP_3 = byte to move.
;
;*****************************************************************************************

PUSH_ARG2:      MOVLW   ARG4_4                  ; \
COPY_FROM_ARG2: MOVWF   TEMP_1                  ;  \
                MOVLW   ARG2_4                  ;   \
                GOTO    COPY_B                  ;    |
                                                ;    |
PUSH_ARG3:      MOVLW   ARG4_4                  ;    |
COPY_FROM_ARG3: MOVWF   TEMP_1                  ;    |
                MOVLW   ARG3_4                  ;    |
COPY_B:         MOVWF   TEMP_2                  ;    |
                GOTO    COPY_A                  ;    |
                                                ;    |
ARG3_TO_ARG2:   MOVLW   ARG3_4                  ;    |
                GOTO    COPY_TO_ARG2            ;     \  set the various source and
                                                ;     /  destination addresses.
PULL_ARG2:      MOVLW   ARG4_4                  ;    |
COPY_TO_ARG2:   MOVWF   TEMP_2                  ;    |
                MOVLW   ARG2_4                  ;    |
                GOTO    COPY                    ;    |
                                                ;    |
ARG2_TO_ARG3:   MOVLW   ARG2_4                  ;    |
                GOTO    COPY_TO_ARG3            ;    |
                                                ;    |
PULL_ARG3:      MOVLW   ARG4_4                  ;   /
COPY_TO_ARG3:   MOVWF   TEMP_2                  ;  /
                MOVLW   ARG3_4                  ; /

COPY:           MOVWF   TEMP_1                  ;
COPY_A:         MOVLW   5                       ; 5 bytes to move.
                MOVWF   COUNT_1
COPY_LOOP:      MOVF    TEMP_2,W                ; Get source address.
                MOVWF   FSR                     ;        "
                MOVF    INDF,W                  ; Get the byte from the source address.
                MOVWF   TEMP_3                  ;                "
                MOVF    TEMP_1,W                ; Get destination address.
                MOVWF   FSR                     ;          "
                MOVF    TEMP_3,W                ; Place the byte at destination address.
                MOVWF   INDF                    ;                   "
                INCF    TEMP_1,F                ; Inc dest address.
                INCF    TEMP_2,F                ; Inc source address.
                DECFSZ  COUNT_1,F
                GOTO    COPY_LOOP               ; Loop until all bytes copied.
                RETURN

;*****************************************************************************************
;
;     NAMES:    EE_TO_ARG2              Copy 5 bytes in EEPROM pointed to by W to ARG2.
;               EE_TO_ARG3              Copy 5 bytes in EEPROM pointed to by W to ARG3.
;               EE_TO_RAM               Copy FREQUENCY (5 bytes) and STEP_SIZE (1 BYTE)
;                                       from EEPROM to RAM.
;               RAM_TO_EE               Copy FREQUENCY (5 bytes) and STEP_SIZE (1 BYTE)
;                                       from RAM to EEPROM.
;               ARG2_TO_EE              Copy ARG2 to the EEPROM address pointed to by W.
;
;                                       All routines use a common routine (EEPROM_MOVE),
;                                       those routines that write to EEPROM set a
;                                       flag (WRITE_FLAG) before calling (EEPROM_MOVE).
;
;*****************************************************************************************

RAM_TO_EE:      BSF     WRITE_FLAG
EE_TO_RAM:      MOVLW   EE_RX_DDS_FREQ          ; The address in EEPROM is EE_RX_DDS_FREQ.
 IF PIC == 84
                MOVWF   EEADR                   ; Save the EEPROM address.
 ELSE
                BSF     RP0                     ; Select bank 1 for EEPROM reg access.
                MOVWF   EEADR                   ; Save the EEPROM address.
                BCF     RP0                     ; Return to bank 0 for port access.
 ENDIF
                MOVLW   FREQUENCY               ; The address in RAM is FREQUENCY.
                MOVWF   FSR
                MOVLW   6                       ; 6 bytes to move.
                GOTO    EEPROM_MOVE

;ARG3_TO_EE:     BSF     WRITE_FLAG
EE_TO_ARG3:
 IF PIC == 84
                MOVWF   EEADR                   ; Save the EEPROM address.
 ELSE
                BSF     RP0                     ; Select bank 1 for EEPROM reg access.
                MOVWF   EEADR                   ; Save the EEPROM address.
                BCF     RP0                     ; Return to bank 0 for port access.
 ENDIF
                MOVLW   ARG3_4                  ; The address in RAM is ARG3.
                GOTO    EEPROM_A

ARG2_TO_EE:     BSF     WRITE_FLAG
EE_TO_ARG2:
 IF PIC == 84
                MOVWF   EEADR                   ; Save the EEPROM address.
 ELSE
                BSF     RP0                     ; Select bank 1 for EEPROM reg access.
                MOVWF   EEADR                   ; Save the EEPROM address.
                BCF     RP0                     ; Return to bank 0 for port access.
 ENDIF
                MOVLW   ARG2_4                  ; The address in RAM is ARG2.
EEPROM_A:       MOVWF   FSR
                MOVLW   5                       ; 5 bytes to move.

EEPROM_MOVE:    MOVWF   COUNT_1                 ; Save the number of bytes to move.
EEPROM_LOOP:    BTFSC   WRITE_FLAG              ; Do we want to write ? (WRITE_FLAG = set)
                CALL    WRITE_EEPROM            ; Y. Write a byte from RAM to EEPROM.
                CALL    READ_EEPROM             ; Get a byte from EEPROM.
                MOVWF   INDF                    ; Place it in RAM.
 IF PIC == 84
                INCF    EEADR,F                 ; Prepare to read the next byte.
 ELSE
                BSF     RP0                     ; Select bank 1 for EEPROM reg access.
                INCF    EEADR,F                 ; Prepare to read the next byte.
                BCF     RP0                     ; Return to bank 0 for port access.
 ENDIF
                INCF    FSR,F
                DECFSZ  COUNT_1,F
                GOTO    EEPROM_LOOP             ; Loop until all bytes moved.
                BCF     WRITE_FLAG
                RETURN

;*****************************************************************************************
;
;      NAME:    WRITE_EEPROM
;
;   PURPOSE:    Write the byte of RAM pointed to by FSR,
;               to the EEPROM address pointed to by EEADR.
;               If the RAM and EEPROM data are the same, the EEPROM is not reprogrammed.
;               Assumes the source address FSR has been set,
;               and the destination address EEADR has been set.
;
;     INPUT:    None.
;
;    OUTPUT:    None.
;
;*****************************************************************************************

WRITE_MULT:     MOVLW   EE_DISPLAY_MULT         ; Dest address is EE_DISPLAY_MULT.
 IF PIC == 84
                MOVWF   EEADR                   ; Save the dest address.
 ELSE
                BSF     RP0                     ; Select bank 1 for EEPROM reg access.
                MOVWF   EEADR                   ; Save the dest address.
                BCF     RP0                     ; Return to bank 0 for port access.
 ENDIF
                MOVLW   ARG2_0                  ; Source address is ARG2.
                MOVWF   FSR                     ; Save it in the indirect register.

WRITE_EEPROM:   CALL    READ_EEPROM             ; Get the current byte from EEPROM.
                SUBWF   INDF,W
                BTFSC   ZERO                    ; Is it the same as it is in RAM ?
                GOTO    NO_CHANGE               ; Y. No need to update this byte.
                MOVF    INDF,W                  ; N. Get the data byte from RAM.
 IF PIC == 84
                MOVWF   EEDATA                  ;    Put it in the buffer.
                BSF     RP0                     ;    Select bank 1 for EEPROM reg access.
                BCF     EEIF                    ;    Clear the EEPROM write done flag.
 ELSE
                BCF     EEIF                    ;    Clear the EEPROM write done flag.
                BSF     RP0                     ;    Select bank 1 for EEPROM reg access.
                MOVWF   EEDATA                  ;    Put it in the buffer.
 ENDIF
                BSF     WREN                    ;    Enable EEPROM write.
                BCF     INTCON,GIE              ;    Disable interupts.
                MOVLW   H'55'                   ;    Perform required safety steps.
                MOVWF   EECON2
                MOVLW   H'AA'
                MOVWF   EECON2
                BSF     WR                      ;    Begin the write.
                BSF     INTCON,GIE              ;    Enable interupts.
 IF PIC == 628
                BCF     RP0                     ;    Return to bank 0 for port access.
 ENDIF
WRITE_LOOP:     BTFSS   EEIF                    ;    Loop until the write is completed.
                GOTO    WRITE_LOOP
 IF PIC == 84
                BCF     RP0                     ;    Return to bank 0 for port access.
 ENDIF
NO_CHANGE:      BCF     EEPROM_UPDATE           ; Clear the update required flag.
                RETURN

;*****************************************************************************************
;
;      NAME:    READ_EEPROM
;
;   PURPOSE:    Read the byte of EEPROM data pointed to by EEADR.
;               Assumes the read address EEADR has been set.
;
;     INPUT:    None.
;
;    OUTPUT:    W = The byte read from EEPROM.
;
;*****************************************************************************************

READ_MULT:      MOVLW   EE_DISPLAY_MULT         ; Source address is EE_DISPLAY_MULT.
 IF PIC == 84
                MOVWF   EEADR                   ; Save the source address.
 ELSE
                BSF     RP0                     ; Select bank 1 for EEPROM reg access.
                MOVWF   EEADR                   ; Save the source address.
                BCF     RP0                     ; Return to bank 0 for port access.
 ENDIF

READ_EEPROM:    BSF     RP0                     ; Select bank 1 for EEPROM reg access.
                BSF     RD                      ; Perform an EEPROM read.
 IF PIC == 84
                BCF     RP0                     ; Return to bank 0 for port access.
                MOVF    EEDATA,W                ; Get the read byte.
 ELSE
                MOVF    EEDATA,W                ; Get the read byte.
                BCF     RP0                     ; Return to bank 0 for port access.
 ENDIF
                RETURN

;*****************************************************************************************
;
;     NAMES:    CHK_CAL_BRIEF   Return a true flag in PRESSED, if the
;               CHK_RIT_BRIEF   button was pressed briefly.
;               CHK_RPT_BRIEF
;
;               CHK_TX          Return a true flag in PRESSED, if the
;                               button has been pressed for a while.
;
;     INPUT:    None.
;
;    OUTPUT:    None. (PRESSSED flag set if button pressed)
;
;*****************************************************************************************

CHK_TX          MOVLW   MIN_ON                  ; Use the minimum on time limit.
                SUBWF   SW_C_CNT,W              ; Has the button been pressed a while ?
                GOTO    RETURN_FLAG

 IF REPEATER == TRUE

CHK_RPT_BRIEF:  MOVF    SW_AB_OLD,W
                MOVWF   TEMP_1                  ; Save the old button state.
                SUBWF   SW_AB_CNT,W             ; Has the button been released ?
                MOVF    SW_AB_CNT,W             ;    (current - old)
                MOVWF   SW_AB_OLD               ;    (copy current state to old state )
                GOTO    CHECK_TIMING
 ENDIF

CHK_RIT_BRIEF:  MOVF    SW_BC_OLD,W
                MOVWF   TEMP_1                  ; Save the old button state.
                SUBWF   SW_BC_CNT,W             ; Has the button been released ?
                MOVF    SW_BC_CNT,W             ;    (current - old)
                MOVWF   SW_BC_OLD               ;    (copy current state to old state )
                GOTO    CHECK_TIMING

MIN_DISP_CAL:   CALL    MINIMUM                 ; Make sure we dont go above that limit.
DISP_CAL:       CALL    DISPLAY_FREQ            ; Display the variable.
                                                ; Fall through and check the CAL button.
CHK_CAL_BRIEF:  MOVF    SW_B_OLD,W
                MOVWF   TEMP_1                  ; Save the old button state.
                SUBWF   SW_B_CNT,W              ; Has the button been released ?
                MOVF    SW_B_CNT,W              ;    (current - old)
                MOVWF   SW_B_OLD                ;    (copy current state to old state )
CHECK_TIMING:   BTFSC   CARRY
                GOTO    FALSE_FLAG              ; N. Return a false flag.
                MOVF    TEMP_1,W                ; Y. Was the button on less than min ?
                SUBLW   MIN_ON                  ;    (min on period - old)
                BTFSC   CARRY                   ;
                GOTO    FALSE_FLAG              ;    Y. Return a false flag.
                MOVF    TEMP_1,W                ;    N. Was the button on less than max ?
                SUBLW   MAX_ON                  ;          (max on period - old)
RETURN_FLAG:    BSF     PRESSED                 ;       Y. Return true flag.
                BTFSS   CARRY
FALSE_FLAG:     BCF     PRESSED                 ;       N. Return false flag.
                RETURN

;*****************************************************************************************
;
;      NAME:    MULTIPLY
;
;   PURPOSE:    40 bit * 40 bit unsigned multiply.    (ARG1 = ARG2 x ARG3)
;
;     INPUT:    ARG2_4..0  (MSD..LSD) (+ or - numbers )
;               ARG3_4..0  (MSD..LSD) (+ numbers only )
;
;    OUTPUT:    ARG1_9..0  (MSD..LSD)
;               (ARG2 and ARG3 unchanged)
;
;*****************************************************************************************

MULTIPLY:       BTFSS   ARG2_4,7                ; Is ARG2 negative ?
                GOTO    MULT                    ; N. Do the multiply.
                CALL    NEG_ARG2                ; Y. Convert to positive.
                CALL    MULT                    ;    Do the multiply.
                GOTO    NEG_ARG2                ;    Convert back to negative.
                                                ;    Return via "GOTO".
MULT:           CALL    CLEAR_ARG1
                MOVLW   40                      ; 40 bits to multiply.
                MOVWF   COUNT_1                 ;
MLOOP:          RRF     ARG3_0,W                ; Rotate the multiplier right 1 bit
                RRF     ARG3_4,F                ; into the carry bit.
                RRF     ARG3_3,F                ;
                RRF     ARG3_2,F                ; ( after 40 rotates ARG3 will be )
                RRF     ARG3_1,F                ; ( returned to its original value )
                RRF     ARG3_0,F                ;

                BTFSC   CARRY                   ; If carry bit is set, add ARG2 to partial
                CALL    ADD_ARG1_ARG2           ; product. ARG1_9..5 = ARG1_9..5 + ARG2_4..0.

                RRF     ARG1_9,F                ; Right shift the partial product.
                RRF     ARG1_8,F                ;
                RRF     ARG1_7,F                ;
                RRF     ARG1_6,F                ;
                RRF     ARG1_5,F                ;
                RRF     ARG1_4,F                ;
                RRF     ARG1_3,F                ;
                RRF     ARG1_2,F                ;
                RRF     ARG1_1,F                ;
                RRF     ARG1_0,F                ;

                DECFSZ  COUNT_1,F               ; All 40 bits done ?
                GOTO    MLOOP                   ; N. Keep going.
                RETURN                          ; Y. All done.

;*****************************************************************************************
;
;      NAME:    CHK_DDS
;
;   PURPOSE:    Update the AD9850/51 DDS chip if the frequency has changed.
;
;               Sends a 40 bit control word to the AD9850/51 DDS chip in serial form.
;               32 bit frequency + 8 bit refclk/phase/power down.
;               LSB of frequency is sent first.
;
;     INPUT:    ARG2 = Frequency.
;               ARG3 = DDS cal constant.  (2^56 / XTAL FREQ)
;
;    OUTPUT:    None.
;               (ARG2 and ARG3 unchanged)
;
;*****************************************************************************************

CHK_DDS:        CALL    PUSH_ARG3               ; Dont destroy ARG3.
                MOVLW   DDS_OLD                 ; Get the previous DDS freq.
                CALL    COPY_TO_ARG3

                CALL    CMP                     ; Is it the same as the current freq ?
                BTFSC   ZERO
                GOTO    PULL_ARG3               ; Y. Restore ARG3 then return via "GOTO".
                                                ;    (do not update the DDS chip)

                CALL    PULL_ARG3               ; N. Recover the original value of ARG3.

                MOVLW   DDS_OLD                 ;    Save it as the new DDS freq.
                CALL    COPY_FROM_ARG2

                CALL    MULTIPLY                ;    Multiply DDS freq by cal constant.
                                                ;    (result ARG1_6..3)
SEND_DDS_WORD:  MOVLW   DDS_WORD                ; Get the refclk/phase/power down byte.
                MOVWF   ARG1_7                  ; Place it in bits 32 to 40.

                MOVLW   40                      ; 40 bits to send.
                MOVWF   COUNT_1
NEXT_BIT:       RRF     ARG1_7,F                ; Shift right the 40 bits,
                RRF     ARG1_6,F                ; to move the LSB into the carry.
                RRF     ARG1_5,F
                RRF     ARG1_4,F
                RRF     ARG1_3,F
                BTFSS   CARRY                   ; Is the carry bit 1 ?
                BCF     DDS_DATA                ; N. Send a 0.
                BTFSC   CARRY                   ; Is the carry bit 1 ?
                BSF     DDS_DATA                ; Y. Send a 1.
                BSF     DDS_CLOCK               ; Toggle the DDS word load clock pin.
                BCF     DDS_CLOCK
                DECFSZ  COUNT_1,F               ; Have all 40 bits been sent?
                GOTO    NEXT_BIT                ; N. Keep going.
                BSF     DDS_LOAD                ; Toggle the DDS frequency update pin.
                BCF     DDS_LOAD
                RETURN

;*****************************************************************************************
;
;      NAME:    BIN_TO_DEC
;
;   PURPOSE:    Convert a 40 bit binary number to a 11 digit ASCII decimal number.
;               First the 40 bit binary number is converted to 11 digit decimal.
;               It is then converted to ASCII, including decimal point, leading zero
;               blanking and negative sign (if NEGATIVE flag set), ready for display on
;               the LCD.
;               (if used on 10 MHz CAL screen, do not include the decimal point.)
;
;     INPUT:    ARG2 = frequency. (+ numbers only )
;
;    OUTPUT:    11 digit ASCII decimal number in ARG1_10..0. (MSD..LSD)
;               (ARG2 unchanged)
;
;*****************************************************************************************

BIN_TO_DEC:     CALL    CLEAR_ARG1
                MOVLW   40
                MOVWF   COUNT_2                 ; 40 bits to process.
                GOTO    SHIFT_TO_DEC

LOOP40:         CALL    SET_ARG1                ; Set FSR = ARG1_10, COUNT_1 = 11.
DEC_LOOP:       MOVLW   3
                ADDWF   INDF,W                  ; If num + 3 > 7 then num = num + 3.
                MOVWF   TEMP_1                  ; Add 3 to num. (adds 6, after next shift)
                BTFSS   TEMP_1,3                ; Is result > 7 ? (bit 3 set)
                GOTO    NO_ADJ
                ADDLW   B'01111000'             ; Y. Move bit 3 to bit 7.
                MOVWF   INDF                    ;    Put the number back in the buffer.
NO_ADJ:         INCF    FSR,F                   ; Point to next byte in buffer.
                DECFSZ  COUNT_1,F
                GOTO    DEC_LOOP                ; Loop until all 11 bytes done.

SHIFT_TO_DEC:   RLF     ARG2_4,W                ; Shift 40 bit number left by one bit
                RLF     ARG2_0,F                ; into dec buffer.
                RLF     ARG2_1,F
                RLF     ARG2_2,F                ; After 40 shifts ARG2 unchanged.
                RLF     ARG2_3,F
                RLF     ARG2_4,F                ; MSD of ARG2
                RLF     ARG1_0,F                ; LSD of ARG1
                RLF     ARG1_1,F
                RLF     ARG1_2,F
                RLF     ARG1_3,F
                RLF     ARG1_4,F
                RLF     ARG1_5,F
                RLF     ARG1_6,F
                RLF     ARG1_7,F
                RLF     ARG1_8,F
                RLF     ARG1_9,F
                RLF     ARG1_10,F               ; MSD of ARG1.
                DECFSZ  COUNT_2,F
                GOTO    LOOP40                  ; Loop until all 40 bits processed.

                BCF     LEADING_FLAG            ; Blank leading zeros until flag is set.

                CALL    SET_ARG1                ; Set FSR = ARG1_10, COUNT_1 = 11.
ASCII_LOOP:     MOVF    INDF,W                  ; Get the number.
                BTFSC   ZERO                    ; Is it = 0 ?
                GOTO    BLANK
NO_BLANK        ADDLW   H'30'                   ; N. Convert it to ASCII.
                MOVWF   INDF
                BSF     LEADING_FLAG            ;    Set the leading blank flag,
                GOTO    NUM_DONE                ;    following zeros will not be blanked.
BLANK:          BTFSC   LEADING_FLAG            ; Y. Is the leading blank flag set.
                GOTO    NO_BLANK                ;    Y. Convert it to ASCII.
                MOVLW   ' '
                MOVWF   INDF                    ;    N. Blank the zero.
                MOVF    FSR,W                   ;       Save the location of the blank
                MOVWF   COUNT_2                 ;       so that a - sign can be inserted.
NUM_DONE:       INCF    FSR,F                   ; Point to the next byte.
                MOVLW   8
                SUBWF   COUNT_1,W               ; Time to cancel leading blanking?
                BTFSC   ZERO
                BSF     LEADING_FLAG            ; Y. Always display char before DP.
                DECFSZ  COUNT_1,F
                GOTO    ASCII_LOOP

                BTFSS   NEGATIVE                ; Was this a negative number?
                RETURN                          ; N. Just return.
                MOVF    COUNT_2,W               ; Y. Get position of last blank.
                MOVWF   FSR
                MOVLW   '-'                     ;    Place - sign in front of the number.
                MOVWF   INDF
                RETURN

;*****************************************************************************************
;
;      NAME:    FREQ_TO_LCD
;
;   PURPOSE:    Update the DDS chip if required, then add the offset freq and multiply
;               by DISP_MULT and display it on the LCD.
;
;     INPUT:    ARG2 = frequency.
;
;    OUTPUT:    None.
;               (ARG1 & ARG3 destroyed)
;
;*****************************************************************************************

FREQ_TO_LCD:    MOVLW   EE_DDS_CAL
                CALL    EE_TO_ARG3              ; Get the XTAL cal constant. (2^56/XTAL)
                CALL    CHK_DDS                 ; Update the DDS chip if required.

                MOVLW   EE_OFFSET_FREQ
                CALL    EE_TO_ARG3              ; Get the offset freq.
                CALL    ADD                     ; Add the offset.

                CALL    CLEAR_ARG3              ; Move the mult byte from EEPROM to ARG3.
                CALL    READ_MULT
                MOVWF   ARG3_0

                CALL    MULTIPLY                ; Do the multiply.
                MOVLW   ARG1_4                  ; Place the result in ARG2.
                CALL    COPY_TO_ARG2            ; Display the frequency, by falling
                                                ; through to DISPLAY_FREQ.

;*****************************************************************************************
;
;      NAME:    DISPLAY_FREQ
;
;   PURPOSE:    Display the frequency in ARG2 on the LCD.
;               Including negative sign, decimal point and leading zero blanking.
;               (if used on 10 MHz CAL screen, do not include the decimal point.)
;
;     INPUT:    ARG2 = frequency.
;
;    OUTPUT:    None.
;
;*****************************************************************************************

DISPLAY_FREQ:   BCF     NEGATIVE                ; Clear the negative flag.
                BTFSS   ARG2_4,7                ; Is ARG2 a negative number?
                GOTO    DISP_FREQ               ; N. Just continue.
                BSF     NEGATIVE                ; Y. Set the negative flag.
                CALL    NEG_ARG2                ;    Convert it to a + number.
DISP_FREQ:      CALL    BIN_TO_DEC              ; Convert the number to ASCII decimal.
                MOVLW   H'C0'                   ; Position the cursor, line 2 pos 1.
                BTFSC   CAL_SCRN                ; Are we in the CAL 10MHz screen ?
                MOVLW   H'C2'                   ; Y. Shift the display to line 2 pos 3.
                CALL    LCD_CMD
                CALL    SET_ARG1                ; Set FSR = ARG1_10, COUNT_1 = 11.
DISP_LOOP:      MOVF    INDF,W                  ; Get char pointed to by FSR.
                CALL    LCD_CHR                 ; Display the char.
                BTFSC   CAL_SCRN                ; Are we in the CAL 10MHz screen ?
                GOTO    NO_DP
                MOVLW   7
                SUBWF   COUNT_1,W               ; N. Is it time to insert a decimal point?
                BTFSC   ZERO
                CALL    LCD_DP                  ;    Y. Print a decimal point.
NO_DP:          INCF    FSR,F                   ; Inc the pointer.
                DECFSZ  COUNT_1,F               ; All done ?
                GOTO    DISP_LOOP               ; N. Continue.

                BTFSC   NEGATIVE                ; Was ARG2 negative ?
                CALL    NEG_ARG2                ; Y. Return it to a negative number.

                BTFSC   CAL_SCRN                ; Are we in the CAL 10MHz screen ?
                RETURN                          ; Y. All done.
                MOVLW   MHZ
                GOTO    LCD_TEXT                ; N. Display ' MHz'.
                                                ;    Return via "GOTO".

;*****************************************************************************************
;
;      NAME:    INIT_LCD
;
;   PURPOSE:    Initialize the 16 x 2 liquid crystal display.
;               The LCD controller chip must be equivalent to the HITACHI 44780.
;
;     INPUT:    None.
;
;    OUTPUT:    None.
;
;*****************************************************************************************

INIT_LCD:       CALL    LCD_RESET               ; Reset the LCD.
                CALL    LCD_RESET               ; Reset the LCD.
                CALL    LCD_RESET               ; Reset the LCD.

                MOVLW   B'00101000'             ; LCD 4 bit mode command.
                MOVWF   PORTB                   ; Also sets LCD_E lead high.
                BCF     LCD_E
                CALL    WAIT_200US
                MOVLW   B'00101000'             ; Set LCD to 2 line, 5x7 dot.
                CALL    LCD_CMD
                MOVLW   B'00001000'             ; Turn the display off.
                CALL    LCD_CMD
                CALL    LCD_CLEAR               ; Clear the display.
                MOVLW   B'00000110'             ; Cursor increments, no display shift.
                CALL    LCD_CMD
LCD_NORM:       MOVLW   B'00001100'             ; Turn the display on.
                GOTO    LCD_CMD                 ; Return via "GOTO".

LCD_RESET:      MOVLW   B'00111000'             ; LCD reset command.
                MOVWF   PORTB                   ; Also sets LCD_E lead high.
                BCF     LCD_E
                GOTO    WAIT_8MS                ; Return via "GOTO".

;*****************************************************************************************
;
;     NAMES:    LCD_CLEAR       Clear the display and home the cursor.
;               LCD_HOME        Home the cursor.
;               LCD_DP          Display a decimal point.
;               LCD_CMD         Send the command byte in W to the LCD.
;               LCD_CHR         Send the data byte in W to the LCD.
;               WAIT_8MS        Do nothing for 8 mS.
;               MS_WAIT         W = The number of mS to wait.
;
;*****************************************************************************************

LCD_CLEAR:      MOVLW   H'01'                   ; Clear the display. Then wait 8mS.
                CALL    LCD_CMD
WAIT_8MS:       MOVLW   8                       ; Set the outer loop to 8mS.
MS_WAIT:        MOVWF   COUNT_0                 ; Save the number of mS delay.
OUTER_LOOP:     CLRW                            ; Set the wait loop to 256.
                CALL    WAIT_LOOP               ; 256 loops x 4 x 1uS = 1.024mS.
                DECFSZ  COUNT_0,F               ; Dec the mS counter.
                GOTO    OUTER_LOOP
                RETURN

LCD_HOME:       MOVLW   H'80'                   ; Position the cursor, line 1 pos 1.
LCD_CMD:        BCF     LCD_RS                  ; Select the instruction register.
                GOTO    LCD_BYTE                ; Send the byte to the LCD.

LCD_DP:         MOVLW   H'A5'                   ; Display a decimal point.
LCD_CHR:        BSF     LCD_RS                  ; Select display data register.
LCD_BYTE:       MOVWF   TEMP_3                  ; Save the byte to send to the LCD.
                CALL    SEND_NIBBLE             ; Send the high nibble to the LCD.
                SWAPF   TEMP_3,F                ; Send the low nibble to the LCD.
SEND_NIBBLE:    MOVLW   B'00001111'
                ANDWF   PORTB,F                 ; Clear the high nibble of port B.
                MOVF    TEMP_3,W                ; Get the byte of data.
                ANDLW   B'11110000'             ; Keep only the high nibble.
                IORWF   PORTB,F                 ; Send the nibble to port B.
                BSF     LCD_E                   ; Pulse the LCD_E lead.
                BCF     LCD_E
WAIT_200US:     MOVLW   256-50                  ; Set the wait loop to 200uS.
WAIT_LOOP:      ADDLW   1                       ; 1 CLK cycles \
                BTFSS   ZERO                    ; 1     "       > 4 CLKs X 1uS
                GOTO    WAIT_LOOP               ; 2     "      /
                RETURN

;*****************************************************************************************
;
;     NAMES:    DISP_RIT        Display 'RIT' on line 1 pos 14. Then display RX or TX.
;               DISP_TX_RX      Display 'RX ' or 'TX ' on line 1 pos 1.
;
;     INPUT:    None.
;
;    OUTPUT:    None.
;
;-----------------------------------------------------------------------------------------
;
;      NAME:    LCD_TEXT
;
;   PURPOSE:    Display on the LCD, the string of characters pointed to by W.
;               The last chr of the string must be a space with the MSB set. (C0)
;
;     INPUT:    W = String offset address. (offset from the base of the text table)
;
;    OUTPUT:    None.
;
;*****************************************************************************************

DISP_RIT:       MOVLW   H'8D'
                CALL    LCD_CMD                 ; Position the cursor, line 1 pos 14.
                MOVLW   RIT_OFF
DISP_TEXT:      CALL    LCD_TEXT                ; Display 'RIT'.
                                                ; Fall through and display RX or TX.
DISP_TX_RX:     CALL    LCD_HOME                ; Goto the start of line 1.
                CALL    CHK_TX                  ; Check the TX button.
                MOVLW   RX                      ; Prepare to display 'RX '.
                BTFSC   PRESSED                 ; Was the TX button pressed ?
                MOVLW   TX                      ; Y. Prepare to display 'TX '.
                                                ; Display the 'TX ' or 'RX '.

LCD_TEXT:       MOVWF   TEMP_1                  ; Save the text start address.
                CALL    TABLE                   ; Get the character
                CALL    LCD_CHR                 ; Display character.
                BTFSC   TEMP_3,3                ; Was it the end of message ? (MSB set)
                RETURN                          ; Y. All done.
                INCF    TEMP_1,W                ; N. Point to next character.
                GOTO    LCD_TEXT                ;    Keep looping until end of message.

;*****************************************************************************************

                END
