
//  Stellex PLL v7, YIG CW manipulation ON/OFF
//  YO4HFU 24.12.2023


#include <SPI.h>
#include <LiquidCrystal_I2C.h>
#include <EEPROM.h>

#define EEPROM_ADDRESS 0

// Define LMX2326 control pins
#define DATA 11
#define LE 10
#define CLK 13

#define AUX_PIN A0            // CW mode ON/OFF
#define STEP_BUTTON_PIN A1    // Frequency Step Selection
#define UP_PIN A3             // Frequency Up
#define DOWN_PIN A2           // Frequency Down
#define LOCK_PIN 8            // PLL lock signal

#define SW_PIN 9              // YIG power supply control
#define MON_PIN A7            // PLL voltage control monitoring, not used

 

const unsigned long REG_R = 0xD0;   // 0xA0 for 10MHz PLL reference
const unsigned long REG_F = 0x92;

const int MIN_FREQ = 1300;      // x4 = 5200MHz
const int MAX_FREQ = 1625;      // x4 = 6500MHz
const int DEFAULT_FREQ = 1440;  // x4 = 5760MHz

const unsigned long DEBOUNCE_DELAY = 50;

unsigned long lastUpButtonTime = 0;
unsigned long lastDownButtonTime = 0;

int SwState = LOW;                     
unsigned long previousMillis = 0;        
const long interval = 800;           // ON/OFF TIME CW MANIPULATION


float freq;
float lastfreq;
int step = 25;

LiquidCrystal_I2C lcd(0x27, 16, 2);

bool cwState = LOW;
bool cwButtonPressed = false;



void init_LMX() {
  pinMode(DATA, OUTPUT);
  pinMode(CLK, OUTPUT);
  pinMode(LE, OUTPUT);
  pinMode(SW_PIN, OUTPUT);

  digitalWrite(DATA, LOW);
  digitalWrite(CLK, LOW);
  digitalWrite(LE, LOW);

}

void init_register(uint32_t reg_value) {
  int bit_pos = 21;
  do {
    bit_pos--;
    digitalWrite(CLK, LOW);
    digitalWrite(LE, LOW);
    digitalWrite(DATA, bitRead(reg_value, bit_pos));
    digitalWrite(CLK, HIGH);
  } while (bit_pos > 0);
  digitalWrite(CLK, LOW);
  digitalWrite(DATA, LOW);
  digitalWrite(LE, HIGH);
}

void init_F() {
  init_register(REG_F);
}

void init_R() {
  init_register(REG_R);
}

void setup() {
  pinMode(UP_PIN, INPUT_PULLUP);
  pinMode(DOWN_PIN, INPUT_PULLUP);
  pinMode(STEP_BUTTON_PIN, INPUT_PULLUP);
  pinMode(LOCK_PIN, INPUT_PULLUP);
  pinMode(AUX_PIN, INPUT_PULLUP);
  pinMode(SW_PIN, OUTPUT);

  EEPROM.get(EEPROM_ADDRESS, lastfreq);

  if (isnan(lastfreq) || lastfreq < MIN_FREQ || lastfreq > MAX_FREQ) {
    freq = DEFAULT_FREQ;
    EEPROM.put(EEPROM_ADDRESS, freq);
  } else {
    freq = lastfreq;
  }

  init_LMX();
  init_register(REG_F);
  delayMicroseconds(50);
  init_register(REG_R);
  delayMicroseconds(50);

  lcd.init();
  lcd.backlight();
  lcd.clear();
  

  uint32_t N = (uint32_t)(freq / 0.25);
  uint32_t P = 32;
  uint32_t B = N / P;
  uint32_t A = N - (B * P);

  if (B >= A && B >= 0 && B <= 8191 && A >= 0 && A <= 31) {
    uint32_t data = 0;
    data |= B << 8;
    data |= A << 3;
    data |= 1UL << 1;

    for (int i = 21; i > 0; i--) {
      digitalWrite(CLK, LOW);
      digitalWrite(LE, LOW);
      digitalWrite(DATA, bitRead(data, i));
      digitalWrite(CLK, HIGH);
    }
    digitalWrite(CLK, LOW);
    digitalWrite(DATA, LOW);
    digitalWrite(LE, HIGH);
  }
}

void loop() {
  bool step_button_reading = digitalRead(STEP_BUTTON_PIN);
  static bool last_step_button_state = HIGH;
  static unsigned long step_button_last_debounce_time = 0;
  if (step_button_reading != last_step_button_state) {
    step_button_last_debounce_time = millis();
  }
  if (millis() - step_button_last_debounce_time > DEBOUNCE_DELAY) {
    if (step_button_reading == LOW) {
      if (step == 25) {
        step = 125;
      } else if (step == 125) {
        step = 250;
      } else {
        step = 25;
      }
    }
  }
  last_step_button_state = step_button_reading;

  bool up_button_reading = digitalRead(UP_PIN);
  bool down_button_reading = digitalRead(DOWN_PIN);

  static bool up_button_state = HIGH;
  static bool down_button_state = HIGH;

  if (up_button_reading != up_button_state) {
    lastUpButtonTime = millis();
  }
  if (down_button_reading != down_button_state) {
    lastDownButtonTime = millis();
  }

  if (millis() - lastUpButtonTime > DEBOUNCE_DELAY) {
    if (up_button_reading == LOW) {
      freq += step / 100.0;
      if (freq > MAX_FREQ) {
        freq = MAX_FREQ;
      }
    }
  }
  if (millis() - lastDownButtonTime > DEBOUNCE_DELAY) {
    if (down_button_reading == LOW) {
      freq -= step / 100.0;
      if (freq < MIN_FREQ) {
        freq = MIN_FREQ;
      }
    }
  }
  up_button_state = up_button_reading;
  down_button_state = down_button_reading;

  lcd.setCursor(0, 0);
  lcd.print("Fq: ");
  lcd.print(freq * 4);
  lcd.print(" MHz     ");

  lcd.setCursor(0, 1);
  lcd.print("Step: ");

  if (step == 25) {
    lcd.print("1 ");
  } else if (step == 125) {
    lcd.print("5 ");
  } else {
    lcd.print("10");
  }

  if (freq != lastfreq) {
    EEPROM.put(EEPROM_ADDRESS, freq);
    delay(5);

    uint32_t N = (uint32_t)(freq / 0.25);
    uint32_t P = 32;
    uint32_t B = N / P;
    uint32_t A = N - (B * P);

    if (B >= A && B >= 0 && B <= 8191 && A >= 0 && A <= 31) {
      uint32_t data = 0;
      data |= B << 8;
      data |= A << 3;
      data |= 1UL << 1;

      for (int i = 21; i > 0; i--) {
        digitalWrite(CLK, LOW);
        digitalWrite(LE, LOW);
        digitalWrite(DATA, bitRead(data, i));
        digitalWrite(CLK, HIGH);
      }
      digitalWrite(CLK, LOW);
      digitalWrite(DATA, LOW);
      digitalWrite(LE, HIGH);

      lastfreq = freq;
    } else {
      lcd.setCursor(0, 1);
      lcd.print("Error!  ");
    }
  }


  // Check if the AUX_PIN state has changed
  if (digitalRead(AUX_PIN) == LOW && !cwButtonPressed) {
    cwState = !cwState;  // Toggle the CW state
    cwButtonPressed = true;
  }
  // Reset the flag when the AUX_PIN is released
  if (digitalRead(AUX_PIN) == HIGH) {
    cwButtonPressed = false;
  }


  if (cwState == HIGH) {
 unsigned long currentMillis = millis();
  if (currentMillis - previousMillis >= interval) {
    previousMillis = currentMillis;
    if (SwState == LOW) {
      SwState = HIGH;
    } else {
      SwState = LOW;
    }
    digitalWrite(SW_PIN, SwState);
  }
  }
  if (cwState == LOW) {
  digitalWrite(SW_PIN, HIGH);
  }
 

  lcd.setCursor(11, 1);
  if (cwState == HIGH) {
    lcd.print("CW   ");
  } else {
    lcd.print("     ");
  }


  int buttonState = digitalRead(LOCK_PIN);
  lcd.setCursor(14, 1);       
  if (buttonState == HIGH) {
    lcd.print("LK "); // print "LK" if the pin is HIGH
  } else {
    lcd.print("UK "); // print "UK" if the pin is LOW
  }

  delayMicroseconds(10);
}
