
//Rotator PWM fw1.1
// YO4HFU - 2025

// Modified from YO3RAK and ON7EQ code

#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <NewEncoder.h>
#include <EEPROM.h>

LiquidCrystal_I2C lcd(0x27, 16, 2);

#define ENCODER_PIN_A 3
#define ENCODER_PIN_B 2
NewEncoder encoder(ENCODER_PIN_B, ENCODER_PIN_A, INT16_MIN, INT16_MAX, 0, FULL_PULSE);
int16_t prevEncoderValue = 0;

#define EEPROM_ADDR_COMAZIM 0

const int AzMin = 99;       // found by Calibration Tool (0 degree value)
const int AzMax = 815;      // found by Calibration Tool (360 degree value)
const int AzErr = 1;
const int Amax = 10;

const int PwAzMin = 40;
const int PwAzMax = 100;

enum PinAssignments {
  POT_PIN = A0,
  OK_BUTTON = 4,
  CW_PIN = A2,
  CCW_PIN = A3,
  PWM_PIN = 5,
  CW_LED_PIN = 12,
  CCW_LED_PIN = 11,
};

bool azMultiplier10 = false;
bool MainControl = true;
int TruAzim = 0, ComAzim = 0, OldTruAzim = 0, OldComAzim = 0;
int PwAzStop = 0, PwAzStar = 0, PwAz = 0, StaAzim = 0;
char AzDir;
bool AzStop = false;
unsigned long NowTime = 0;

const int numReadings = 5;
int azimuth[numReadings];
int totalAz = 0, readIndex = 0;

bool isSwitchingDirection = false;
unsigned long switchStartTime = 0;
const unsigned long switchDelay = 200;
char newAzDir = '\0';
char lastAzDir = '\0';

String ComputerRead = "", Azimuth = "", ComputerWrite = "";

void setup() {
  Serial.begin(9600);
  Serial.setTimeout(50);

  lcd.init();
  lcd.clear();
  lcd.backlight();
  lcd.setCursor(0, 0);
  lcd.print("Rotator fw1.1");
  lcd.setCursor(0, 1);
  lcd.print("YO4HFU 2025");
  delay(2000);  
  lcd.clear();
  

  pinMode(POT_PIN, INPUT);
  pinMode(OK_BUTTON, INPUT_PULLUP);
  pinMode(CW_PIN, OUTPUT);
  pinMode(CCW_PIN, OUTPUT);
  pinMode(PWM_PIN, OUTPUT);
  pinMode(CW_LED_PIN, OUTPUT);
  pinMode(CCW_LED_PIN, OUTPUT);

  encoder.begin();

  encoder.setValue(0);

  totalAz = 0;
for (int i = 0; i < numReadings; i++) {
  delay(5); // slight delay between reads
  int raw = analogRead(POT_PIN);
  azimuth[i] = map(raw, AzMin, AzMax, 0, 359);
  azimuth[i] = constrain(azimuth[i], 0, 359);
  totalAz += azimuth[i];
}
TruAzim = totalAz / numReadings;

  EEPROM.get(EEPROM_ADDR_COMAZIM, ComAzim);
  if (ComAzim < 0 || ComAzim > 359) {
    ComAzim = TruAzim;
    EEPROM.put(EEPROM_ADDR_COMAZIM, ComAzim);
  }

  OldTruAzim = TruAzim;
  OldComAzim = ComAzim;
}

void loop() {
  totalAz -= azimuth[readIndex];
  azimuth[readIndex] = map(analogRead(POT_PIN), AzMin, AzMax, 0, 359);
  totalAz += azimuth[readIndex];
  readIndex = (readIndex + 1) % numReadings;
  TruAzim = constrain(totalAz / numReadings, 0, 359);

  ReadAzimEncoder();
  Display();

  if (Serial.available()) SerComm();

  if ((millis() - NowTime) > 300) {
    if (OldTruAzim != TruAzim) OldTruAzim = TruAzim;
    NowTime = millis();
  }

  if (OldComAzim != ComAzim) {
    OldComAzim = ComAzim;
    if (!MainControl) {
      MainControl = true;
      lcd.setCursor(14, 1);
      lcd.print(" ");
    }
  }

  if (TruAzim == ComAzim) {
    AzStop = true;
    analogWrite(PWM_PIN, 0);
    StaAzim = TruAzim;
    lcd.setCursor(14, 0);
    lcd.print("=");
    digitalWrite(CW_LED_PIN, LOW);
    digitalWrite(CCW_LED_PIN, LOW);
  } else if ((abs(TruAzim - ComAzim) <= AzErr) && !AzStop) {
    AzimRotate();
  } else if (abs(TruAzim - ComAzim) > AzErr) {
    AzStop = false;
    AzimRotate();
  }
}

void ReadAzimEncoder() {
  static bool lastButtonState = HIGH;
  static unsigned long lastDebounceTime = 0;
  const unsigned long debounceDelay = 50;

  static int16_t lastSavedAzim = -1;
  static unsigned long lastEncoderChange = 0;
  static bool pendingSave = false;

  bool reading = digitalRead(OK_BUTTON);
  if (reading != lastButtonState) lastDebounceTime = millis();

  if ((millis() - lastDebounceTime) > debounceDelay) {
    static bool buttonPressed = false;
    if (reading == LOW && !buttonPressed) {
      azMultiplier10 = !azMultiplier10;
      buttonPressed = true;
    } else if (reading == HIGH) {
      buttonPressed = false;
    }
  }
  lastButtonState = reading;

  NewEncoder::EncoderState state;
  if (encoder.getState(state)) {
    int16_t diff = state.currentValue - prevEncoderValue;
    if (diff != 0) {
      int step = azMultiplier10 ? 10 : 1;
      ComAzim = (ComAzim + diff * step) % 360;
      if (ComAzim < 0) ComAzim += 360;
      prevEncoderValue = state.currentValue;

      lastEncoderChange = millis();
      pendingSave = true;
    }
  }

  if (pendingSave && (millis() - lastEncoderChange > 1000)) {
    ComAzim = constrain(ComAzim, 0, 359);
    if (ComAzim != lastSavedAzim) {
      EEPROM.put(EEPROM_ADDR_COMAZIM, ComAzim);
      lastSavedAzim = ComAzim;
    }
    pendingSave = false;
  }
}

void Display() {
  lcd.setCursor(0, 0);
  lcd.print("Azimut: ");
  DisplValue(TruAzim, 8, 0);
  lcd.setCursor(0, 1);
  lcd.print("Target: ");
  DisplValue(ComAzim, 8, 1);
  lcd.setCursor(15, 1);
  lcd.print(azMultiplier10 ? "*" : " ");
}

void DisplValue(int value, int col, int row) {
  char buffer[4];
  sprintf(buffer, "%03d", value);
  lcd.setCursor(col, row);
  lcd.print(buffer);
  lcd.print((char)223);
}

void AzimRotate() {
  if (!MainControl) {
    analogWrite(PWM_PIN, 0);
    digitalWrite(CW_LED_PIN, LOW);
    digitalWrite(CCW_LED_PIN, LOW);
    return;
  }

  char desiredDir = (ComAzim > TruAzim) ? char(126) : char(127);

  if (isSwitchingDirection) {
    if (millis() - switchStartTime >= switchDelay) {
      bool dir = (desiredDir == char(126));
      digitalWrite(CW_PIN, dir ? LOW : HIGH);
      digitalWrite(CCW_PIN, dir ? HIGH : LOW);
      isSwitchingDirection = false;
      AzDir = desiredDir;
      if (AzDir != lastAzDir) {
        lcd.setCursor(14, 0);
        lcd.print(AzDir);
        lastAzDir = AzDir;
      }
    }
    analogWrite(PWM_PIN, 0);
    digitalWrite(CW_LED_PIN, LOW);
    digitalWrite(CCW_LED_PIN, LOW);
    return;
  }

  if (desiredDir != AzDir) {
    analogWrite(PWM_PIN, 0);
    digitalWrite(CW_LED_PIN, LOW);
    digitalWrite(CCW_LED_PIN, LOW);
    StaAzim = TruAzim;
    switchStartTime = millis();
    isSwitchingDirection = true;
    return;
  }

  bool dir = (desiredDir == char(126));
  digitalWrite(CW_PIN, dir ? LOW : HIGH);
  digitalWrite(CCW_PIN, dir ? HIGH : LOW);

  float errorStop = abs(ComAzim - TruAzim);
  float errorStart = abs(StaAzim - TruAzim);
  float scaledStop = constrain(errorStop / Amax, 0.0, 1.0);
  float scaledStart = constrain(errorStart / Amax, 0.0, 1.0);

  PwAzStop = PwAzMin + round((PwAzMax - PwAzMin) * scaledStop * scaledStop);
  PwAzStar = PwAzMin + round((PwAzMax - PwAzMin) * scaledStart * scaledStart);
  PwAz = min(PwAzStop, PwAzStar);
  PwAz = constrain(PwAz, PwAzMin, PwAzMax);

  analogWrite(PWM_PIN, round(PwAz * 2.55));
  digitalWrite(CW_LED_PIN, dir ? LOW : HIGH);
  digitalWrite(CCW_LED_PIN, dir ? HIGH : LOW);
}

void SerComm() {
  ComputerRead = Serial.readStringUntil('\n');
  Azimuth = "";

  for (int i = 0; i < ComputerRead.length(); i++) {
    if (ComputerRead.charAt(i) == 'M') {
      for (int j = i + 1; j < ComputerRead.length(); j++) {
        if (isDigit(ComputerRead.charAt(j)))
          Azimuth += ComputerRead.charAt(j);
        else break;
      }
    }

    if (ComputerRead.charAt(i) == 'S') {
      MainControl = false;
      lcd.setCursor(14, 1);
      lcd.print("S");
    }
  }

  if (Azimuth != "") {
    ComAzim = Azimuth.toInt() % 360;
  }

  for (int i = 0; i < ComputerRead.length(); i++) {
    if (ComputerRead.charAt(i) == 'C' && millis() > 6000) {
      ComputerWrite = "+0" + String(TruAzim);
      Serial.println(ComputerWrite);
    }
  }
}
