/*-----------------------------------------------------------------
  All code is provided for EDUCATIONAL purposes ONLY. There is no
  warranty and no support. Licensed under the MIT license - see
  http://www.qsl.net/ve3lny/MIT%20License.html

  slc_charger : main.c

  This program functions as a monitor for a UC3906 SLC charger,
  displaying status on a standard parallel 2x16 LCD display.
  
  1. UC3906 pin 9 !OC is sampled directly by PA3. !OC is
     an open collector output. I tie it to +5V with a 2.2K
     resistor (R108).
	 
  2. UC3906 pin 7 !PI goes to a LM393 comparator, then to PB6 with
     pullup, where it becomes PI.
	 
  3. UC3906 pin 10 SLC goes to a LM393 comparator where it becomes
     !SLC, then to a pullup resistor and an isolating resistor to PB1.
	 
  4. Vout from an ACS712 hall-effect current sensor is directed to
     ADC0, to measure the charge current. A digital filter is used
	 because the sensor output is a but unsteady.

  5. Vout is directed to a 5:1 voltage divider then to PA2. It
     is read directly by the A/D converter to provide a readout
	 of Vout on the display.
		
  Processor: ATtiny461
      Clock: Internal, 8.0 MHz

  Change log
    20160725jb 1. Fix buffer overflow bug
               2. Added 'alive' indicator on display
               3. fix up fault error message handling.
    20161104jb fix up fault error message handling some more.
	20161107jb replace digital filter with averager. The qsort
			   function gobbled up too much memory.
-----------------------------------------------------------------*/
#include <avr/io.h>
#include <avr/interrupt.h>
#define F_CPU 8000000UL
#include <util/delay.h>
#include <string.h>
#include <stdlib.h>
#include "common.h"
#include "lcd.h"

#define TOS       282		// Temperature Sensor Offset (nominal 300) - see spec
                            // sheet section 15.12 Temperature Measurement

#define VADC_MEASURED_MV	  4970L	   // measured ADC Vcc reference (MV) (Nominal 5000)

#define FLOAT_FAILURE_Vout_HUNDREDTHS   1300	// Vout below which float mode assumed to fail

// Discussion: I had the current threshold at 350 before which under
// some circumstances, when the current was around 400 to 500, caused a
// fault to be reported. So the value chosen is kind of arbitrary.
#define FLOAT_CURRENT_MAX_MA		600

#define MAX_MINUTES_IN_HIGH_CHARGE   2880L		// max time allowed in bulk and overcharge modes
   
#define ACS712_SENS_MV_A       180     // (nominal 185)

#define FAN_TEMP_THRESHOLD       32   // degrees C
#define FAN_CURRENT_THRESHOLD  1000   // milliamps

#define AVERAGER_ENTRIES		8

// Global variables
static volatile uint16_t uiAverager_array[AVERAGER_ENTRIES];

static volatile uint8_t  ucBulkOverchgModeEntered=0;	// turned on when mode entered
static volatile uint32_t ulBulkOverchgMins=0;			// minutes in bulk or overcharge mode

// Internal Functions

static uint16_t averager_get( void );
static void     averager_add( uint16_t );
static uint8_t  determine_state( uint8_t ucOC, uint8_t ucPI, uint8_t ucSLC, uint16_t uiCurrent, 
                                 uint16_t uiVout_HV, char *buffer );
static uint16_t measure_current(void);
static int16_t  measure_temperature(void);
static uint16_t measure_Vout(void);
static void     run_fan( int16_t iTemp, uint16_t uiCurrent );
static void     show_live_indicator( void );
static void     sound_beeper( void );
static void     system_init(void);

/*-----------------------------------------------------------------
  main()
-----------------------------------------------------------------*/
int main(void)
{
  static const char *msg1 = "Chrg Monitor 1.2";
  uint8_t		ii, nn, ucFault;
  uint8_t       ucPI, ucSLC, ucOC;
  int16_t		iTemp;			// internal temperature sensor, degrees C
  uint16_t		uiCurrent;
  uint16_t      uiVoutHV;		// Vout, hundredths volts
  char		 	buffer[20];     // 20160725jb

  system_init();			// initialize system

  lcd_init( LCD_DISP_ON_CURSOR );
  lcd_clrscr();
  lcd_gotoxy( 0, 0 );
  lcd_puts( msg1 );
  lcd_gotoxy( 0, 1 );

  // clock = 8mHz = 125ns
  for ( ii=0; ii<80; ++ii )		// 2 second
	_delay_loop_2( 50000 );		// 25 ms

  while( 1 )
  {
    lcd_clrscr();
    lcd_gotoxy( 0, 0 );
	
	iTemp = measure_temperature();		// degrees C
    itoa( iTemp, buffer, 10 );
    lcd_puts( "T=" );
    lcd_puts( buffer );
    lcd_puts( "C " );

	uiVoutHV = measure_Vout();			// hundredths of volts
    utoa( uiVoutHV, buffer, 10 );
	nn = strlen( buffer );
	if ( nn > 1 )
	{
	  // insert a decimal before 2nd last digit
	  buffer[nn+1] = '\0';
	  buffer[nn] = buffer[nn-1];
	  buffer[nn-1] = buffer[nn-2];
	  buffer[nn-2] = '.';
	}
    lcd_puts( "Vo=" );
    lcd_puts( buffer );
    lcd_puts( "V" );
	
	uiCurrent = measure_current();		// milliamps
	averager_add( uiCurrent );			// 20161107jb
	uiCurrent = averager_get();			// use an average value
    utoa( uiCurrent, buffer, 10 );
	if ( strlen( buffer ) > 4 ) buffer[4] = '\0';  //
    lcd_gotoxy( 0, 1 );
    lcd_puts( buffer );
    lcd_puts( "mA " );

    // Get statuses
	ucOC = ( test_bit( PINA, PA3 ) ) ? 0 : 1;     // get OC positive logic
	ucPI = test_bit( PINB, PB6 );				  // PI already positive logic
	ucSLC = ( test_bit( PINB, PB1 ) ) ? 0 : 1;    // Get SLC positive logic

    ucFault = determine_state( ucOC, ucPI, ucSLC, uiCurrent, uiVoutHV, buffer );
    // reorganize the logic  20161104jb
	if ( buffer[0] )			// there is a message?
      lcd_puts( buffer );		// show message

    if ( ucFault )				// if a fault occurred
	{
	  sound_beeper();				// sound beeper
      for ( ii=0; ii<240; ++ii )	// 6 second
        _delay_loop_2( 50000 );		// 25 ms
	}
	else
	  show_live_indicator();		// otherwise show alive indicator

    run_fan( iTemp, uiCurrent );	// manage the fan

    for ( ii=0; ii<60; ++ii )		// 1.5 second
      _delay_loop_2( 50000 );		// 25 ms
  }
}

/*-----------------------------------------------------------------
  system_init()

  PORTB: PB0 (output) (ISP-MOSI) beeper
         PB1 (input) (ISP-MISO) !SLC
         PB2 (output) (ISP-SCK) Fan
         PB3 (output) LCD RS
         PB4 (output) LCD R/W
         PB5 (output) LCD E
         PB6 (input) PI
         PB7 (n/a) (ISP-!Reset)

  PORTA: PA0 (input) ADC0 (current sensor) 
         PA1 (input) not used
         PA2 (input) ADC2 (Vout)
         PA3 (input) !OC
         PA4 (output) LCD DB4
         PA5 (output) LCD DB5
         PA6 (output) LCD DB6
         PA7 (output) LCD DB7
-----------------------------------------------------------------*/
static void system_init()
{
  // The fuses are set to provide an internal clock, running at
  // 8.0 MHz with a prescaler of 1. No change to the prescaler
  // is required.

  DDRB  = 0b00111101;       // PortB: 0=input, 1=output
  PORTB = 0b01000000;       // initialize PortB, PB6 pullup

  DDRA  = 0b11110000;       // PortA: 0=input, 1=output
  PORTA = 0b00001010;       // initialize PortA; PA1, PA3 pullup

  DIDR0  = 0b00000101;		// disable digital buffer: ADC0 ADC2
  DIDR1  = 0b00000000;

  // Initialize Timer0. Timer0 is used to produce 1s ticks
  // so we can count time.
  TCCR0A = 0b10000000;       // normal 16-bit counter mode
  TCCR0B = 0b00000100;       // prescaler = 128 (turns timer0 on)
  TCNT0L = INIT_CTR0L;
  TCNT0H = INIT_CTR0H;
  set_bit( TIMSK, TOIE0 );	 // enable timer/counter0 overflow interrupt
  
  sei();                     // enable interrupts
}

/*-----------------------------------------------------------------
  Timer 0 overflow interrupt handler, for timing ADC conversions

  Since the clock is running at 8.0 mHz, the counter increments
  every 125.0 ns. Using a prescaler of 256 gives a cycle time of
  32us. So if we count to 31,250 we get 1s interrupts.
-----------------------------------------------------------------*/
ISR(TIMER0_OVF_vect)
{
  static volatile uint16_t uiSeconds=0;
  
  ++uiSeconds;
  if ( uiSeconds > 59 )
  {
	 uiSeconds = 0;

    if ( ucBulkOverchgModeEntered )
      ++ulBulkOverchgMins;              // keep minutes in bulk/overcharge
  }

  TCNT0L = INIT_CTR0L;					// reinitialize counter
  TCNT0H = INIT_CTR0H;
}

/*-----------------------------------------------------------------
  measure_temperature()
  
  Puts the ADC into temperature measuring mode and does A/D
  conversion. According to the documentation an ADC reading of
  325 means a temperature of 25C, so you need to subtract 300,
  the Temperature Sensor Offset. Note that the Tos changes with
  temperature! (Table 15-2) I'm not sure how you are supposed to 
  manage that. Also the temperature could be up to 10 degrees off 
  unless you calibrate it somehow. Anyway the purpose for measuring
  the temperature is only to turn the fan on and off, so it doesn't
  really matter much. You can adjust #define TOS if you like or
  feel free to rewrite the code to do it properly if you know how.
-----------------------------------------------------------------*/
static int16_t measure_temperature()
{
  uint16_t		uiAdc;
  uint8_t		ii;
  
  ADMUX  = 0b10011111;		// selects 1.1V reference and temperature sensor
  ADCSRA = 0b10000111;		// enable ADC; prescaler=128; disable interrupt
  ADCSRB = 0b00001000;		// REFS2=0, MUX5=1

  for ( ii=0; ii<2; ++ii )
  {
    set_bit( ADCSRA, ADSC );				// start conversion
    while ( 1 )
    {
	  if ( !test_bit( ADCSRA, ADSC ) )	    // conversion complete
	    break;
    }
  
    uiAdc = (uint16_t) ADCL | ( (uint16_t) ADCH << 8 );
  }
    
  return (int16_t) ( uiAdc - TOS );
}

/*-----------------------------------------------------------------
  measure_Vout()
  
  Puts the ADC into single-ended mode on port ADC2 and does A/D
  conversion. This measures Vout, reduced by a 5:1 voltage
  divider. 
  
  Returns Vout in hundredths of volts.
-----------------------------------------------------------------*/
static uint16_t measure_Vout()
{
  uint16_t		uiAdc, uiVout_HV;
  uint32_t		ulVoutMV;
  uint8_t		ii;
  
  ADMUX  = 0b00000010;		// selects Vcc reference; ADC2
  ADCSRA = 0b10000111;		// enable ADC; prescaler=128; disable interrupt
  ADCSRB = 0b00000000;		// REFS2=0, MUX5=0

  for ( ii=0; ii<2; ++ii )
  {
    set_bit( ADCSRA, ADSC );				// start conversion
    while ( 1 )
    {
	  if ( !test_bit( ADCSRA, ADSC ) )	// conversion complete
	    break;
    }
  
    uiAdc = (uint16_t) ADCL | ( (uint16_t) ADCH << 8 );
  }

  // The adc output is converted to volts as follows:
  // Vin = (ADC)(Vref) / 1024
  // Since we're using integer arithmetic, its better to express voltage
  // as millivolts to avoid truncation of important results.
  
  ulVoutMV = ( (uint32_t) uiAdc * VADC_MEASURED_MV ) / 1024L;

  // Account for 5:1 divider and convert MV to HV
  uiVout_HV = (uint16_t) ( ( ulVoutMV * 5L ) / 10L );

  return uiVout_HV;
}

/*-----------------------------------------------------------------
  run_fan(...)
  
  Uses charge current and temperature to turn the fan on or off.
  Fan is on if temperature is above our threshold or current is
  above our threshold. However change in fan state is suppressed
  until 30 seconds have elapsed. Called approximately every second.
-----------------------------------------------------------------*/
static void run_fan( int16_t iTemp, uint16_t uiCurrent )
{
  static volatile uint16_t   iSecondCtr=0;
  uint8_t                    ucOn;	

  if ( iSecondCtr < INT16_MAX )	
    ++iSecondCtr;
	
  if ( iSecondCtr < 30 ) return;	// wait awhile before doing anything

  ucOn = ( test_bit( PINB, PB2 ) ) ? 1 : 0;  // is fan on now?

  if ( ucOn )
  {
	if ( iTemp <= FAN_TEMP_THRESHOLD && uiCurrent < FAN_CURRENT_THRESHOLD )
	{
	  clear_bit( PORTB, PB2 );	// turn the fan relay off
	  iSecondCtr = 0; 
	}
  }
  else
  {
	if ( iTemp > FAN_TEMP_THRESHOLD || uiCurrent > FAN_CURRENT_THRESHOLD )
	{
	  set_bit( PORTB, PB2 );	// turn the fan relay on
	  iSecondCtr = 0;
	}
  }
}

/*-----------------------------------------------------------------
  measure_current()
  
  Measures the output of the ACS712 hall-effect current sensor on
  ADC0. Because the sensor is bipolar, it outputs 1/2 Vcc (2.5V)
  with zero current. To get the best resolution, the internal
  2.56V reference voltage is used. The sensor is wired with the
  plus on the low side and the minus on the high side. The effect
  is that the sensor voltage will drop as the current rises, 
  keeping the sensor voltage below the 2.56V reference.
  
  Converts sensor output to current in mA.
  
  Returns current in milliamps.
-----------------------------------------------------------------*/
static uint16_t measure_current()
{
  uint16_t		uiAdc;
  uint32_t		ulMV, ulMA;
  uint32_t      ulOneHalfVAdcMV;
  uint8_t		ii;
  
  ADMUX  = 0b10000000;		// Vref=2.56V; ADLAR-0; ADC0
  ADCSRA = 0b10000111;		// enable ADC; prescaler=128; disable interrupt
  ADCSRB = 0b00010000;		// bipolar=0, GSEL=0, REFS2=1, MUX5=0

  for ( ii=0; ii<2; ++ii )
  {
    set_bit( ADCSRA, ADSC );				// start conversion
    while ( 1 )
    {
	  if ( !test_bit( ADCSRA, ADSC ) )	// conversion complete
	    break;
    }
  
    uiAdc = (uint16_t) ADCL | ( (uint16_t) ADCH << 8 );
  }

  // The adc output is converted to volts as follows:
  // mV = (2560)(ADC) / 1024
  
  ulMV = ( ( (uint32_t) uiAdc * 2560 ) / 1024L );
  
  // Observation: at 0mA, ulMV is 2500. At 2.5A, 2050.
  // Convert to mV referenced to zero. Keep in mind the hall
  // sensor shares Vcc with the ADC Vcc so we can be a little
  // more accurate if we measure Vcc.
  
  ulOneHalfVAdcMV = (unsigned long) VADC_MEASURED_MV / 2;
  
  if ( ulMV > ulOneHalfVAdcMV )
    ulMV = 0;
  else	
    ulMV = ulOneHalfVAdcMV - ulMV;
	
  // The spec sheet says that the ACS712 outputs 185 mV per Amp
  // (nominal).
  
  ulMA = ( ulMV * 1000L ) / (uint32_t) ACS712_SENS_MV_A;

  return (uint16_t) ulMA;
}

/*-----------------------------------------------------------------
  determine_state(...)
  
  Interpret state information and return a descriptive string.
  
  Returns: 0 = OK, 1 = Fault
           Also returns a short message in buffer.
-----------------------------------------------------------------*/
static uint8_t determine_state( uint8_t ucOC, uint8_t ucPI, uint8_t ucSLC, 
                                uint16_t uiCurrent, uint16_t uiVout_HV, char *buffer )
{
  uint8_t ucFault=0;
  
  *buffer = '\0';     // 20160725jb
  
  if ( !ucPI )
  {
	// As long as the unit is operating PI should be high
	strcpy( buffer, " Fault PI");
	return 1;
  }

  if ( !uiCurrent && !ucSLC )
  {
	strcpy( buffer, " No Batt" );
	return 0;
  }
  
  if ( ucSLC && uiCurrent < FLOAT_CURRENT_MAX_MA )
  {
	// I seen SLC high and current at Imax!
	if ( uiVout_HV < FLOAT_FAILURE_Vout_HUNDREDTHS )
	{
	  // This targets one failure mode I seen where the UC3906 fails
	  // but SLC stays high, making you think that the battery is in
	  // normal float mode but the charger has failed. Eventually the
	  // Battery voltage will drop below Vf.	
	  strcpy( buffer, " Fault VO");
	  ucFault = 1;
	}
	else
	{
	  strcpy( buffer, " Float" );
	  ucBulkOverchgModeEntered = 0;
	}

	return ucFault;  
  }
  
  if ( ucOC )
  {
	if ( uiCurrent > 2300 )
	  strcpy( buffer, "Bulk-Chg" );
	else
	  strcpy( buffer, "Over-Chg" );

    // The following code targets a failure mode I seen when a cell
	// shorts out in the battery, and the charger stays on bulk or
	// overcharge (probably bulk) forever.
	if ( ucBulkOverchgModeEntered )
	{
	  if ( ulBulkOverchgMins > MAX_MINUTES_IN_HIGH_CHARGE )
	  {
	    strcpy( buffer, " Fault TM");
		ucFault = 1;
	  }
	}
	else
	{  
	  ucBulkOverchgModeEntered = 1;
	  ulBulkOverchgMins = 0L;
	}
	
	return ucFault;  
  }

  // Unknown state!
  strcpy( buffer, " O=" );
  strcat( buffer, ( ucOC ) ? "1" : "0" );
  strcat( buffer, " SL=" );
  strcat( buffer, ( ucSLC ) ? "1" : "0" );
  return 1;	
}

/*-----------------------------------------------------------------
  sound_beeper()
  
  Sounds the alarm briefly.
-----------------------------------------------------------------*/
static void sound_beeper()
{
  uint8_t			ii;
  
  set_bit( PORTB, PB0 );        // beeper on
	
  for ( ii=0; ii<4; ++ii )	    // 100 ms
    _delay_loop_2( 50000 );		// 25 ms
	  
  clear_bit( PORTB, PB0 );      // beeper off
  
  for ( ii=0; ii<160; ++ii )	// 4 seconds
    _delay_loop_2( 50000 );		// 25 ms
}

/*-----------------------------------------------------------------
  show_live_indicator()
  
  20160725jb Shows a changing character in row 2 column 16.
-----------------------------------------------------------------*/
static void show_live_indicator()
{
  static volatile uint8_t ndx=0;
  static const char symbols[4] = { '.', 'o', 'O', 'o' };
  char cc;
  
  cc = symbols[ndx];
  lcd_gotoxy( 15, 1 );
  lcd_putc( cc );
  ++ndx;
  if ( ndx > 3 ) ndx = 0;
}

/*-----------------------------------------------------------------
  averager_add(...)

  Add a value to the averager array.
-----------------------------------------------------------------*/
static void averager_add( uint16_t uiValue )
{
  static volatile uint8_t ucFirst=1;
  uint8_t		ii, jj;
	
  if ( ucFirst )	
  {
	// Initialize the array with the 1st value  
	for ( ii=0; ii<AVERAGER_ENTRIES; ++ii )
	  uiAverager_array[ii] = uiValue;
	ucFirst = 0;  
  }

  // Move entries down one position, discarding the last entry
  ii = AVERAGER_ENTRIES - 2;
  jj = AVERAGER_ENTRIES - 1;
  for ( ; jj > 0; --ii, --jj )
    uiAverager_array[jj] = uiAverager_array[ii];

  uiAverager_array[0] = uiValue;	
}

/*-----------------------------------------------------------------
  averager_get()

  Get a value from the averager array.
-----------------------------------------------------------------*/
static uint16_t averager_get()
{
  uint8_t		ii;
  uint16_t		uiValue=0;
  
  for ( ii=0; ii<AVERAGER_ENTRIES; ++ii )
    uiValue += uiAverager_array[ii];
	
  return ( uiValue / AVERAGER_ENTRIES );
}
