/*-----------------------------------------------------------------
  LED_Milliohmmeter Copyright (c) 2008 by Jack Botner

  main.c

  This program implements a milliohmmeter using a 4 digit LED display.
  The principle is to have a constant current source providing
  approxamately 250ma passing through the unknown resistor, and 
  measuring the voltage dropped with the A/D converter. Since we're
  dealing with small voltages, we use the Vref=1.1V feature of the
  microprocessor.

  Processor: ATMega168
      Clock: Internal, 8.0 MHz
-----------------------------------------------------------------*/
#include <avr/io.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdlib.h>
#include <string.h>
#include "common.h"
#include "led.h"

#define CURRENT_MA		250			// current in R unknown (measured)
#define VREF_MV			1100		// Vref (measured)
//#define CURRENT_MA		259			// current in R unknown (measured)
//#define VREF_MV			1088		// Vref (measured)

#define EVENT_NONE			0
#define EVENT_DISPLAY_DATA	1
#define EVENT_LED_REFRESH	2
#define EVENT_DEBOUNCE		3
#define EVENT_BUTTON_PRESS	4
#define EVENT_SORT_ARRAY	5

#define DIG_SAMPLES		15
#define DIG_ADD			0
#define DIG_GET			1
#define DIG_SORT		2

// Global variables

static volatile uint8_t ucLed[4] = { 0, 0, 0, 0 };

//static volatile uint16_t uiLastAdc = 0;
static volatile uint16_t uiAdcOffset;

static volatile uint8_t ucEventFlags = EVENT_NONE;

// Internal Functions

static int Compare( const void *pElement1, const void *pElement2 );
static void convert_display( uint16_t uiAdc );
static uint16_t digital_filter( uint8_t ucRequest, uint16_t uiSample );
static void system_init(void);

/*-----------------------------------------------------------------
  main()
-----------------------------------------------------------------*/
int main( void )
{
  static volatile uint8_t ucIndex=0;	// digit selection index
  static volatile uint8_t ucPress=0;	// button press counter
  uint8_t	ucNr;
  uint16_t	uiAdc;

  system_init();                		// initialise system

  // Get previous ADC offset from eeprom
  uiAdcOffset = eeprom_read_word( 0 );
  if ( uiAdcOffset > 199 )				// probably not initialized
  {
    uiAdcOffset = 0;
	eeprom_write_word( 0, uiAdcOffset );
  }

  while (1)
  {
	if ( test_bit( ucEventFlags, EVENT_SORT_ARRAY ) )
	{
	  digital_filter( DIG_SORT, 0 );
	  clear_bit( ucEventFlags, EVENT_SORT_ARRAY );
	}

	if ( test_bit( ucEventFlags, EVENT_LED_REFRESH ) )
	{
	  ucNr = ucLed[ucIndex];
	  ledOutputNumber( ucNr );

	  ledSelectDigit( ucIndex );

	  ++ucIndex;
	  if ( ucIndex > 3 )
	    ucIndex = 0;  

	  clear_bit( ucEventFlags, EVENT_LED_REFRESH );
	}

	if ( test_bit( ucEventFlags, EVENT_DEBOUNCE ) )
	{
	  if ( !test_bit( PINB, PB0 ) )
	  {
	    ++ucPress;
		if ( ucPress == 8 )		// ~20ms
		{
		  set_bit( ucEventFlags, EVENT_BUTTON_PRESS );
		}
	  }
	  else
	    ucPress = 0;

	  clear_bit( ucEventFlags, EVENT_DEBOUNCE );
	}

	if ( test_bit( ucEventFlags, EVENT_DISPLAY_DATA ) )
	{
	  uiAdc = digital_filter( DIG_GET, 0 );
	  convert_display( uiAdc );
	  clear_bit( ucEventFlags, EVENT_DISPLAY_DATA );
	}

	if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESS ) )
	{
	  // In response to a button press we want to validate the
	  // ADC reading and update the ADC offset, which will be
	  // subtracted from subsequent calculations.
	  uiAdc = digital_filter( DIG_GET, 0 );
	  if ( uiAdc < 100 )
	  {
	    uiAdcOffset = uiAdc;
		eeprom_write_word( 0, uiAdcOffset );
	  }

	  clear_bit( ucEventFlags, EVENT_BUTTON_PRESS );
	}
  }
}

/*-----------------------------------------------------------------
  ISR(TIMER2_OVF_vect)

  Timer 2 overflow interrupt handler

  Since the clock is running at 8.0 mHz, the counter increments
  every 125 ns. The counter overflows after 256 counts, so an
  interrupt would occur every 32.0 us. We specify a prescaler of
  128, so the interrupt actually occurs every 4.096 ms.
  This is a good interval to use to switch LED digits.
-----------------------------------------------------------------*/
ISR(TIMER2_OVF_vect)
{
  static volatile uint8_t ucCtr1=0;		// interrupt counter
  static volatile uint8_t ucCtr2=0;		// interrupt counter

  // Signal an LED refresh event
  set_bit( ucEventFlags, EVENT_LED_REFRESH );

  // Signal a button debounce event
  set_bit( ucEventFlags, EVENT_DEBOUNCE );

  // Every 4 interrupts (~16 ms) start an ADC conversion
  // FYI, the adc clock is running at 125kHz (8 us) and
  // a conversion takes 13 clock cycles (104 us).
  ++ucCtr1;
  if ( ucCtr1 == 4 )
  {
  	set_bit( ADCSRA, ADSC );	// start an ADC conversion
	ucCtr1 = 0;
  }

  // Every 244 interrupts (~1 second) post an event
  // to update the display
  ++ucCtr2;
  if ( ucCtr2 == 244 )
  {
  	set_bit( ucEventFlags, EVENT_DISPLAY_DATA );
	ucCtr2 = 0;
  }
}

/*-----------------------------------------------------------------
  ISR( ADC_vect ) - ADC conversion complete interrupt handler

  ADC0 is monitoring the voltage across the unknown resistor.
-----------------------------------------------------------------*/
ISR( ADC_vect )
{
  uint16_t	uiAdc;

  // ADCL must be read before ADCH.
  uiAdc = (uint16_t) ADCL | ( (uint16_t) ADCH << 8 );

  digital_filter( DIG_ADD, uiAdc );
}

/*-----------------------------------------------------------------
  system_init()

  PORTB: PB0 (input) "zero" pushbutton
         PB1 (output) LED DP2
         PB2 (output) LED DP1
         PB3 (output) MOSI ISP
         PB4 (output) MISO ISP
         PB5 (output) SCK ISP
         PB6 (output) not used
         PB7 (output) not used

  PORTC: PC0 (input) ADC0 Rx
         PC1 (output) LED Digit 1 select
         PC2 (output) LED Digit 2 select
         PC3 (output) LED Digit 3 select
         PC4 (output) LED Digit 4 select
         PC5 (output) diagnostic LED
         PC6 (input) !Reset

  PORTD: PD0 (output) LED A segment
         PD1 (output) LED B segment
         PD2 (output) LED C segment
         PD3 (output) LED D segment
         PD4 (output) LED E segment
         PD5 (output) LED F segment
         PD6 (output) LED G segment
         PD7 (output) not used
-----------------------------------------------------------------*/
static void system_init()
{
  // The fuses are set to provide an internal clock, running at
  // 8.0 MHz with a prescaler of 8. Change the prescaler from
  // 8 to 1. The procedure requires two steps as per the device
  // specifications.
  CLKPR = 0b10000000;       // set clock prescaler change enable
  CLKPR = 0b00000000;       // set prescaler to 1

  // Set up the I/O ports directions, pull-ups
  DDRB  = 0b11111110;       // PortB0=input, 1..7 = output
  PORTB = 0b00000001;       // initialise PortB, PB0=pullup

  DDRC  = 0b01111110;       // PortC0 = input, C1..6 = output, C7 = input
  PORTC = 0b00000000;       // initialise PortC7

  DDRD  = 0b11111111;       // PortD0..7 = output
  PORTD = 0b00000000;       // initialise PortD7

  // Set up Timer/Counter2 to produce an interrupt every 4 ms.
  TCCR2A = 0b00000000;		// normal operation
  TCCR2B = 0b00000101;		// use prescaler=128
  TCNT2 = 0;
  TIMSK2 = 0b00000001;		// enable interrupt on overflow	

  // Set up PC0/ADC0 as analog input, Vref=1.1V
  ADMUX  = 0b11000000;		// Vref=1.1V; result right adjusted; select ADC0
  ADCSRA = 0b10001110;		// Enable ADC; Enable interrupt; prescaler = 64
  ADCSRB = 0b00000000;		// Auto trigger source = none
  DIDR0  = 0b00000001;		// Disable PC0 digital input buffer

  PRR = 0b10000110;			// power save: no TWI, no SPI, no USART0
  sei();					// enable global interrupts
}

/*-----------------------------------------------------------------
  convert_display(...)

  Input: current A/D reading

  Output: ucLed[0..3]
-----------------------------------------------------------------*/
static void convert_display( uint16_t uiAdc )
{
  uint16_t	uiRes, uiNr, uiTemp, uiMV;
  uint8_t	ucNonZeroIndex=0;

  if ( uiAdc == 1023 )
  {
	// Probably there is no resistor in the test socket. Show zero.
	ucLed[0] = 0x0f;
	ucLed[1] = 0x0f;
	ucLed[2] = 0x0f;
	ucLed[3] = 0x00;
	return;
  }

  // The ADC measures a higher voltage than actually exists
  // across the unknown resistor, caused mainly by resistence
  // of wiring and connections inside the unit. 
  if ( uiAdc > uiAdcOffset )
    uiAdc -= uiAdcOffset;
  else
    uiAdc = 0;

  // Convert the A/D output to millivolts.
  // A/D is in the range 0..1023
  // Vref is 1100 mV

  uiMV = (uint16_t) ( ( (uint32_t) uiAdc * VREF_MV ) / 1024 );

  // Convert millivolts to milliohms

  uiRes = (uint16_t) ( (uint32_t) uiMV * 1000L / CURRENT_MA );
  uiTemp = uiRes;

  uiNr = uiTemp / 1000;
  ucLed[0] = (uint8_t) uiNr;
  if ( ucLed[0] )
    ucNonZeroIndex = 1;
  else
	ucLed[0] = 0x0f;			// suppress leading zero
  uiTemp -= ( uiNr * 1000 );

  uiNr = uiTemp / 100;
  ucLed[1] = (uint8_t) uiNr;
  if ( !ucNonZeroIndex )
  {
	if ( ucLed[1] )
      ucNonZeroIndex = 1;
	else
	  ucLed[1] = 0x0f;			// suppress leading zero
  }
  uiTemp -= ( uiNr * 100 );

  uiNr = uiTemp / 10;
  ucLed[2] = (uint8_t) uiNr;
  if ( !ucNonZeroIndex )
  {
	if ( ucLed[2] )
      ucNonZeroIndex = 1;
	else
	  ucLed[2] = 0x0f;			// suppress leading zero
  }
  uiTemp -= ( uiNr * 10 );

  ucLed[3] = (uint8_t) uiTemp;
}

/*-----------------------------------------------------------------
  digital_filter(...)

  ucRequest = DIG_ADD

	Set ucRequest to 0 and pass the sample in uiSample.

  ucRequest = DIG_SORT

	Pass 2 in ucRequest and the array is sorted.

  ucRequest = DIG_GET

	Pass 1 in ucRequest and the median sample is returned.
-----------------------------------------------------------------*/
static uint16_t digital_filter( uint8_t ucRequest, uint16_t uiSample )
{
  static volatile uint16_t uiSamples[DIG_SAMPLES];
  static volatile uint8_t  ucFirst=1;
  uint8_t	ii;
  uint16_t	uiRet;

  if ( ucRequest == DIG_ADD )
  {
	if ( ucFirst )
	{
	  // Initialize the table with first value
	  for ( ii=0; ii<DIG_SAMPLES; ++ii )
	    uiSamples[ii] = uiSample;
	  ucFirst = 0;
	  return 0;
	}

	// A median filtering method is used, in which the new sample
	// is placed in the first and last positions of the array, and
	// the array is sorted in ascending order. Since this code is
	// running from an interrupt handler, schedule the sort later.

    uiSamples[0] = uiSample;
    uiSamples[DIG_SAMPLES-1] = uiSample;
	set_bit( ucEventFlags, EVENT_SORT_ARRAY );
	return 0;
  }

  if ( ucRequest == DIG_GET )
  {
  	// Return the median element
	uiRet = uiSamples[DIG_SAMPLES/2];
	return uiRet;
  }

  if ( ucRequest == DIG_SORT )
	qsort( (void *) &uiSamples, DIG_SAMPLES, sizeof( uint16_t ), Compare );

  return 0;
}

/*--------------------------------------------------------------------------
  Compare(...)
--------------------------------------------------------------------------*/
static int Compare( const void *pElement1, const void *pElement2 )
{
  uint16_t *pui1, *pui2;

  pui1 = (uint16_t *) pElement1;
  pui2 = (uint16_t *) pElement2;

  if ( *pui1 > *pui2 ) return 1;
  if ( *pui1 < *pui2 ) return -1;
  return 0;
}
