/*-----------------------------------------------------------------
  All code is provided for EDUCATIONAL purposes ONLY. There is no
  warranty and no support.

  Copyright (c) Jack Botner and licensed under the MIT license; see
  http://opensource.org/licenses/mit-license.php

  ICOM commander by Jack Botner

  This program implements 4 pushbuttons to send 4 commands to an
  ICOM radio CIV port. The coomands are
  - set power to 10 watts
  - set power to 100 watts
  - select paddle
  - select keyer

  Processor: ATtiny84A
      Clock: External, 8.0 MHz

  20201225 program written
-----------------------------------------------------------------*/
#define F_CPU 8000400UL			// 8 MHz
#include <stdint.h>
#include <avr/io.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <util/delay_basic.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"

// Global Variables

static volatile unsigned char ucEventFlags=0;		 // program event flags

static unsigned char ucDebouncedState=0;    // debounced state of the switches
static unsigned char ucState[MAX_CHECKS] =  // array containing bounce status
{ 0, 0, 0, 0, 0, 0, 0, 0 };
static unsigned char ucIndex=0;             // index into ucState

// Local functions

static void serial_send_cmd( unsigned char ucIndex );
static void system_init(void);

/*-----------------------------------------------------------------
  main()
-----------------------------------------------------------------*/
int main(void)
{
  int           ii;
  unsigned char ucSwIn;
  unsigned char ucResult;

  system_init();                	// initialize system

  while (1)
  {
    if ( test_bit( ucEventFlags, EVENT_1MS ) )
    {
      // We use the technique documented in "A Guide to Debouncing" by
      // Jack Ganssle to debounce up to 8 inputs (4 in this case).

      // Read current switch states
      ucSwIn = 0;
      if ( !test_bit( PINA, PA0 ) )
        set_bit( ucSwIn, SW1 );
      if ( !test_bit( PINA, PA1 ) )
        set_bit( ucSwIn, SW2 );
      if ( !test_bit( PINA, PA2 ) )
        set_bit( ucSwIn, SW3 );
      if ( !test_bit( PINA, PA3 ) )
        set_bit( ucSwIn, SW4 );

      ucState[ucIndex] = ucSwIn;

      ++ucIndex;
      if ( ucIndex == MAX_CHECKS )
        ucIndex = 0;

      // And all columns of the array
      for ( ii=0, ucResult=0xff; ii<(MAX_CHECKS-1); ++ii )
        ucResult = ucResult & ucState[ii];

      ucDebouncedState = ucResult;

      if ( test_bit( ucDebouncedState, SW1 ) )
        set_bit( ucEventFlags, EVENT_SW1 );
      if ( test_bit( ucDebouncedState, SW2 ) )
        set_bit( ucEventFlags, EVENT_SW2 );
      if ( test_bit( ucDebouncedState, SW3 ) )
        set_bit( ucEventFlags, EVENT_SW3 );
      if ( test_bit( ucDebouncedState, SW4 ) )
        set_bit( ucEventFlags, EVENT_SW4 );

      clear_bit( ucEventFlags, EVENT_1MS );
    }

    if ( test_bit( ucEventFlags, EVENT_SW1 ) )
    {
      serial_send_cmd( 0 );
      for ( ii=0; ii<20; ++ii )
        _delay_loop_2( 0 );          // 32.7ms delay
      clear_bit( ucEventFlags, EVENT_SW1 );
    }

    if ( test_bit( ucEventFlags, EVENT_SW2 ) )
    {
      serial_send_cmd( 1 );
      for ( ii=0; ii<20; ++ii )
        _delay_loop_2( 0 );          // 32.7ms delay
      clear_bit( ucEventFlags, EVENT_SW2 );
    }

    if ( test_bit( ucEventFlags, EVENT_SW3 ) )
    {
      serial_send_cmd( 2 );
      for ( ii=0; ii<20; ++ii )
        _delay_loop_2( 0 );          // 32.7ms delay
      clear_bit( ucEventFlags, EVENT_SW3 );
    }

    if ( test_bit( ucEventFlags, EVENT_SW4 ) )
    {
      serial_send_cmd( 3 );
      for ( ii=0; ii<20; ++ii )
        _delay_loop_2( 0 );          // 32.7ms delay
      clear_bit( ucEventFlags, EVENT_SW4 );
    }
  }
}

/*-----------------------------------------------------------------
  system_init()

  PORTA: PA0 (input)  SW1
		 PA1 (input)  SW2
		 PA2 (input)  SW3
		 PA3 (input)  SW4
		 PA4 (input)  programming
		 PA5 (input)  programming
		 PA6 (input)  programming
		 PA7 (output) CIV Output

  PORTB: PB0 (input)  xtal
         PB1 (input)  xtal
         PB2 (output) LED
         PB3 (input)  programming
----------------------------------------------------------------*/
static void system_init()
{
  // Set up the I/O ports directions, pull-ups
  DDRA  = 0b10001000;
  PORTA = 0b01111111;       // initialize PortA

  DDRB  = 0b00000100;
  PORTB = 0b00001100;       // initialize PortB

  // Initialize Timer1.
  // Timer1 is used to generate interrupts every 1 millisecond for switch debouncing.
  TCCR1A = 0b00000000;		  // timer used as counter only
  TCCR1B = 0b00000001;        // internal clock, no prescaler
  TCNT1 = INIT_TIMER1;
  set_bit( TIMSK1, TOIE1 );	  // enable timer/counter1 overflow interrupt

  PRR = 0b00000011;	         // power save: no USI, no ADC

  UART_init();               // note this enables interrupts
}

/*-----------------------------------------------------------------
  ISR(TIMER1_OVF_vect)

  Timer 1 overflow interrupt handler - to test for button presses

  System clock frequency = 8 mHz
  System clock period = 125 ns
  Prescaler = 1
  Counts = 8000
  Interrupt period = 1 ms
------------------------------------------------------------------*/
ISR(TIM1_OVF_vect)
{
  set_bit( ucEventFlags, EVENT_1MS );

  TCNT1 = INIT_TIMER1;
}

/*-----------------------------------------------------------------
  serial_send_cmd(...)

  Selects a command from the menu index and sends it out.
-----------------------------------------------------------------*/
static void serial_send_cmd( unsigned char ucIndex )
{
  static const unsigned char pucCIV_10W[] =
  { 0xfe, 0xfe, TO_ADDRESS, FROM_ADDRESS, 0x14, 0x0a, 0x00, 0x28, 0xfd };
  static const unsigned char pucCIV_100W[] =
  { 0xfe, 0xfe, TO_ADDRESS, FROM_ADDRESS, 0x14, 0x0a, 0x02, 0x55, 0xfd };
  static const unsigned char pucCIV_keyer[] =
  { 0xfe, 0xfe, TO_ADDRESS, FROM_ADDRESS, 0x1a, 0x05, 0x01, 0x64, 0x02, 0xfd };
  static const unsigned char pucCIV_SK[] =
  { 0xfe, 0xfe, TO_ADDRESS, FROM_ADDRESS, 0x1a, 0x05, 0x01, 0x64, 0x00, 0xfd };

  unsigned char *ptr;

  clear_bit( LED_PORT, LED_PIN );  // make the LED wink

  switch ( ucIndex )
  {
    case 0:      // 10W
      ptr = (unsigned char *) pucCIV_10W;
      UART_tx_data( ptr, 9 );
      break;
    case 1:      // 100W
      ptr = (unsigned char *) pucCIV_100W;
      UART_tx_data( ptr, 9 );
      break;
    case 2:      // keyer
      ptr = (unsigned char *) pucCIV_keyer;
      UART_tx_data( ptr, 10 );
      break;
    case 3:      // straight key
      ptr = (unsigned char *) pucCIV_SK;
      UART_tx_data( ptr, 10 );
      break;
  }

  set_bit( LED_PORT, LED_PIN );
}
