/*  Software ("bit-bang") UART Transmitter (8 data bits, 1 stop bit, no parity)
    for Attiny24A/44A/84A using the internal 8MHz oscillator as clock source
    (c) 2018 Marcel Meyer-Garcia
    see LICENCE.txt

    20201230JB Added UART_tx_data function to make it easier to send a block
               of data which may contain nulls.
    20201231JB Adjust initialization value of OCR0A to fine tune baud rate,
               since I'm using 8MHz crystal oscillator.
 */

 /* NOTE: since the internal 8MHz oscillator is not very accurate, the value for OCR0A can be tuned
    to achieve the desired baud rate (nominal value is 103)
 */

#include <avr/io.h>
#include <avr/interrupt.h>
#define F_CPU 8000400UL			// 8 MHz
#include <util/delay.h>

// change these to use another pin
#define TX_PORT PORTA
#define TX_PIN  PA7
#define TX_DDR  DDRA
#define TX_DDR_PIN DDA7

static volatile uint16_t tx_shift_reg = 0;


void UART_tx(unsigned char character)
{
   uint16_t local_tx_shift_reg = tx_shift_reg;

   //if sending the previous character is not yet finished, return
   //transmission is finished when tx_shift_reg == 0
   if(local_tx_shift_reg) return;

   //fill the TX shift register with the character to be sent and the start & stop bits (start bit (1<<0) is already 0)
   local_tx_shift_reg = ( (uint16_t) character<<1) | (1<<9); //stop bit (1<<9)
   tx_shift_reg = local_tx_shift_reg;

   //start timer0 with a prescaler of 8
   TCCR0B = (1<<CS01);
}

void UART_tx_str(char* string)
{
    while( *string )
    {
        UART_tx( *string++ );
        //wait until transmission is finished
        while(tx_shift_reg);
    }
}

void UART_tx_data(unsigned char* data, char length)
{
    int ii;

    for ( ii=0; ii<(int) length; ++ii )
    {
        //UART_tx( *data++ );
        UART_tx( data[ii] );
        //wait until transmission is finished
        while(tx_shift_reg);
    }
}

void UART_init()
{
   //set TX pin as output
   TX_DDR |= (1<<TX_DDR_PIN);
   TX_PORT |= (1<<TX_PIN);
   //set timer0 to CTC mode
   TCCR0A = (1<<WGM01);
   //enable output compare 0 A interrupt
   TIMSK0 |= (1<<OCF0A);
   //set compare value to 103 to achieve a 9600 baud rate (i.e. 104µs)
   //together with the 8MHz/8=1MHz timer0 clock
   /*NOTE: since the internal 8MHz oscillator is not very accurate, this value can be tuned
     to achieve the desired baud rate, so if it doesn't work with the nominal value (103), try
     increasing or decreasing the value by 1 or 2 */
   OCR0A = 105;
   //enable interrupts
   sei();
}

#if 0
int main(void)
{
   UART_init();

   while(1)
   {
    UART_tx_str("Hello world!\n");
    _delay_ms(100);
   }

   return 0;
}
#endif

//timer0 compare A match interrupt
ISR(TIM0_COMPA_vect )
{
   uint16_t local_tx_shift_reg = tx_shift_reg;
   //output LSB of the TX shift register at the TX pin
   if( local_tx_shift_reg & 0x01 )
   {
      TX_PORT |= (1<<TX_PIN);
   }
   else
   {
      TX_PORT &=~ (1<<TX_PIN);
   }
   //shift the TX shift register one bit to the right
   local_tx_shift_reg >>= 1;
   tx_shift_reg = local_tx_shift_reg;
   //if the stop bit has been sent, the shift register will be 0
   //and the transmission is completed, so we can stop & reset timer0
   if(!local_tx_shift_reg)
   {
      TCCR0B = 0;
      TCNT0 = 0;
   }
}