/*-----------------------------------------------------------------
  Electric Blanket Controller  by Jack Botner

  main.c

  This program implements an electric blanket controller using a
  2 digit LED display and a rotary encoder. PWM is used to control
  a IRL640 power FET. The blanket's thermistor is sampled via ADC0.

  Processor: ATtiny461
      Clock: Internal, 4.0 MHz

  jb041210 - Added watchdog timer support.
-----------------------------------------------------------------*/
#include <avr/io.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>
#include <string.h>
#include <stdlib.h>
#include "main.h"

// Global variables

volatile uint8_t ucPowerState=1;			// power: 0=off, 1=on
volatile uint8_t ucDebouncedState;			// debounced state of the switches
volatile uint8_t ucPreviousState;			// previous switch state
volatile uint8_t ucEventFlags;				// event flags
volatile uint16_t uiEepromDelay=0;			// delay updating eeprom factor
volatile uint16_t uiAdc=0;					// ADC sample
volatile uint16_t uiAdcDelay=ADC_DELAY_2MS;	// time between adc samples

volatile char cDisplay[2] = { 0, 0 }; 		// led digits to display
static volatile uint8_t ucHeatSetting=0;	// blanket heat setting 0..99

// Internal Functions

static uint16_t adc_to_resistance( void );
static void read_eeprom(void);
static void setting_to_display(void);
static void system_init(void);
static void update_eeprom(void);

/*-----------------------------------------------------------------
  main()

  After initialization, the mainline processes events scheduled
  elsewhere in the program.
-----------------------------------------------------------------*/
int main(void)
{
  static volatile uint8_t ucFirst=1;		// encoder state managment
  uint16_t	uiOhms=0;						// thermistor resistance
  uint8_t	ucPwmCtr=99;

  system_init();			// initialize system

  read_eeprom();			// retrieve saved variables

  setting_to_display();		// initialize cDisplay

  set_bit( ADCSRA, ADSC );	// start first adc conversion

  wdt_enable( WDTO_60MS );

  while( 1 )				// do forever
  {
	if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
	{
	  if ( ucPowerState )
	  {
		ucPowerState = 0;			// power down
		ucPwmCtr = 99;				// stop the heat
	  	uiEepromDelay = EEPROM_DELAY_2MS;
	  }
	  else
	  {
		ucPowerState = 1;			// power up
	  	uiEepromDelay = EEPROM_DELAY_2MS;
		ucPwmCtr = 0;
		set_bit( ADCSRA, ADSC );	// start an adc conversion
		uiAdcDelay = ADC_DELAY_2MS;
	  }

	  clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
	}

	if ( test_bit( ucEventFlags, EVENT_ENCODER_UP ) )
	{
	  if ( ucPowerState )
	  {
		if ( ucHeatSetting < 98 )
		{
	      ucHeatSetting += 2;
		  uiEepromDelay = EEPROM_DELAY_2MS;
		}

		setting_to_display();
	  }

	  clear_bit( ucEventFlags, EVENT_ENCODER_UP );
	}

	if ( test_bit( ucEventFlags, EVENT_ENCODER_DOWN ) )
	{
	  if ( ucPowerState )
	  {
	    if ( ucHeatSetting > 1 )
	    {
	      ucHeatSetting -= 2;
	  	  uiEepromDelay = EEPROM_DELAY_2MS;
	    }

	    setting_to_display();
	  }

	  clear_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	}

	if ( test_bit( ucEventFlags, EVENT_UPDATE_EEPROM ) )
	{
	  update_eeprom();
	  clear_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
	}

	if ( test_bit( ucEventFlags, EVENT_ADC_COMPLETE ) )
	{
	  uiOhms = adc_to_resistance();
	  clear_bit( ucEventFlags, EVENT_ADC_COMPLETE );
	}

	if ( test_bit( ucEventFlags, EVENT_REFRESH_LED ) )
	{
	  // This event occurs every 4ms.
	  ledActivateNext();		// output the next LED digit

	  clear_bit( ucEventFlags, EVENT_REFRESH_LED );
	}

	if ( test_bit( ucEventFlags, EVENT_DEBOUNCE_IO ) )
	{
	  // Process the rotary encoder

	  debounce_switch();

	  // The 1st time only, set the previous switch state to the
	  // current switch state.
	  if ( ucFirst )
	  {
	    ucPreviousState = ucDebouncedState;
		ucFirst = 0;
	  }

	  if ( ucDebouncedState != ucPreviousState )
	  {
	    analyze_switch();           // convert switch change to event
		ucPreviousState = ucDebouncedState;
	  }

	  clear_bit( ucEventFlags, EVENT_DEBOUNCE_IO );
	}

	if ( test_bit( ucEventFlags, EVENT_10MS_PWM ) )
	{
	  // Every 10 ms we need to decide if the heat i/o should
	  // be high or low. Since ucHeatSetting is in the range 0..99,
	  // a counter is used and compared to the heat setting. If less,
	  // heat is turned on, otherwise off.
	  if ( ucHeatSetting > ucPwmCtr )
	  {
	    // heat should be on
		if ( !test_bit( PORTB, PB3 ) )
		  set_bit( PORTB, PB3 );
	  }
	  else
	  {
		// heat should be off
		if ( test_bit( PORTB, PB3 ) )
		  clear_bit( PORTB, PB3 );
	  }
	  	  	
	  if ( ucPowerState )
	  {
		++ucPwmCtr;
		if ( ucPwmCtr > 98 )
	      ucPwmCtr = 0;
	  }

	  clear_bit( ucEventFlags, EVENT_10MS_PWM );

	  wdt_reset();
	}
  }
}

/*-----------------------------------------------------------------
  system_init()

  PORTB: PB0 (output) LED DIG1
         PB1 (output) LED DIG2
         PB2 (input) (test jumper)
         PB3 (output) PWM
         PB4 (input) Rotary Encoder Output B *
         PB5 (input) Rotary Encoder Output A *
         PB6 (input) Rotary Encoder Pushbutton *
         PB7 (n/a) !Reset (ISP)

  PORTA: PA0 (input) ADC0 Thermistor sensor
         PA1 (output) LED 'A' segments
         PA2 (output) LED 'B' segments
         PA3 (output) LED 'C' segments
         PA4 (output) LED 'D' segments
         PA5 (output) LED 'E' segments
         PA6 (output) LED 'F' segments 
         PA7 (output) LED 'G' segments

  * = pull-up required
-----------------------------------------------------------------*/
static void system_init()
{
  // The fuses are set to provide an internal clock, running at
  // 8.0 MHz with a prescaler of 8. Change the prescaler from
  // 8 to 2. The procedure requires two steps as per the device
  // specifications.
  CLKPR = 0b10000000;       // set clock prescaler change enable
  CLKPR = 0b00000001;       // set prescaler to 2

  DDRB  = 0b00001011;       // PB0..3 = output, PB4..7 = input
  PORTB = 0b01110111;       // initialise PortB, PB2, PB4..6 = pull-up

  DDRA  = 0b11111110;       // PA0 = input, PA1..7 = output
  PORTA = 0b11111110;       // initialise PortA

  // Set up PA0/ADC0 as analog input, Vref=Vdd
  ADMUX  = 0b00000000;		// Vref=Vdd; result right adjusted; ADC0/no gain
  ADCSRA = 0b10001101;		// Enable ADC; Enable interrupt; prescaler = 32
  ADCSRB = 0b00000000;		// free running mode
  DIDR0  = 0b00000001;		// disable digital I/O on PA0

  // Initialize Timer1. Timer1 is used to generate interrupts
  // every 2 milliseconds for switch debouncing.
  TCNT1 = INIT_TIMER1;
  TC1H = 0;
  TCCR1A = 0b00000000;          // normal counter operation
  TCCR1B = 0b00000110;          // normal operation, prescaler=32
  set_bit( TIMSK, TOIE1 );		// enable timer/counter1 overflow interrupt

  sei();                    // enable global interrupts
}

/*-----------------------------------------------------------------
  setting_to_display()

   Input: ucHeatSetting (binary, 0..99)

  Output: cDisplay[2] (2 digits ascii)
-----------------------------------------------------------------*/
static void setting_to_display()
{
  char	cBuf[4];

  utoa( (unsigned int) ucHeatSetting, cBuf, 10 );

  if ( ucHeatSetting < 10 )
  {
	cDisplay[0] = 0x0f;		// digit off
	cDisplay[1] = cBuf[0];
  }
  else
  {
	cDisplay[0] = cBuf[0];
	cDisplay[1] = cBuf[1];
  }
}

/*-----------------------------------------------------------------
  adc_to_resistance()

    Input: uiAdc

  Returns: resistance of the thermistor in ohms
-----------------------------------------------------------------*/
static uint16_t adc_to_resistance()
{
  uint16_t	uimV, uiOhms, uiuA;

  // The thermistor is connected from Vdd to PA0/ADC0, and a 1K
  // resistor is connected from PA0/ADC0 to ground. (The resistance
  // of the thermistor at room temperature is around 970 ohms, rising
  // to around 1050 ohms at warm operating temperature.)
  // Vdd is in this case around 4.85 volts, or 4850 mV.
  // First, convert the adc reading to mV.

  uimV = (uint16_t) ( VDD_MV * (uint32_t) uiAdc ) / 1024;

  // Calculate the current in uA

  uiuA = (uint16_t) ( ( (uint32_t) uimV * 1000L ) / R8_OHMS );

  // Calculate the thermistor resistance

  uiOhms = (uint16_t) ( ( ( (uint32_t) VDD_MV - (uint32_t) uimV ) * 1000L ) / uiuA );

  return uiOhms;
}

/*-----------------------------------------------------------------
  read_eeprom()

  Reads our saved state variables from eeprom.
-----------------------------------------------------------------*/
static void read_eeprom()
{
  // Get the previous power setting
  ucPowerState = eeprom_read_byte( (uint8_t *) 0 );
  if ( ucPowerState > 1 )				// EEPROM not initialized?
  {
    ucPowerState = 0;					// initially off
	set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
  }

  // Get the previous heat setting
  ucHeatSetting = eeprom_read_byte( (uint8_t *) 1 );
  if ( ucHeatSetting > 99 )				// EEPROM not initialized?
  {
    ucHeatSetting = 0;
	set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
  }
}

/*-----------------------------------------------------------------
  update_eeprom()

  Updates our saved state variables in eeprom.
-----------------------------------------------------------------*/
static void update_eeprom()
{
  eeprom_write_byte( (uint8_t *) 0, ucPowerState );
  eeprom_write_byte( (uint8_t *) 1, ucHeatSetting );
}
