/*--------------------------------------------------------------------
  led.c  by Jack Botner
--------------------------------------------------------------------*/
#include <avr/io.h>
#include "main.h"

extern volatile uint8_t ucPowerState;		// power: 0=off, 1=on
extern volatile char cDisplay[2]; 			// led digits to display

/*---------------------------------------------------------------------
  ledActivateNext()

  This function gets called every 4 ms by the mainline. The purpose is
  to switch off the current LED digit and activate the next digit.

  This function is also used to manage the brightness of the LED
  display. If we just switch from digit 1 to digit 2 and back, the
  display is too bright, even with a large (1K) resistor in series
  with each segment. One way to manage the brightness is to have each
  digit on less than 50% of the time. This is done by imagining one or
  more phantom digits, and allocating time slices to each of them.
  The number of phantom digits ought to determine the brightness.
---------------------------------------------------------------------*/
void ledActivateNext()
{
  static volatile uint8_t ucLedIndex=0;		// to select the led digit

  ledSelectDigit( NO_DIGIT );               // turn all digits off

  if ( ucPowerState )
  {
	if ( ucLedIndex < NUMBER_REAL_DIGITS )
	{
	  ledOutputNumber( cDisplay[ucLedIndex] );	// output the number

	  ledSelectDigit( ucLedIndex );				// select the digit
	}


	++ucLedIndex;
	if ( ucLedIndex == TOTAL_DIGITS ) ucLedIndex = 0;
  }
}

/*---------------------------------------------------------------------
  ledSelectDigit(...)

  Selects a digit of the led display. Ports PB0 and PB1 drive each
  digit of the LED display, where PB0 drives the high order digit and
  PB1 drives the low order digit. Clearing the port selects the 
  corresponding digit on.

  Input: a single numeric digit, 0 or 1. 0 drives the high order
         digit, 1 drives the low order digit. To turn off all digits,
         pass NO_DIGIT.
---------------------------------------------------------------------*/
void ledSelectDigit( char cDigit )
{
  switch ( cDigit )
  {
    case 0:
      clear_bit( PORTB, PB0 );
      break;
    case 1:
      clear_bit( PORTB, PB1 );
      break;
    case NO_DIGIT:
      set_bit( PORTB, PB0 );
      set_bit( PORTB, PB1 );
  }
}

/*---------------------------------------------------------------------
  ledOutputNumber(...)

  Outputs a number to the led display. Ports PA1..PA7 are connected
  to the A..G segments of the LED display. Clearing the port turns
  the corresponding segment on.

  Input: a single numeric digit, range 0..9 (ascii or binary, only the
         low-order nibble is processed.) Also the value of x'0f'
		 which means blank the display.
---------------------------------------------------------------------*/
void ledOutputNumber( char cNumber )
{
  // Segment A = bit 0, and so on, for each digit 0..9
  static const uint8_t ucDigTab[10] =
  {
	0b00111111, 0b00000110, 0b01011011, 0b01001111, 0b01100110,
	0b01101101, 0b01111101, 0b00000111, 0b11111111, 0b01100111
  };

  uint8_t	ucNibble, ucSegments;

  ucNibble = (uint8_t) cNumber & 0x0f;
  ucSegments = ( ucNibble > 9 ) ? 0 : ucDigTab[ucNibble];

  if ( test_bit( ucSegments, SEG_A ) )
    clear_bit( PORTA, PA1 );
  else
    set_bit( PORTA, PA1 );

  if ( test_bit( ucSegments, SEG_B ) )
    clear_bit( PORTA, PA2 );
  else
    set_bit( PORTA, PA2 );

  if ( test_bit( ucSegments, SEG_C ) )
    clear_bit( PORTA, PA3 );
  else
    set_bit( PORTA, PA3 );

  if ( test_bit( ucSegments, SEG_D ) )
    clear_bit( PORTA, PA4 );
  else
    set_bit( PORTA, PA4 );

  if ( test_bit( ucSegments, SEG_E ) )
    clear_bit( PORTA, PA5 );
  else
    set_bit( PORTA, PA5 );

  if ( test_bit( ucSegments, SEG_F ) )
    clear_bit( PORTA, PA6 );
  else
    set_bit( PORTA, PA6 );

  if ( test_bit( ucSegments, SEG_G ) )
    clear_bit( PORTA, PA7 );
  else
    set_bit( PORTA, PA7 );
}
