/*-----------------------------------------------------------------
  interrupt.c  by Jack Botner

  This module contains the interrupt handlers.
-----------------------------------------------------------------*/
#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>
#include "main.h"

extern volatile uint8_t ucPowerState;		// power: 0=off, 1=on
extern volatile uint8_t  ucEventFlags;		// event flags
extern volatile uint16_t uiEepromDelay;		// delay updating eeprom factor
extern volatile uint16_t uiAdc;				// ADC sample
extern volatile uint16_t uiAdcDelay;		// time between adc samples
extern volatile uint16_t uiHeatOn2ms;		// time to have the heat on
extern volatile uint16_t uiHeatOff2ms;		// time to have the heat off

// Internal Functions

/*-----------------------------------------------------------------
  ISR( ADC_vect ) - ADC conversion complete interrupt handler

  Capture the A/D result registers into static memory uiAdc.
  Schedule an event to process the result.
----------------------------------------------------------------*/
ISR( ADC_vect )
{
  // ADCL must be read before ADCH.
  uiAdc = (uint16_t) ADCL | ( (uint16_t) ADCH << 8 );

  set_bit( ucEventFlags, EVENT_ADC_COMPLETE );
}

/*-----------------------------------------------------------------
  ISR(TIMER1_OVF_vect)

  Timer 1 overflow interrupt handler. Generates 2ms interrupts for
  timing purposes. 

  System clock frequency = 4 mHz
  System clock period = 250 ns
  Prescaler = 32
  Timer count period = 16 us
  Counts = 250
  Interrupt period = 2 ms
-----------------------------------------------------------------*/
ISR(TIMER1_OVF_vect)
{
  static volatile uint8_t ucToggle=0, ucPwmCtr=0;

  // Alternate the LED update and switch debounce so that
  // each occur every 4 ms

  if ( ucToggle )	
  {
	set_bit( ucEventFlags, EVENT_DEBOUNCE_IO );
    ucToggle = 0;
  }
  else
  {
    ucToggle = 1;
	set_bit( ucEventFlags, EVENT_REFRESH_LED );
  }

  // If an eeprom update has been scheduled, manage the
  // programmed delay.

  if ( uiEepromDelay )
  {
    --uiEepromDelay;		// decrement the eeprom delay counter
	if ( !uiEepromDelay  )
	  set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
  }

  // Schedule another ADC sample after the programmed delay.

  if ( ucPowerState )				// powered up
  {
	--uiAdcDelay;					// decrement the adc delay counter
	if ( !uiAdcDelay  )
	{
	  set_bit( ADCSRA, ADSC );		// start a conversion
	  uiAdcDelay = ADC_DELAY_2MS;	// start another cycle
	}
  }

  //  Generate a 10 ms event for pwm managment.

  ++ucPwmCtr;
  if ( ucPwmCtr == 5 )		// 5 x 2ms
  {
	set_bit( ucEventFlags, EVENT_10MS_PWM );
	ucPwmCtr = 0;
  }

  TCNT1 = INIT_TIMER1;		// reset the timer1 counter
}

