/*--------------------------------------------------------------------
  encoder.c  by Jack Botner
--------------------------------------------------------------------*/
#include <avr/io.h>
#include "main.h"

// Global variables

extern volatile uint8_t ucEventFlags;		// event flags
extern volatile uint8_t ucDebouncedState;	// debounced state of the switches
extern volatile uint8_t ucPreviousState;	// previous switch state

/*-----------------------------------------------------------------
  debounce_switch()

  Called every 4 ms from the mainline, debounces inputs from
  the rotary encoder.

  Output: ucDebouncedState
-----------------------------------------------------------------*/
void debounce_switch()
{
  static volatile uint8_t ucIndex=0;                    // index into ucState
  static volatile uint8_t ucState[MAX_CHECKS] =         // array containing bounce status
                { 0, 0, 0, 0, 0, 0 };

  uint8_t ii, ucResult, ucSwIn=0;

  // Use the technique documented in "A Guide to Debouncing" by
  // Jack Ganssle to debounce up to 8 inputs (3 in this case).

  // Read current switch states
  if ( !test_bit( ENCODER_PORT, ENCODER_A ) )
    set_bit( ucSwIn, ENC_SWITCH_A );
  if ( !test_bit( ENCODER_PORT, ENCODER_B ) )
    set_bit( ucSwIn, ENC_SWITCH_B );
  if ( !test_bit( ENCODER_PORT, ENCODER_PB ) )
    set_bit( ucSwIn, ENC_SWITCH_PB );

  ucState[ucIndex] = ucSwIn;

  ++ucIndex;
  if ( ucIndex == MAX_CHECKS )
    ucIndex = 0;

  // And all columns of the array
  for ( ii=0, ucResult=0xff; ii<(MAX_CHECKS-1); ++ii )
    ucResult = ucResult & ucState[ii];

  ucDebouncedState = ucResult;
}

/*-----------------------------------------------------------------
  analyze_switch()

  Analyzes the rotary encoder to determine if the shaft rotation
  is up or down, or if the pushbutton has been pressed, and
  generate the corresponding event.

  Output: ucEventFlags
-----------------------------------------------------------------*/
void analyze_switch()
{
  uint8_t       ucNow, ucThen;

  if ( test_bit( ucDebouncedState, ENC_SWITCH_PB ) &&
	!test_bit( ucPreviousState, ENC_SWITCH_PB ) )
  {
    // The encoder button has been pressed
	set_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
  }

  // Convert the position of the encoder to a number 0..3
  ucNow = ucDebouncedState & 0x03;
  ucThen = ucPreviousState & 0x03;

  // Given the A output in bit 0 and B output in bit 1 positions, the
  // encoder generates the values 0, 1, 3, 2 as it is rotated clockwise.

  if ( ucNow == 0 )
  {
    if ( ucThen == 2 )
	  set_bit( ucEventFlags, EVENT_ENCODER_UP );
	else
	  if ( ucThen == 1 )
		set_bit( ucEventFlags, EVENT_ENCODER_DOWN );
  }
  else
	if ( ucNow == 1 )
	{
      if ( ucThen == 0 )
		set_bit( ucEventFlags, EVENT_ENCODER_UP );
	  else
		if ( ucThen == 3 )
		  set_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	}
	else
	  if ( ucNow == 3 )
	  {
        if ( ucThen == 1 )
		  set_bit( ucEventFlags, EVENT_ENCODER_UP );
		else
		  if ( ucThen == 2 )
			set_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	  }
	  else  // ucNow == 2
	  {
        if ( ucThen == 3 )
			set_bit( ucEventFlags, EVENT_ENCODER_UP );
		else
		  if ( ucThen == 0 )
			set_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	  }
}

