/*-----------------------------------------------------------------
  Released under the MIT License  - see
  http://www.opensource.org/licenses/mit-license.php

  travel_clock by Jack Botner

  rtc.c - DS1302 RTC routines

  20141011 - program written.
-----------------------------------------------------------------*/
#include <string.h>
#include <avr/io.h>
#include <util/delay_basic.h>
#include "main.h"
#include "rtc.h"

// internal functions
static void RTC_read_data( uint8_t ucCmd, unsigned char *ucData, uint8_t ucLength );
static void RTC_write_data( unsigned char *ucData, uint8_t ucLength );

/*-----------------------------------------------------------------
  RTC_init()
  
  Initialize the DS1302
-----------------------------------------------------------------*/
void RTC_init()
{
  unsigned char ucData[10];
  uint8_t		ucNr, mm, ss;
  int8_t		iInit=0;

  _delay_loop_2(0);				// 16 ms delay

  // Clear the write protect bit
  ucData[0] = 0x8E;				// set write protect bit
  ucData[1] = 0;				// high order WP bit = 0 (write allowed)
  RTC_write_data( ucData, 1 );

  // Turn the trickle charger off
  ucData[0] = 0x90;				// set trickle charger state
  ucData[1] = 0;				// trickle charger off
  RTC_write_data( ucData, 1 );
  
  // Read 8 clock registers in burst mode
  RTC_read_data( 0xbf, ucData, 8 );
  
  // Does clock need initialization?
  if ( ucData[0] & 0x80 )			// clock halted?
    iInit = 1;
  else
  {
    ucNr = ucData[0] & 0x7F;		// get seconds
	ss = ( ( ucNr >> 4 ) * 10 ) + ( ucNr & 0x0f );
	ucNr = ucData[1] & 0x7f;		// get minutes
	mm = ( ( ucNr >> 4 ) * 10 ) + ( ucNr & 0x0f );

    if ( ss > 59 || mm > 59 )
      iInit = 1;
  }
  
  if ( iInit )
  {
    // Initialize the clock to 2014-01-01 12:0:0
    memset( ucData, '\0', 9 );
    ucData[0] = 0xbe;				// cmd = burst write to clock
	ucData[2] = 0;					// minutes
    ucData[3] = 0x80 | 0x20 | 0x12;	// 12 hour format, 12PM
    ucData[4] = 1;					// day 1
    ucData[5] = 1;					// month 1
    ucData[6] = 4;					// day of week
    ucData[7] = 0x14;				// year
    RTC_write_data( ucData, 8 );
	_delay_loop_2(0);				// 16 ms delay
  }
}

/*-----------------------------------------------------------------
  RTC_get_time(...)
  
  Returns hours and minutes from the DS1302
-----------------------------------------------------------------*/
uint8_t RTC_get_time( uint8_t *pucHours, uint8_t *pucMinutes, uint8_t *pucSeconds )
{
  unsigned char ucData[4];
   
  RTC_read_data( 0x81, ucData, 1 );		// read seconds
  *pucSeconds = ( ( ( ucData[0] >> 4 ) & 0x07 ) * 10 ) + ( ucData[0] & 0x0f );
	
  // Read hours and minutes
  RTC_read_data( 0x83, ucData, 1 );		// read minutes
  *pucMinutes = ( ucData[0] & 0x7f );
  
  RTC_read_data( 0x85, ucData, 1 );		// read hours
  *pucHours = ( ucData[0] & 0x1f );
 
  return ( ( ucData[0] & 0x20 ) ? 1 : 0 );
}

/*-----------------------------------------------------------------
  RTC_set_time(...)
  
  Sets the DS1302 time
-----------------------------------------------------------------*/
void RTC_set_time( uint8_t ucH1, uint8_t ucH2, uint8_t ucM1, uint8_t ucM2, uint8_t ucPMind )
{
  unsigned char ucData[4];
	
  // Read hours and minutes
  ucData[0] = 0x84;					// write hours
  ucData[1] = 0x80 | ( ucH1 << 4 ) | ucH2;
  if ( ucPMind ) ucData[1] |= 0x20;
  RTC_write_data( ucData, 1 );
  
  ucData[0] = 0x82;					// write minutes
  ucData[1] = ( ucM1 << 4 ) | ucM2;
  RTC_write_data( ucData, 1 ); 
}

/*-----------------------------------------------------------------
  RTC_write_data(...)
  
  Writes a command byte and 0 or more data bytes to the RTC.
  
  ucData - A byte buffer containing command byte and 0 or more
           data bytes
			
  ucLength - The number of data bytes (0..31)
-----------------------------------------------------------------*/
static void RTC_write_data( unsigned char *ucData, uint8_t ucLength )
{
  uint8_t    ii, jj, nn, byte;
  
  if ( ucLength > 31 ) ucLength = 0;
  nn = ucLength + 1;		// include command byte
  
  clear_bit( RTC_PORT, RTC_SCLK );
  clear_bit( RTC_PORT, RTC_IO );
  set_bit( RTC_PORT, RTC_CE );
  nop();
  
  for ( ii=0; ii<nn; ++ii )
  {
    byte = ucData[ii];

    for ( jj=0; jj<8; ++jj )
	{
      if ( byte & 0x01 )
		set_bit( RTC_PORT, RTC_IO );
	  else	
		clear_bit( RTC_PORT, RTC_IO );

	  set_bit( RTC_PORT, RTC_SCLK );
      DELAY_1US();
      clear_bit( RTC_PORT, RTC_SCLK );
      nop();		

      byte = byte >> 1;		// access next higher bit
	}
  }
  
  clear_bit( RTC_PORT, RTC_CE );
}

/*-----------------------------------------------------------------
  RTC_read_data(...)
  
  Writes a command byte then reads 0 or more data bytes from the RTC.
  
  ucCmd - The command byte to write to the RTC
  
  ucData - A byte buffer to receive the data read
			
  ucLength - The number of data bytes to read (0..31)
-----------------------------------------------------------------*/
static void RTC_read_data( uint8_t ucCmd, unsigned char *ucData, uint8_t ucLength )
{
  uint8_t    ii, jj, byte;
  
  if ( ucLength > 31 ) ucLength = 0;
  
  clear_bit( RTC_PORT, RTC_SCLK );
  clear_bit( RTC_PORT, RTC_IO );
  set_bit( RTC_PORT, RTC_CE );
  nop();
  byte = ucCmd;
  
  for ( ii=0; ii<8; ++ii )
  {
    if ( byte & 0x01 )
	  set_bit( RTC_PORT, RTC_IO );
	else	
	  clear_bit( RTC_PORT, RTC_IO );

	set_bit( RTC_PORT, RTC_SCLK );
    DELAY_1US();
    clear_bit( RTC_PORT, RTC_SCLK );
    nop();		

    byte = byte >> 1;		// access next higher bit
  }

  if ( ucLength )
  {
    clear_bit( RTC_PORT, RTC_IO );
	clear_bit( RTC_DDR, RTC_IO );		// set to input
	
	for ( ii=0; ii<ucLength; ++ii )
	{
	  byte = 0;
	  for ( jj=0; jj<8; ++jj )
	  {	
	    if ( test_bit( RTC_PIN, RTC_IO ) )
	      byte |= 0x80;

	    set_bit( RTC_PORT, RTC_SCLK );
	    DELAY_1US();
	    clear_bit( RTC_PORT, RTC_SCLK );
		
	    nop();
		if ( jj < 7 )			// don't shift the last time
          byte = byte >> 1;		// access next higher bit
	  }

      ucData[ii] = byte;
	}
  }
  
  clear_bit( RTC_PORT, RTC_CE );
  set_bit( RTC_DDR, RTC_IO );		// set back to output
}


