/*-----------------------------------------------------------------
  Released under the MIT License  - see
  http://www.opensource.org/licenses/mit-license.php

  travel_clock by Jack Botner

  program.c - clock programming routines

  To program the clock's time or the alarm time.
  
  20141021 - program written.
-----------------------------------------------------------------*/
#include <avr/io.h>
#include "main.h"
#include "rtc.h"

// Global variables
extern volatile uint8_t  ucEventFlags;		// system event flags
extern volatile uint8_t  ucBlink;			// to make digits blink
extern volatile uint8_t  ucAlarmState;	 	// alarm state

extern volatile uint8_t ucLed[4];			// led digit values
extern volatile uint8_t ucDp[4];			// which decimals are on
extern volatile uint8_t ucAlarm[4];			// alarm digit values
extern volatile uint8_t ucClockPM, ucAlarmPM;	// 0=AM, 1=PM

// Internal functions
static void program_hours( uint8_t *pucPmind );
static void program_minutes( void );

/*-----------------------------------------------------------------
  program_clock()
  
  Programs the clock or alarm time.
  
  If the alarm is on, the alarm time is programmed.
  If the alarm is off, the clock time is programmed and set
  in the RTC.
  Turning the alarm on and off is handled in the main module.
-----------------------------------------------------------------*/
void program_clock()
{
  uint8_t		ii, ucPmind;
  
  clear_bit( ucEventFlags, EVENT_ENCODER_CW );
  clear_bit( ucEventFlags, EVENT_ENCODER_CCW );
  clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );

  if ( ucAlarmState != ALARM_OFF )
  {
	// Since we're programming the alarm, show it
	for ( ii=0; ii<4; ++ii )
	  ucLed[ii] = ucAlarm[ii];
	  
	ucPmind = ucAlarmPM; 
  }
  else
	ucPmind = ucClockPM;

  ucDp[3] = ( ucPmind ) ? 1 : 0;

  program_hours( &ucPmind );

  program_minutes();

  // At this point ucLed[] contains updated alarm or clock time.
  if ( ucAlarmState == ALARM_OFF )
  {
	ucClockPM = ucPmind;  
    RTC_set_time( ucLed[0], ucLed[1], ucLed[2], ucLed[3], ucClockPM );
  }
  else
  {
	ucAlarmPM = ucPmind;
	  
	for ( ii=0; ii<4; ++ii )
	  ucAlarm[ii] = ucLed[ii];
	  
    set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
  }
}

/*-----------------------------------------------------------------
  program_hours(...)
-----------------------------------------------------------------*/
static void program_hours( uint8_t *pucPmind )
{
  uint8_t ucHours;
  
  ucBlink = BLINK_HOURS;				// blink on
  
  while (1)
  {
    if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
    {
	  // Pressing the button now ends the programming for this segment
      clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
	  break;						// done
	}

    if ( test_bit( ucEventFlags, EVENT_ENCODER_CW ) )
	{
      ucHours = ucLed[0] * 10 + ucLed[1];
	  if ( ucHours == 12 )
	  {
	    ucHours = 1;
		*pucPmind = ( *pucPmind ) ? 0 : 1;
	  }
	  else
	    ++ucHours;	

      ucDp[3] = ( *pucPmind ) ? 1 : 0;
      ucLed[0] = ucHours / 10;
	  ucLed[1] = ucHours % 10;
	  clear_bit( ucEventFlags, EVENT_ENCODER_CW );
	}

    if ( test_bit( ucEventFlags, EVENT_ENCODER_CCW ) )
    {
      ucHours = ucLed[0] * 10 + ucLed[1];
	  if ( ucHours > 1 )
        --ucHours;
	  else
	  {
	    ucHours = 12;
		*pucPmind = ( *pucPmind ) ? 0 : 1;
	  }

      ucDp[3] = ( *pucPmind ) ? 1 : 0;
      ucLed[0] = ucHours / 10;
      ucLed[1] = ucHours % 10;
      clear_bit( ucEventFlags, EVENT_ENCODER_CCW );
    }
  }

  ucBlink = BLINK_NONE;				// blink off
}

/*-----------------------------------------------------------------
  program_minutes()
-----------------------------------------------------------------*/
static void program_minutes()
{
  uint8_t ucMins;
  
  ucBlink = BLINK_MINUTES;				// blink on
  
  while (1)
  {
	  if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
	  {
		  // Pressing the button now ends the programming for this segment
		  clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
		  break;						// done
	  }

	  if ( test_bit( ucEventFlags, EVENT_ENCODER_CW ) )
	  {
		  ucMins = ucLed[2] * 10 + ucLed[3];
		  if ( ucMins == 59 )
		    ucMins = 0;
		  else
		    ++ucMins;

		  ucLed[2] = ucMins / 10;
		  ucLed[3] = ucMins % 10;
		  clear_bit( ucEventFlags, EVENT_ENCODER_CW );
	  }

	  if ( test_bit( ucEventFlags, EVENT_ENCODER_CCW ) )
	  {
		  ucMins = ucLed[2] * 10 + ucLed[3];
		  if ( ucMins > 0 )
		    --ucMins;
		  else
		    ucMins = 59;

		  ucLed[2] = ucMins / 10;
		  ucLed[3] = ucMins % 10;
		  clear_bit( ucEventFlags, EVENT_ENCODER_CCW );
	  }
  }

  ucBlink = BLINK_NONE;				// blink off
}
