/*-----------------------------------------------------------------
  travel_clock by Jack Botner

  main.c

  This project consists of the following hardware:

  - 4 digits LED numeric display
  - pushbutton rotary encoder model EC11-1B-18T
    (Changzhou Xinze Electronic Co.)
  - DS1302 Real Time Clock 

  Processor: ATMega88
      Clock: Internal, 4 MHz (250 ns)

  Released under the MIT License  - see
  http://www.opensource.org/licenses/mit-license.php

  20141006 - program written.
-----------------------------------------------------------------*/
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <util/delay_basic.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "main.h"
#include "led.h"
#include "rtc.h"

#define STATE_INITIAL		1
#define STATE_PROGRAM		2
#define STATE_RUN			3

// Global variables

static volatile uint8_t	ucSystemState=STATE_INITIAL;

volatile uint8_t ucLed[4] = { 9, 9, 9, 9 };		// led digit values
volatile uint8_t ucDp[4] = { 0, 1, 1, 0 };		// which decimals are on
volatile uint8_t ucBlink=BLINK_NONE;			// blink during programming
volatile uint8_t ucAlarm[4] = { 0, 7, 0, 0 };	// alarm digit values
volatile uint8_t ucClockPM=0, ucAlarmPM=0;		// 0=AM, 1=PM

volatile uint8_t  ucEventFlags=0;			// system event flags
volatile uint8_t  ucAlarmState=ALARM_OFF; 	// alarm state
static volatile uint8_t ucSeconds=0;		// clock seconds

MY_EEPROM_DATA stEepromVars;			// variables kept in EEPROM

// Internal functions

static void check_alarm( void);
static void system_init(void);
static void update_time(void);

/*-----------------------------------------------------------------
  main()
-----------------------------------------------------------------*/
int main( void )
{
  static volatile uint8_t ucTctr=0;		// update time counter

  while (1)
  {
    switch ( ucSystemState )
    {
      case STATE_INITIAL:
	    system_init();              // initialize cpu
		RTC_init();					// initialize ds1302 rtc
		
		// Get the saved state variables from eeprom
		eeprom_read_block( &stEepromVars, 0, sizeof( MY_EEPROM_DATA ) );
		if ( stEepromVars.ucAlarmTime[0] > 1 || stEepromVars.ucAlarmTime[1] > 9 ||
		     stEepromVars.ucAlarmTime[2] > 5 || stEepromVars.ucAlarmTime[3] > 9 ||
		     stEepromVars.ucAlarmPm > 1      || stEepromVars.ucAlarm_State > ALARM_ON_ACTIVE )
		{
		  // Initialize all eeprom variables	
		  stEepromVars.ucAlarmTime[0] = 0; 	
		  stEepromVars.ucAlarmTime[1] = 7;
		  stEepromVars.ucAlarmTime[2] = 0;
		  stEepromVars.ucAlarmTime[3] = 0;
		  stEepromVars.ucAlarmPm = 0;
		  stEepromVars.ucAlarm_State = ALARM_OFF;
	      set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
		}
		else
		{
		  // Initialize clock with saved eeprom variables
		  ucAlarm[0] = stEepromVars.ucAlarmTime[0];
		  ucAlarm[1] = stEepromVars.ucAlarmTime[1];
		  ucAlarm[2] = stEepromVars.ucAlarmTime[2];
		  ucAlarm[3] = stEepromVars.ucAlarmTime[3];
		  ucAlarmPM  = stEepromVars.ucAlarmPm;
		  ucAlarmState = stEepromVars.ucAlarm_State;		  	
	      if ( ucAlarmState != ALARM_OFF )
			ucDp[0] = 1;
		}		 
		
	    ucSystemState = STATE_RUN;
	    break;

      case STATE_RUN:
        if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
        {
		  if ( ucAlarmState == ALARM_ON_ACTIVE )
		  {
			// If the alarm is sounding, just shut it up
			ucAlarmState = ALARM_ON_QUIET;
	        set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
			set_bit( BEEPER_PORT, BEEPER_PIN );		// beeper off
		  }
		  else
	        ucSystemState = STATE_PROGRAM;
			
	      clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
        }

        if ( test_bit( ucEventFlags, EVENT_ENCODER_CW ) ||
		     test_bit( ucEventFlags, EVENT_ENCODER_CCW ) )
        {
		  if ( ucAlarmState == ALARM_ON_ACTIVE )
		  {
			// If the alarm is sounding, just shut it up  
			set_bit( BEEPER_PORT, BEEPER_PIN );		// beeper off
			ucAlarmState = ALARM_ON_QUIET; 
	        set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
		  }
		  else
		  {
			// If alarm is off, turn it on
	        if ( ucAlarmState == ALARM_OFF )
			{
			  ucAlarmState = ALARM_ON_QUIET;
	          set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
			  ucDp[0] = 1;
			}
			else
			  // If alarm is on, turn it off
	          if ( ucAlarmState == ALARM_ON_QUIET )
			  {
	            ucAlarmState = ALARM_OFF;
			    set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
			    ucDp[0] = 0;
			  }
	          else
			  {
			    // Alarm is sounding, shut up
	            ucAlarmState = ALARM_ON_QUIET;
				set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
			    set_bit( BEEPER_PORT, BEEPER_PIN );		// beeper off
			  }
		  }
			  
	      clear_bit( ucEventFlags, EVENT_ENCODER_CW );
	      clear_bit( ucEventFlags, EVENT_ENCODER_CCW );
        }
		
        if ( test_bit( ucEventFlags, EVENT_HALFSEC ) )
        {
		  // if the beeper is active, turn it on and off
	      if ( ucAlarmState == ALARM_ON_ACTIVE )
		  {
			if ( test_bit( BEEPER_PORT, BEEPER_PIN ) )
			  clear_bit( BEEPER_PORT, BEEPER_PIN );		// beeper on
			else
			  set_bit( BEEPER_PORT, BEEPER_PIN );		// beeper off
		  }

		  if ( ucTctr == 1 )
		  {
			// Update the time periodically    
		    update_time();
			ucTctr = 0;
		  }
		  else
		  {
			// Check the alarm periodically
			check_alarm();
		    ++ucTctr;
		  }
		  		  
	      clear_bit( ucEventFlags, EVENT_HALFSEC );
        }
		
		if ( test_bit( ucEventFlags, EVENT_UPDATE_EEPROM ) )
		{
		  stEepromVars.ucAlarmTime[0] = ucAlarm[0];
		  stEepromVars.ucAlarmTime[1] = ucAlarm[1];
		  stEepromVars.ucAlarmTime[2] = ucAlarm[2];
		  stEepromVars.ucAlarmTime[3] = ucAlarm[3];
		  stEepromVars.ucAlarmPm = ucAlarmPM;
		  stEepromVars.ucAlarm_State = ucAlarmState;
	      eeprom_write_block( &stEepromVars, 0, sizeof( MY_EEPROM_DATA ) );		
	      clear_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
		}
		
	    break;

      case STATE_PROGRAM:
	    program_clock();
	    ucSystemState = STATE_RUN;
	    break;
	}
  }
}

/*-----------------------------------------------------------------
  system_init()

  PORTB: PB0 (input) Encoder pushbutton
         PB1 (output) Beeper
         PB2 (output) DS1302 !RST
         PB3 (output) ISP MOSI
         PB4 (input) ISP MISO
         PB5 (output) ISP SCK
         PB6 (output) DS1302 I/O
         PB7 (output) DS1302 SCLK

  PORTC: PC0 (output) LED-A1
         PC1 (output) LED-A2
         PC2 (output) LED-A3
         PC3 (output) LED-A4
         PC4 (input) Encoder "A"
         PC5 (input) Encoder "B"
         PC6 (input) ISP !Reset

  PORTD: PD0 (output) LED-a
         PD1 (output) LED-b
         PD2 (output) LED-c
         PD3 (output) LED-d
         PD4 (output) LED-e
         PD5 (output) LED-f
         PD6 (output) LED-g
         PD7 (output) LED-dp
-----------------------------------------------------------------*/
static void system_init()
{
  // Make sure the clock prescaler is set to 2.
  CLKPR = 0b10000000;       // set clock prescaler change enable
  CLKPR = 0b00000001;       // set prescaler to 2
  
  // Set up the I/O ports directions, pull-ups
  DDRB  = 0b11101110;		// port B 0=input, 1=output
  PORTB = 0b00000011;       // port B initialization, pullups

  DDRC  = 0b00001111;		// port C 0=input, 1=output
  PORTC = 0b00111111;       // port C initialization, pullups

  DDRD  = 0b11111111;		// port D 0=input, 1=output
  PORTD = 0b11111111;       // port D initialization

  // Initialize timer0 for button debouncing
  TCNT0 = INIT_CTR0;
  TCCR0A = 0b00000000;       // timer/counter normal mode
  TCCR0B = 0b00000011;       // prescaler = 64 (turns timer0 on)
  set_bit( TIMSK0, TOIE0 );	 // enable timer/counter0 overflow interrupt

  // Initialize timer1 for 1/2 second timing
  TCNT1 = INIT_CTR1;
  TCCR1A = 0b00000000;		// normal operation
  TCCR1B = 0b00000011;		// prescaler = 64 (turns timer1 on)
  TCCR1C = 0b00000000;		// normal operation
  set_bit( TIMSK1, TOIE1 );	// enable timer/counter1 overflow interrupt

  // Initialize timer2 for LED management
  TCNT2 = INIT_CTR2;
  TCCR2A = 0b00000000;       // timer/counter normal mode
  TCCR2B = 0b00000100;       // prescaler = 64 (turns timer2 on)
  set_bit( TIMSK2, TOIE2 );	 // enable timer/counter0 overflow interrupt

  sei();					 // enable global interrupts
}

/*-----------------------------------------------------------------
  check_alarm()
  
  If the alarm is on, compares the alarm time to the clock time.
  If match, sounds the alarm. Seconds are checked to prevent the
  alarm from sounding again after being silenced.
-----------------------------------------------------------------*/
static void check_alarm()
{
  if ( ucAlarmState != ALARM_ON_QUIET ) return;
  
  if ( ucLed[0] == ucAlarm[0] && ucLed[1] == ucAlarm[1] &&
       ucLed[2] == ucAlarm[2] && ucLed[3] == ucAlarm[3] &&
	   ucClockPM == ucAlarmPM && ucSeconds < 2 )
  {
	ucAlarmState = ALARM_ON_ACTIVE;
	set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
  }	   
}

/*-----------------------------------------------------------------
  update_time()
  
  Get the time from the RTC and put HH, MM into ucLed array.
  Also returns seconds for alarm logic.
-----------------------------------------------------------------*/
static void update_time()
{
  uint8_t ucHours, ucMinutes, ucSecs;

  ucClockPM = RTC_get_time( &ucHours, &ucMinutes, &ucSecs );

  ucLed[0] = ucHours >> 4;
  ucLed[1] = ucHours & 0x0f;
  ucLed[2] = ucMinutes >> 4;
  ucLed[3] = ucMinutes & 0x0f;
  
  ucSeconds = ucSecs;
  
  ucDp[3] = ( ucClockPM ) ? 1 : 0;
}


