/*-----------------------------------------------------------------
  Released under the MIT License  - see
  http://www.opensource.org/licenses/mit-license.php

  travel_clock by Jack Botner

  led.c - led display routines

  The display is 4 numeric LED digits. It is an oddity of this
  display that the 3rd digit is upside down. The designer wanted
  the decimal point to appear at the top left for this digit,
  to make a crude colon symbol between hours and minutes. The
  result is that digit 3 requires a different look-up table in
  order to appear ri8ght. Please note that in the segment table,
  1 means off and 0 means on. This is also the case for the
  digit selection port bits.
  
  20141008 - program written.
-----------------------------------------------------------------*/
#include <avr/io.h>
#include "main.h"
#include "led.h"

/*-----------------------------------------------------------------
  LedOutputData(...)
  
  Sets the 8 segments of the LED display.
  
  ucDigit - which digit 1..4
            This parameter is not needed in the sense that the 
			digit is not selected here, but since digit 3 is
			upside down, it is needed to select the right
			look-up table.
			
  ucNumber - 0..9 (anything else shows blank)
  
  ucDp - 0 = no DP, 1 = show DP
  
  Note: in the look-up tables, 0=segment on, 1=segment off.	
-----------------------------------------------------------------*/
void LedOutputData( uint8_t ucDigit, uint8_t ucNumber, uint8_t ucDp )
{
  uint8_t		portval=0xff;
  
  static const uint8_t ucLUT1[10] = 
  {
	 0b11000000, 0b11111001, 0b10100100, 0b10110000, 0b10011001,
	 0b10010010, 0b10000011, 0b11111000, 0b10000000, 0b10011000
  };

  static const uint8_t ucLUT2[10] =
  {
	 0b11000000, 0b11001111, 0b10100100, 0b10000110, 0b10001011,
	 0b10010010, 0b10011000, 0b11000111, 0b10000000, 0b10000011
  };
  
  if ( ucNumber >= 0 && ucNumber <= 9 )
    portval = ( ucDigit == 3 ) ? ucLUT2[ucNumber] : ucLUT1[ucNumber];
  
  // To blank out high-order hours digit when zero
  if ( ucDigit == 1 && !ucNumber ) portval = 0xff;
  
  if ( ucDp )
    portval = portval & 0x7F;
  
  LED_SEGMENT_PORT = portval;
}

/*---------------------------------------------------------------------
  LedSelectDigit(...)
  
  Selects a digit of the led display, by turning on the requested
  driver transistor. Turns off all the other digits.
  
  Input: 1) a single numeric digit, range 1..4. 1 drives the high order
            digit, 4 drives the low order digit.
		 2) the digit to blink, 1..4, 5 = no blink
		 
		 Blinking a digit is accomplished by forgetting to enable it
		 when its turn comes up, as specified by ucBlinkLED. Which
		 is under the control of the 1/2 second timer tick.
---------------------------------------------------------------------*/
void LedSelectDigit( uint8_t ucDigit, uint8_t ucBlinkLED )
{
  if ( ucDigit == 1 && ucBlinkLED != BLINK_HOURS )
    clear_bit( LED_DIGIT_SELECT_PORT, LED_DIGIT_SELECT_A1 );
  else
    set_bit( LED_DIGIT_SELECT_PORT, LED_DIGIT_SELECT_A1 );

  if ( ucDigit == 2 && ucBlinkLED != BLINK_HOURS )
    clear_bit( LED_DIGIT_SELECT_PORT, LED_DIGIT_SELECT_A2 );
  else
    set_bit( LED_DIGIT_SELECT_PORT, LED_DIGIT_SELECT_A2 );

  if ( ucDigit == 3 && ucBlinkLED != BLINK_MINUTES )
    clear_bit( LED_DIGIT_SELECT_PORT, LED_DIGIT_SELECT_A3 );
  else
    set_bit( LED_DIGIT_SELECT_PORT, LED_DIGIT_SELECT_A3 );

  if ( ucDigit == 4 && ucBlinkLED != BLINK_MINUTES )
    clear_bit( LED_DIGIT_SELECT_PORT, LED_DIGIT_SELECT_A4 );
  else
    set_bit( LED_DIGIT_SELECT_PORT, LED_DIGIT_SELECT_A4 );
}


