/*-----------------------------------------------------------------
  Released under the MIT License  - see
  http://www.opensource.org/licenses/mit-license.php

  travel_clock by Jack Botner

  interrupt.c

  This module contains all the interrupt handlers.
-----------------------------------------------------------------*/
#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "led.h"

// Global variables

extern volatile uint8_t  ucEventFlags;			// system event flags
extern volatile uint8_t ucLed[];				// led digit values
extern volatile uint8_t ucDp[];					// which decimals are on
extern volatile uint8_t ucBlink;				// blink during programming

static volatile uint8_t  ucDebouncedState;		// debounced state of the switches
static volatile uint8_t ucBlinkLED=BLINK_NONE;	// tell LED routine which LED to blink
static volatile uint16_t uiMsPressed;			// how long encoder button pressed
static volatile uint8_t  ucPreviousState=0; 		// previous switch state
static volatile uint8_t  ucNrUps=0, ucNrDowns=0; 	// Encoder logic

// Internal Functions

static void analyze_switch(void);

/*-----------------------------------------------------------------
  ISR(TIMER0_OVF_vect)

  Timer 0 overflow interrupt handler - tests for button presses

  System clock frequency = 4 mHz
  System clock period = 250 ns
  Prescaler = 64
  Counts = 62
  Interrupt period = 1 ms
------------------------------------------------------------------*/
ISR(TIMER0_OVF_vect)
{
  static volatile uint8_t ucFirst=1;
  static volatile uint8_t ucIndex=0;             	// index into ucState
  static volatile uint8_t ucState[MAX_CHECKS] =  	// array containing bounce status
        	{ 0, 0, 0, 0, 0, 0 };

  uint8_t ii, ucResult, ucSwIn=0;

  // We use the technique documented in "A Guide to Debouncing" by
  // Jack Ganssle to debounce up to 8 inputs (3 in this case).

  // Read current switch states
  if ( !test_bit( ENCODER_PORT, ENCODER_A ) )
    set_bit( ucSwIn, ENC_SWITCH_A );
  if ( !test_bit( ENCODER_PORT, ENCODER_B ) )
    set_bit( ucSwIn, ENC_SWITCH_B );
  if ( !test_bit( ENC_SW_PORT, ENC_SW_PB ) )
    set_bit( ucSwIn, ENC_SWITCH_PB );

  ucState[ucIndex] = ucSwIn;

  ++ucIndex;
  if ( ucIndex == MAX_CHECKS )
    ucIndex = 0;

  // And all columns of the array
  for ( ii=0, ucResult=0xff; ii<(MAX_CHECKS-1); ++ii )
    ucResult = ucResult & ucState[ii];

  ucDebouncedState = ucResult;

  // The 1st time only, set the previous switch state to the
  // current switch state.
  if ( ucFirst )
  {
    ucPreviousState = ucDebouncedState;
	ucFirst = 0;
  }

  // Since we're here every millisecond, keep count
  // of how many milliseconds the encoder button is pressed
  if ( test_bit( ucDebouncedState, ENC_SWITCH_PB ) )
    ++uiMsPressed;
  else
    uiMsPressed = 0;

  if ( ucDebouncedState != ucPreviousState )
    analyze_switch();		// convert switch change to event
	
  TCNT0 = INIT_CTR0;
}

/*-----------------------------------------------------------------
  ISR(TIMER1_OVF_vect)

  Timer 1 overflow interrupt handler for 1/2 second events
  - beeper
  - blinking digits
  - time refresh

  System clock frequency = 4 mHz
  System clock period = 250 ns
  Prescaler = 64
  Counts = 31250
  Interrupt period = 500 ms
------------------------------------------------------------------*/
ISR(TIMER1_OVF_vect)
{	
  if (  ucBlink!= BLINK_NONE )
  {
    // Want to blink the specified digit
	if ( ucBlinkLED == ucBlink )
	  ucBlinkLED = BLINK_NONE;			// turn the digits on
	else
	  ucBlinkLED = ucBlink;				// turn the digits off
  }
  else
    ucBlinkLED = BLINK_NONE;			// turn the digits on

  set_bit( ucEventFlags, EVENT_HALFSEC );

  TCNT1 = INIT_CTR1;
}

/*-----------------------------------------------------------------
  ISR(TIMER2_OVF_vect)

  Timer 2 overflow interrupt handler - LED digit switching

  System clock frequency = 4 mHz
  System clock period = 250 ns
  Prescaler = 64
  Counts = 125
  Interrupt period = 2 ms
------------------------------------------------------------------*/
ISR(TIMER2_OVF_vect)
{
  // Every 2 ms refresh the LED
  // Any longer and it flickers

  static volatile uint8_t ucIndex=0;	// digit selection index

  // Show the next digit on the LED display.
  LedOutputData( ( ucIndex + 1 ), ucLed[ucIndex], ucDp[ucIndex] );
  LedSelectDigit( ucIndex + 1, ucBlinkLED );

  ++ucIndex;
  if ( ucIndex > 3 )
    ucIndex = 0;
  
  TCNT2 = INIT_CTR2;
}

/*-----------------------------------------------------------------
  analyze_switch()

  Analyzes the rotary encoder to determine if the shaft rotation
  is up or down, or if the pushbutton has been pressed.

  Note. Like the CTS Series 290 Encoder, the EC11-1B-18T encoder
  outputs four states for each rotary detent instead of one as the
  Greyhill encoder does. So in effect you get four "ups" or four 
  "downs" for each detent. To compensate, we count to 4.
-----------------------------------------------------------------*/
static void analyze_switch()
{
  uint8_t	ucNow, ucThen;

  if ( test_bit( ucDebouncedState, ENC_SWITCH_PB ) && 
       !test_bit( ucPreviousState, ENC_SWITCH_PB ) )
  {
    // The encoder button has been pressed
	set_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
  }

  // Convert the position of the encoder to a number 0..3
  ucNow = ucDebouncedState & 0x03;
  ucThen = ucPreviousState & 0x03;

  // Given the A output in bit 0 and B output in bit 1 positions, the
  // encoder generates the values 0, 1, 3, 2 as it is rotated clockwise.

  if ( ucNow == 0 )
  {
    if ( ucThen == 2 )
	{
	  ++ucNrUps;
	  ucNrDowns = 0;
	}
	else
	  if ( ucThen == 1 )
	  {
	    ++ucNrDowns;
		ucNrUps = 0;
	  }
  }
  else
	if ( ucNow == 1 )
	{
      if ( ucThen == 0 )
	  {
	    ++ucNrUps;
		ucNrDowns = 0;
	  }
	  else
	    if ( ucThen == 3 )
		{
		  ++ucNrDowns;
		  ucNrUps = 0;
		}
	}
	else
	  if ( ucNow == 3 )
	  {
    	if ( ucThen == 1 )
		{
	      ++ucNrUps;
		  ucNrDowns = 0;
		}
		else
	 	  if ( ucThen == 2 )
		  {
		    ++ucNrDowns;
		    ucNrUps = 0;
		  }
	  }
	  else	// ucNow == 2
	  {
    	if ( ucThen == 3 )
		{
	      ++ucNrUps;
		  ucNrDowns = 0;
		}
		else
	 	  if ( ucThen == 0 )
		  {
			++ucNrDowns;
			ucNrUps = 0;
		  }
	  }

  if ( ucNrUps == 4 )
  {
	set_bit( ucEventFlags, EVENT_ENCODER_CCW );
	ucNrUps = ucNrDowns = 0;
  }

  if ( ucNrDowns == 4 )
  {
	set_bit( ucEventFlags, EVENT_ENCODER_CW );
	ucNrUps = ucNrDowns = 0;
  }

  ucPreviousState = ucDebouncedState;
}
