/*-----------------------------------------------------------------
  All code is provided for EDUCATIONAL purposes ONLY. There is no
  warranty and no support.

  thermostat2 by Jack Botner

  program.c

  Routines to run the programming dialogs.
-----------------------------------------------------------------*/
#include <stdint.h>
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <util/delay.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "lcd.h"
#include "nju6355.h"

#define UPDATE_THE_DISPLAY	1
#define WAITFOR_EVENT		2
#define ST_UPDATE			5
#define ST_DONE				9

// Global variables

extern MY_EEPROM_DATA stEepromVars;		// variables kept in EEPROM

extern uint16_t uiSetTemp10C_Stage;		// used during adjustments and for display
extern uint16_t uiSetTemp10C_Final;		// used for state analysis
extern uint8_t ucEventFlags;
extern uint16_t uiMsPressed;
extern uint8_t ucDelaySec;				// system response delay to temp. change
extern RTCPARMS stRtc;

// Internal functions

static uint8_t update_number( uint8_t, uint8_t, uint8_t );
static void strextnd( char *s1, uint8_t fillchar, uint8_t newlength );

/*-----------------------------------------------------------------
  program_hysteresis()

  Manage programming the hysteresis dialog.
-----------------------------------------------------------------*/
void program_hysteresis()
{
  const char *msg1  = "Heat ";
  const char *msg2  = "Cool ";
  const char *msg3  = "Hysteresis:";
  uint8_t ucHigh, ucLow, ucInit, ucRc;

  // Program the heat hysteresis
  lcd_clrscr();
  lcd_puts( msg1 );
  lcd_puts( msg3 );
  ucLow = 1;
  ucHigh = 20;
  ucInit = stEepromVars.ucHysteresis_Heat;
  ucRc = update_number( ucInit, ucLow, ucHigh );
  if( ucRc != stEepromVars.ucHysteresis_Heat )
  {
	stEepromVars.ucHysteresis_Heat = ucRc;
	set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
  }

  // Program the cool hysteresis
  lcd_clrscr();
  lcd_puts( msg2 );
  lcd_puts( msg3 );
  ucLow = 1;
  ucHigh = 15;
  ucInit = stEepromVars.ucHysteresis_Cool;
  ucRc = update_number( ucInit, ucLow, ucHigh );
  if( ucRc != stEepromVars.ucHysteresis_Cool )
  {
	stEepromVars.ucHysteresis_Cool = ucRc;
	set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
  }
}

/*-----------------------------------------------------------------
  program_mode()

  Manage programming the mode dialog.
-----------------------------------------------------------------*/
void program_mode()
{
  const char *msg1a = "Select Mode:";
  const char *msg2a = "*Heat* ";
  const char *msg2b = "Heat ";
  const char *msg3a = "*Cool* ";
  const char *msg3b = "Cool ";
  const char *msg4a = "*Off*";
  const char *msg4b = "Off";

  uint8_t ucTempMode = stEepromVars.ucSystemMode, ucState = UPDATE_THE_DISPLAY;

  // We have 3 modes: "heat", "cool" and "off". Highlight the
  // current mode by bracketing it with *.

  while ( ucState != ST_DONE )
  {
	wdt_reset();

	if ( uiMsPressed > 1000 )
	{
	  ucState = ST_DONE;
	  continue;
	}

    if ( ucState == UPDATE_THE_DISPLAY )
	{
	  lcd_clrscr();
	  lcd_puts( msg1a );
	  lcd_gotoxy( 0, 1 );

      // Display the current state
      if ( test_bit( ucTempMode, MODE_HEAT_ON ) )
        lcd_puts( msg2a );
	  else
        lcd_puts( msg2b );
      if ( test_bit( ucTempMode, MODE_COOL_ON ) )
        lcd_puts( msg3a );
	  else
        lcd_puts( msg3b );
      if ( !test_bit( ucTempMode, MODE_HEAT_ON ) && !test_bit( ucTempMode, MODE_COOL_ON ) )
        lcd_puts( msg4a );
	  else
        lcd_puts( msg4b );

	  ucState = WAITFOR_EVENT;
	}

    if ( ucState == WAITFOR_EVENT )
	{
      if ( test_bit( ucEventFlags, EVENT_ENCODER_UP ) )
	  {
	    clear_bit( ucEventFlags, EVENT_ENCODER_UP );

	    // Move up to the next item
	    if ( test_bit( ucTempMode, MODE_HEAT_ON ) )
	    {
	      clear_bit( ucTempMode, MODE_HEAT_ON );
		  set_bit( ucTempMode, MODE_COOL_ON );
		  ucState = UPDATE_THE_DISPLAY;
		  continue;
	    }
	    if ( test_bit( ucTempMode, MODE_COOL_ON ) )
	    {
	      clear_bit( ucTempMode, MODE_COOL_ON );
		  ucState = UPDATE_THE_DISPLAY;
		  continue;
	    }
	    set_bit( ucTempMode, MODE_HEAT_ON );
		ucState = UPDATE_THE_DISPLAY;
	    continue;
	  }

  	  if ( test_bit( ucEventFlags, EVENT_ENCODER_DOWN ) )
	  {
	    clear_bit( ucEventFlags, EVENT_ENCODER_DOWN );

	    // Move down to the next item
	    if ( test_bit( ucTempMode, MODE_HEAT_ON ) )
	    {
	      clear_bit( ucTempMode, MODE_HEAT_ON );
		  ucState = UPDATE_THE_DISPLAY;
		  continue;
	    }
	    if ( test_bit( ucTempMode, MODE_COOL_ON ) )
	    {
	      clear_bit( ucTempMode, MODE_COOL_ON );
	      set_bit( ucTempMode, MODE_HEAT_ON );
		  ucState = UPDATE_THE_DISPLAY;
		  continue;
	    }
	    set_bit( ucTempMode, MODE_COOL_ON );
		ucState = UPDATE_THE_DISPLAY;
	    continue;
	  }

	  if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
	  {
	    clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
	    if ( stEepromVars.ucSystemMode != ucTempMode )
		{
	      stEepromVars.ucSystemMode = ucTempMode;
		  if ( test_bit( stEepromVars.ucSystemMode, MODE_COOL_ON ) )
			uiSetTemp10C_Stage = stEepromVars.uiSetTemp10C_Cool;
		  else
			uiSetTemp10C_Stage = stEepromVars.uiSetTemp10C_Heat;
		  set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );

		  uiSetTemp10C_Final = uiSetTemp10C_Stage;  
		}
	    ucState = ST_DONE;
	  }
	}
  }
}

/*-----------------------------------------------------------------
  program_fan()

  Manage programming the fan dialog.
-----------------------------------------------------------------*/
void program_fan()
{
  const char *msg1a = "Select Fan:";
  const char *msg2a = "*On* ";
  const char *msg2b = "On ";
  const char *msg3a = "*Off*";
  const char *msg3b = "Off";

  uint8_t ucTempMode = stEepromVars.ucSystemMode, ucState = UPDATE_THE_DISPLAY;

  // We have 2 states: "on" and "off". Highlight the
  // current state by bracketing it with *.

  while ( ucState != ST_DONE )
  {
	wdt_reset();

	if ( uiMsPressed > 1000 )
	{
	  ucState = ST_DONE;
	  continue;
	}

    if ( ucState == UPDATE_THE_DISPLAY )
	{
	  lcd_clrscr();
	  lcd_puts( msg1a );
	  lcd_gotoxy( 0, 1 );

      // Display the current state
      if ( test_bit( ucTempMode, MODE_FAN_ON ) )
        lcd_puts( msg2a );
	  else
        lcd_puts( msg2b );
      if ( !test_bit( ucTempMode, MODE_FAN_ON ) )
        lcd_puts( msg3a );
	  else
        lcd_puts( msg3b );

	  ucState = WAITFOR_EVENT;
	}

    if ( ucState == WAITFOR_EVENT )
	{
  	  if ( test_bit( ucEventFlags, EVENT_ENCODER_UP ) )
	  {
	    clear_bit( ucEventFlags, EVENT_ENCODER_UP );

	    // Move up to the next item
	    if ( test_bit( ucTempMode, MODE_FAN_ON ) )
	    {
	      clear_bit( ucTempMode, MODE_FAN_ON );
		  ucState = UPDATE_THE_DISPLAY;
		  continue;
	    }
	    set_bit( ucTempMode, MODE_FAN_ON );
		ucState = UPDATE_THE_DISPLAY;
	    continue;
	  }

  	  if ( test_bit( ucEventFlags, EVENT_ENCODER_DOWN ) )
	  {
	    clear_bit( ucEventFlags, EVENT_ENCODER_DOWN );

	    // Move down to the next item
	    if ( test_bit( ucTempMode, MODE_FAN_ON ) )
	    {
	      clear_bit( ucTempMode, MODE_FAN_ON );
		  ucState = UPDATE_THE_DISPLAY;
		  continue;
	    }
	    set_bit( ucTempMode, MODE_FAN_ON );
		ucState = UPDATE_THE_DISPLAY;
	    continue;
	  }

	  if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
	  {
	    clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
	    if ( stEepromVars.ucSystemMode != ucTempMode )
		{
	      stEepromVars.ucSystemMode = ucTempMode;
		  set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
	      ucState = ST_UPDATE;
		}
		else
	      ucState = ST_DONE;
	  }
	}

    if ( ucState == ST_UPDATE )
	{
  	  // Make the fan relay follow the fan setting
	  if ( test_bit( stEepromVars.ucSystemMode, MODE_FAN_ON ) )
		clear_bit( RELAY_PORT, RELAY_FAN );
	  else
		set_bit( RELAY_PORT, RELAY_FAN );
	  ucState = ST_DONE;
	}
  }
}

/*-----------------------------------------------------------------
  pgm_clock_query()

  Manage programming the clock query dialog.
-----------------------------------------------------------------*/
uint8_t pgm_clock_query()
{
  const char *msg1  = "Program Clock?";
  const char *msg2a = "*No* ";
  const char *msg2b = "No ";
  const char *msg3a = "*Yes*";
  const char *msg3b = "Yes";

  uint8_t ucResult=0, ucState=UPDATE_THE_DISPLAY;

  // We have 2 states: "no" and "yes". Highlight the
  // current state by bracketing it with *.

  while ( ucState != ST_DONE )
  {
	wdt_reset();

	if ( uiMsPressed > 1000 )
	{
	  ucState = ST_DONE;
	  continue;
	}

    if ( ucState == UPDATE_THE_DISPLAY )
	{
	  lcd_clrscr();
	  lcd_puts( msg1 );
	  lcd_gotoxy( 0, 1 );

      // Display the current state
      if ( ucResult )
        lcd_puts( msg2b );
	  else
        lcd_puts( msg2a );
      if ( !ucResult )
        lcd_puts( msg3b );
	  else
        lcd_puts( msg3a );

	  ucState = WAITFOR_EVENT;
	}

    if ( ucState == WAITFOR_EVENT )
	{
  	  if ( test_bit( ucEventFlags, EVENT_ENCODER_UP ) )
	  {
	    clear_bit( ucEventFlags, EVENT_ENCODER_UP );
	    // Move to the previous item
		ucResult = ( ucResult ) ? 0 : 1;
		ucState = UPDATE_THE_DISPLAY;
	    continue;
	  }

  	  if ( test_bit( ucEventFlags, EVENT_ENCODER_DOWN ) )
	  {
	    clear_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	    // Move to the next item
		ucResult = ( ucResult ) ? 0 : 1;
		ucState = UPDATE_THE_DISPLAY;
	    continue;
	  }

	  if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
	  {
	    clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
	    ucState = ST_DONE;
	  }
	}
  }

  return ucResult;
}

/*-----------------------------------------------------------------
  program_clock()

  Manage programming the clock dialogs. 
-----------------------------------------------------------------*/
void program_clock()
{
  const char *msg1  = "Year:";
  const char *msg2  = "Month:";
  const char *msg3  = "Day:";
  const char *msg4  = "Hour:";
  const char *msg5  = "Minute:";
  uint8_t ucHigh, ucLow, ucInit, ucRc, ucUpdated=0;

  // Program the year
  lcd_clrscr();
  lcd_puts( msg1 );
  ucLow = 0;
  ucHigh = 50;
  ucInit = stRtc.rtc_year;
  ucRc = update_number( ucInit, ucLow, ucHigh );
  if( ucRc != stRtc.rtc_year )
  {
	stRtc.rtc_year = ucRc;
	ucUpdated = 1;
  }

  // Program the month
  lcd_clrscr();
  lcd_puts( msg2 );
  ucLow = 1;
  ucHigh = 12;
  ucInit = stRtc.rtc_month;
  ucRc = update_number( ucInit, ucLow, ucHigh );
  if( ucRc != stRtc.rtc_month )
  {
	stRtc.rtc_month = ucRc;
	ucUpdated = 1;
  }

  // Program the day
  lcd_clrscr();
  lcd_puts( msg3 );
  ucLow = 1;
  ucHigh = 31;
  ucInit = stRtc.rtc_day;
  ucRc = update_number( ucInit, ucLow, ucHigh );
  if( ucRc != stRtc.rtc_day )
  {
	stRtc.rtc_day = ucRc;
	ucUpdated = 1;
  }

  // Program the hour
  lcd_clrscr();
  lcd_puts( msg4 );
  ucLow = 1;
  ucHigh = 24;
  ucInit = stRtc.rtc_hours;
  ucRc = update_number( ucInit, ucLow, ucHigh );
  if( ucRc != stRtc.rtc_hours )
  {
	stRtc.rtc_hours = ucRc;
	ucUpdated = 1;
  }

  // Program the minute
  lcd_clrscr();
  lcd_puts( msg5 );
  ucLow = 1;
  ucHigh = 60;
  ucInit = stRtc.rtc_mins;
  ucRc = update_number( ucInit, ucLow, ucHigh );
  if( ucRc != stRtc.rtc_mins )
  {
	stRtc.rtc_mins = ucRc;
	ucUpdated = 1;
  }

  if ( ucUpdated )
	rtc_write();
}

/*-----------------------------------------------------------------
  update_number(...)

  Show a number on row 2. Allow the encoder to increase or
  decrease the number. 
-----------------------------------------------------------------*/
static uint8_t update_number( uint8_t initial_value, uint8_t low_bound,
							  uint8_t high_bound )
{
  uint8_t	ucResult=255, ucTemp=initial_value;
  char		buffer[6];

  while ( 1 )
  {
	wdt_reset();

	if ( ucResult != ucTemp )
	{
	  // Display current value
	  lcd_gotoxy( 5, 1 );
	  utoa( (uint16_t) ucTemp, buffer, 10 );
	  strextnd( buffer, ' ', 2 );
	  lcd_puts( buffer );
	  ucResult = ucTemp;
	}

  	if ( test_bit( ucEventFlags, EVENT_ENCODER_UP ) )
	{
	  clear_bit( ucEventFlags, EVENT_ENCODER_UP );
	  // increment the value
	  ++ucTemp;
	  if ( ucTemp > high_bound )
	    ucTemp = low_bound;
	  continue;
	}

  	if ( test_bit( ucEventFlags, EVENT_ENCODER_DOWN ) )
	{
	  clear_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	  // decrement the value
	  if ( ucTemp == low_bound )
		ucTemp = high_bound;
	  else
		--ucTemp;
	  continue;
	}

	if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
	{
	  clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
	  break;
	}
  }

  return ucResult;
}

/*-----------------------------------------------------------------
  strextnd(...)

  extend a string with fill character
-----------------------------------------------------------------*/
static void strextnd( char *s1, uint8_t fillchar, uint8_t newlength )
{
  uint8_t ii, nn;

  nn = strlen( s1 );
  if ( nn >= newlength ) return;

  for ( ii=nn; ii<newlength; ++ii ) *(s1+ii) = fillchar;
  *( s1+newlength ) = '\0';
}
