/*---------------------------------------------------------------------
  All code is provided for EDUCATIONAL purposes ONLY. There is no
  warranty and no support.

  Module: rtc.c

  This module supports a NJU6355 real time clock chip connected to
  PORTC as per the defines in the header file.
---------------------------------------------------------------------*/
#include <avr/io.h>
#include <util/delay.h>
#include <stdlib.h>
#include <string.h>
#include "common.h"
#include "nju6355.h"

// Global variables

extern RTCPARMS stRtc;

// Internal functions

static uint8_t BCD2Hex( uint8_t ucBCD );
static uint8_t Hex2BCD( uint8_t cBinary );
static uint8_t read_RTC( uint8_t nbits );
static void write_RTC( uint8_t cData, uint8_t nBits );

/*---------------------------------------------------------------------
  rtc_init()

  The first power up, the rtc contains invalid data. If this is
  detected, initialize the clock to Jan. 1, 2000, 00:00:00.
---------------------------------------------------------------------*/
void rtc_init()
{
  rtc_read();

  if ( stRtc.rtc_month > 0 && stRtc.rtc_month < 13 )
    return;                             // looks like its already initialized

  stRtc.rtc_year = 0;
  stRtc.rtc_month = 1;
  stRtc.rtc_day = 1;
  stRtc.rtc_dayweek = 6;
  stRtc.rtc_hours = 0;
  stRtc.rtc_mins = 0;
  stRtc.rtc_secs = 0;
  rtc_write();
}

/*---------------------------------------------------------------------
  rtc_read()

  This routine grabs the current time from the RTC and STORES IT IN
  BCD FORMAT in the registers Year, Month, Date, Day, Hours, Minutes,
  Seconds.
---------------------------------------------------------------------*/
void rtc_read()
{
  uint8_t ucNr;

  clear_bit( RTC_DATA_DDR, RTC_DATA_DDR_BIT );  // make the data pin input
  clear_bit( RTC_DATA_PORT_OUT, RTC_DATA_BIT ); // no pull-up

  clear_bit( RTC_CLK_PORT, RTC_CLK_BIT );
  nop();
  clear_bit( RTC_IO_PORT, RTC_IO_BIT );  // see timing diagram from NJU6355 datasheet
  nop();
  set_bit( RTC_CE_PORT, RTC_CE_BIT );

  // Read all 52 bits (Year[8],Month[8],Date[8],Day[4],Hours[8],Minutes[8],Seconds[8])
  ucNr = read_RTC( 8 );
  stRtc.rtc_year = BCD2Hex( ucNr );
  ucNr = read_RTC( 8 );
  stRtc.rtc_month = BCD2Hex( ucNr );
  ucNr = read_RTC( 8 );
  stRtc.rtc_day = BCD2Hex( ucNr );

  // Day value is only 4 bits (1 nibble).
  stRtc.rtc_dayweek = read_RTC( 4 );

  ucNr = read_RTC( 8 );
  stRtc.rtc_hours = BCD2Hex( ucNr );
  ucNr = read_RTC( 8 );
  stRtc.rtc_mins = BCD2Hex( ucNr );
  ucNr = read_RTC( 8 );
  stRtc.rtc_secs = BCD2Hex( ucNr );

  clear_bit( RTC_CE_PORT, RTC_CE_BIT );    // Turn off RTC output
}

/*---------------------------------------------------------------------
  rtc_write()

  This routine sets the time for the RTC. Note that upon power-up, the
  RTC has NO TIME SET. You must define the time and set it in the RTC
  before attempting to read it. Otherwise, no values will be returned.
  Note: the Get_Time routine reads the variables from the RTC and stores
  them in BCD format, not Hex. If you want to "reset" the time with the
  Set_Time routine, then you don't have to use the Hex2BCD conversion.
  In fact, if the original variables are defined in BCD format, you don't
  need to convert, either. But, since Hex is easier to work with and
  necessary for table lookup, it might be worthwhile to convert back and
  forth if you need to manipulate the variables or use a table.
---------------------------------------------------------------------*/
void rtc_write()
{
  uint8_t temp1;

  set_bit( RTC_DATA_DDR, RTC_DATA_DDR_BIT );  // make the data pin output

  clear_bit( RTC_CLK_PORT, RTC_CLK_BIT );
  nop();
  set_bit( RTC_IO_PORT, RTC_IO_BIT );
  nop();
  set_bit( RTC_CE_PORT, RTC_CE_BIT );

  // Send 44 bits total (Year[8],Month[8],Date[8],Day[4],Hours[8],Minutes[8])

  temp1 = Hex2BCD( stRtc.rtc_year );    // convert each variable to BCD
  write_RTC( temp1, 8 );
  temp1 = Hex2BCD( stRtc.rtc_month );
  write_RTC( temp1, 8 );
  temp1 = Hex2BCD( stRtc.rtc_day );
  write_RTC( temp1, 8 );
  temp1 = Hex2BCD( stRtc.rtc_dayweek );
  write_RTC( temp1, 4 );
  temp1 = Hex2BCD( stRtc.rtc_hours );
  write_RTC( temp1, 8 );
  temp1 = Hex2BCD( stRtc.rtc_mins );
  write_RTC( temp1, 8 );

  // Ignore Seconds - RTC does not accept "seconds" input
  clear_bit( RTC_CLK_PORT, RTC_CLK_BIT );   // pull the clock low for CE toggle
  nop();
  clear_bit( RTC_CE_PORT, RTC_CE_BIT );    // send the final write command to the RTC
  nop();
  set_bit( RTC_DATA_DDR, RTC_DATA_DDR_BIT ); // make the data pin input again
}

/*---------------------------------------------------------------------
  Hex2BCD(...)

  Convert a (binary) number in the range 0..99 to BCD.

---------------------------------------------------------------------*/
static uint8_t Hex2BCD( uint8_t cBinary )
{
  uint8_t	rc;
  char		buffer[4];
  int16_t	nn;

  // Convert number to decimal
  rc = ( cBinary > 99 ) ? 99 : cBinary;
  utoa( (uint16_t) rc, buffer, 10 );
  nn = strlen( buffer );

  // Make sure we have 2 digits
  if ( !nn ) return 0;

  if ( nn == 1 )
  {
    buffer[1] = buffer[0];
    buffer[0] = '0';
  }

  rc = ( buffer[0] << 4 ) | ( buffer[1] & 0x0f );

  return rc;
}

/*---------------------------------------------------------------------
  read_RTC(...)

  This just clocks the next RTC function into Time_Temp and returns it.
  Call with: 4 or 8 (bits to be read)
  Returns with: time function byte
---------------------------------------------------------------------*/
static uint8_t read_RTC( uint8_t nBits )
{
  uint8_t	Time_Temp, mask;
  int8_t	ii;

  for ( ii=0, mask=0x01, Time_Temp=0; ii<nBits; ++ii )
  {
    set_bit( RTC_CLK_PORT, RTC_CLK_BIT );   // signal we want to read a bit
    _delay_loop_1( 3 );        // wait 1 us

    if ( test_bit( RTC_DATA_PORT_IN, RTC_DATA_BIT ) )
      Time_Temp |= mask;

    clear_bit( RTC_CLK_PORT, RTC_CLK_BIT );
    _delay_loop_1( 3 );        // wait 1 us

    mask = mask << 1;
  }

  return Time_Temp;
}

/*---------------------------------------------------------------------
  write_RTC(...)

  Call with: Data, 4 or 8 (bits to be written)
---------------------------------------------------------------------*/
static void write_RTC( uint8_t cData, uint8_t nBits )
{
  int8_t	ii;

  for ( ii=0; ii<nBits; ++ii )
  {
    clear_bit( RTC_CLK_PORT, RTC_CLK_BIT );
    _delay_loop_1( 3 );                     // wait 1 us

    if ( test_bit( cData, ii ) )
      set_bit( RTC_DATA_PORT_OUT, RTC_DATA_BIT );
    else
      clear_bit( RTC_DATA_PORT_OUT, RTC_DATA_BIT );

    _delay_loop_1( 3 );                     // wait 1 us
    set_bit( RTC_CLK_PORT, RTC_CLK_BIT );
  }
}

/*---------------------------------------------------------------------
  BCD2Hex(...)

  Convert a (BCD) number in the range 0..99 to binary.
---------------------------------------------------------------------*/
static uint8_t BCD2Hex( uint8_t ucBCD )
{
  uint8_t ucLow, ucHigh, ucResult;

  ucLow = ucBCD & 0x0f;
  ucHigh = ucBCD >> 4;
  ucResult = ( ucHigh * 10 ) + ucLow;

  return ucResult;
}
