/*-----------------------------------------------------------------
  All code is provided for EDUCATIONAL purposes ONLY. There is no
  warranty and no support.

  thermostat3 by Jack Botner

  main.c

  This program implements a thermostat with the following hardware:
  - a 16 x 2 LCD display
  - a rotary encoder with pushbutton
  - a TC1047A temperature sensor
  - three relays that control the furnace/AC/Fan
  - NJU6355 real time clock

  Note: All temperatures in tenths degrees C

  Processor: ATMega164
      Clock: Internal, 8.0 MHz

  20090526 (v2.1.2) - Made Vref a define so it could be more easily
  					  customized if the cpu is changed
-----------------------------------------------------------------*/
#include <stdint.h>
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "lcd.h"
#include "nju6355.h"

#define DIGITAL_FILTER_SAMPLES 25	// to reduce jitter in the display

// The TC1047A is specified to plus/minus 2 degrees C, so some
// correction may be required.
#define TEMP_CORRECTION		0		// tenths of degrees C

// To improve the accuracy of the A/D, measure Vref at pin 32 (AREF).
// This varies from one cpu to another. 
#define VREF_MV				1083L	// actual 1.1V Vref, mV

#define COOL_LOCKOUT_SECS 180		// to prevent A/C from starting too soon
#define HEAT_LOCKOUT_SECS 120		// to prevent furnace from starting too soon

// System states
#define STATE_RUN			1
#define STATE_PROGRAM		2

// Global variables

MY_EEPROM_DATA stEepromVars;			// variables kept in EEPROM

uint8_t ucDebouncedState=0;    			// debounced state of the switches

static uint16_t uiActualTemp10C=0;		// currnt temperature, tenths degrees C
uint16_t uiMsPressed=0;					// how long encoder button pressed

// The following are the set temperatures in tenths degrees C. Normally
// they are identical, except when the user is pressing the up or down
// buttons. This is to prevent the user from changing the state of the
// system until the button pressing is complete. After a few seconds,
// uiSetTemp10C_Final is set from uiSetTemp10C_Stage.
uint16_t uiSetTemp10C_Stage=210;			// used during adjustments and for display
uint16_t uiSetTemp10C_Final=210;			// used for state analysis

uint8_t ucEventFlags=0;
uint16_t uiAdc=0;							// last ADC sample
static uint16_t uiSmoothedAdc=0;			// smoothed ADC value

uint8_t ucDelaySec=0;						// system response delay to temp. change
uint8_t ucHeatLockoutSec=0;					// delay in restarting furnace
uint8_t ucCoolLockoutSec=0;					// delay in restarting A/C
static uint8_t ucSystemState=STATE_RUN;		// system states

RTCPARMS stRtc;

// Internal Functions

static void convert_temperature(void);
static void digital_filter_adc(void);
static void display_state(void);
static void display_temp(void);
static void format_and_display_time(void);
static void state_analysis(void);
static void state_run(void);
static void state_program(void);
static void system_init(void);

/*-----------------------------------------------------------------
  main()
-----------------------------------------------------------------*/
int main( void )
{
  const char *msg1a = "Thermostat";
  const char *msg1b = "Version 2.1.2";

  uint8_t	 ii;

  system_init();          		// initialise system

  // Display the firmware version
  lcd_init( LCD_DISP_ON );
  lcd_home();
  lcd_puts( msg1a );
  lcd_gotoxy( 0, 1 );
  lcd_puts( msg1b );

  // Get the saved state variables from eeprom
  eeprom_read_block( &stEepromVars, 0, sizeof( MY_EEPROM_DATA ) );
  if ( stEepromVars.uiSetTemp10C_Heat > 400 ||		// probably EEPROM not initialized
	   stEepromVars.uiSetTemp10C_Cool > 400 ||
	   stEepromVars.ucSystemMode > 5 ||
	   stEepromVars.ucHysteresis_Heat > 20 ||
	   stEepromVars.ucHysteresis_Cool > 20 )
  {
    stEepromVars.uiSetTemp10C_Heat = 210;
    stEepromVars.uiSetTemp10C_Cool = 280;
    stEepromVars.ucSystemMode = 0;
    stEepromVars.ucHysteresis_Heat = HYSTERESIS_HEAT;
    stEepromVars.ucHysteresis_Cool = HYSTERESIS_COOL;
	set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
  }

  // Initialize the temperature states
  if ( test_bit( stEepromVars.ucSystemMode, MODE_COOL_ON ) )
	uiSetTemp10C_Stage = stEepromVars.uiSetTemp10C_Cool;
  else
	uiSetTemp10C_Stage = stEepromVars.uiSetTemp10C_Heat;

  uiSetTemp10C_Final = uiSetTemp10C_Stage;
  	
  // Initialize the fan relay
  if ( test_bit( stEepromVars.ucSystemMode, MODE_FAN_ON ) )
	clear_bit( RELAY_PORT, RELAY_FAN );

  // Initialize the rtc
  rtc_init();

  // Wait a bit then clear the screen
  for ( ii=0; ii<60; ++ii )		// 2 sec delay
    _delay_loop_2( 65000 );		// 32 ms delay
  lcd_clrscr();

  // Initialize the watchdog
  wdt_enable( WDTO_500MS );
  wdt_reset();

  // Now down to business...
  while (1)
  {
    if ( ucSystemState == STATE_RUN )
	  state_run();
	else
	  state_program();
  }
}

/*-----------------------------------------------------------------
  state_run()

  The mainline when the state is "run".
-----------------------------------------------------------------*/
static void state_run()
{
  wdt_reset();

  if ( test_bit( ucEventFlags, EVENT_CONVERT_TEMP ) )
  {
	digital_filter_adc();
	convert_temperature();
	clear_bit( ucEventFlags, EVENT_CONVERT_TEMP );
  }

  if ( test_bit( ucEventFlags, EVENT_UPDATE_LCD ) )
  {
    display_temp();
    display_state();
	clear_bit( ucEventFlags, EVENT_UPDATE_LCD );
  }

  if ( test_bit( ucEventFlags, EVENT_ANALYZE_STATE ) )
  {
	state_analysis();
	clear_bit( ucEventFlags, EVENT_ANALYZE_STATE );
  }

  if ( test_bit( ucEventFlags, EVENT_UPDATE_EEPROM ) )
  {
	eeprom_write_block( &stEepromVars, 0, sizeof( MY_EEPROM_DATA ) );
	clear_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
  }

  if ( test_bit( ucEventFlags, EVENT_ENCODER_UP ) )
  {
	uiSetTemp10C_Stage += SET_TEMP_DELTA;
	ucDelaySec = SYSTEM_DELAY_SEC;
    set_bit( ucEventFlags, EVENT_UPDATE_LCD );
  	clear_bit( ucEventFlags, EVENT_ENCODER_UP );
  }

  if ( test_bit( ucEventFlags, EVENT_ENCODER_DOWN ) )
  {
	uiSetTemp10C_Stage -= SET_TEMP_DELTA;
	ucDelaySec = SYSTEM_DELAY_SEC;
    set_bit( ucEventFlags, EVENT_UPDATE_LCD );
	clear_bit( ucEventFlags, EVENT_ENCODER_DOWN );
  }

  if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
  {
    ucSystemState = STATE_PROGRAM;
	clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
  }

  if ( test_bit( ucEventFlags, EVENT_READ_RTC ) )
  {
	rtc_read();
	format_and_display_time();

	clear_bit( ucEventFlags, EVENT_READ_RTC );
  }
}

/*-----------------------------------------------------------------
  state_program()

  The mainline when the state is "program".
-----------------------------------------------------------------*/
static void state_program()
{
  uint8_t ucRc=0;

  program_mode();

  if ( uiMsPressed < 1000 )
    program_fan();

  if ( uiMsPressed < 1000 )
	program_hysteresis();

  if ( uiMsPressed < 1000 )
	ucRc = pgm_clock_query();

  if ( ucRc )	// update clock requested
  {
	if ( uiMsPressed < 1000 )
	  program_clock();
  }

  lcd_clrscr();
  ucDelaySec = SYSTEM_DELAY_SEC;
  ucSystemState = STATE_RUN;
}

/*-----------------------------------------------------------------
  system_init()

  PORTA: PA0 (input) (ADC0) temperature sensor
         PA1 (output) LCD RS
         PA2 (output) LCD RW
         PA3 (output) LCD E
         PA4 (output) LCD DB4
         PB5 (output) LCD DB5
         PA6 (output) LCD DB6
         PA7 (output) LCD DB7

  PORTB: PB0 (output) RTC data
         PB1 (output) RTC I/O
         PB2 (output) RTC CLK
         PB3 (output) RTC CE
         PB4 (output)
         PB5 (input) ISP MOSI
         PB6 (input) ISP MISO
         PB7 (input) ISP SCK

  PORTC: PC0 (output)
         PC1 (output)
         PC2 (output)
         PC3 (output)
         PC4 (output) (do not use)
         PC5 (output) (do not use)
         PC6 (output) (do not use) 
		 PC7 (output) 

  PORTD: PD0 (output) HEAT relay
         PD1 (output) COOL relay
         PD2 (output) FAN relay
         PD3 (output)
         PD4 (output)
         PD5 (input) Rotary Encoder "B"
         PD6 (input) Rotary Encoder "A"
         PD7 (input) Rotary Encoder "PB"
-----------------------------------------------------------------*/
static void system_init()
{
  // The fuses are set to provide an internal clock, running at
  // 8.0 MHz with a prescaler of 8. Change the prescaler from
  // 8 to 1. The procedure requires two steps as per the device
  // specifications.
  CLKPR = 0b10000000;       // set clock prescaler change enable
  CLKPR = 0b00000000;       // set prescaler to 1

  // Set up the I/O ports directions, pull-ups
  DDRA  = 0b11111110;		// pa0=input, pa1..7=output
  PORTA = 0b00000000;       // initialize PortA

  DDRB  = 0b00011111;		// pb0..4=output, pb5..7=input
  PORTB = 0b00000000;       // initialize PortB

  DDRC  = 0b11111111;		// pc0..7=output
  PORTC = 0b00000000;       // initialize PortC

  DDRD  = 0b00011111;		// pd0..4=output, pd5..7=input
  PORTD = 0b11101111;       // initialize PortD; pull-ups on pd3,5..7

  // Set up PA0/ADC0 as analog input, Vref=1.1V
  ADMUX  = 0b10000000;		// Vref=1.1V; result right adjusted; ADC0/no gain
  ADCSRA = 0b10001110;		// Enable ADC; Enable interrupt; prescaler = 64
  ADCSRB = 0b00000000;		// free running mode
  DIDR0  = 0b00000001;		// disable digital I/O on PA0

  // Initialize Timer0. 
  // Timer0 is used to create 32 ms intervals for scheduling events
  TCCR0A = 0b00000000;		// normal port operation
  TCCR0B = 0b00000101;		// prescaler = 1024
  TCNT0 = INIT_TIMER0;
  set_bit( TIMSK0, TOIE0 );	// enable timer/counter0 overflow interrupt

  // Initialize Timer1.
  // Timer1 is used to create 1 s intervals for scheduling events
  TCCR1A = 0b00000000;		// normal operation, no OC1A, no OC1B
  TCCR1B = 0b00000100;      // internal clock, prescaler = 256
  TCNT1 = INIT_TIMER1;
  set_bit( TIMSK1, TOIE1 );	// enable timer/counter1 overflow interrupt

  // Initialize Timer2. Timer2 is used to generate interrupts
  // every 2 milliseconds for switch debouncing.
  TCCR2A = 0b00000000;		// normal port operation
  TCCR2B = 0b00000100;		// normal operation, prescaler=64
  TCNT2 = INIT_TIMER2;
  set_bit( TIMSK2, TOIE2 );	// enable timer/counter2 overflow interrupt

  // Initialize external interrupt on PD3/INT1
  EICRA = 0b00001000;		// falling edge of INT1 produces interrupt
  EIMSK = 0b00000010;		// enable the interrupt on INT1

  sei();					// enable global interrupts
}

/*-----------------------------------------------------------------
  convert_temperature()

  Convert the temperature from the A/D output to 10ths of degrees C.
  ADC0 is monitoring TC1047A Vout signal, and Vref = 1.1V (internal).

  First we need to convert the A/D voltage to millivolts, then
  to tenths of degrees C.
  
  The formula for voltage to temperature conversion is 
  
				  V - 500
			T = ------------
				   10.0
				   
  where T is temperature in degrees C, and V is Vout in millivolts.

  Input: uiSmoothedAdc from the adc filter.

  Output: uiActualTemp10C
-----------------------------------------------------------------*/
static void convert_temperature()
{
  uint32_t ulVoutMv;
  uint16_t uiTemp10C;

  // Input is in the range 0..1023 which represents a voltage of 0..1.10
  // volts (Vref). Scale the adc value to a voltage value (millivolts).

  ulVoutMv = ( (uint32_t) uiSmoothedAdc * VREF_MV ) / 1024L;

  // Derive temperature (tens of degrees C) from the formula above.
  // Multiply numerator and denominator by 100 to handle decimals.
  // Multiply numerator by 10 to get tens of degrees.
  
  //uiTempTensC = (unsigned int) ( ( ( ulVoutMv - 500L ) * 1000L ) / 1000L );
  uiTemp10C = (uint16_t) ( ulVoutMv - 500L );

  if ( uiTemp10C > 999 )
    uiTemp10C = 999;		// sensor may be disconnected

  uiActualTemp10C = uiTemp10C - TEMP_CORRECTION;
}

/*-----------------------------------------------------------------
  display_temp()

  Format the tenths of degrees and display on LCD.
-----------------------------------------------------------------*/
static void display_temp()
{
  const char 	*msg2a = "Tmp        ";
  const char 	*msg2b = "Set";
  char		 	buffer[16];
  uint16_t	 	uiLen;

  // Display the actual temperature on line 1

  lcd_gotoxy( 0, 0 );
  lcd_puts( msg2a );

  utoa( uiActualTemp10C, buffer, 10 );

  // Insert a decimal point in the result
  uiLen = strlen( buffer );
  if ( uiLen )
    uiLen -= 1;
  buffer[uiLen+1] = buffer[uiLen];
  buffer[uiLen] = '.';
  buffer[uiLen+2] = '\0';

  lcd_gotoxy( 4, 0 );
  lcd_puts( buffer );

  // Display the set temperature on line 2

  lcd_gotoxy( 0, 1 );
  lcd_puts( msg2b );

  utoa( uiSetTemp10C_Stage, buffer, 10 );

  // Insert a decimal point in the result
  uiLen = strlen( buffer );
  if ( uiLen )
    uiLen -= 1;
  buffer[uiLen+1] = buffer[uiLen];
  buffer[uiLen] = '.';
  buffer[uiLen+2] = '\0';

  lcd_gotoxy( 4, 1 );
  lcd_puts( buffer );
}

/*-----------------------------------------------------------------
  display_state()

  Display the system mode on the LCD.
  In the top right part of the display.
-----------------------------------------------------------------*/
static void display_state()
{
  const char *msg3a = "Heat";
  const char *msg3b = "Cool";
  const char *msg3c = " Off";
  const char *msg3d = "On";
  const char *msg3e = "  ";

  lcd_gotoxy( 9, 0 );
  if ( test_bit( stEepromVars.ucSystemMode, MODE_HEAT_ON ) )
    lcd_puts( msg3a );
  else
    if ( test_bit( stEepromVars.ucSystemMode, MODE_COOL_ON ) )
      lcd_puts( msg3b );
    else
      lcd_puts( msg3c );

  lcd_gotoxy( 14, 0 );
  if ( test_bit( RELAY_PORT, RELAY_HEAT ) && test_bit( RELAY_PORT, RELAY_COOL ) )
    lcd_puts( msg3e );
  else
    lcd_puts( msg3d );
}

/*-----------------------------------------------------------------
  state_analysis()

  Decides if the cool/heat relays should be on or off.
-----------------------------------------------------------------*/
static void state_analysis()
{
  if ( test_bit( stEepromVars.ucSystemMode, MODE_HEAT_ON ) )
  {
	// mode=Heat
    if ( test_bit( RELAY_PORT, RELAY_HEAT ) )
	{
	  // The heat relay is off. If the temperature drops 1/2 degree C
	  // below our setting, turn it on.
	  if ( uiActualTemp10C < ( uiSetTemp10C_Final - stEepromVars.ucHysteresis_Heat ) )
	  {
	    if ( !ucHeatLockoutSec )				// if not locked out
		  clear_bit( RELAY_PORT, RELAY_HEAT );	// turn heat on
	  }
	}
	else
	{
	  // The heat relay is on. If the temperature is more than specified
	  // degrees C above our setting, turn it off.
	  if ( uiActualTemp10C > ( uiSetTemp10C_Final + stEepromVars.ucHysteresis_Heat ) )
	  {
		set_bit( RELAY_PORT, RELAY_HEAT );		// turn heat off
		ucHeatLockoutSec = HEAT_LOCKOUT_SECS;	// keep it off for awhile	
	  }
	}
  }
  else
  {
    // !mode=Heat
    if ( !test_bit( RELAY_PORT, RELAY_HEAT ) )	// the heat relay is on
	{
	  set_bit( RELAY_PORT, RELAY_HEAT );		// turn it off
	  ucHeatLockoutSec = HEAT_LOCKOUT_SECS;		// keep it off for awhile	
	}
  }

  if ( test_bit( stEepromVars.ucSystemMode, MODE_COOL_ON ) )
  {
	// The cool switch is on
    if ( test_bit( RELAY_PORT, RELAY_COOL ) )
	{
	  // The cool relay is off. If the temperature rises specified degrees C
	  // above our setting, turn it on.
	  if ( uiActualTemp10C > ( uiSetTemp10C_Final + stEepromVars.ucHysteresis_Cool ) )
	  {
	    if ( !ucCoolLockoutSec )				// if not locked out
		  clear_bit( RELAY_PORT, RELAY_COOL );	// turn cool on
	  }
	}
	else
	{
	  // The cool relay is on. If the temperature drops more than 1/2
	  // degree C below our setting, turn it off.
	  if ( uiActualTemp10C < ( uiSetTemp10C_Final - stEepromVars.ucHysteresis_Cool ) )
	  {
		set_bit( RELAY_PORT, RELAY_COOL );		// turn cool off
		ucCoolLockoutSec = COOL_LOCKOUT_SECS;	// keep it off for awhile	
      }
	}
  }
  else
  {
    // The cool switch is off
    if ( !test_bit( RELAY_PORT, RELAY_COOL ) )	// the cool relay is on
	{
	  set_bit( RELAY_PORT, RELAY_COOL );		// turn it off
	  ucCoolLockoutSec = COOL_LOCKOUT_SECS;		// keep it off for awhile	
	}
  }
}

/*-----------------------------------------------------------------
  format_and_display_time()

  Display the time on the LCD.
  In the bottom right part of the display.
-----------------------------------------------------------------*/
static void format_and_display_time()
{
  char	buffer[6];

  lcd_gotoxy( 11, 1 );
  utoa( (uint16_t) stRtc.rtc_hours, buffer, 10 );
  if ( strlen( buffer ) == 1 )
  {
    buffer[2] = '\0';
	buffer[1] = buffer[0];
	buffer[0] = '0';
  }
  lcd_puts( buffer );
  lcd_putc( ':' );
  utoa( (uint16_t) stRtc.rtc_mins, buffer, 10 );
  if ( strlen( buffer ) == 1 )
  {
    buffer[2] = '\0';
	buffer[1] = buffer[0];
	buffer[0] = '0';
  }
  lcd_puts( buffer );
}

/*---------------------------------------------------------------------
  digital_filter_adc()

  Why do we need this? Because the output of the adc jitters one or
  two counts each sample, which translates to one or two tenths of a
  degree C on the display. It goes up and down apparently randomly,
  which is very annoying. According to the Atmel spec, the adc has a
  number of error modes which result in the lsb being uncertain.
---------------------------------------------------------------------*/
static void digital_filter_adc()
{
  static uint16_t uiTable[DIGITAL_FILTER_SAMPLES];
  static uint8_t  ucIndex=0, ucEntries=0;
  uint8_t		  ii;
  uint32_t		  ulSum;

  // Add the entry to the table, wrap-around
  uiTable[ucIndex] = uiAdc;
  ++ucIndex;
  if ( ucIndex >= DIGITAL_FILTER_SAMPLES )
    ucIndex = 0;
  if ( ucEntries < DIGITAL_FILTER_SAMPLES )
    ++ucEntries;

  // Average the table
  ulSum = 0;
  for ( ii=0; ii<ucEntries; ++ii )
    ulSum += uiTable[ii];

  uiSmoothedAdc = (unsigned int) ( ulSum / ucEntries );
}
