/*-----------------------------------------------------------------
  All code is provided for EDUCATIONAL purposes ONLY. There is no
  warranty and no support.

  thermostat2 by Jack Botner

  interrupt.c

  This module contains all the interrupt handlers.
-----------------------------------------------------------------*/
#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"

// Global variables

extern MY_EEPROM_DATA stEepromVars;		// variables kept in EEPROM

extern uint8_t  ucEventFlags;			// system event flags
extern uint8_t  ucDelaySec;				// system response delay to temp. change
extern uint16_t uiSetTemp10C_Stage;		// used during adjustments and for display
extern uint16_t uiSetTemp10C_Final;		// used for state analysis
extern uint8_t  ucHeatLockoutSec;		// delay in restarting furnace
extern uint8_t  ucCoolLockoutSec;		// delay in restarting A/C
extern uint8_t  ucDebouncedState;		// debounced state of the switches
extern uint16_t uiAdc;					// last ADC sample
extern uint16_t uiMsPressed;			// how long encoder button pressed

static uint8_t  ucPreviousState=0; 		// previous switch state

// Internal Functions

static void analyze_switch(void);

/*-----------------------------------------------------------------
  ISR(TIMER0_OVF_vect)

  Timer 0 overflow interrupt handler 

  Used to provide 32 ms intervals for various timings.

  System clock frequency = 8 mHz
  System clock period = 125 ns
  Prescaler = 1024
  Timer count period = 128 us
  Counts = 250
  Interrupt period = 32 ms
-----------------------------------------------------------------*/
ISR(TIMER0_OVF_vect)
{
  static uint8_t ucLcdCtr=13;

  // Schedule an update of the LCD periodically

  ++ucLcdCtr;
  if ( ucLcdCtr >= 15 )			// every 1/2 second
  {
    set_bit( ucEventFlags, EVENT_UPDATE_LCD );
	ucLcdCtr = 0;
  }

  TCNT0 = INIT_TIMER0;
}

/*-----------------------------------------------------------------
  ISR(TIMER1_OVF_vect)

  Timer/Counter 1 overflow interrupt handler
  Used to provide 1 second intervals for various timings.

  System clock frequency = 8 mHz
  System clock period = 125 ns
  Prescaler = 256
  Timer count period = 32 us
  Counts = 31250
  Interrupt period = 1.0 s
-----------------------------------------------------------------*/
ISR(TIMER1_OVF_vect)
{
  static uint8_t ucAdcCnt=9, ucAnalCnt=1;

  // Initiate an A/D conversion periodically

  ++ucAdcCnt;
  if ( ucAdcCnt >= 5 )			// every 5 seconds
  {
    set_bit( ADCSRA, ADSC );	// initiate the ADC conversion
	ucAdcCnt = 0;
  }

  // Initiate a system state analysis periodically

  ++ucAnalCnt;
  if ( ucAnalCnt == 1 )
    set_bit( ucEventFlags, EVENT_READ_RTC );
  else
	if ( ucAnalCnt == 2 )		// every 2 seconds
	{
	  set_bit( ucEventFlags, EVENT_ANALYZE_STATE );
      ucAnalCnt = 0;
	}

  // If uiSetTemp10C_Stage has changed, update uiSetTemp10C_Final after
  // the specified period.

  if ( ucDelaySec )
  {
    -- ucDelaySec;
	if ( !ucDelaySec )
	{
	  uiSetTemp10C_Final = uiSetTemp10C_Stage;

	  if ( test_bit( stEepromVars.ucSystemMode, MODE_COOL_ON ) )
	    stEepromVars.uiSetTemp10C_Cool = uiSetTemp10C_Final;
	  else
	    stEepromVars.uiSetTemp10C_Heat = uiSetTemp10C_Final;

	  set_bit( ucEventFlags, EVENT_UPDATE_EEPROM );
    }
  }

  // If a lockout timer is on, decrement it
  if ( ucHeatLockoutSec )
    --ucHeatLockoutSec;
  if ( ucCoolLockoutSec )
    --ucCoolLockoutSec;

  TCNT1 = INIT_TIMER1;
}

/*-----------------------------------------------------------------
  ISR(TIMER2_OVF_vect)

  Timer 2 overflow interrupt handler - tests for button presses

  System clock frequency = 8 mHz
  System clock period = 125 ns
  Prescaler = 64
  Timer count period = 8 us
  Counts = 250
  Interrupt period = 2 ms
------------------------------------------------------------------*/
ISR(TIMER2_OVF_vect)
{
  static volatile uint8_t ucFirst=1;
  static volatile uint8_t ucIndex=0;             	// index into ucState
  static volatile uint8_t ucState[MAX_CHECKS] =  	// array containing bounce status
        	{ 0, 0, 0, 0, 0, 0 };

  uint8_t ii, ucResult, ucSwIn=0;

  // We use the technique documented in "A Guide to Debouncing" by
  // Jack Ganssle to debounce up to 8 inputs (5 in this case).

  // Read current switch states
  if ( !test_bit( ENCODER_PORT, ENCODER_A ) )
    set_bit( ucSwIn, ENC_SWITCH_A );
  if ( !test_bit( ENCODER_PORT, ENCODER_B ) )
    set_bit( ucSwIn, ENC_SWITCH_B );
  if ( !test_bit( ENCODER_PORT, ENCODER_PB ) )
    set_bit( ucSwIn, ENC_SWITCH_PB );

  ucState[ucIndex] = ucSwIn;

  ++ucIndex;
  if ( ucIndex == MAX_CHECKS )
    ucIndex = 0;

  // And all columns of the array
  for ( ii=0, ucResult=0xff; ii<(MAX_CHECKS-1); ++ii )
    ucResult = ucResult & ucState[ii];

  ucDebouncedState = ucResult;

  // The 1st time only, set the previous switch state to the
  // current switch state.
  if ( ucFirst )
  {
    ucPreviousState = ucDebouncedState;
	ucFirst = 0;
  }

  // Since we're here every 2 milliseconds, keep count
  // of how many milliseconds the encoder button is pressed

  if ( test_bit( ucDebouncedState, ENC_SWITCH_PB ) )
    uiMsPressed += 2;
  else
    uiMsPressed = 0;

  if ( ucDebouncedState != ucPreviousState )
    analyze_switch();		// convert switch change to event

  TCNT2 = INIT_TIMER2;
}

/*-----------------------------------------------------------------
  ISR( ADC_vect ) - ADC conversion complete interrupt handler

  Capture the A/D result registers into static memory uiAdc.
  Schedule an event to process the result.
----------------------------------------------------------------*/
ISR( ADC_vect )
{
  // ADCL must be read before ADCH.
  uiAdc = (uint16_t) ADCL | ( (uint16_t) ADCH << 8 );

  set_bit( ucEventFlags, EVENT_CONVERT_TEMP );
}

/*-----------------------------------------------------------------
  ISR( INT1_vect ) - External Interrupt handler routine

  This interrupt occurs every second courtesy of the DS1305 RTC.
----------------------------------------------------------------*/
ISR( INT1_vect )
{
  set_bit( ucEventFlags, EVENT_READ_RTC );
}


/*-----------------------------------------------------------------
  analyze_switch()

  Analyzes the rotary encoder to determine if the shaft rotation
  is up or down, or if the pushbutton has been pressed.
-----------------------------------------------------------------*/
static void analyze_switch()
{
  uint8_t	ucNow, ucThen;

  if ( test_bit( ucDebouncedState, ENC_SWITCH_PB ) && 
       !test_bit( ucPreviousState, ENC_SWITCH_PB ) )
  {
    // The encoder button has been pressed
	set_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
  }

  // Convert the position of the encoder to a number 0..3
  ucNow = ucDebouncedState & 0x03;
  ucThen = ucPreviousState & 0x03;

  // Given the A output in bit 0 and B output in bit 1 positions, the
  // encoder generates the values 0, 1, 3, 2 as it is rotated clockwise.

  if ( ucNow == 0 )
  {
    if ( ucThen == 2 )
	  set_bit( ucEventFlags, EVENT_ENCODER_UP );
	else
	  if ( ucThen == 1 )
		set_bit( ucEventFlags, EVENT_ENCODER_DOWN );
  }
  else
	if ( ucNow == 1 )
	{
      if ( ucThen == 0 )
	    set_bit( ucEventFlags, EVENT_ENCODER_UP );
	  else
	    if ( ucThen == 3 )
		  set_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	}
	else
	  if ( ucNow == 3 )
	  {
    	if ( ucThen == 1 )
		  set_bit( ucEventFlags, EVENT_ENCODER_UP );
		else
	 	  if ( ucThen == 2 )
			set_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	  }
	  else	// ucNow == 2
	  {
    	if ( ucThen == 3 )
		  set_bit( ucEventFlags, EVENT_ENCODER_UP );
		else
	 	  if ( ucThen == 0 )
			set_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	  }

  ucPreviousState = ucDebouncedState;
}
