/*---------------------------------------------------------------------
  Module: rtc.c

  This module supports a NJU6355 real time clock chip connected to
  PORTC as per the defines in the header file.
---------------------------------------------------------------------*/
#include <system.h>
#include "rtc.h"

// Global variables

extern RTCPARMS stRtc;

// Internal functions

static unsigned char BCD2Hex( unsigned char ucBCD );
static unsigned char Hex2BCD( unsigned char cBinary );
static unsigned char read_RTC( unsigned char nbits );
static void write_RTC( unsigned char cData, unsigned char nBits );

/*---------------------------------------------------------------------
  rtc_init()

  The first power up, the rtc contains invalid data. If this is
  detected, initialize the clock to Jan. 1, 2000, 00:00:00.
---------------------------------------------------------------------*/
void rtc_init()
{
  rtc_read();

  if ( stRtc.rtc_month > 0 && stRtc.rtc_month < 13 )
    return;				// looks like its already initialized

  stRtc.rtc_year = 0;
  stRtc.rtc_month = 1;
  stRtc.rtc_day = 1;
  stRtc.rtc_dayweek = 6;
  stRtc.rtc_hours = 0;
  stRtc.rtc_mins = 0;
  stRtc.rtc_secs = 0;
  rtc_write();
}

/*---------------------------------------------------------------------
  rtc_read()

  This routine grabs the current time from the RTC and STORES IT IN
  BCD FORMAT in the registers Year, Month, Date, Day, Hours, Minutes,
  Seconds.
---------------------------------------------------------------------*/
void rtc_read()
{
  unsigned char ucNr;

  set_bit( RTC_TRIS, RTC_DATA );	// make the data pin input
 
  clear_bit( RTC_PORT, RTC_CLK );
  nop();
  clear_bit( RTC_PORT, RTC_IO );    // see timing diagram from NJU6355 datasheet
  nop();
  set_bit( RTC_PORT, RTC_CE );

  // Read all 52 bits (Year[8],Month[8],Date[8],Day[4],Hours[8],Minutes[8],Seconds[8])
  //stRtc.rtc_year    = read_RTC( 8 );
  //stRtc.rtc_month   = read_RTC( 8 );
  //stRtc.rtc_day     = read_RTC( 8 );
  ucNr = read_RTC( 8 );
  stRtc.rtc_year = BCD2Hex( ucNr );
  ucNr = read_RTC( 8 );
  stRtc.rtc_month = BCD2Hex( ucNr );
  ucNr = read_RTC( 8 );
  stRtc.rtc_day = BCD2Hex( ucNr );
  
  // Day value is only 4 bits (1 nibble).
  stRtc.rtc_dayweek = read_RTC( 4 );

  //stRtc.rtc_hours   = read_RTC( 8 );
  //stRtc.rtc_mins    = read_RTC( 8 );
  //stRtc.rtc_secs    = read_RTC( 8 );
  ucNr = read_RTC( 8 );
  stRtc.rtc_hours = BCD2Hex( ucNr );
  ucNr = read_RTC( 8 );
  stRtc.rtc_mins = BCD2Hex( ucNr );
  ucNr = read_RTC( 8 );
  stRtc.rtc_secs = BCD2Hex( ucNr );

  clear_bit( RTC_PORT, RTC_CE );    // Turn off RTC output
}

/*---------------------------------------------------------------------
  rtc_write()

  This routine sets the time for the RTC. Note that upon power-up, the
  RTC has NO TIME SET. You must define the time and set it in the RTC
  before attempting to read it. Otherwise, no values will be returned.
  Note: the Get_Time routine reads the variables from the RTC and stores
  them in BCD format, not Hex. If you want to "reset" the time with the
  Set_Time routine, then you don't have to use the Hex2BCD conversion.
  In fact, if the original variables are defined in BCD format, you don't
  need to convert, either. But, since Hex is easier to work with and
  necessary for table lookup, it might be worthwhile to convert back and
  forth if you need to manipulate the variables or use a table.
---------------------------------------------------------------------*/
void rtc_write()
{
  unsigned char temp1;

  clear_bit( RTC_TRIS, RTC_DATA );	// make the data pin output

  clear_bit( RTC_PORT, RTC_CLK );
  nop();
  set_bit( RTC_PORT, RTC_IO );
  nop();
  set_bit( RTC_PORT, RTC_CE );

  // Send 44 bits total (Year[8],Month[8],Date[8],Day[4],Hours[8],Minutes[8])

  temp1 = Hex2BCD( stRtc.rtc_year );    // convert each variable to BCD
  write_RTC( temp1, 8 );
  temp1 = Hex2BCD( stRtc.rtc_month );
  write_RTC( temp1, 8 );
  temp1 = Hex2BCD( stRtc.rtc_day );
  write_RTC( temp1, 8 );
  temp1 = Hex2BCD( stRtc.rtc_dayweek );
  write_RTC( temp1, 4 );
  temp1 = Hex2BCD( stRtc.rtc_hours );
  write_RTC( temp1, 8 );
  temp1 = Hex2BCD( stRtc.rtc_mins );
  write_RTC( temp1, 8 );

  // Ignore Seconds - RTC does not accept "seconds" input
  clear_bit( RTC_PORT, RTC_CLK );   // pull the clock low for CE toggle
  nop();
  clear_bit( RTC_PORT, RTC_CE );    // send the final write command to the RTC
  nop();
  clear_bit( RTC_PORT, RTC_IO );    // leave the RTC in output mode
}

/*---------------------------------------------------------------------
  Hex2BCD(...)

  Convert a (binary) number in the range 0..99 to BCD.

---------------------------------------------------------------------*/
static unsigned char Hex2BCD( unsigned char cBinary )
{
  unsigned char temp1, temp2;

  temp1 = cBinary;

  asm
  {
    clrw
    btfsc   _temp1, 4
    addlw   0x16
    btfsc   _temp1, 5
    addlw   0x32
    addlw   0x06
    btfss   _status, DC        ; skpdc
    addlw   0xfa               ; -0x06
    btfsc   _temp1, 6
    addlw   0x64
    addlw   0x06
    btfss   _status, DC        ; skpdc
    addlw   0xfa               ; -0x06
    movwf   _temp2
    movf    _temp1, 0
    andlw   0x0F
    addwf   _temp2, 0
    btfsc   _status, DC        ; skpndc
    addlw   0x06
    addlw   0x06
    btfss   _status, DC        ; skpdc
    addlw   0xfa               ; -0x06
    movwf   _temp2
  }

  return temp2;
}

/*---------------------------------------------------------------------
  read_RTC(...)

  This just clocks the next RTC function into Time_Temp and returns it.
  Call with: 4 or 8 (bits to be read)
  Returns with: time function byte
---------------------------------------------------------------------*/
static unsigned char read_RTC( unsigned char nBits )
{
  unsigned char Time_Temp, mask;
  char          ii;

  for ( ii=0, mask=0x01, Time_Temp=0; ii<nBits; ++ii )
  {
    set_bit( RTC_PORT, RTC_CLK );   // signal we want to read a bit
    delay_10us( 1 );

    if ( test_bit( RTC_PORT, RTC_DATA ) )
      Time_Temp |= mask;

    clear_bit( RTC_PORT, RTC_CLK );
    delay_10us( 1 );

    mask = mask << 1;
  }

  return Time_Temp;
}

/*---------------------------------------------------------------------
  write_RTC(...)

  Call with: Data, 4 or 8 (bits to be written)
---------------------------------------------------------------------*/
static void write_RTC( unsigned char cData, unsigned char nBits )
{
  char          ii;

  for ( ii=0; ii<nBits; ++ii )
  {
    clear_bit( RTC_PORT, RTC_CLK );
    nop();

    if ( test_bit( cData, ii ) )
      set_bit( RTC_PORT, RTC_DATA );
    else
      clear_bit( RTC_PORT, RTC_DATA );

    nop();
    set_bit( RTC_PORT, RTC_CLK );
  }
}

/*---------------------------------------------------------------------
  BCD2Hex(...)

  Convert a (BCD) number in the range 0..99 to binary.
---------------------------------------------------------------------*/
static unsigned char BCD2Hex( unsigned char ucBCD )
{
  unsigned char ucLow, ucHigh, ucResult;
  
  ucLow = ucBCD & 0x0f;
  ucHigh = ucBCD >> 4;
  ucResult = ( ucHigh * 10 ) + ucLow;
  
  return ucResult;
}