/*--------------------------------------------------------------------
  clock1: a 24-hour clock using a PIC16F685 and a 2x16 LCD display
          and a NJU6355 real-time clock chip.
  
  RA0/AN0/ICSPDAT = in-circuit seial programming
  RA1/AN1/Vref/ICSPCLK = in-circuit seial programming
  RA2/AN2 = Program/Run switch
  RA3/!MCLR/Vpp = in-circuit seial programming 
  RA4/AN3 = LCD En
  RA5 = LCD R/S
  RB4-RB5-RB6-RB7 = LCD DB4..DB7
  RC0 = Date (Time) button
  RC1 = Seconds/Year button
  RC2 = Minutes/Day button
  RC3 = Hours/Month button
  RC4 = RTC CE
  RC5 = RTC CLK
  RC6 = RTC I/O
  RC7 = RTC Data

  This clock shows date and time (local and UTC):
  
    HH:MM:SS  HH:MMz
    APR 30 2006  SUN
    
  HH:MM:SS on the top left is local time, all others are UTC.
  
  The clock is set in UTC and local time is computed. Local time is 
  assumed to be eastern time and can be computed based on the date.
  
  SW5 is the program/run mode switch. In run mode the date and time  
  are obtained from the RTC continuously and displayed. The four
  buttons don't do anything in this mode. In program mode, the
  buttons advance hours/minutes and zero seconds, or if SW4 is
  down, month/day/year. When SW5 is returned to run mode, the RTC
  is updated.
--------------------------------------------------------------------*/  
#include <system.h>
#include "calendar.h"
#include "rtc.h"

#pragma CLOCK_FREQ      4000000

//#pragma DATA _CONFIG, _FCMEN_OFF & _IESO_OFF & _BOD_OFF & _WDT_OFF & _INTRC_OSC_NOCLK 
// Config Bits: 11 0000 1110 0100
// bit 13-12: unimplemented, set to 1
// bit 11:    FCMEN  0 = fail safe clock disabled
// bit 10:    IESO   0 = internal external switchover disabled
// bits 9-8:  BODEN  00 = Brown out detect disabled
// bit 7:     !CPD   1 = data memory code protection disabled
// bit 6:     !CP    1 = program memory code protection disabled
// bit 5:     MCLRE  1 = RA3/!MCLR pin function is !MCLR
// bit 4:     !PWRTE 0 = PWRT enabled
// bit 3:     WDTE   0 = watchdog timer disabled
// bits 2-0:  FOSC   100 = INTOSCIO oscillator, I/O on RA4/OSC2/CLKOUT, I/O on RA5/OSC1/CLKIN

#pragma DATA _CONFIG, 0x30e4

// LCD settings (see lcd_driver.h)
#define LCD_ARGS 	2,	/* Interface type: mode 0 = 8bit, 1 = 4bit(low nibble), 2 = 4bit(upper nibble) */ \
		0, 				/* Use busy signal: 1 = use busy, 0 = use time delays */\
		PORTB, TRISB, 	/* Data port and data port tris register */ \
		PORTA, TRISA, 	/* Control port and control port tris register */ \
		5,				/* Bit number of control port is connected to RS */ \
		6,				/* Bit number of control port is connected to RW */ \
		4 				/* Bit number of control port is connected to Enable */

// Note that RW is not connected to the processor in this project. RA6 is specified
// for R/W but doesn't exist anyway.

#include <lcd_driver.h>

#define TIMER2_PR2		250		// for initializing pr2

#define SW1_HR_MO		0		// bit positions for switches
#define SW2_MN_DY		1
#define SW3_SC_YR		2
#define SW4_DT_TM		3
#define SW5_RN_PG		4

// Global variables

RTCPARMS stRtc;             // Real Time Clock I/O

#define MAX_CHECKS		10		// number of tests before a switch is debounced

#define HOLD_DOWN_MS	250		// repeat detect rate, milliseconds

static unsigned char ucDebouncedState=0;	// debounced state of the switches
static unsigned char ucState[MAX_CHECKS];	// array containing bounce status
static unsigned char ucIndex=0;				// index into ucState

static bool bRtcUpdateReqd=false;

static bool bButton1State=false;		// used for repeat delay
static bool bButton2State=false;
static bool bButton3State=false;
static unsigned int uiMsPressed[3] = { 0, 0, 0 };	// how long buttons sw1..3 pressed

// Internal Functions

static void displayTime( void );
static void initCPU( void );
//static void timer1Overflow( void );
static void timer2Overflow( void );
static void processButtons( void );

/*--------------------------------------------------------------------
  main()
--------------------------------------------------------------------*/  
void main()
{
  initCPU();                	// initialize hardware
  lcd_setup();					// initialize lcd
  lcd_clear();
  
  lcd_datamode();
  lprintf( "VE3LNY Clock 1.2" );
  delay_s( 2 );
  lcd_clear();
  
  rtc_init();					// initialize the rtc
  
  while ( 1 )
  {
    if ( test_bit( ucDebouncedState, SW5_RN_PG ) )
    {
      // SW5 is in 'program' position
      processButtons();
      displayTime();
      delay_ms( 10 );
    }
    else
    {
      // SW5 is in 'run' position
           
      if ( bRtcUpdateReqd )
      {
        rtc_write();
        bRtcUpdateReqd = false;
        bButton1State = bButton2State = bButton3State = false;
      }
      
      displayTime();
      delay_ms( 100 );
      rtc_read();
    }
  }
}

/*---------------------------------------------------------------------
  interrupt()
---------------------------------------------------------------------*/
void interrupt( void )
{
  if ( test_bit( pir1, TMR2IF ) )
  {
    timer2Overflow();
    clear_bit( pir1, TMR2IF );		// reset timer2 overflow interrupt flag
  }
}

/*---------------------------------------------------------------------
  timer2Overflow()
  
  The timer2 interrupt occurs ( pr2 * prescale * postscale * Fosc/4 ).
  
  Fosc/4	prescale	postscale	pr2		period
  ------	--------	---------	-----	------
   1 us			4			1		1		4 us
	"			4			1		25		100 us
	"			4			1		250		1000 us
  
  For switch debouncing we use a period of 1 millisecond.
---------------------------------------------------------------------*/
static void timer2Overflow( void )
{
  unsigned char ucSwIn=0;
  unsigned char ii, ucResult;

  // We use the technique documented in "A Guide to Debouncing" by
  // Jack Ganssle to debounce up to 8 inputs (5 in this case).
  
  // Read current switch states
  if ( !test_bit( portc, 3 ) )
    set_bit( ucSwIn, SW1_HR_MO );
  if ( !test_bit( portc, 2 ) )
    set_bit( ucSwIn, SW2_MN_DY );
  if ( !test_bit( portc, 1 ) )
    set_bit( ucSwIn, SW3_SC_YR );
  if ( !test_bit( portc, 0 ) )
    set_bit( ucSwIn, SW4_DT_TM );
  if ( !test_bit( porta, 2 ) )
    set_bit( ucSwIn, SW5_RN_PG );

  ucState[ucIndex] = ucSwIn;

  ++ucIndex;
  if ( ucIndex == MAX_CHECKS )
    ucIndex = 0;

  // And all columns of the array  
  for ( ii=0, ucResult=0xff; ii<(MAX_CHECKS-1); ++ii )
    ucResult = ucResult & ucState[ii];
  
  ucDebouncedState = ucResult;
  
  // Since we're here every millisecond, keep count of
  // how many milliseconds SW1..3 are pressed
  
  if ( test_bit( ucDebouncedState, SW1_HR_MO ) )
    ++uiMsPressed[0];
  else
    uiMsPressed[0] = 0;
    
  if ( test_bit( ucDebouncedState, SW2_MN_DY ) )
    ++uiMsPressed[1];
  else
    uiMsPressed[1] = 0;

  if ( test_bit( ucDebouncedState, SW3_SC_YR ) )
    ++uiMsPressed[2];
  else
    uiMsPressed[2] = 0;
}

/*---------------------------------------------------------------------
  initCPU()
---------------------------------------------------------------------*/
static void initCPU()
{
  // Initialize the internal clock:
  osccon = 01100000b;		// internal clock @ 4MHz, use FOSC (config)

  // PORTC connections as follows:
  //  RC0 - (in) Date (Time) button 
  //  RC1 - (in) Seconds/Year button
  //  RC2 - (in) Minutes/Day button
  //  RC3 - (in) Hours/Month button
  //  RC4 - (out) RTC CE 
  //  RC5 - (out) RTC CLK 
  //  RC6 - (out) RTC I/O 
  //  RC7 - (in/out) RTC Data 

  portc = 0;
  trisc = 00001111b;            // 1=input, 0=output
  ansel = anselh = 0;			// all ports digital I/O
  
  // PORTB connections as follows:
  //  RB4 - (out) LCD DB4
  //  RB5 - (out) LCD DB5
  //  RB6 - (out) LCD DB6
  //  RB7 - (out) LCD DB7
  
  portb = 0;
  trisb = 00000000b;			// 1=input, 0=output
  wpub = 00000000b;				// disable all weak pull-ups

  // PORTA connections as follows:
  //  RA0 - (in) ICSP Data
  //  RA1 - (in) ICSP Clock / Vref input (0.5V)
  //  RA2 - (in) Program/Run switch SW5
  //  RA3 - (in) ICSP Vpp
  //  RA4 - (out) LCD En
  //  RA5 - (out) LCD R/S

  trisa = 00001111b;			// 1=input, 0=output
  porta = 0;

#if 0
  // Timer1 initialization

  tmr1l = 0;
  tmr1h = TIMER1_H;
  t1con = 00000001b;            // 1:1 prescale; int. clock; timer enabled
  set_bit( pie1, TMR1IE );      // Timer 1 interrupts
#endif

  // Timer2 initialization. Timer2 is used to generate interrupts
  // every 1 millisecond (or so) for switch debouncing. 

  pr2 = TIMER2_PR2;				// load pr2 limit
  t2con = 00000101b;			// 1:1 postscale; timer2 on; 1:4 prescale
  set_bit( pie1, TMR2IE );      // Timer2 interrupts

  // Enable interrupts for Timer2

  set_bit( intcon, GIE );       // enable global interrupts
  set_bit( intcon, PEIE );      // enable peripheral interrupts
}

/*---------------------------------------------------------------------
  displayTime()

  Display the date and time.
---------------------------------------------------------------------*/
static void displayTime()
{
  rom char *dayTable = "SUNMONTUEWEDTHUFRISAT";
  rom char *monthTable = "JANFEBMARAPRMAYJUNJULAUGSEPOCTNOVDEC";
  int       ii;
  char      cc;
  unsigned char hh, oo, n1, n2;

  // Display the HH:MM:SS local 24-hour format in row 1, column 1

  lcd_gotoxy( 0, 0 );       // line 1 column 1
  lcd_datamode();
  
  hh = stRtc.rtc_hours;
  oo = offset_to_local();
  if ( hh >= oo )
    hh -= oo;
  else
    hh = ( 24 + hh ) - oo;  

  n1 = ( hh > 9 ) ? ( hh / 10 ) : 0;
  cc = n1 + 0x30;
  lcd_write( cc );
  n2 = hh - ( n1 * 10 );
  cc = n2 + 0x30;
  lcd_write( cc );
  cc = ':';
  lcd_write( cc );
  
  n1 = ( stRtc.rtc_mins > 9 ) ? ( stRtc.rtc_mins / 10 ) : 0;
  cc = n1 + 0x30;
  lcd_write( cc );
  n2 = stRtc.rtc_mins - ( n1 * 10 );
  cc = n2 + 0x30;
  lcd_write( cc );
  cc = ':';
  lcd_write( cc );
  
  n1 = ( stRtc.rtc_secs > 9 ) ? ( stRtc.rtc_secs / 10 ) : 0;
  cc = n1 + 0x30;
  lcd_write( cc );
  n2 = stRtc.rtc_secs - ( n1 * 10 );
  cc = n2 + 0x30;
  lcd_write( cc );
  
  // Display the HH:MMz UTC 24-hour format in row 1, column 11

  lcd_gotoxy( 10, 0 );       // line 1 column 12
  lcd_datamode();
  
  hh = stRtc.rtc_hours;
  n1 = ( hh > 9 ) ? ( hh / 10 ) : 0;
  cc = n1 + 0x30;
  lcd_write( cc );
  n2 = hh - ( n1 * 10 );
  cc = n2 + 0x30;
  lcd_write( cc );
  cc = ':';
  lcd_write( cc );
  
  n1 = ( stRtc.rtc_mins > 9 ) ? ( stRtc.rtc_mins / 10 ) : 0;
  cc = n1 + 0x30;
  lcd_write( cc );
  n2 = stRtc.rtc_mins - ( n1 * 10 );
  cc = n2 + 0x30;
  lcd_write( cc );
  cc = 'z';
  lcd_write( cc );

  // Display the date "APR 30 2006" in row 2, column 1

  lcd_gotoxy( 0, 1 );       // line 2 column 1
  lcd_datamode();
  
  if ( stRtc.rtc_month )
    ii = stRtc.rtc_month - 1;
  else
    ii = 0;  
  ii *= 3;
  cc = monthTable[ii];
  lcd_write( cc );
  ++ii;
  cc = monthTable[ii];
  lcd_write( cc );
  ++ii;
  cc = monthTable[ii];
  lcd_write( cc );

  lcd_gotoxy( 4, 1 );       // line 2 column 5
  lcd_datamode();
  
  n1 = ( stRtc.rtc_day > 9 ) ? ( stRtc.rtc_day / 10 ) : 0;
  if ( n1 )
    cc = n1 + 0x30;
  else
    cc = ' ';  
  lcd_write( cc );
  n2 = stRtc.rtc_day - ( n1 * 10 );
  cc = n2 + 0x30;
  lcd_write( cc );
  
  lcd_gotoxy( 7, 1 );       // line 2 column 8
  lcd_datamode();
  lcd_write( '2' );
  lcd_write( '0' );

  n1 = ( stRtc.rtc_year > 9 ) ? ( stRtc.rtc_year / 10 ) : 0;
  cc = n1 + 0x30;
  lcd_write( cc );
  n2 = stRtc.rtc_year - ( n1 * 10 );
  cc = n2 + 0x30;
  lcd_write( cc );

  // Display the day of week as 3 character mnemonic in row 2, column 14
  
  lcd_gotoxy( 13, 1 );       // line 2 column 14
  lcd_datamode();

  // Keep in mind Sunday is usually 7 but may be 0!
  ii = ( stRtc.rtc_dayweek > 6 ) ? 0 : (int) stRtc.rtc_dayweek;
  ii *= 3;
  cc = dayTable[ii];
  lcd_write( cc );
  ++ii;
  cc = dayTable[ii];
  lcd_write( cc );
  ++ii;
  cc = dayTable[ii];
  lcd_write( cc );
}

/*---------------------------------------------------------------------
  processButtons()

  Driven by the mainline periodically. Tests the switches and
  updates the clock parameters.
  
  There is a problem we try to solve. We want to acknowledge a button
  press immediately but have a programmed wait before recognizing a
  repeat if the button is held down. We use a counter in the timer
  routine to remember how many milliseconds a button has been down,
  and a flag to tell whether this is the first press.
---------------------------------------------------------------------*/
static void processButtons()
{
  unsigned char ucDaysMonth;
  
  if ( test_bit( ucDebouncedState, SW1_HR_MO ) )
  {
    if ( !bButton1State || uiMsPressed[0] > HOLD_DOWN_MS )
    {
      if ( test_bit( ucDebouncedState, SW4_DT_TM ) )
      {
        // Update month
        if ( stRtc.rtc_month == 12 )
          stRtc.rtc_month = 1;
        else
          ++stRtc.rtc_month;
          
        stRtc.rtc_dayweek = dow( stRtc.rtc_year, stRtc.rtc_month, stRtc.rtc_day );
      }
      else
      {
        // Update hour
        if ( stRtc.rtc_hours == 23 )
          stRtc.rtc_hours = 0;
        else
          ++stRtc.rtc_hours;  
      }
    
      bRtcUpdateReqd = true;
      bButton1State = true;		// activate repeat delay now
      uiMsPressed[0] = 0;		// reset counter
    }  
    return;
  }

  bButton1State = false;		// cancel repeat delay now

  if ( test_bit( ucDebouncedState, SW2_MN_DY ) )
  {
    if ( !bButton2State || uiMsPressed[1] > HOLD_DOWN_MS )
    {
      if ( test_bit( ucDebouncedState, SW4_DT_TM ) )
      {
        // Update day
        ucDaysMonth = days_in_month( stRtc.rtc_month, stRtc.rtc_year );
        if ( stRtc.rtc_day >= ucDaysMonth )
          stRtc.rtc_day = 1;
        else
          ++stRtc.rtc_day;  

        stRtc.rtc_dayweek = dow( stRtc.rtc_year, stRtc.rtc_month, stRtc.rtc_day );
      }
      else
      {
        // Update minute
        if ( stRtc.rtc_mins == 59 )
          stRtc.rtc_mins = 0;
        else
          ++stRtc.rtc_mins;  
      }
      
      bRtcUpdateReqd = true;
      bButton2State = true;		// activate repeat delay now
      uiMsPressed[1] = 0;		// reset counter
    }    
    return;
  }

  bButton2State = false;		// cancel repeat delay now

  if ( test_bit( ucDebouncedState, SW3_SC_YR ) )
  {
    if ( !bButton3State || uiMsPressed[2] > HOLD_DOWN_MS )
    {
      if ( test_bit( ucDebouncedState, SW4_DT_TM ) )
      {
        // Update year
        if ( stRtc.rtc_year == 40 )
          stRtc.rtc_year = 0;
        else
          ++stRtc.rtc_year;  

        stRtc.rtc_dayweek = dow( stRtc.rtc_year, stRtc.rtc_month, stRtc.rtc_day );
      }
      else
      {
        // Update (zero) second
        stRtc.rtc_secs = 0; 
      }
      
      bRtcUpdateReqd = true;
      bButton3State = true;		// activate repeat delay now
      uiMsPressed[2] = 0;		// reset counter
    }

    return;
  }
  
  bButton3State = false;		// cancel repeat delay now
}