/*---------------------------------------------------------------------
  Module: calendar.c

  Some calendar routines.

  jb080809 - Fix problem with determining when DST begins.
---------------------------------------------------------------------*/
#include <system.h>
#include "calendar.h"
#include "rtc.h"

#define OFFSET_STANDARD         5
#define OFFSET_SAVINGS          4

// Global variables

extern RTCPARMS stRtc;

static unsigned char daysmonth[12] =
{
  31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
};

static unsigned char months_table[12] =
{
  0, 3, 3, 6, 1, 4, 6, 2, 5, 0, 3, 5
};

static unsigned char months_table_ly[12] =
{
  6, 2, 3, 6, 1, 4, 6, 2, 5, 0, 3, 5
};

/*---------------------------------------------------------------------
  days_in_month(...)

  Returns the number of days in the month, given the month and year.
---------------------------------------------------------------------*/
unsigned char days_in_month( unsigned char ucMonth, unsigned char ucYear )
{
  unsigned char ii, rcode;

  if ( ucMonth < 1 || ucMonth > 12 )          // bad input
    return 0;

  ii = ucMonth - 1;
  rcode = daysmonth[ii];

  if ( ucMonth == 2 )
  {
    if ( isleapyear( ucYear ) )
      ++rcode;
  }

  return rcode;
}

/*--------------------------------------------------------------------------
  offset_to_local()

  Returns the number of hours between GMT and EST/EDT as a positive number.
  The EST offset is -5 hours and EDT is -4 hours. EDT is deemed to begin on
  the 1st Sunday of April and end on the last Sunday of October.

  UPDATE as of 2007, EDT begins on the 2nd Sunday in March and ends on the
  1st Sunday in November.
--------------------------------------------------------------------------*/
unsigned char offset_to_local()
{
  unsigned char ucDow, ucDay;

  if ( stRtc.rtc_month < 3 || stRtc.rtc_month > 11 ) return OFFSET_STANDARD;
  if ( stRtc.rtc_month > 3 && stRtc.rtc_month < 11 ) return OFFSET_SAVINGS;

  if ( stRtc.rtc_month == 3 )
  {
    // Its March so find the 2nd Sunday.
    // What day is March 1st?
    ucDay = 1;
    ucDow = dow( stRtc.rtc_year, stRtc.rtc_month, ucDay );
    if ( ucDow )                        // not Sunday
      ucDay = 8 - ucDow;
    ucDay += 7;                         // 2nd Sunday
    if ( stRtc.rtc_day > ucDay ) return OFFSET_SAVINGS;
    if ( stRtc.rtc_day < ucDay ) return OFFSET_STANDARD;
    if ( stRtc.rtc_hour > 1 ) return OFFSET_SAVINGS;  // after 2:00
    return OFFSET_STANDARD;
  }

  // Its November, find the 1st Sunday.
  ucDay = 1;
  ucDow = dow( stRtc.rtc_year, stRtc.rtc_month, ucDay );
  if ( ucDow )                          // not Sunday
      ucDay = 8 - ucDow;
  if ( stRtc.rtc_day > ucDay ) return OFFSET_STANDARD;
  if ( stRtc.rtc_day < ucDay ) return OFFSET_SAVINGS;
  // Check the time for 0200.
  if ( stRtc.rtc_hours >= 2 ) return OFFSET_STANDARD;
  return OFFSET_SAVINGS;
}

/*--------------------------------------------------------------------------
  isleapyear(...)

  Leap Year
  The rules for a leap year are:

    1. every four years is a leap year, except
    2. every century is not a leap year, except
    3. every fourth century is a leap year.

  Code that converts dates from a Julian style format to an external format
  and vice versa must incorporate the above three rules in order to deal with
  leap years correctly.

  Typically you should find #4, #100 and #400 in the conversion algorithm.
  Alternatively you may find \4, \100 and \400 or even /4, /100 and /400.

  If your code is missing the #400 expression then it may not cater for the
  leap year in the year 2000.

  If your code only has a #4 expression then it may not cater for the absence
  of a leap year in 1800, 1900 and 2100. Such code is good for dates between
  1 March 1900 and 31 December 2099. This is acceptable for many
  applications, however it is unlikely to be found in cases where a date
  before 1900 is required, such as for entry and validation of a person's
  date of birth.

  Input: iYear = the year (yy), we make assumption about century

  Returns: 0 = no, 1 = yes
--------------------------------------------------------------------------*/
unsigned char isleapyear( unsigned char ucYear )
{
  unsigned int  uiRemainder, uiYear, uiResult;

  if ( ucYear > 60 )
    uiYear = 1900 + (unsigned int) ucYear;
  else
    uiYear = 2000 + (unsigned int) ucYear;

  uiRemainder = uiYear % 100;
  if ( !uiRemainder )
  {
    // This is a century.

    uiResult = uiYear / 100;          // which century
    uiRemainder = uiResult % 4;       // is it a 4th century?
                                      // not / is a 4th century
    return ( uiRemainder ) ? 0 : 1;
  }

  uiRemainder = uiYear % 4;

  return ( uiRemainder ) ? 0 : 1;    // not / is a 4th year
}

/*---------------------------------------------------------------
  dow(...)

  Calculates the day of the week given the year, month, day.
  Based on the method documented in Wikipedia:
  http://en.wikipedia.org/wiki/Calculating_the_day_of_the_week
  We make one simplification: The century is always 2000-2099,
  so C = 6.

  Input: Year  - 0..99
         Month - 1..12
         Day   - 1..31

  Returns: 0..6 where 0 = Sunday, 6 = Saturday
---------------------------------------------------------------*/
unsigned char dow( unsigned char ucYear, unsigned char ucMonth,
                   unsigned char ucDay )
{
  unsigned int CC, YY, MM, DD, RR;
  unsigned char ii, ly;

  // Look up 2000s in the centuries table
  CC = 6;

  // Divide last 2 digits of yea by 4 and drop fractional part.
  YY = ucYear / 4;

  // Look up the month in the months table
  ly = isleapyear( ucYear );
  ii = ucMonth - 1;
  if ( ly )
    MM = months_table_ly[ii];
  else
    MM = months_table[ii];

  // Add all numbers from steps 1-4 to the day of the month

  DD = CC + (unsigned int) ucYear + YY + MM + (unsigned int) ucDay;

  // Divide the sum from step 5 by 7 and find the remainder:

  RR = DD % 7;

  return (unsigned char) RR;
}
