/*-----------------------------------------------------------------
  All code is provided for EDUCATIONAL purposes ONLY. There is no
  warranty and no support. Licensed under the MIT license - see
  http://www.qsl.net/ve3lny/MIT%20License.html

  LED_voltmeter

  main.c

  This program implements a milliammeter using a 3 digit LED display.

  Processor: ATtiny861
      Clock: Internal, 4.0 MHz
-----------------------------------------------------------------*/
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <util/delay.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "common.h"
#include "led.h"

#define INIT_CTR0   	131			// 256 - 125
#define INIT_CTR1	 	156			// 256 - 100

#define RESISTOR		0.1536

#define DIGITAL_FILTER_SAMPLES 10       // to reduce jitter in the display

// Global variables
extern uint8_t uiDecimal_digit;		// 0..2

static volatile char cDisplay[3] = { 0, 0, 0 };	// digits to display

static volatile uint16_t	uiAdc=0;			// last adc sample
static volatile uint8_t		ucComplete=0;		// ADC complete flag
static volatile uint8_t		ucAutoRange=0;

static volatile double		dMilliamps=0.0;

static volatile uint16_t	uiOffsetError1=0;
static volatile uint16_t	uiOffsetError8=0;

static volatile uint16_t 	uiSmoothedAdc=0;        // smoothed adc reading

// Internal Functions

static void calculate_current(void);
static void digital_filter(void);
static void display_number( uint16_t uiNumber );
static void measure_offset_error(void);
static void system_init(void);

/*-----------------------------------------------------------------
  main()

  The mainline initiates the A/D conversion, processes the result
  and formats the output for display on the LEDs. The actual
  output to the LED display is handled by the timer0 interrupt
  handler. 
-----------------------------------------------------------------*/
int main(void)
{
  uint8_t	ctr=0, gain;
  uint16_t 	uiMilliamps;

  system_init();			// initialise system

  // Perform one ADC conversion before calibration
  ucComplete = 0;
  set_bit( ADCSRA, ADSC );	// initiate conversion
  while ( !ucComplete );

  measure_offset_error();
#if 0
  // display the offset errors
  display_number( uiOffsetError1 );
  for ( ii=0; ii<50; ++ii )
    _delay_loop_2( 0 );		// 32 ms
  display_number( uiOffsetError8 );
  for ( ii=0; ii<50; ++ii )
    _delay_loop_2( 0 );		// 32 ms
#endif

  // Perform an ADC conversion to determine gain setting
  ucComplete = 0;
  set_bit( ADCSRA, ADSC );		// initiate conversion
  while ( !ucComplete );

  if ( uiAdc < 1000 )
  {
    clear_bit( ADCSRA, ADEN );		// disable the ADC
    set_bit( ADCSRB, GSEL );		// gain = 8
    set_bit( ADCSRA, ADEN );		// enable the ADC
  }

  while( 1 )
  {
#if 0
	// This code does not appear to be necessary.

    // Turn counter0 and counter1 off
    TCCR0B = 0b00000000;
    TCCR1B = 0b00000000;

    // Set the sleep mode to ADC noise reduction mode. Then enter
	// sleep mode. This automatically initiates the ADC conversion.
	// Sleep mode is ended when the conversion complete interrupt
	// occurs.
    cli();
    set_sleep_mode( SLEEP_MODE_ADC );
	sleep_enable();
	sei();
    sleep_mode();
	sleep_disable();

    // Turn counter0 and counter1 on
    TCCR0B = 0b00000011;		// prescaler = 64 (turns timer0 on)
    TCCR1B = 0b00001111;		// prescaler = 16384 (turns timer1 on)
#endif

    ucComplete = 0;
    set_bit( ADCSRA, ADSC );	// initiate conversion
    while ( !ucComplete );

	digital_filter();			// outputs uiSmoothedAdc from uiAdc

    calculate_current();		// computes dMilliamps from uiAdc
    uiMilliamps = (unsigned int) dMilliamps;

	// OK I don't like this but the milliammeter reads around 3 too
	// high, even when the voltage is 0, and I don't know why.
	if ( uiMilliamps > 2 )
	  uiMilliamps -= 3;
	else
	  uiMilliamps = 0;  	 

	if ( ctr > 7 )
	{
	  // we want to update the display less often
	  display_number( uiMilliamps );
	  ctr = 0;
	}

    _delay_loop_2( 0 );				// 32 ms

    clear_bit( ADCSRA, ADEN );		// disable the ADC
    gain = test_bit( ADCSRB, GSEL );
    if ( uiSmoothedAdc > 1000  && gain )
      clear_bit( ADCSRB, GSEL );	// gain = 1
    if ( uiSmoothedAdc < 125  && !gain )
      set_bit( ADCSRB, GSEL );		// gain = 8
    set_bit( ADCSRA, ADEN );		// enable the ADC
	
	++ctr;
  }
}

/*-----------------------------------------------------------------
  Timer 0 overflow interrupt handler - updates LED display

  To achieve an interval of approximately 2 ms for selecting
  the next LED digit. By sequencing each digit fast enough, the LED
  digits appear to be all on.

  Since the clock is running at 4.0 mHz, the counter increments
  every 250.0 ns. Using a count of 125 and prescaler of 64 we get
  an interrupt interval of 2.0 ms.
-----------------------------------------------------------------*/
ISR(TIMER0_OVF_vect)
{
  static uint8_t ucIndex=0;

  ledSelectDigit( NO_DIGIT );               // turn all digits off

  ledOutputNumber( cDisplay[ucIndex] );		// output the number

  ledSelectDigit( ucIndex );                // select the digit

  ++ucIndex;
  if ( ucIndex == 3 ) ucIndex = 0;

  TCNT0L = INIT_CTR0;						// reinitialize counter
}

/*-----------------------------------------------------------------
  Timer 1 overflow interrupt handler - flashes alive LED

  Since the clock is running at 4.0 mHz, the counter increments
  every 250.0 ns. Using a count of 100 and prescaler of 16384 we
  get an interrupt interval of 409.6 ms.
-----------------------------------------------------------------*/
ISR(TIMER1_OVF_vect)
{
  // Flash the alive LED
  if ( test_bit( PORTB, PB6 ) )
    clear_bit( PORTB, PB6 );
  else
    set_bit( PORTB, PB6 );

  TCNT1 = INIT_CTR1;				// reinitialize counter
}

/*-----------------------------------------------------------------
  ISR( ADC_vect ) - ADC conversion complete interrupt handler
-----------------------------------------------------------------*/
ISR( ADC_vect )
{
  // ADCL must be read before ADCH.
  uiAdc = (uint16_t) ADCL | ( (uint16_t) ADCH << 8 );

  // Account for offset error
  if ( test_bit( ADCSRB, GSEL ) )
    uiAdc -= uiOffsetError8;
  else
    uiAdc -= uiOffsetError1;

  ucComplete = 1;
}

/*-----------------------------------------------------------------
  system_init()

  PORTB: PB0 (n/a) MOSI (ISP)
         PB1 (n/a) MISO (ISP)
         PB2 (n/a) SCK (ISP)
         PB3 (output) LED DP
         PB4 (output) LED DIG3
         PB5 (output) LED DIG1
         PB6 (output) Diagnostic LED
         PB7 (n/a) !Reset (ISP)

  PORTA: PA0 (input) ADC0 Voltage sensor 
         PA1 (input) ADC1 (not used)
         PA2 (output) LED DIG2
         PA3 (input) AREF (not used)
         PA4 (output) LED BCD "A"
         PA5 (output) LED BCD "B"
         PA6 (output) LED BCD "C" 
         PA7 (output) LED BCD "D" 

-----------------------------------------------------------------*/
static void system_init()
{
  // The fuses are set to provide an internal clock, running at
  // 8.0 MHz with a prescaler of 8. Change the prescaler from
  // 8 to 2. The procedure requires two steps as per the device
  // specifications.
  CLKPR = 0b10000000;       // set clock prescaler change enable
  CLKPR = 0b00000001;       // set prescaler to 2

  DDRB  = 0b01111000;       // PortB3..6 = output
  PORTB = 0b01000000;       // initialise PortB

  DDRA  = 0b11110100;       // PortA2, A4..7 = output
  PORTA = 0b00000000;       // initialise PortA

  // Initialize Timer0. Timer0 is used to output digits on the LED
  // display every 2 milliseconds.
  TCCR0A = 0b00000000;       // normal 8-bit counter mode
  TCCR0B = 0b00000011;       // prescaler = 64 (turns timer0 on)
  TCNT0L = INIT_CTR0;
  TCNT0H = 0;
  set_bit( TIMSK, TOIE0 );	// enable timer/counter0 overflow interrupt

  // Initialize Timer1. Timer1 is used to flash the alive LED
  // every 300 milliseconds.

  TCCR1A = 0b00000000;		// normal operation
  TCCR1B = 0b00001111;		// prescaler = 16384 (turns timer1 on)
  TCNT1 = INIT_CTR1;
  set_bit( TIMSK, TOIE1 );	// enable timer/counter1 overflow interrupt

  DIDR0  = 0b00001011;		// disable digital buffer: ADC0 ADC1 AREF
  DIDR1  = 0b00000000;

  // Set up the A/D: single-ended mode, input=ADC0
  //ADMUX  = 0b10000000;		// REFS1=1, REFS0=0, Vref=1.1; Input=ADC0
  //ADCSRA = 0b10001101;		// enable ADC; prescaler=32; enable interrupt
  //ADCSRB = 0b00000000;		// REFS2=0; no auto trigger

  // Set up the A/D: differential mode, +input=ADC0, -input=ADC1
  ADMUX  = 0b10000001;		// REFS1=1, REFS0=0, Vref=1.1, Input=ADC0/ADC1
  ADCSRA = 0b00001101;		// enable interrupt; prescaler=32
  ADCSRB = 0b00001000;		// BIN=0, GSEL=0, REFS2=0, MUX5=1; no auto trigger
  set_bit( ADCSRA, ADEN );	// enable the ADC

  sei();                    // enable global interrupts
}

/*---------------------------------------------------------------------
  calculate_current()

  Convert the A/D input value (0..1023) to milliamps. In this imple-
  mentation, the source is a voltage in the range 0 to about 0.7V. 
  This is applied to ADC0 with a Vref 0f 1.1V. Ohm's law will tell us
  what the current is, given the voltage and resistance. The result 
  is placed in global variable dMilliamps.
---------------------------------------------------------------------*/
static void calculate_current()
{
  //uint32_t ulADCmv;
  double dAdcMv, dGain;

  dGain = ( test_bit( ADCSRB, GSEL ) ) ? 8.0 : 1.0;

  // Calculate millivolts from uiAdc (adresh:adresl).
  // The formula is: Vin = ( ADC * Vref * 1000 ) / ( 1024 * gain )

  dAdcMv = ( (double) uiSmoothedAdc * 1100.0 ) / ( 1024.0 * dGain );

  // I = E / R

  dMilliamps = dAdcMv / RESISTOR;
}

/*---------------------------------------------------------------------
  display_number()

  Format the given number and store in cDisplay array.
---------------------------------------------------------------------*/
static void display_number( uint16_t uiNumber )
{
  uint8_t  buffer[6];
  uint8_t  nn;

  if ( uiNumber > 999 )
  {
    uiNumber /= 10;
    uiDecimal_digit = 0;
  }
  else
    uiDecimal_digit = 2;

  utoa( uiNumber, (char *) buffer, 10 );
  nn = (char) strlen( (char *) buffer );
  if ( !nn )
  {
    cDisplay[0] = 0x0f;
    cDisplay[1] = 0x0f;
    cDisplay[2] = '0';
  }
  else
    if ( nn == 1 )
    {
      cDisplay[0] = 0x0f;
      cDisplay[1] = 0x0f;
      cDisplay[2] = buffer[0];
    }
    else
      if ( nn == 2 )
      {
        cDisplay[0] = 0x0f;
        cDisplay[1] = buffer[0];
        cDisplay[2] = buffer[1];
      }
	  else
        if ( nn == 3 )
        {
          cDisplay[0] = buffer[0];
          cDisplay[1] = buffer[1];
          cDisplay[2] = buffer[2];
        }
	    else
        {
          cDisplay[0] = '9';
          cDisplay[1] = '9';
          cDisplay[2] = '9';
        }
}

/*---------------------------------------------------------------------
  measure_offset_error()

  Measure the ofset error and store result in uiOffsetError1/8.
---------------------------------------------------------------------*/
static void measure_offset_error()
{
  uint8_t	ii;
  uint16_t	uiSum;

  // The idea is to set both inputs to the same pin and
  // take an ADC reading.

  clear_bit( ADCSRA, ADEN );	// disable the ADC
  set_bit( ADMUX, MUX4 );
  set_bit( ADMUX, MUX3 );		// both inputs=ADC0
  clear_bit( ADCSRB, GSEL );	// gain = 1
  set_bit( ADCSRA, ADEN );		// enable the ADC

  for ( ii=0, uiSum=0; ii<10; ++ii )
  {
    ucComplete = 0;
    set_bit( ADCSRA, ADSC );		// initiate a conversion
    while ( !ucComplete );
	uiSum += uiAdc;
  }

  uiOffsetError1 = uiSum / 10;

  clear_bit( ADCSRA, ADEN );	// disable the ADC
  set_bit( ADCSRB, GSEL );		// gain = 8
  set_bit( ADCSRA, ADEN );		// enable the ADC

  for ( ii=0, uiSum=0; ii<10; ++ii )
  {
    ucComplete = 0;
    set_bit( ADCSRA, ADSC );		// initiate a conversion
    while ( !ucComplete );
	uiSum += uiAdc;
  }

  uiOffsetError8 = uiSum / 10;

  clear_bit( ADCSRA, ADEN );	// disable the ADC
  clear_bit( ADMUX, MUX4 );
  clear_bit( ADMUX, MUX3 );		// input=ADC0 and ADC1
  clear_bit( ADCSRB, GSEL );	// gain = 1
  set_bit( ADCSRA, ADEN );		// enable the ADC
}

/*---------------------------------------------------------------------
  digital_filter()

  Why do we need this? Because the output of the adc jitters one or
  two counts each sample, which translates to one or two tenths of a
  degree C on the display. It goes up and down apparently randomly,
  which is very annoying. According to the Atmel spec, the adc has a
  number of error modes which result in the lsb being uncertain.
---------------------------------------------------------------------*/
static void digital_filter()
{
  static uint16_t uiTable[DIGITAL_FILTER_SAMPLES];
  static uint8_t  ucIndex=0, ucEntries=0;
  uint8_t                 ii;
  uint32_t                ulSum;

  // Add the entry to the table, wrap-around
  uiTable[ucIndex] = uiAdc;
  ++ucIndex;
  if ( ucIndex >= DIGITAL_FILTER_SAMPLES )
    ucIndex = 0;
  if ( ucEntries < DIGITAL_FILTER_SAMPLES )
    ++ucEntries;

  // Average the table
  ulSum = 0;
  for ( ii=0; ii<ucEntries; ++ii )
    ulSum += uiTable[ii];

  uiSmoothedAdc = (unsigned int) ( ulSum / ucEntries );
}
