/*-----------------------------------------------------------------
  Heating Pad Controller  by Jack Botner

  main.c

  This program implements an electric blanket controller using a
  1 digit LED display and a rotary encoder. PWM is used to control
  an IRL640 power FET.

  Processor: ATtiny861A
      Clock: Internal, 4.0 MHz

  20231010JB - created from eblanket project
-----------------------------------------------------------------*/
#define F_CPU 4000000UL
#include <avr/io.h>
#include <avr/interrupt.h>
//#include <avr/wdt.h>
#include <util/delay_basic.h>
#include <string.h>
#include <stdlib.h>
#include "main.h"

// Global variables

volatile uint8_t ucPowerState=0;			// power: 0=off, 1=on
volatile uint8_t ucDebouncedState;			// debounced state of the switches
volatile uint8_t ucPreviousState;			// previous switch state
volatile uint8_t ucEventFlags;				// event flags

static volatile uint8_t ucHeatSetting=1;	// blanket heat setting 1..9

// Internal Functions

static void system_init(void);

/*-----------------------------------------------------------------
  main()

  After initialization, the mainline processes events scheduled
  elsewhere in the program.
-----------------------------------------------------------------*/
int main(void)
{
  static volatile uint8_t ucFirst=1;		// encoder state management
  uint8_t	ucPwmCtr=99;

  system_init();			// initialize system

//  wdt_enable( WDTO_60MS );
  ledOutputNumber( ucHeatSetting );
  
  while( 1 )				// do forever
  {
	if ( test_bit( ucEventFlags, EVENT_BUTTON_PRESSED ) )
	{
	  if ( ucPowerState )
	  {
		ucPowerState = 0;			// power down
		ucPwmCtr = 9;				// stop the heat
	  }
	  else
	  {
		ucPowerState = 1;			// power up
		ucPwmCtr = 0;
	  }

      ledOutputNumber( ucHeatSetting );

	  clear_bit( ucEventFlags, EVENT_BUTTON_PRESSED );
	}

	if ( test_bit( ucEventFlags, EVENT_ENCODER_UP ) )
	{
	  if ( ucPowerState )
	  {
		if ( ucHeatSetting < 9 )
		{
	      ++ucHeatSetting;
		}

        ledOutputNumber( ucHeatSetting );
	  }

	  clear_bit( ucEventFlags, EVENT_ENCODER_UP );
	}

	if ( test_bit( ucEventFlags, EVENT_ENCODER_DOWN ) )
	{
	  if ( ucPowerState )
	  {
	    if ( ucHeatSetting > 1 )
	    {
	      --ucHeatSetting;
	    }

        ledOutputNumber( ucHeatSetting );
	  }

	  clear_bit( ucEventFlags, EVENT_ENCODER_DOWN );
	}

	if ( test_bit( ucEventFlags, EVENT_DEBOUNCE_IO ) )
	{
	  // Process the rotary encoder

	  debounce_switch();

	  // The 1st time only, set the previous switch state to the
	  // current switch state.
	  if ( ucFirst )
	  {
	    ucPreviousState = ucDebouncedState;
		ucFirst = 0;
	  }

	  if ( ucDebouncedState != ucPreviousState )
	  {
	    analyze_switch();           // convert switch change to event
		ucPreviousState = ucDebouncedState;
	  }

	  clear_bit( ucEventFlags, EVENT_DEBOUNCE_IO );
	}

	if ( test_bit( ucEventFlags, EVENT_100MS_PWM ) )
	{
	  // Every 10 ms we need to decide if the heat i/o should
	  // be high or low. Since ucHeatSetting is in the range 0..9,
	  // a counter is used and compared to the heat setting. If less,
	  // heat is turned on, otherwise off.
	  if ( ucHeatSetting > ucPwmCtr )
	  {
	    // heat should be on
		if ( !test_bit( HEAT_PORT, HEAT_PIN ) )
		  set_bit( HEAT_PORT, HEAT_PIN );
	  }
	  else
	  {
		// heat should be off
		if ( test_bit( HEAT_PORT, HEAT_PIN ) )
		  clear_bit( HEAT_PORT, HEAT_PIN );
	  }
	  	  	
	  if ( ucPowerState )
	  {
		++ucPwmCtr;
		if ( ucPwmCtr > 9 )
	      ucPwmCtr = 0;
	  }

	  clear_bit( ucEventFlags, EVENT_100MS_PWM );

//	  wdt_reset();
	}
  }
}

/*-----------------------------------------------------------------
  system_init()

  PORTB: PB0 (input)  MOSI
         PB1 (input)  MISO
         PB2 (input)  SCK
         PB3 (output) PWM
         PB4 (input)  Rotary Encoder Output B *
         PB5 (input)  Rotary Encoder Output A *
         PB6 (input)  Rotary Encoder Pushbutton *
         PB7 (input) !Reset (ISP)

  PORTA: PA0 (output) LED DP
         PA1 (output) LED 'A' segments
         PA2 (output) LED 'B' segments
         PA3 (output) LED 'C' segments
         PA4 (output) LED 'D' segments
         PA5 (output) LED 'E' segments
         PA6 (output) LED 'F' segments 
         PA7 (output) LED 'G' segments

  * = pull-up required
-----------------------------------------------------------------*/
static void system_init()
{
  // The fuses are set to provide an internal clock, running at
  // 8.0 MHz with a prescaler of 8. Change the prescaler from
  // 8 to 2. The procedure requires two steps as per the device
  // specifications.
  CLKPR = 0b10000000;       // set clock prescaler change enable
  CLKPR = 0b00000001;       // set prescaler to 2

  DDRB  = 0b00001000;       // set port B directions
  PORTB = 0b11110111;       // initialise PortB / set pullups

  DDRA  = 0b11111111;       // PA0 = input, PA1..7 = output
  PORTA = 0b11111110;       // initialise PortA

  // Initialize Timer1. Timer1 is used to generate interrupts
  // every 2 milliseconds for switch debouncing.
  TCNT1 = INIT_TIMER1;
  TC1H = 0;
  TCCR1A = 0b00000000;          // normal counter operation
  TCCR1B = 0b00000110;          // normal operation, prescaler=32
  set_bit( TIMSK, TOIE1 );		// enable timer/counter1 overflow interrupt

  PRR = 0b0000111;          // using only timer1
  sei();                    // enable global interrupts
}
