/* winstrct.h
 *
 * Defines Windows "Unix-like" error number and error message variables and
 * functions.
 * C++: Encapsulates in C++ classes some Windows API that uses stuctures.
 * Written by Olof Lagerkvist 2000.
 */

#ifndef __WINERRNO_H
#define __WINERRNO_H

// Include Winsock API 2.0
#ifndef _INC_WINDOWS_
#  // Winsock2 is not compatible with 4.0 comp.
#  if !defined(_WINSOCKAPI_) && (_WIN32_WINNT < 0x0400)
#	   include <winsock2.h>
#  else
#     include <windows.h>
#  endif
#endif

#ifndef _MSWSOCK_
# include <mswsock.h>
#endif

#ifndef _LM_
# include <lm.h>
#endif

#ifdef __DLL__
# define DLLEXPORT __declspec(dllexport)
#else
# define DLLEXPORT
#endif

// Include the I/O stream class deinitions, so we can stream out error messages
// to the cerr object
#if defined(__cplusplus) && !defined(__IOSTREAM_H)
#	include <iostream.h>
#endif

// Windows version macros
#define WinVer_Major       (GetVersion()&0x000000FF)
#define WinVer_Minor       ((GetVersion()&0x0000FF00)>>8)
#define WinVer_Build       ((GetVersion()&0x7FFF0000)>>16)

// Windows platform macros (boolean return values)
#define WinVer_WindowsNT	(!(GetVersion()&0x80000000))
#define WinVer_Windows95	((bool)((GetVersion()&0x800000FF) >= 0x80000004))
#define WinVer_Win32s		((bool)(((GetVersion()^0x80000000)&0x800000FF) < 4))
#define WinVer_Win95orNT4	(WinVer_Major >= 4)
#define WinVer_Win32sor95	((bool)(GetVersion()&0x80000000))
#define WinVer_Win32sorNT3	(WinVer_Major < 4)

#ifndef _QWORD_DEFINED
typedef DWORDLONG QWORD;
#define _QWORD_DEFINED
#endif

#ifdef __cplusplus

#ifdef UNICODE
# define WINSOCK_MODULE L"WSOCK32"
#else
# define WINSOCK_MODULE "WSOCK32"
#endif

#pragma option -Ve -Vx

/* win_errno
 *
 * Used similar to errno in Unix environments.
 *
 * Used to get or set the error code from the most recently non-successful call
 * to a Windows API function. Gets the value when converted to DWORD, sets the
 * value when used with the assignment operator =.
 *
 * To make sure it is not taken for a real variable, the address-of operator &
 * returns just type void, which is illegal in any assignment.
 */
extern const class
{
	public:
	DWORD operator=( const DWORD& new_win_errno ) const
	{
		SetLastError( new_win_errno );
   	return new_win_errno;
   }
   void operator&() const{}
   operator DWORD() const
   {
   	return GetLastError();
   }
} win_errno;

// winsock.h and winsock2.h defines h_errno to be WSAGetLastError(). In this
// header, we provide a better way to implement h_errno
#	ifdef h_errno
#		undef h_errno
#	endif

/* h_errno
 *
 * Used similar to h_errno in Unix environments.
 *
 * Used to get or set the error code from the most recently non-successful call
 * to a Winsock API function. Gets the value when converted to int, sets the
 * value when used with the assignment operator =.
 *
 * To make sure it is not taken for a real variable, the address-of operator &
 * returns just type void, which is illegal in any assignment.
 */
extern const class
{
	public:
	int operator=( const int& new_wsa_errno ) const
   {
   	WSASetLastError( new_wsa_errno );
      return new_wsa_errno;
   }
   void operator&() const{}
   operator int() const
   {
   	return WSAGetLastError();
   }
} h_errno;

/* cmdlg_errno
 *
 * Used to get the error code from the most recently called Windows Common
 * Dialog API function. Gets the value when converted to int, the value can't
 * be set.
 *
 * To make sure it is not taken for a real variable, the address-of operator &
 * returns just type void, which is illegal in any assignment.
 */
extern const class
{
	public:
   void operator&() const{}
	operator int() const
   {
   	return CommDlgExtendedError();
   }
} cmdlg_errno;

/* wnet_errno
 *
 * Used to get the most recent error code with description reported by a
 * Windows network provider.
 */
extern const class
{
	public:
   void operator&() const{}
   operator DWORD() const
   {
		DWORD dwError = 0;
      TCHAR cBuf[2];

		WNetGetLastError( &dwError, cBuf, sizeof(cBuf), cBuf, sizeof(cBuf) );

		return dwError;
   }
   DWORD operator()( LPTSTR pcErrorBuf = NULL, DWORD dwErrorBufSize = 0,
   	LPTSTR pcNameBuf = NULL, DWORD dwNameBufSize = 0 )
	{
   	DWORD  dwError;
      TCHAR cBuf[2];

		WNetGetLastError( &dwError,
                   pcErrorBuf && dwErrorBufSize ? pcErrorBuf : cBuf,
                   pcErrorBuf && dwErrorBufSize ? dwErrorBufSize : sizeof(cBuf),
                   pcNameBuf && dwNameBufSize ? pcNameBuf : cBuf,
                   pcNameBuf && dwNameBufSize ? dwNameBufSize : sizeof(cBuf) );

      return dwError;
   }
} wnet_errno;

/* win_sys_errlist
 *
 * Used similar to sys_errlist in Unix environments. Note that Windows error
 * messages end with a CR LF NULL. Unix error messages end with only a NULL.
 *
 * Used to get the error message associated with the Windows API error code
 * given as array index.
 *
 * The the returned error message buffer is dynamically allocated, the calling
 * process must use the LocalFree() function to free the buffer when it is no
 * longer needed. If not, a new buffer for the error message is allocated each
 * time win_sys_errlist is used.
 *
 * To make sure it is not taken for a real variable, the address-of operator &
 * returns just type void, which is illegal in any assignment.
 */
extern const class
{
	public:
   void operator&() const{}
   LPTSTR operator[]( const int &__win_errno ) const
   {
		LPTSTR errmsg;
      if( FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM |
      		FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, __win_errno,
            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
            (LPTSTR)&errmsg, 0, NULL ) )
			return errmsg;
      else
      	return NULL;
   }
} win_sys_errlist;

/* h_sys_errlist
 *
 * Completes the win_sys_errlist object with errors messages from Windows
 * Socket 32-bit system. Note: To use this object the calling process must
 * have made a successful call to WSAStartup(). If not, caller would have to
 * assign the error number to h_errno before using h_sys_errlist[].
 */
extern const class
{
	public:
   void operator&() const{}
   LPTSTR operator[]( const int &__h_errno ) const
   {
		LPTSTR errmsg;
      HMODULE hWinSockDLL = GetModuleHandle( WINSOCK_MODULE );
      if( FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM |
            FORMAT_MESSAGE_FROM_HMODULE |
      		FORMAT_MESSAGE_ALLOCATE_BUFFER, hWinSockDLL, __h_errno,
            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
            (LPTSTR)&errmsg, 0, NULL ) )
			return errmsg;
      else
      	return NULL;
   }
} h_sys_errlist;

/* win_error
 *
 * Just a better way than using win_sys_errlist[win_errno]. Used to get the
 * error message associated with the error code of the most recent non-
 * sucessful call to a Windows API function. The buffer is dynamically
 * allocated, the calling process must use the LocalFree() function to free the
 * buffer when it is no longer needed. If not, a new buffer for the error
 * message is allocated each time win_error is used.
 *
 * To make sure it is not taken for a real variable, the address-of operator &
 * returns just type void, which is illegal in any assignment.
 */
extern const class
{
	public:
   void operator&() const{}
   operator LPTSTR() const
   {
		LPTSTR errmsg;
      if( FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM |
      		FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, GetLastError(),
            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
            (LPTSTR)&errmsg, 0, NULL ) )
			return errmsg;
      else
      	return NULL;
	}
} win_error;

/* h_error
 *
 * Completes the win_error object with errors messages from Windows
 * Socket 32-bit system.
 */
extern const class
{
	public:
   void operator&() const{}
   operator LPTSTR() const
   {
		LPTSTR errmsg;
      HMODULE hWinSock = GetModuleHandle( WINSOCK_MODULE );
      if( FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM |
      		FORMAT_MESSAGE_FROM_HMODULE | FORMAT_MESSAGE_ALLOCATE_BUFFER,
            hWinSock, WSAGetLastError(),
            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
            (LPTSTR)&errmsg, 0, NULL ) )
			return errmsg;
      else
      	return NULL;
	}
} h_error;

/* win_perror()
 *
 * Used similar to perror() in Unix environments.
 *
 * Used to print to stderr the error message associated with the error code of
 * the most recently non-sucessful call to a Windows API function. The error
 * message buffer is only used internally and is automatically freed by this
 * function.
 */
inline void win_perror( LPCTSTR __errmsg = NULL )
{
	if( __errmsg ? (bool)*__errmsg : false )
  		cerr << __errmsg << ": ";

   DWORD __win_errno = win_errno;
   LPTSTR win_sys_errmsg = win_error;
   if( win_sys_errmsg )
   {
		cerr << win_sys_errmsg << flush;
   	LocalFree( win_sys_errmsg );
   }
   else
   	cerr << "Error " << __win_errno << endl;
}

/* h_perror()
 *
 * Completes the win_perror function with errors messages from Windows
 * Socket 32-bit system.
 */
inline void h_perror( LPTSTR __errmsg = NULL )
{
	if( __errmsg ? (bool)*__errmsg : false )
  		cerr << __errmsg << ": ";

   int __h_errno = h_errno;
   LPTSTR h_sys_errmsg = h_error;
   if( h_sys_errmsg )
   {
      cerr << h_sys_errmsg << endl;
      LocalFree( h_sys_errmsg );
   }
   else
   	cerr << "Socket Error " << __h_errno << endl;
}

/* hStdIn, hStdOut, hStdErr
 *
 * Used to get or set the standard input, output or error handle respectively.
 *
 * To make sure it is not taken for a real variable, the address-of operator &
 * returns just type void, which is illegal in any assignment.
 */
extern const class
{
	public:
	HANDLE operator=( const HANDLE &hNew ) const
   {
   	if( SetStdHandle( STD_INPUT_HANDLE, hNew ) )
      	return hNew;
      else
      	return INVALID_HANDLE_VALUE;
   }
	void operator&() const{}
   operator HANDLE() const
   {
   	return GetStdHandle( STD_INPUT_HANDLE );
   }
} hStdIn;

extern const class
{
	public:
	HANDLE operator=( const HANDLE &hNew ) const
   {
   	if( SetStdHandle( STD_OUTPUT_HANDLE, hNew ) )
      	return hNew;
      else
      	return INVALID_HANDLE_VALUE;
   }
	void operator&() const{}
   operator HANDLE() const
   {
   	return GetStdHandle( STD_OUTPUT_HANDLE );
   }
} hStdOut;

extern const class
{
	public:
	HANDLE operator=( const HANDLE &hNew ) const
   {
   	if( SetStdHandle( STD_ERROR_HANDLE, hNew ) )
      	return hNew;
      else
      	return INVALID_HANDLE_VALUE;
   }
	void operator&() const{}
   operator HANDLE() const
   {
   	return GetStdHandle( STD_ERROR_HANDLE );
   }
} hStdErr;

// Complement to the lstr*** API functions
inline int lstrncmpi(LPCTSTR lpString1, LPCTSTR lpString2, int iMaxLength)
{
  int iLength1 = lstrlen(lpString1);
  int iLength2 = lstrlen(lpString2);
  int iCompare = CompareString(LOCALE_SYSTEM_DEFAULT, NORM_IGNORECASE|
    SORT_STRINGSORT, lpString1, iLength1<iMaxLength ? iLength1 : iMaxLength,
    lpString2, iLength2<iMaxLength ? iLength2 : iMaxLength);

  if( iCompare == 1 )
    return -1;

  if( iCompare == 3 )
    return 1;

  return 0;
}

// API encapsulation classes

// Enhanced FILETIME stucture with encapsulated API functions,
// compare operators and type conversions
struct WFileTime : public FILETIME
{
   // Get current time function
   void GetUTC()
   {
      GetSystemTimeAsFileTime(this);
   }

   // Time compare operators
   bool operator ==(CONST WFileTime &wft) const
   {
      return !CompareFileTime(this, &wft);
   }

   bool operator >(CONST WFileTime &wft) const
   {
      return CompareFileTime(this, &wft) > 0;
   }

   bool operator >=(CONST WFileTime &wft) const
   {
      return CompareFileTime(this, &wft) >= 0;
   }

   bool operator <(CONST WFileTime &wft) const
   {
      return CompareFileTime(this, &wft) < 0;
   }

   bool operator <=(CONST WFileTime &wft) const
   {
      return CompareFileTime(this, &wft) <= 0;
   }

   // Convert to DOS FAT date and time
   BOOL ToDOS(LPWORD lpFatDate, LPWORD lpFatTime) const
   {
      return FileTimeToDosDateTime(this, lpFatDate, lpFatTime);
   }

   BOOL ToLocal(LPFILETIME lpft) const
   {
      return FileTimeToLocalFileTime(this, lpft);
   }

   BOOL ToUTC(LPFILETIME lpft) const
   {
      return LocalFileTimeToFileTime(this, lpft);
   }

   // Type conversions
   operator QWORD() const
   {
      return *(QWORD*)this;
   }

   operator FILETIME() const
   {
      return *(FILETIME*)this;
   }

   // Construct from SYSTEMTIME stucture
   WFileTime(CONST SYSTEMTIME &st)
   {
      SystemTimeToFileTime(&st, this);
   }

   // Construct from QWORD value
   WFileTime(QWORD qw)
   {
      *(QWORD*)this = qw;
   }

   // Construct from DOS FAT date and time
   WFileTime(WORD wFatDate, WORD wFatTime)
   {
      DosDateTimeToFileTime(wFatDate, wFatTime, this);
   }
};

// Enhanced SYSTEMTIME stucture with encapsulated API functions and type
// conversions
struct WSystemTime : public SYSTEMTIME
{
   // Get/set current time functions
   void GetLocal()
   {
      GetLocalTime(this);
   }

   void GetUTC()
   {
      GetSystemTime(this);
   }

   BOOL SetLocal() const
   {
      return SetLocalTime(this);
   }

   BOOL SetUTC() const
   {
      return SetSystemTime(this);
   }

   // Special for Windows NT
   BOOL GetUTCAsTzSpecificLocal(LPTIME_ZONE_INFORMATION lpTimeZoneInformation,
      LPSYSTEMTIME lpUniversalTime)
   {
      return SystemTimeToTzSpecificLocalTime(lpTimeZoneInformation,
         lpUniversalTime, this);
   }

   // Type conversions
   operator SYSTEMTIME()
   {
      return *(SYSTEMTIME*)this;
   }

   // Construct from FILETIME structure
   WSystemTime(CONST FILETIME &ft)
   {
      FileTimeToSystemTime(&ft, this);
   }
};

// Enhanced TIME_ZONE_INFORMATION stucture with encapsulated API functions
struct WTimeZoneInformation : public TIME_ZONE_INFORMATION
{
   operator TIME_ZONE_INFORMATION() const
   {
      return *(TIME_ZONE_INFORMATION*)this;
   }

   DWORD Get()
   {
      return GetTimeZoneInformation(this);
   }

   BOOL Set() const
   {
      return SetTimeZoneInformation(this);
   }
};

// WFind class. Encapsulation of Windows API functions FindFirstFile(),
// FindNextFile() and FindClose().
class WFind : public WIN32_FIND_DATA
{
   protected:
   HANDLE hFind;   // Open find handle used in internal API calls

   public:
   // bool conversion returns status of object
   operator bool() const
   {
      return hFind != INVALID_HANDLE_VALUE;
   }

   // Start find with file name pattern
   WFind(LPCTSTR lpFileName) : hFind(FindFirstFile(lpFileName, this))
   {
   }

   // Restart find with new file name pattern
   WFind& operator =(LPCTSTR lpFileName)
   {
      if( hFind != INVALID_HANDLE_VALUE )
         FindClose(hFind);

      hFind = FindFirstFile(lpFileName, this);

      return *this;
   }

   ~WFind()
   {
      if( hFind != INVALID_HANDLE_VALUE )
         FindClose(hFind);
   }

   // Find next file
   BOOL Next()
   {
      return FindNextFile(hFind, this);
   }
};

// WFindFiltered adds filtered search functionality to WFind
class WFindFiltered : public WFind
{
   protected:
   // Check found file against filter
   BOOL Filter()
   {
      if( !(dwFileAttributes & dwFileAttrMask) &
         !(~dwFileAttributes & dwFileAttrForce) &
         (*(QWORD*)&ftCreationTime >= ftCreationTime1) &
         (*(QWORD*)&ftCreationTime <= ftCreationTime2) &
         (*(QWORD*)&ftLastAccessTime >= ftLastAccessTime1) &
         (*(QWORD*)&ftLastAccessTime <= ftLastAccessTime2) &
         (*(QWORD*)&ftLastWriteTime >= ftLastWriteTime1) &
         (*(QWORD*)&ftLastWriteTime <= ftLastWriteTime2) &
         ((((QWORD)nFileSizeHigh * MAXDWORD) + nFileSizeLow) >= qwMinSize) &
         ((((QWORD)nFileSizeHigh * MAXDWORD) + nFileSizeLow) <= qwMaxSize) )
         return true;
      else
      {
         if( !WFind::Next() )  // Find next if this didn't match
            return false;

#ifdef NO_RECURSE_SLEEPS
         Sleep(0);
#endif
         return Filter();
      }
   }

   public:
   DWORD dwFileAttrMask;     // Specify attributes files must not have
   DWORD dwFileAttrForce;   // Specify attributes files must have
   QWORD qwMinSize;                      // File size interval
   QWORD qwMaxSize;
   WFileTime ftLastWriteTime1;  // Last write time interval
   WFileTime ftLastWriteTime2;
   WFileTime ftLastAccessTime1; // Last access time interval
   WFileTime ftLastAccessTime2;
   WFileTime ftCreationTime1;   // Creation time interval
   WFileTime ftCreationTime2;

   // Overload with filter function
   WFindFiltered& operator =(LPCTSTR lpFileName)
   {
      WFind::operator=(lpFileName);

      if( hFind == INVALID_HANDLE_VALUE )
         return *this;

      if( !Filter() )
         if( hFind != INVALID_HANDLE_VALUE )
         {
            FindClose(hFind);
            hFind = INVALID_HANDLE_VALUE;
         }
      return *this;
   }

   // Begin find with file name pattern and initialize filter data
   WFindFiltered(LPCTSTR lpFileName,
      DWORD dwSetFileAttrMask = 0,     // Specify attributes not to find
      DWORD dwSetFileAttrForce = 0,   // Specify attributes to find
      QWORD qwSetMinSize = 0,                      // File size interval
      QWORD qwSetMaxSize = (QWORD)-1,
      WFileTime ftSetLastWriteTime1 = 0,  // Last write time interval
      WFileTime ftSetLastWriteTime2 = (QWORD)-1,
      WFileTime ftSetLastAccessTime1 = 0, // Last access time interval
      WFileTime ftSetLastAccessTime2 = (QWORD)-1,
      WFileTime ftSetCreationTime1 = 0,   // Creation time interval
      WFileTime ftSetCreationTime2 = (QWORD)-1
   ) : WFind(lpFileName),
   dwFileAttrMask(dwSetFileAttrMask), dwFileAttrForce(dwSetFileAttrForce),
   qwMinSize(qwSetMinSize), qwMaxSize(qwSetMaxSize),
   ftLastWriteTime1(ftSetLastWriteTime1), ftLastWriteTime2(ftSetLastWriteTime2),
   ftLastAccessTime1(ftSetLastAccessTime1), ftLastAccessTime2(ftSetLastAccessTime2),
   ftCreationTime1(ftSetCreationTime1), ftCreationTime2(ftSetCreationTime2)
   {
      if( hFind == INVALID_HANDLE_VALUE )
         return;

      // The only way we can report error condition is by closing handle and
      // set hFind invalid. This makes bool operator return false, so caller
      // can check if anything found.
      if( !Filter() )
         if( hFind != INVALID_HANDLE_VALUE )
         {
            FindClose(hFind);
            hFind = INVALID_HANDLE_VALUE;
         }
   }

   BOOL Next()
   {
      if( !WFind::Next() )
         return false;

      return Filter();
   }
};

// Enhanced DCB stucture with encapsulated API functions
struct WDCB : public DCB
{
   // Get/set functions
   BOOL Get(HANDLE hComm)
   {
      return GetCommState(hComm, this);
   }

   BOOL Set(HANDLE hComm)
   {
      return SetCommState(hComm, this);
   }

   // Convert pointer to DCB
   operator DCB() const
   {
      return *(DCB*)this;
   }

   // Fill data members using MODE command line style parameter string
   BOOL Build(LPCTSTR lpDef)
   {
      ZeroMemory(this, sizeof(DCB));
      DCBlength = sizeof(DCB);   // Init length member
      return BuildCommDCB(lpDef, this);
   }

   // Empty stucture constructor
   WDCB()
   {
      ZeroMemory(this, sizeof(DCB));
      DCBlength = sizeof(DCB);   // Init length member
   }

   // Construct from MODE command line style parameter string
   WDCB(LPCTSTR lpDef)
   {
      Build(lpDef);
   }
};

#pragma option -Ve- -Vx-

#else /*_cplusplus*/

// Do the best possible if not C++
#  define win_errno      (GetLastError())
#  define cmdlg_errno    (CommDlgExtendedError())
#  ifndef h_errno
#     define h_errno        (WSAGetLastError())
#  endif
#  define hStdIn         (GetStdHandle(STD_INPUT_HANDLE))
#  define hStdOut        (GetStdHandle(STD_OUTPUT_HANDLE))
#  define hStdErr        (GetStdHandle(STD_ERROR_HANDLE))

#endif /*_cplusplus*/

#endif /*__WINERRNO_H*/

