#ifndef _FILEMAP_H
#define _FILEMAP_H
#include <windows>

enum FMAP_ACCESS // Access type for file mapping views
{
	FMAP_RO,      // Read only view
	FMAP_RW,      // Read/write view
	FMAP_WC       // Copy to writeable view
};

#ifndef _QWORD_DEFINED
typedef DWORDLONG QWORD;
#define _QWORD_DEFINED
#endif

class FileMap
{
	private:

	protected:
   HANDLE hFile;           // Handle to open file
   LPCTSTR lpMapName;      // Name of file mapping object
   HANDLE hMap;            // Handle to open file mapping
   LPVOID lpView;          // Pointer to view of mapped file
   QWORD  qwFileSize;      // New size of file
   QWORD  qwMapOffset;     // File offset for data mapped into view
   DWORD  dwBytesMapped;   // File bytes mapped into view
   FMAP_ACCESS Access;     // Access type
   LPSECURITY_ATTRIBUTES lpSecurityAttributes;  // Security Attributes object

   void openfilemap()      // Function to open an existing file mapping
   {
   	// Evaluate the file access parameter from our Access parameter
	   DWORD dwDesiredAccess = 0;
   	if( Access == FMAP_RO )
      	dwDesiredAccess |= FILE_MAP_READ;
      else if( Access == FMAP_RW )
	  	   dwDesiredAccess |= FILE_MAP_ALL_ACCESS;
      else if( Access == FMAP_WC )
      	dwDesiredAccess |= FILE_MAP_COPY;

	   // Open an existing file mapping object
   	if( (hMap = OpenFileMapping(dwDesiredAccess, lpSecurityAttributes ?
         lpSecurityAttributes->bInheritHandle : false, lpMapName)) == NULL )
      {
	   	dwLastError = GetLastError();
         freeall();
	      return;
      }

      // Now make the view
      makeview();
   }

   void makefilemap()      // Function to create the file mapping and view
   {
   	// Evaluate the file access parameter from our Access parameter
	   DWORD flProtect = SEC_COMMIT;
   	if( Access == FMAP_RO )
      	flProtect |= PAGE_READONLY;
   	else if( Access == FMAP_RW )
  		   flProtect |= PAGE_READWRITE;
      else if( Access == FMAP_WC )
      	flProtect |= PAGE_WRITECOPY;

	   // Create a file mapping object
   	if( (hMap = CreateFileMapping(hFile, lpSecurityAttributes, flProtect,
      	(DWORD)(qwFileSize >> 32), (DWORD)qwFileSize, lpMapName)) == NULL )
      {
	   	dwLastError = GetLastError();
         freeall();
	      return;
      }

      // Now make the view
      makeview();
   }

   void makeview()         // Function to create just the view
   {
	   DWORD dwDesiredAccess = 0;
   	if( Access == FMAP_RO )
      	dwDesiredAccess |= FILE_MAP_READ;
      else if( Access == FMAP_RW )
	  	   dwDesiredAccess |= FILE_MAP_ALL_ACCESS;
      else if( Access == FMAP_WC )
      	dwDesiredAccess |= FILE_MAP_COPY;

      // Map view into memory
      if( (lpView = MapViewOfFile(hMap, dwDesiredAccess,
   	   (DWORD)(qwMapOffset >> 32), (DWORD)qwMapOffset, dwBytesMapped)) ==
         NULL )
      {
      	dwLastError = GetLastError();
         freeall();
         return;
      }

      if( !qwFileSize )
      	qwFileSize |= GetFileSize(hFile, (LPDWORD)&qwFileSize+1);

      if( !dwBytesMapped )
	   	dwBytesMapped = (DWORD)(qwFileSize - qwMapOffset);
   }

   void unmakefilemap()    // Function to free the view, close the map and file
   {
   	unmakeview();

      if( hMap )
      {
	  	   CloseHandle(hMap);
         hMap = NULL;
      }
   }

   void unmakeview()     // Just unmap the view
   {
      if( lpView )
         UnmapViewOfFile(lpView);

      lpView = NULL;
	}

   void freeall()   // Free all resources used by object
   {
   	unmakefilemap();

      if( (hFile != NULL) & (hFile != INVALID_HANDLE_VALUE) )
	  	{
	   	CloseHandle( hFile );
      	hFile = INVALID_HANDLE_VALUE;
   	}

   	if( lpSecurityAttributes )
      {
      	delete lpSecurityAttributes;
         lpSecurityAttributes = NULL;
      }
   }

   public:
   DWORD dwLastError;    // Last error code
   DWORD dwRecLen;       // Record length used by index operator

   // Cast to void pointer (or handle) returns pointer (handle) to the view
   operator LPVOID() const
   {
      if( this )
      	return lpView;
      else
         return NULL;
   }

   // Cast to LPSTR returns pointer to the view
   operator LPSTR() const
   {
      if( this )
      	return (LPSTR)lpView;
      else
         return NULL;
   }

   // Cast to LPWSTR pointer returns pointer to the view
   operator LPWSTR() const
   {
      if( this )
      	return (LPWSTR)lpView;
      else
         return NULL;
   }

   // Index operator. Used to get a record from a set of fixed-length record
   // in a mapped file. Record length is set in the dwRecLen public data
   // member.
   LPVOID operator[](DWORD dwRecNr) const
   {
      if( !this )
         return NULL;

      if( dwBytesMapped < (dwRecNr*dwRecLen)+dwRecLen )
         return NULL;

   	return (LPVOID)(((LPSTR)lpView)+dwRecNr*dwRecLen);
   }

   // Returns void pointer to the view
   LPVOID GetView() const
   {
      if( this )
      	return lpView;
      else
         return NULL;
   }

   // Get size of the file mapping object
   QWORD GetSizeOfMappedFile() const
   {
      if( this )
         return qwFileSize;
      else
         return 0;
   }

   // Get number of bytes that was read from physical file into the view
   DWORD GetBytesMapped() const
   {
      if( this )
      	return dwBytesMapped;
      else
         return 0;
   }

   // Get the byte offset in the file, where the view byte read started
   QWORD GetMapOffset() const
   {
      if( this )
      	return qwMapOffset;
      else
         return -1;
   }

   // Make a new view with different offset and length from the mapped file.
   void Resize(DWORD dwBytesToMap = 0, // Length of new view. 0 = Entire file
   				QWORD qwNewMapOffset = 0, // File offset for new view to begin
   				QWORD qwNewFileSize = 0 // New size of file. 0 = Current size.
               )
   {
      if( !this )
         return;

   	// Initialize data members with new values
      qwMapOffset = qwNewMapOffset;
      dwBytesMapped = dwBytesToMap;

      // If file size should be changed, we must close the file mapping object
      if( qwNewFileSize )
      {
		   qwFileSize = qwNewFileSize;
      	unmakefilemap();
         makefilemap();
      }
      else
      {
	      // No file size change, just remake the view
   	   unmakeview();
      	makeview();
      }
   }

   // Flush number of bytes of the view to the physical file. 0 = Entire view.
   BOOL Flush(DWORD dwNumberOfBytesToFlush = 0) const
   {
      if( this )
      	return FlushViewOfFile(lpView, dwNumberOfBytesToFlush);
      else
         return false;
   }

   // Constructor by file name
	FileMap(LPCTSTR lpFileName,   // Name of file to map
   		  FMAP_ACCESS NewAccess = FMAP_RW,  // Access to map and view
           DWORD dwShareMode = 0,   // Share mode for open file. FILE_SHARE_READ
                                    // and/or FILE_SHARE_WRITE is valid.
           QWORD qwNewFileSize = 0,  // New size of file set by mapping object
           QWORD qwNewMapOffset = 0, // Byte offset of view in physical file
           DWORD dwBytesToMap = 0,  // Bytes to map from file

           // Thise is used in CreateFile() call
           DWORD dwCreationDistribution = OPEN_EXISTING,

           // Name of file mapping object used in CreateFileMapping() call
           LPCTSTR lpNewMapName = NULL,

           // These are used in CreateFile() call
           DWORD dwFileAttributes = FILE_ATTRIBUTE_NORMAL|FILE_FLAG_SEQUENTIAL_SCAN,
           LPSECURITY_ATTRIBUTES lpNewSecurityAttributes = NULL, // Also used by CreateFileMapping()
           HANDLE hTemplateFile = NULL
          )
   	// Construct data members
      : hFile(INVALID_HANDLE_VALUE), hMap(NULL), lpView(NULL),
      Access(NewAccess), qwFileSize(qwNewFileSize), qwMapOffset(qwNewMapOffset),
      dwBytesMapped(dwBytesToMap), lpMapName(lpNewMapName),
      lpSecurityAttributes(NULL), dwLastError(NO_ERROR), dwRecLen(1)
	{
		// Evaluate the file access parameter from our Access parameter
		DWORD dwDesiredAccess = 0;

      // Copy security attributes
      if( lpNewSecurityAttributes )
      	lpSecurityAttributes = new SECURITY_ATTRIBUTES(*lpNewSecurityAttributes);

      if( lpMapName )   // If an object name was supplied
         openfilemap();  // First attempt to open an existing object

      if( lpView )
         return;

    	// Open the file
     	if( Access == FMAP_RO )
  	      dwDesiredAccess = GENERIC_READ;
      else if( (Access == FMAP_RW) | (Access == FMAP_WC) )
     	   dwDesiredAccess = GENERIC_READ|GENERIC_WRITE;

		if( (hFile = CreateFile(lpFileName, dwDesiredAccess, dwShareMode,
   		lpSecurityAttributes, dwCreationDistribution, dwFileAttributes,
   	   hTemplateFile)) == INVALID_HANDLE_VALUE )
	   {
   		dwLastError = GetLastError();
         freeall();
	   }
      else
      	// Create the mapping and view using this file handle
	      makefilemap();
	}

   // Constructor by already open file handle
	FileMap(HANDLE hOpenFile = INVALID_HANDLE_VALUE,  // Open file handle
   		  FMAP_ACCESS NewAccess = FMAP_RW,  // Access to map and view
           QWORD qwNewFileSize = 0, // New size of file set by mapping object
           QWORD qwNewMapOffset = 0,  // Byte offset of view in physical file
           DWORD dwBytesToMap = 0,  // Bytes to map from file

           // This is used by call to CreateFileMapping().
           LPCTSTR lpNewMapName = NULL,
           LPSECURITY_ATTRIBUTES lpNewSecurityAttributes = NULL
          )
      // Construct data members
      : hFile(INVALID_HANDLE_VALUE), hMap(NULL), lpView(NULL),
      Access(NewAccess), qwFileSize(qwNewFileSize), qwMapOffset(qwNewMapOffset),
      dwBytesMapped(dwBytesToMap), lpMapName(lpNewMapName),
      lpSecurityAttributes(NULL), dwLastError(NO_ERROR), dwRecLen(1)
   {
      // Copy security attributes
      if( lpNewSecurityAttributes )
      	lpSecurityAttributes = new SECURITY_ATTRIBUTES(*lpNewSecurityAttributes);

   	// Duplicate the incoming file handle
      if( hOpenFile != INVALID_HANDLE_VALUE )
      {
         HANDLE hProc = GetCurrentProcess();
         
      	if( !DuplicateHandle(hProc, hOpenFile, hProc, &hFile, 0,
            lpSecurityAttributes ? lpSecurityAttributes->bInheritHandle : false,
         	DUPLICATE_SAME_ACCESS) )
      	   dwLastError = GetLastError();
      }

   	// Create the mapping and view using this file handle
      if( (hFile != INVALID_HANDLE_VALUE)|(hOpenFile == INVALID_HANDLE_VALUE) )
	      makefilemap();
   }

   // Copy constructor. Duplicates existing object's file handle and creates
   // a new mapping and view for the new object. Data members are copied into
   // the new object.
   FileMap(FileMap& filemap)
      // Construct data members by copy the other object's data
   	: hFile(INVALID_HANDLE_VALUE), lpMapName(filemap.lpMapName), hMap(NULL),
      lpView(NULL), Access(filemap.Access), qwFileSize(filemap.qwFileSize),
      qwMapOffset(filemap.qwMapOffset), dwBytesMapped(filemap.dwBytesMapped),
      lpSecurityAttributes(NULL), dwLastError(NO_ERROR), dwRecLen(1)
   {
      // Copy security attributes into new struct
      if( filemap.lpSecurityAttributes )
      	lpSecurityAttributes = new SECURITY_ATTRIBUTES(*filemap.lpSecurityAttributes);

   	// Duplicate the source object's file handle
      if( filemap.hFile != INVALID_HANDLE_VALUE )
      {
         HANDLE hProc = GetCurrentProcess();

      	if( !DuplicateHandle(hProc, filemap.hFile, hProc, &hFile, 0,
            lpSecurityAttributes ? lpSecurityAttributes->bInheritHandle : false,
         	DUPLICATE_SAME_ACCESS) )
      	   dwLastError = GetLastError();
      }

   	// Create the mapping and view using this file handle
      if( (hFile != INVALID_HANDLE_VALUE)|(filemap.hFile == INVALID_HANDLE_VALUE) )
	      makefilemap();
   }

   ~FileMap()
   {
      freeall();
   }
};
#endif

