// GRABRX01 2021-06-13
// Set the frequency of CLK0 and CLK1 of a SI5351a module. CLK0 locked to PLL A. CLK1 locked to PLL B
// Based on the OE1CGS (Christophe) Arduino sketch, modified by Onno, PA2OHH to use it as 2xVFO for 2 grabber receivers

#include <Wire.h>
#include <avr/sleep.h>

#define F_XTAL  24999989                    // Exact frequency of Xtal in Hz, 25 MHz or 27 MHz
#define c       1048574                     // "c" part of Feedback-Multiplier from XTAL to PLL

unsigned long frequencyA = 28118000;        // Frequency CLK0
unsigned long powerA = 3;                   // Power CLK0 0 = off; 1 = -8 dBm; 2 = -3 dBm; 3 = 0 dBm
unsigned long frequencyB = 14104600;        // frequency CLK1
unsigned long powerB = 3;                   // Power CLK1 0 = off; 1 = -8 dBm; 2 = -3 dBm; 3 = 0 dBm
unsigned long MS0_P1old = 0;                // Old setting of MS0_P1 (CLK0)
unsigned long MS1_P1old = 0;                // Old setting of MS1_P1 (CLK1)
int LEDpin = 13;                            // PIN of LED

void setup() {
  Wire.begin();                             // Initialize I2C-communication as master  SDA on pin ADC04 -  SCL on pin ADC05
  pinMode(LEDpin, OUTPUT);                  // sets the digital pin as output
  
  Si5351a_Write_Reg (16, 128);              // Power down CLK0
  Si5351a_Write_Reg (17, 128);              // Power down CLK1
  Si5351a_Write_Reg (18, 128);              // Power down CLK2 Necessary, also for 10 pin version
  Si5351a_Write_Reg (19, 128);              // Power down CLK3 to switch off all unused CLKn
  Si5351a_Write_Reg (20, 128);              // Power down CLK4 to reduce supply current
  Si5351a_Write_Reg (21, 128);              // Power down CLK5
  Si5351a_Write_Reg (22, 128);              // Power down CLK6
  Si5351a_Write_Reg (23, 128);              // Power down CLK7
}


void loop() {
  delay(1000);                             // Wait 1 second
  digitalWrite(LEDpin, HIGH);              // sets the LED on
  SetFrequencyA(frequencyA);               // Set TX-Frequency CLK0
  SetFrequencyA(powerA);                   // Set CLK0 power
  delay(1000);                             // Wait 1 second
  digitalWrite(LEDpin, LOW);               // sets the LED off

  delay(1000);                             // Wait 1 second
  digitalWrite(LEDpin, HIGH);              // sets the LED on
  SetFrequencyB(frequencyB);               // Set TX-Frequency CLK1
  SetFrequencyB(powerB);                   // Set CLK1 power
  delay(1000);                             // Wait 1 second
  digitalWrite(LEDpin, LOW);               // sets the LED off

  delay(30000);                            // Extra delay of 30s for interference tests
  sleep_enable();                          // Enable sleep mode
  set_sleep_mode(SLEEP_MODE_PWR_DOWN);     // Sleep mode power down
  sleep_cpu();                             // Activate sleep mode
}


void SetFrequencyA(unsigned long frequency) { // Frequency in Hz; must be within [7810 Hz to 200 MHz]
  unsigned long fvco;                  // VCO frequency (600-900 MHz) of PLL
  unsigned long outdivider;            // Output divider in range [4,6,8-900], even numbers preferred
  byte R = 1;                          // Additional Output Divider in range [1,2,4,...128]
  byte a;                              // "a" part of Feedback-Multiplier from XTAL to PLL in range [15,90]
  unsigned long b;                     // "b" part of Feedback-Multiplier from XTAL to PLL
  float f;                             // floating variable, needed in calculation
  unsigned long MS0_P1;                // Si5351a Output Divider register MS0_P1, P2 and P3 are hardcoded below
  unsigned long MSNA_P1;               // Si5351a Feedback Multisynth register MSNA_P1
  unsigned long MSNA_P2;               // Si5351a Feedback Multisynth register MSNA_P2
  unsigned long MSNA_P3;               // Si5351a Feedback Multisynth register MSNA_P3

  if (frequency == 0) {
    Si5351a_Write_Reg (16, 128);       // Switch off output CLK0
  }
  if (frequency == 1) {
    Si5351a_Write_Reg (16, 76);        // CLK0 drive strength = 2mA; power level ~ -8dB
    Si5351a_Write_Reg (177, 32);       // This resets PLL A
  }
  if (frequency == 2) {
    Si5351a_Write_Reg (16, 77);        // CLK0 drive strength = 4mA; power level ~ -3dB
    Si5351a_Write_Reg (177, 32);       // This resets PLL A
  }
  if (frequency == 3) {
    Si5351a_Write_Reg (16, 78);        // CLK0 drive strength = 6mA; power level ~ -1dB
    Si5351a_Write_Reg (177, 32);       // This resets PLL A
  }
  if (frequency == 4) {
    Si5351a_Write_Reg (16, 79);        // CLK0 drive strength = 8mA; power level := 0dB
    Si5351a_Write_Reg (177, 32);       // This resets PLL A
  }

  if (frequency < 7900) {              // Frequency > 7900 Hz
    return;
  }
  if (frequency > 200000000) {         // Frequency < 200 MHz
    return;
  }

  frequencyA = frequency;

  outdivider = 900000000 / frequency;  // With 900 MHz beeing the maximum internal PLL-Frequency

  while (outdivider > 900) {           // If output divider out of range (>900) use additional Output divider
    R = R * 2;
    outdivider = outdivider / 2;
  }
  if (outdivider % 2) outdivider--;    // finds the even divider which delivers the intended Frequency

  fvco = outdivider * R * frequency;   // Calculate the PLL-Frequency (given the even divider)

  switch (R) {                         // Convert the Output Divider to the bit-setting required in register 44
    case 1: R = 0; break;              // Bits [6:4] = 000
    case 2: R = 16; break;             // Bits [6:4] = 001
    case 4: R = 32; break;             // Bits [6:4] = 010
    case 8: R = 48; break;             // Bits [6:4] = 011
    case 16: R = 64; break;            // Bits [6:4] = 100
    case 32: R = 80; break;            // Bits [6:4] = 101
    case 64: R = 96; break;            // Bits [6:4] = 110
    case 128: R = 112; break;          // Bits [6:4] = 111
  }

  a = fvco / F_XTAL;                   // Multiplier to get from Quartz-Oscillator Freq. to PLL-Freq.
  f = fvco - a * F_XTAL;               // Multiplier = a+b/c
  f = f * c;                           // this is just "int" and "float" mathematics
  f = f / F_XTAL;
  b = f;

  MS0_P1 = 128 * outdivider - 512;     // Calculation of Output Divider registers MS0_P1 to MS0_P3
  // MS0_P2 = 0 and MS0_P3 = 1; these values are hardcoded, see below

  f = 128 * b / c;                     // Calculation of Feedback Multisynth registers MSNA_P1 to MSNA_P3
  MSNA_P1 = 128 * a + f - 512;
  MSNA_P2 = f;
  MSNA_P2 = 128 * b - MSNA_P2 * c;
  MSNA_P3 = c;

  // Si5351a_Write_Reg (16, 128);                      // Disable output during the following register settings
  Si5351a_Write_Reg (26, (MSNA_P3 & 65280) >> 8);   // Bits [15:8] of MSNA_P3 in register 26
  Si5351a_Write_Reg (27, MSNA_P3 & 255);            // Bits [7:0]  of MSNA_P3 in register 27
  Si5351a_Write_Reg (28, (MSNA_P1 & 196608) >> 16); // Bits [17:16] of MSNA_P1 in bits [1:0] of register 28
  Si5351a_Write_Reg (29, (MSNA_P1 & 65280) >> 8);   // Bits [15:8]  of MSNA_P1 in register 29
  Si5351a_Write_Reg (30, MSNA_P1 & 255);            // Bits [7:0]  of MSNA_P1 in register 30
  Si5351a_Write_Reg (31, ((MSNA_P3 & 983040) >> 12) | ((MSNA_P2 & 983040) >> 16)); // Parts of MSNA_P3 and MSNA_P1
  Si5351a_Write_Reg (32, (MSNA_P2 & 65280) >> 8);   // Bits [15:8]  of MSNA_P2 in register 32
  Si5351a_Write_Reg (33, MSNA_P2 & 255);            // Bits [7:0]  of MSNA_P2 in register 33

  if (MS0_P1 != MS0_P1old) {
    MS0_P1old = MS0_P1;
    Si5351a_Write_Reg (42, 0);                        // Bits [15:8] of MS0_P3 (always 0) in register 42
    Si5351a_Write_Reg (43, 1);                        // Bits [7:0]  of MS0_P3 (always 1) in register 43
    Si5351a_Write_Reg (44, ((MS0_P1 & 196608) >> 16) | R);  // Bits [17:16] of MS0_P1 in bits [1:0] and R in [7:4]
    Si5351a_Write_Reg (45, (MS0_P1 & 65280) >> 8);    // Bits [15:8]  of MS0_P1 in register 45
    Si5351a_Write_Reg (46, MS0_P1 & 255);             // Bits [7:0]  of MS0_P1 in register 46
    Si5351a_Write_Reg (47, 0);                        // Bits [19:16] of MS0_P2 and MS0_P3 are always 0
    Si5351a_Write_Reg (48, 0);                        // Bits [15:8]  of MS0_P2 are always 0
    Si5351a_Write_Reg (49, 0);                        // Bits [7:0]   of MS0_P2 are always 0

    if (outdivider == 4) {
      Si5351a_Write_Reg (44, 12 | R);                 // Special settings for R = 4 (see datasheet)
      Si5351a_Write_Reg (45, 0);                      // Bits [15:8]  of MS0_P1 must be 0
      Si5351a_Write_Reg (46, 0);                      // Bits [7:0]  of MS0_P1 must be 0
    }
    Si5351a_Write_Reg (177, 32);                      // This resets PLL A
  }
}


void SetFrequencyB(unsigned long frequency) { // Frequency in Hz; must be within [7810 Hz to 200 MHz]
  unsigned long fvco;                  // VCO frequency (600-900 MHz) of PLL
  unsigned long outdivider;            // Output divider in range [4,6,8-900], even numbers preferred
  byte R = 1;                          // Additional Output Divider in range [1,2,4,...128]
  byte a;                              // "a" part of Feedback-Multiplier from XTAL to PLL in range [15,90]
  unsigned long b;                     // "b" part of Feedback-Multiplier from XTAL to PLL
  float f;                             // floating variable, needed in calculation
  unsigned long MS1_P1;                // Si5351a Output Divider register MS1_P1, P2 and P3 are hardcoded below
  unsigned long MSNB_P1;               // Si5351a Feedback Multisynth register MSNB_P1
  unsigned long MSNB_P2;               // Si5351a Feedback Multisynth register MSNB_P2
  unsigned long MSNB_P3;               // Si5351a Feedback Multisynth register MSNB_P3

  if (frequency == 0) {
    Si5351a_Write_Reg (17, 128);       // Switch off output CLK1
  }
  if (frequency == 1) {
    Si5351a_Write_Reg (17, 108);       // CLK0 drive strength = 2mA; power level ~ -8dB
    Si5351a_Write_Reg (177, 128);      // This resets PLL B
  }
  if (frequency == 2) {
    Si5351a_Write_Reg (17, 109);       // CLK1 drive strength = 4mA; power level ~ -3dB
    Si5351a_Write_Reg (177, 128);      // This resets PLL B
  }
  if (frequency == 3) {
    Si5351a_Write_Reg (17, 110);       // CLK1 drive strength = 6mA; power level ~ -1dB
    Si5351a_Write_Reg (177, 128);      // This resets PLL B
  }
  if (frequency == 4) {
    Si5351a_Write_Reg (17, 111);       // CLK1 drive strength = 8mA; power level := 0dB
    Si5351a_Write_Reg (177, 128);      // This resets PLL B
  }

  if (frequency < 7900) {              // Frequency > 7900 Hz
    return;
  }
  if (frequency > 200000000) {         // Frequency < 200 MHz
    return;
  }

  frequencyB = frequency;

  outdivider = 900000000 / frequency;  // With 900 MHz beeing the maximum internal PLL-Frequency

  while (outdivider > 900) {           // If output divider out of range (>900) use additional Output divider
    R = R * 2;
    outdivider = outdivider / 2;
  }
  if (outdivider % 2) outdivider--;    // finds the even divider which delivers the intended Frequency

  fvco = outdivider * R * frequency;   // Calculate the PLL-Frequency (given the even divider)

  switch (R) {                         // Convert the Output Divider to the bit-setting required in register 44
    case 1: R = 0; break;              // Bits [6:4] = 000
    case 2: R = 16; break;             // Bits [6:4] = 001
    case 4: R = 32; break;             // Bits [6:4] = 010
    case 8: R = 48; break;             // Bits [6:4] = 011
    case 16: R = 64; break;            // Bits [6:4] = 100
    case 32: R = 80; break;            // Bits [6:4] = 101
    case 64: R = 96; break;            // Bits [6:4] = 110
    case 128: R = 112; break;          // Bits [6:4] = 111
  }

  a = fvco / F_XTAL;                   // Multiplier to get from Quartz-Oscillator Freq. to PLL-Freq.
  f = fvco - a * F_XTAL;               // Multiplier = a+b/c
  f = f * c;                           // this is just "int" and "float" mathematics
  f = f / F_XTAL;
  b = f;

  MS1_P1 = 128 * outdivider - 512;     // Calculation of Output Divider registers MS0_P1 to MS0_P3
  // MS1_P2 = 0 and MS1_P3 = 1; these values are hardcoded, see below

  f = 128 * b / c;                     // Calculation of Feedback Multisynth registers MSNA_P1 to MSNA_P3
  MSNB_P1 = 128 * a + f - 512;
  MSNB_P2 = f;
  MSNB_P2 = 128 * b - MSNB_P2 * c;
  MSNB_P3 = c;

  // Si5351a_Write_Reg (17, 128);                      // Disable output during the following register settings
  Si5351a_Write_Reg (34, (MSNB_P3 & 65280) >> 8);   // Bits [15:8] of MSNB_P3 in register 34
  Si5351a_Write_Reg (35, MSNB_P3 & 255);            // Bits [7:0]  of MSNB_P3 in register 35
  Si5351a_Write_Reg (36, (MSNB_P1 & 196608) >> 16); // Bits [17:16] of MSNB_P1 in bits [1:0] in register 36
  Si5351a_Write_Reg (37, (MSNB_P1 & 65280) >> 8);   // Bits [15:8]  of MSNB_P1 in register 37
  Si5351a_Write_Reg (38, MSNB_P1 & 255);            // Bits [7:0]  of MSNB_P1 in register 38
  Si5351a_Write_Reg (39, ((MSNB_P3 & 983040) >> 12) | ((MSNB_P2 & 983040) >> 16)); // Parts of MSNB_P3 and MSNB_P1 in register 39
  Si5351a_Write_Reg (40, (MSNB_P2 & 65280) >> 8);   // Bits [15:8]  of MSNB_P2 in register 40
  Si5351a_Write_Reg (41, MSNB_P2 & 255);            // Bits [7:0]  of MSNB_P2 in register 41

  if (MS1_P1 != MS1_P1old) {
    MS1_P1old = MS1_P1;
    Si5351a_Write_Reg (50, 0);                        // Bits [15:8] of MS1_P3 (always 0) in register 50 Bits [15:8]
    Si5351a_Write_Reg (51, 1);                        // Bits [7:0]  of MS1_P3 (always 1) in register 51
    Si5351a_Write_Reg (52, ((MS1_P1 & 196608) >> 16) | R);  // Bits [17:16] of MS1_P1 in bits [1:0] and R in [7:4]
    Si5351a_Write_Reg (53, (MS1_P1 & 65280) >> 8);    // Bits [15:8]  of MS1_P1 in register 53
    Si5351a_Write_Reg (54, MS1_P1 & 255);             // Bits [7:0]  of MS1_P1 in register 54
    Si5351a_Write_Reg (55, 0);                        // Bits [19:16] of MS1_P2 and MS1_P3 are always 0
    Si5351a_Write_Reg (56, 0);                        // Bits [15:8]  of MS1_P2 are always 0
    Si5351a_Write_Reg (57, 0);                        // Bits [7:0]   of MS1_P2 are always 0

    if (outdivider == 4) {
      Si5351a_Write_Reg (60, 12 | R);                 // Special settings for R = 4 (see datasheet)
      Si5351a_Write_Reg (61, 0);                      // Bits [15:8]  of MS1_P1 must be 0
      Si5351a_Write_Reg (62, 0);                      // Bits [7:0]  of MS1_P1 must be 0
    }
    Si5351a_Write_Reg (177, 128);                     // This resets PLL B
  }
}


void Si5351a_Write_Reg (byte regist, byte value) {  // Writes "byte" into "regist" of Si5351a via I2C
  Wire.beginTransmission(96);                       // Starts transmission as master to slave 96, which is the
  // I2C address of the Si5351a (see Si5351a datasheet)
  Wire.write(regist);                               // Writes a byte containing the number of the register
  Wire.write(value);                                // Writes a byte containing the value to be written in the register
  Wire.endTransmission();                           // Sends the data and ends the transmission
}
