********* SYSTEM EQUATES *********

PORTA   EQU     $00     ; Port A register
PORTB   EQU     $01     ; Port B register
DDRA    EQU     $04     ; Port A Data Direction register
DDRB    EQU     $05     ; Port B Data Direction register
ERROR   EQU     0       ; Error Bit
DQ      EQU     5       ; 1820 DQ signal
DQ_CTRL EQU     5
SKIPROM EQU     $CC     ; 1820 Skip ROM command byte
CONVERT EQU     $44     ; 1820 Temperature Convert command byte
READRAM EQU     $BE     ; 1820 Read RAM command byte
CE      EQU     $02     ; ISD2560 chip enable bit
PD      EQU     $03     ; ISD2560 powerdown bit
EOM     EQU     $04     ; ISD2560 end of message bit
DDRAMSK EQU     $FF     ; Port A Data Direction register mask
DDRBMSK EQU     $2F     ; Port B Data Direction register mask
PORTAMSK EQU    $00     ; Port A mask
PORTBMSK EQU    $2C     ; Port B mask
POSITIVE_SIGN   EQU     $00    ; MSB of a positive temperature reading
NEGATIVE_SIGN   EQU     $FF    ; MSB of a neagtive temperature reading
POSITIVE_LIMIT  EQU     $FA    ; The highest LSB for a positive temperature.
NEGATIVE_LIMIT  EQU     $92    ; The lowest LSB for a negative temperature.


********* VARIABLES *********

                ORG $C0

SYS_STATUS      DS      1       ; System status variable
TEMP_HI         DS      1       ; Stores the temperature reading high byte
TEMP_LO         DS      1       ; Stores the temperature reading low byte
TEMP            DS      1       ; Temporary storage space
TEMPA           DS      1       ; Register A tempoary storage space
TEMPX           DS      1       ; Register X temporary storage space

RAW_TEMP        EQU   TEMP_HI   ; Storage space for converted reading
PHRASE_BUFFER   DS      $11     ; Stores addresses of phrases to be output
POINT_FLAG      DS      1       ; Flag indicating a .5 increment in temperature
QUOTIENT        DS      1       ; Storage space for the result of division
PHRASE_POINTER  DS      1       ; Pointer to current address in phrase buffer



                ORG $300

START:          JSR     INITIALIZE      ; Initialize J1A's I/O ports
WAIT4INT        STOP                    ;Stop
                BRA      WAIT4INT

IRQ_INT:        CLR     SYS_STATUS      ; Clear the error bit
                JSR     DEBOUNCE        ; Debounce the activation switch
                BRSET   ERROR,SYS_STATUS,IRQ_INT_EXIT ; If the error bit is
                                        ; set, the exit routine
                JSR     GET_TEMP        ; Get a temperature reading from the 1820
                BRSET   ERROR,SYS_STATUS,IRQ_INT_EXIT ; If the error bit is
                                        ; set, the exit routine
                JSR     FORM_PHRASE     ; Form table of addresses of the phrases to be output
                JSR     OUTPUT_TEMP     ; Audibly output temperature
IRQ_INT_EXIT    BCLR    ERROR,SYS_STATUS ; Clear the error bit
                RTI

***********************************************************
*                                                         *
*  Function Name: OUTPUT_TEMP                             *
*  Function Inputs: None                                  *
*  Functions Outputs: None                                *
*                                                         *
*  Purpose: This function outputs the contents of the     *
*  phrase_buffer to the ISD2560 which outputs them        *
*  audibly.                                               *
*                                                         *
***********************************************************

OUTPUT_TEMP:  BCLR  PD,PORTB            ; Take the ISD2560 out of powerdown mode.
              LDX   #PHRASE_BUFFER      ; Point to the phrase buffer.
OUT_PHRASE:   LDA   PORTB
              AND   #$FC
              ORA   ,X
              STA   PORTB
              INCX
              LDA   ,X          ; Put the address of the next phrase to
              STA   PORTA       ; be output on the address bus of the ISD2560
              INCX
              BCLR  CE,PORTB    ; Pulse the ISD2560's chip enable pin to start
              BSET  CE,PORTB    ; outputting the current phrase.
EOM_H_WAIT:   BRSET EOM,PORTB,EOM_H_WAIT ; Wait for the ISD2560's End of Message
EOM_L_WAIT:   BRCLR EOM,PORTB,EOM_L_WAIT ; pulse before continuing
              LDA   ,X          ; Look for the end of the phrases to be output
              CMP   #$FF        ; if it is found exit the routine. Otherwise
              BNE   OUT_PHRASE  ; continue outputting phrases.
              BSET  PD,PORTB    ; Put the ISD2560 into powerdown mode.
              RTS

***********************************************************
*                                                         *
*  Function Name: FORM_PHRASE                             *
*  Function Inputs: None                                  *
*  Functions Outputs: None                                *
*                                                         *
*  Purpose: This function converts the temperature read   *
*  from the 1820 to the addresses of the phrases in       *
*  the ISD2560 that match the individual digits in the    *
*  reading. These addresses are stored in the phrase      *
*  buffer.                                                *
*                                                         *
***********************************************************


FORM_PHRASE:  CLR   POINT_FLAG  ; Check to see if the temperature reading is a
                                ; a .5 increment, if it is set the POINT_FLAG.
              BRCLR 0,(RAW_TEMP+1),NOT_POINT
              INC   POINT_FLAG
NOT_POINT:    LDX   #PHRASE_BUFFER
              LDA   RAW_TEMP    ; Check to see if the temperature is negative
              BEQ   NOT_NEG     ; if it is, place the address of the "Negative"
              LDA   NEG_ADDR    ; phrase at the start of the phrase buffer. Otherwise
              STA   ,X          ; convert the temperature into its positive equivalent.
              INCX
              LDA   (NEG_ADDR+1)
              STA   ,X
              INCX
              COM   (RAW_TEMP+1)
              INC   (RAW_TEMP+1)
NOT_NEG:      LSR   (RAW_TEMP+1) ; Check for the temperature being lower than 100 degrees
              LDA   (RAW_TEMP+1) ; Celcius.
              CMP   #$64
              BLO   BELOW_100
              SUB   #$64
              STA   (RAW_TEMP+1)
              LDA   HUNDRED_ADDR ; If the temperature is greater than or equal to 100 degrees
              STA   ,X           ; put the address of the "One hundred" phrase in the phrase
              INCX               ; buffer and subtract the equivalent value of 100 from the value.
              LDA   (HUNDRED_ADDR+1)
              STA   ,X
              INCX
              LDA   (RAW_TEMP+1)
              BEQ   POINT
BELOW_100:    LDA   (RAW_TEMP+1) ; Check to see if the remaining temperature value is less than 20
              CMP   #$14         ; degrees. If it is, search for it in the TB0_19 table.
              BLO   BELOW_20     ; Otherwise divide the data by ten. Store the quotient in the
              CLR   QUOTIENT     ; quotient variable and the remainder in (RAW_TEMP+1).
              SUB   #$14
DIV10         CMP   #$A
              BLO   DIV_DONE
              INC   QUOTIENT
              SUB   #$A
              BRA   DIV10
DIV_DONE      STA   (RAW_TEMP+1)
              ASL   QUOTIENT
              STX   PHRASE_POINTER ; Find the address of the quotient's phrase in
              LDX   QUOTIENT       ;the TBL20_90 table and store it in the phrase buffer.
              LDA   TBL20_90,X
              INCX
              STX   TEMP
              LDX   PHRASE_POINTER
              STA   ,X
              INCX
              STX   PHRASE_POINTER
              LDX   TEMP
              LDA   TBL20_90,X
              LDX   PHRASE_POINTER
              STA   ,X
              INCX
              LDA   (RAW_TEMP+1)
              BEQ   POINT
BELOW_20      LDA   (RAW_TEMP+1) ; Find the address of the remainder's phrase in the
              ASLA               ; TBL0_19 table and store it in the phrase buffer.
              STX   PHRASE_POINTER
              TAX
              LDA   TBL0_19,X
              INCX
              STX   TEMP
              LDX   PHRASE_POINTER
              STA   ,X
              INCX
              STX   PHRASE_POINTER
              LDX   TEMP
              LDA   TBL0_19,X
              LDX   PHRASE_POINTER
              STA   ,X
              INCX
POINT         TST   POINT_FLAG          ; If the temperature is a .5 increment reading
              BEQ   END_RAWTEMP         ; load the phrase buffer with the addresses for the
              LDA   POINT_ADDR          ; "Point" and "Five" phrases.
              STA   ,X
              INCX
              LDA   (POINT_ADDR+1)
              STA   ,X
              INCX
              LDA   FIVE_ADDR
              STA   ,X
              INCX
              LDA   (FIVE_ADDR+1)
              STA   ,X
              INCX
END_RAWTEMP   LDA   DEGREE_ADDR         ; Load the phrase buffer with the address for
              STA   ,X                  ; the "Degrees" phrase.
              INCX
              LDA   (DEGREE_ADDR+1)
              STA   ,X
              INCX
              LDA   CELCIUS_ADDR        ; Load the phrase buffer with the address for
              STA   ,X                  ; the "Celcius" phrase.
              INCX
              LDA   (CELCIUS_ADDR+1)
              STA   ,X
              INCX
              CLR   ,X
              DEC   ,X
              RTS

***********************************************************
*                                                         *
*  Function Name: INITIALIZE                              *
*  Function Inputs: None                                  *
*  Functions Outputs: None                                *
*                                                         *
*  Purpose: This function configures PORT A and PORT B    *
*  and their data direction registers.                    *
*                                                         *
***********************************************************

INITIALIZE    LDA   #DDRAMSK
              STA   DDRA
              LDA   #PORTAMSK
              STA   PORTA
              LDA   #DDRBMSK
              STA   DDRB
              LDA   #PORTBMSK
              STA   PORTB
              RTS

***********************************************************
*                                                         *
*  Function Name: GET_TEMP                                *
*  Function Inputs: None                                  *
*  Functions Outputs: None                                *
*                                                         *
*  Purpose: This function performs the required reads and *
*  writes to the 1820 to perform a temperature conversion *
*  and acquisition. The temperature read is returned in   *
*  TEMP variable.                                         *
*                                                         *
***********************************************************

GET_TEMP        JSR     RESET_1820                ; Reset the 1820.
                BRSET   ERROR,SYS_STATUS,GET_ERROR
                LDA     #SKIPROM        ; Send the 1820's SKIP ROM command.
                STA     TEMP
                JSR     WRITE_1820
                LDA     #CONVERT        ; Send the 1820's CONVERT T command.
                STA     TEMP
                JSR     WRITE_1820
READ_LOOP       JSR     READ_1820
                LDA     TEMP
                CMP     #$FF
                BNE     READ_LOOP
                JSR     RESET_1820                 ; Reset the 1820.
                BRSET   ERROR,SYS_STATUS,GET_ERROR ; If the reset fails set the
                LDA     #SKIPROM                   ; error bit and exit the routine.
                STA     TEMP            ; Send the 1820's SKIP ROM command.
                JSR     WRITE_1820
                LDA     #READRAM        ; Read the 1820's RAM to get the temperature
                STA     TEMP            ; reading.
                JSR     WRITE_1820
                JSR     READ_1820
                LDA     TEMP
                STA     TEMP_LO
                JSR     READ_1820
                LDA     TEMP
                STA     TEMP_HI
                CMP     #POSITIVE_SIGN   ; Check for an invalid positive
                BEQ     CHK_POSITIVE     ; data value.
                CMP     #NEGATIVE_SIGN   ; Check for an invalid negative
                BNE     GET_ERROR        ; data value.
                LDA     TEMP_LO
                CMP     #NEGATIVE_LIMIT
                BLO     GET_ERROR
                BRA     GET_EXIT
CHK_POSITIVE    LDA     TEMP_LO
                CMP     #POSITIVE_LIMIT
                BLS     GET_EXIT
GET_ERROR       BSET    ERROR,SYS_STATUS ; Set the error bit if an error
GET_EXIT        JSR     RESET_1820       ; occurs.
                RTS

***********************************************************
*                                                         *
*  Function Name: RESET_1820                              *
*  Function Inputs: None                                  *
*  Functions Outputs: None                                *
*                                                         *
*  Purpose: This function resets the 1820. If the 1820    *
*  resets properly, it will return a response pulse. If   *
*  a pulse is not received, the error bit is set in       *
*  system status.                                         *
*                                                         *
***********************************************************

RESET_1820      STA     TEMPA           ; Save the CPU registers
                STX     TEMPX
                BSET    DQ,PORTB        ; Send a reset pulse to
                BSET    DQ_CTRL,DDRB    ; the 1820
                BCLR    DQ,PORTB
                JSR     DELAY_500uS
                BSET    DQ,PORTB
                BCLR    DQ_CTRL,DDRB    ; Set the J1A to receive the
                JSR     DELAY_100uS     ; response pulse from the 1820
                BRSET   DQ,PORTB,RESET_ERR      ; If the start of the pulse
                JSR     DELAY_500uS     ;is not received, handle the error
                BRSET   DQ,PORTB,RESET_EXIT
RESET_ERR       BSET    ERROR,SYS_STATUS        ;Set the error bit
RESET_EXIT      BSET    DQ,PORTB        ; Set the J1A for transmission
                BSET    DQ_CTRL,DDRB
                LDA     TEMPA           ; Restore CPU registers
                LDX     TEMPX
                RTS

***********************************************************
*                                                         *
*  Function Name: WRITE_1820                              *
*  Function Inputs: None                                  *
*  Functions Outputs: None                                *
*                                                         *
*  Purpose: This function writes the data stored in the   *
*  TEMP variable to the 1820.                             *
*                                                         *
***********************************************************

WRITE_1820      STA     TEMPA           ; Save CPU registers.
                STX     TEMPX
                LDX     #8              ; Load X with count.
WRITE_SHIFT     LSR     TEMP            ; Shift out the bit to be sent
                BCS     WRITE_ONE
WRITE_ZERO      BCLR    DQ,PORTB        ; Send a zero to the 1820
                JSR     DELAY_80uS
                BSET    DQ,PORTB
                BRA     DEC_WRITE
WRITE_ONE       BCLR    DQ,PORTB        ; Send a one to the 1820
                NOP
                NOP
                NOP
                BSET    DQ,PORTB
                JSR     DELAY_80uS
DEC_WRITE       DECX
                BNE     WRITE_SHIFT
                LDA     TEMPA           ; Restore CPU registers
                LDX     TEMPX
                RTS

***********************************************************
*                                                         *
*  Function Name: READ_1820                               *
*  Function Inputs: None                                  *
*  Functions Outputs: None                                *
*                                                         *
*  Purpose: This function reads data from the 1820 and    *
*  returns the data in the TEMP variable.                 *
*                                                         *
***********************************************************

READ_1820       STA     TEMPA           ; Save CPU registers
                STX     TEMPX
                LDX     #8              ; Load X registers with count
READ_BIT        BSET    DQ,PORTB        ; Set up the DQ line for read
                BSET    DQ_CTRL,DDRB
                BCLR    DQ,PORTB
                NOP
                NOP
                NOP
                NOP
                NOP
                BCLR    DQ_CTRL,DDRB    ; Set the DQ line to receive data
                BRSET   DQ,PORTB,READ_ONE       ; Read bit
                CLC
                BRA     READ_SHIFT
READ_ONE        SEC
READ_SHIFT      ROR     TEMP            ; Rotate the bit in the TEMP variable
                JSR     DELAY_80uS
                DECX
                BNE     READ_BIT
                BSET    DQ,PORTB
                BSET    DQ_CTRL,DDRB
                LDA     TEMPA           ; Restore CPU registers
                LDX     TEMPX
                RTS

***********************************************************
*                                                         *
*  Function Name: DEBOUNCE                                *
*  Function Inputs: None                                  *
*  Functions Outputs: None                                *
*                                                         *
*  Purpose: This function debounces the pushbutton switch.*
*                                                         *
***********************************************************

DEBOUNCE        LDX     #$FF
DEBOUNCE_LOOP   JSR     DELAY_500uS
                DECX
                BNE     DEBOUNCE_LOOP
                BIL     DEBOUNCE_EXIT    ; If the interrupt is valid, exit
                                         ; the routine
                BSET    ERROR,SYS_STATUS ; If the interrupt is invalid, set
                                         ; the error bit and exit
DEBOUNCE_EXIT   RTS

***********************************************************
*                                                         *
*  Function Inputs: None                                  *
*  Functions Outputs: None                                *
*                                                         *
*  Purpose: This function provides delays.                *
*                                                         *
***********************************************************

DELAY_80uS      LDA     #$0C
                BRA     DELAY_LOOP
DELAY_100uS     LDA     #$0F
                BRA     DELAY_LOOP
DELAY_500uS     LDA     #$52
                BRA     DELAY_LOOP
DELAY_LOOP      NOP
                NOP
                NOP
                DECA
                BNE     DELAY_LOOP
                RTS




***********************************************************
*                                                         *
*                 PHRASE ADDRESS TABLE                    *
*                                                         *
***********************************************************

              org $700

TBL0_19:      DW  $0000         ; Address for the phrase "Zero".
              DW  $0010         ; Address for the phrase "One".
              DW  $0020         ; Address for the phrase "Two".
              DW  $0030         ; Address for the phrase "Three".
              DW  $0040         ; Address for the phrase "Four".
FIVE_ADDR:    DW  $0050         ; Address for the phrase "Five".
              DW  $0060         ; Address for the phrase "Six".
              DW  $0070         ; Address for the phrase "Seven".
              DW  $0080         ; Address for the phrase "Eight".
              DW  $0090         ; Address for the phrase "Nine".
              DW  $00A0         ; Address for the phrase "Ten".
              DW  $00B0         ; Address for the phrase "Eleven".
              DW  $00C0         ; Address for the phrase "Twelve".
              DW  $00D0         ; Address for the phrase "Thirteen".
              DW  $00E0         ; Address for the phrase "Fourteen".
              DW  $00F0         ; Address for the phrase "Fifteen".
              DW  $0100         ; Address for the phrase "Sixteen".
              DW  $0110         ; Address for the phrase "Seventeen".
              DW  $0120         ; Address for the phrase "Eighteen".
              DW  $0130         ; Address for the phrase "Nineteen".

TBL20_90:     DW  $0140         ; Address for the phrase "Twenty".
              DW  $0150         ; Address for the phrase "Thirty".
              DW  $0160         ; Address for the phrase "Forty".
              DW  $0170         ; Address for the phrase "Fifty".
              DW  $0180         ; Address for the phrase "Sixty".
              DW  $0190         ; Address for the phrase "Seventy".
              DW  $01A0         ; Address for the phrase "Eighty".
              DW  $01B0         ; Address for the phrase "Ninety".

HUNDRED_ADDR: DW  $01C0         ; Address for the phrase "One Hundred".

POINT_ADDR:   DW  $01D0         ; Address for the phrase "Point".

DEGREE_ADDR:  DW  $01E0         ; Address for the phrase "Degree".

NEG_ADDR:     DW  $01F0         ; Address for the phrase "Negative".

CELCIUS_ADDR: DW  $0200         ; Address for the phrase "Celcius".


              ORG     $7FA
              DW      IRQ_INT

              ORG     $7FE
              DW      START

              END
