#define CSPSYS_ERR_NOERROR            0
#define CSPSYS_ERR_GENERALFAILURE     1
#define CSPSYS_ERR_ACQUIRED           2
#define CSPSYS_ERR_UNACQUIRED         3
#define CSPSYS_ERR_UNSUPPORTEDMSG     4
#define CSPSYS_ERR_INVALIDUSER        5
#define CSPSYS_ERR_INVALIDDEVICEID    6
#define CSPSYS_ERR_UNSUPPORTEDPARAM   7
#define CSPSYS_ERR_INVALIDPARAMVALUE  8

#define CSPSYS_PARAM_DMAWIDTH      5
#define CSPSYS_PARAM_DRIVERVERSION 1
#define CSPSYS_PARAM_NCHANNELS     4

#define CSPSYS_ACQUIRE    1
#define CSPSYS_RELEASE    2
#define CSPSYS_DOWNLOAD   3
#define CSPSYS_START      4
#define CSPSYS_STOP       5
#define CSPSYS_SETPARAM   6
#define CSPSYS_GETPARAM   7


#define AUTO_INIT                  1
#define BITS_PER_SAMPLE_8     0x0400
#define BITS_PER_SAMPLE_16    0x0800
#define BLOCK_0               0x0001  // Bit 0 = Block Type 0
#define BLOCK_1               0x0002  // Bit 1 = Block Type 1
#define BLOCK_2               0x0004  // Bit 2 = Block Type 2
#define BLOCK_8               0x0100  // Bit 8 = Block Type 8
#define BLOCK_9               0x0200  // Bit 9 = Block Type 9
#define COMMAND_PAUSE           0x01
#define COMMAND_PLAY            0x02
#define COMMAND_QUIT            0x04
#define COMMAND_REC             0X08
#define DMA8_FF_REG           0x000C
#define DMA8_MASK_REG         0x000A
#define DMA8_MODE_REG         0x000B
#define DMA16_FF_REG          0x00D8
#define DMA16_MASK_REG        0x00D4
#define DMA16_MODE_REG        0x00D6
#define DSP_DATA_AVAIL        0x000E
#define DSP_GET_VERSION       0x00E1
#define DSP_READ_PORT         0x000A
#define DSP_READY             0x00AA
#define DSP_RESET_PORT        0x0006
#define DSP_WRITE_PORT        0x000C
#define END_OF_INTERRUPT      0x0020
#define FAIL                       0
#define FALSE                      0
#define FILE_NOT_DONE_PLAYING 0x8000
#define INVALID_BLOCK_TYPE    0x4000
#define INVALID_FILE_FORMAT        1
#define PIC0_COMMAND_REG        0x20
#define PIC1_COMMAND_REG        0xA0
#define PIC0_MASK_REG           0x21
#define PIC1_MASK_REG           0xA1
#define REMEMBER_VOLUME            1
#define RESTORE_VOLUME             2
#define SB2_LO                     1
#define SB2_HI                     2
#define SBPRO                      3
#define SB16                       4
#define SINGLE_CYCLE               2
#define SUCCESS                    1
#define TRUE                  !FALSE
#define UNUSED                     0
// #define USE_CSP                        // Comment out if not using CSP chip.
#define VOC_FILE                   2
#define WAVE_FILE                  3

#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <mem.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\stat.h>

extern void error(int error_number,char *from);

extern unsigned char            REC_BUF_A[];
extern unsigned char            REC_BUF_B[];
extern char                     REC_BUF_AB;
extern char 			XSET;

/*------------------- TYPEDEFS -----------------------------------------*/
/*----------------------------------------------------------------------*/
typedef struct _BLASTER
{
  short int BaseIOPort,
	    DMAChan8Bit,
	    DMAChan16Bit,
	    DSPVersion,
	    IRQNumber,
		 MIDIPort;
} BLASTER;

typedef struct _FILEINFO
{
  unsigned char      BitsPerSample,
		     Channels;      // MONO = 1, STEREO = 2
  unsigned short int FileFormat,    // Determines BitsPerSample in old .VOC
		     TimeConstant;  // Old .VOC version of SampPerSec
  signed long int    SampPerSec;
} FILEINFO;



typedef unsigned long int DWORD;

/*------------------- FUNCTION PROTOTYPES ------------------------------*/
/*----------------------------------------------------------------------*/
char GetBlastEnv(BLASTER *Blast),
     ResetDSP(short int),
     VerifyFileType(FILE *);


short int DSPRead(short int),
	  LoadAndPlay(unsigned long int, int , unsigned char *, char );

unsigned long int AllocateDMABuffer(unsigned char **, unsigned short int *),
		  OnSamePage(unsigned char *, unsigned short int);

void interrupt DmaISR(void);
void ContinueDMA(unsigned char),
     DSPWrite(short int, short int),
     KillVolume(void),
     PauseDMA(unsigned char),
     ProgramDMA(unsigned long int, FILEINFO *, unsigned short int),
     ProgramDSP(unsigned short int, FILEINFO *, unsigned char),
     RestoreOldISR(BLASTER *),
     RestoreOrRememberVolume(char),
     SetDmaISR(BLASTER *),
	  SetMixer(void);


short int RecAndSave(unsigned long int   DMABufPhysAddr,
		      int                 File,
		      unsigned char      *DMABuf,
		      char                Command
		      );


/*------------------- GLOBAL VARIABLES ---------------------------------*/
/*----------------------------------------------------------------------*/
BLASTER           gBlastSet;

signed short int   gBitsPerSample,
		   gChannels;


unsigned short int gCardType,
		   gDMABufNowPlaying,
		   gDMABufSize,       // See "WARNING #1" above.
		   gHighSpeed,
		   gIRQMaskSave0,
		   gIRQMaskSave1;

unsigned long int  gUserNo;

signed long int gBytesLeftToPlay;

void interrupt (*gOldISRFuncPtr)(void);

/*************************************************************************
*
* FUNCTION: main()
*
* DESCRIPTION: READ IT!
*
*************************************************************************/
long int play_file(char *pfile,long int posi)
{
  char               Command,
		     KeyPressed;
  short int          Status;
  unsigned char far  *DMABuf;
  unsigned long int  DMABufPhysAddr;
  unsigned long int  pos;
  int                File;

  /*--- GET BLASTER ENVIRONMENT VARIABLES AND DSP VERSION --------------*/
  if (GetBlastEnv(&gBlastSet) == FAIL)
  {
	 error(300,"Playing");
	 close(File);
	 return 0;
  }

  /*--- OPEN THE FILE TO PLAY OR RECORD --------------------------------*/
  File=open(pfile,O_BINARY, S_IREAD);
  if (File<0)
		{
		 printf("Can Not open file:\n\r");
		 perror("Dos Error");
		 printf("\n\r");
		 close(File);
		 return 0;
		}

  lseek(File,posi,SEEK_SET);
  if (File == 0)
  {
	 printf("\n\rfopen() FAILS!  %s",strerror(errno));
	 getch();
	 close(File);
	 return 0;
  }
  if (filelength(File)<10000l)
  {
	 printf("\n\rFAILS! Not long enough to play");
	 getch();
	 close(File);
	 return 0;
  }
  /*--- CONVERT COMMAND-LINE PARAMETER TO DMA BUFFER SIZE. -------------*/
  gDMABufSize = 10000;


  /*--- ALLOCATE DMA BUFFER --------------------------------------------*/
  /*--------------------------------------------------------------------*/
  DMABufPhysAddr = AllocateDMABuffer(&DMABuf, &gDMABufSize);
  if (DMABufPhysAddr == FAIL)
  {
	 puts("\n\rAllocateDMABuffer() FAILS!");
	 getch();
	 close(File);
	 return 0;
  }

	 /*--- THE FOLLOWING printf()s ARE FOR DEBUG ONLY! --------------------*/
/*  printf("DMA Buffer = %u bytes.\n", gDMABufSize);
  printf("I/O        = %x (hex)\n",  gBlastSet.BaseIOPort);
  printf("DMA 8-bit  = %d\n",        gBlastSet.DMAChan8Bit);
  printf("DMA 16-bit = %d\n",        gBlastSet.DMAChan16Bit);
  printf("IRQ        = %d\n",        gBlastSet.IRQNumber);
  printf("DSP Ver.   = %d.%02d\n",  (gBlastSet.DSPVersion >> 8) & 0x00FF,
					 (gBlastSet.DSPVersion & 0x00FF));

  puts("PRESS:  Space Bar  to Pause/Resume");
  puts("PRESS:  Esc or' Q' to Quit");

*/

  if (XSET==1) SetMixer();

  SetDmaISR(&gBlastSet);


  Command = COMMAND_PLAY;  // "Command" MUST equal this to begin playing!
  ResetDSP(gBlastSet.BaseIOPort);  // Always avoid problems--reset SB card!


  // LOOP HERE WHILE FILE IS PLAYING.
  do
  {
	 Status = LoadAndPlay(DMABufPhysAddr,File,DMABuf,Command);

	 /*--- DETECT KEYS HIT.  PAUSE, RESUME, AND QUIT SUPPORTED. -------*/
	 /*----------------------------------------------------------------*/
	 pos=tell(File);
	 gotoxy(1,14);
	 printf("%6li",pos);

    if (kbhit())
    {
     KeyPressed = getch();
     if (KeyPressed == 0)     // If 1st byte is 0, key pressed sent 2 bytes.
	 KeyPressed = getch();  // Flush the 2nd byte from the buffer.

     switch (KeyPressed)
	   {
	    case 13: {
		      Command = COMMAND_QUIT;
		      pos=pos;
		      break;
			  }


	    case 32: {
		      pos=posi;
		      lseek(File,posi,SEEK_SET);
		      Command = COMMAND_QUIT;
		      break;
		     }
	    case 27: {
		      Command = COMMAND_QUIT;
		      pos=posi;
		      lseek(File,posi,SEEK_SET);
				break;
		     }
    }
    if (eof(File)) Command = COMMAND_QUIT;
   }

  } while (Status & FILE_NOT_DONE_PLAYING);


  RestoreOldISR(&gBlastSet);

  close(File);
return pos;
}




/*************************************************************************
*
* FUNCTION: LoadAndPlay()
*
* DESCRIPTION:
*
* RETURN: "ReturnStatus", where the bits (bit 0 to bit 15) have the
*         significance shown.
*
*         Bit No.     | If bit set HI (equals 1), it means...
*         ---------------------------------------------------
*          0 (LSBit)  | Block 0 detected (File Done Playing)
*          1          | Block 1 detected
*          2 - 7      | Unused
*          8          | Block 8 detected
*          9          | Block 9 detected
*          10         | BitsPerSample = 8
*          11         | BitsPerSample = 16
*          12 - 13    | Unused
*          14         | Invalid Block Type detected (Fatal Error)
*          15 (MSBit) | File NOT done playing
*
*************************************************************************/
short int LoadAndPlay(unsigned long int   DMABufPhysAddr,
		      int                 File,
		      unsigned char      *DMABuf,
		      char                Command
		      )
{
  static FILEINFO           FileHdrInfo;
  static signed long int    BlockByteCount;
  static char	            FirstTimeFuncCalled = TRUE;
  static unsigned short int Count,
			    DMABufToLoad;
  char                      BlockType;
  short int                 ReturnStatus = FILE_NOT_DONE_PLAYING;
  unsigned int byte;

  /*--- CHECK FOR COMMANDS SENT AFTER FILE HAS BEGUN PLAYING. ----------*/
  /*--------------------------------------------------------------------*/
  switch(Command)
  {
    case COMMAND_PAUSE:
      if (FileHdrInfo.BitsPerSample == 8)
	ReturnStatus |= BITS_PER_SAMPLE_8;
      else
	ReturnStatus |= BITS_PER_SAMPLE_16;
	 return(ReturnStatus);

    case COMMAND_PLAY:
      if (FileHdrInfo.BitsPerSample == 8)
	ReturnStatus |= BITS_PER_SAMPLE_8;
      else
	ReturnStatus |= BITS_PER_SAMPLE_16;
    break;

    case COMMAND_QUIT:
      ResetDSP(gBlastSet.BaseIOPort);          // Stops DMA transfer!
      ReturnStatus &= ~FILE_NOT_DONE_PLAYING;  // File IS DONE playing!
      FirstTimeFuncCalled = TRUE;              // Reset for next file.
	 return(ReturnStatus);                      // RETURN!
  }


  if (FirstTimeFuncCalled == TRUE)
  {
    FirstTimeFuncCalled = FALSE;
    goto read_new_block;
  }
  else
  {
    // Prevent loading DMA buffer when 1/2 is loaded and other 1/2 is playing.
    while (DMABufToLoad == gDMABufNowPlaying);  // WAIT!
  }


  /*--- Point DMABuf to top 1/2 of DMA buffer if needed. ---------------*/
  /*--------------------------------------------------------------------*/
  if (DMABufToLoad == 1)
    DMABuf += (gDMABufSize / 2);  // Load top 1/2 of DMA buffer.

  if (BlockByteCount > gDMABufSize / 2)
  {
    /*--- LOAD EXACTLY 1/2 DMA BUFFER FROM THE FILE. -------------------*/
    /*------------------------------------------------------------------*/
    Count = gDMABufSize / 2;
	 _dos_read(File,DMABuf, Count, &byte);
    if (byte<Count)
	 {
	  ResetDSP(gBlastSet.BaseIOPort);          // Stops DMA transfer!
	  ReturnStatus &= ~FILE_NOT_DONE_PLAYING;  // File IS DONE playing!
	  FirstTimeFuncCalled = TRUE;              // Reset for next file.
	  return(ReturnStatus);                    // RETURN!
	 }
    DMABufToLoad ^= 1;  // Get ready to load next 1/2 DMA buffer.
  }
  else if (BlockByteCount >= 0)
  {
    /*--- LOAD ALL DATA REMAINING IN CURRENT BLOCK INTO DMA BUFFER. ----*/
	 /*------------------------------------------------------------------*/
    Count = (unsigned short int) BlockByteCount;
    if (Count > 0)
    {
      _dos_read(File,DMABuf,Count, &byte);
      if (byte<Count)
	 {
	  ResetDSP(gBlastSet.BaseIOPort);          // Stops DMA transfer!
	  ReturnStatus &= ~FILE_NOT_DONE_PLAYING;  // File IS DONE playing!
	  FirstTimeFuncCalled = TRUE;              // Reset for next file.
	  return(ReturnStatus);                    // RETURN!
	 }

		if (byte<Count)
      if (Count == gDMABufSize / 2)
      DMABufToLoad ^= 1;  // Get ready to load next 1/2 DMA buffer.
      ProgramDSP(Count, &FileHdrInfo, SINGLE_CYCLE);
    }

read_new_block:

    /*--- READ A NEW BLOCK TYPE AND STORE ITS INFORMATION IN THE -----*/
    /*--- THE LOCAL STATIC STRUCTURE "FileHdrInfo".              -----*/
    /*----------------------------------------------------------------*/
  BlockByteCount = 0L;  // Clears MSByte
  BlockByteCount=filelength(File);
  FileHdrInfo.TimeConstant  = 156;
  FileHdrInfo.FileFormat    = 0;
  BlockByteCount           -= 2;
  FileHdrInfo.Channels      = 1;
  FileHdrInfo.BitsPerSample = 8;
  FileHdrInfo.SampPerSec    = 10000;
  ReturnStatus              |= BLOCK_1;

  gBitsPerSample = FileHdrInfo.BitsPerSample;
  gChannels      = FileHdrInfo.Channels;

  DMABufToLoad      = 1;     // Next 1/2 DMA buffer to load is top 1/2.
  gBytesLeftToPlay  = BlockByteCount;  // Altered by DmaISR().
  gDMABufNowPlaying = 0;     // Altered by ISR when 1/2 buffer done playing.
  gHighSpeed        = FALSE; // Initialize to NOT high-speed DMA.

  /*--- LOAD DMA BUFFER AND BEGIN PLAYING THE FILE. --------*/
  /*--------------------------------------------------------*/
  ProgramDMA(DMABufPhysAddr, &FileHdrInfo, gDMABufSize);

  if (BlockByteCount > gDMABufSize / 2)
	{
	  Count = gDMABufSize / 2;
	  _dos_read(File,DMABuf, Count, &byte);
	  if (byte<Count)
	    {
		  ResetDSP(gBlastSet.BaseIOPort);          // Stops DMA transfer!
	     ReturnStatus &= ~FILE_NOT_DONE_PLAYING;  // File IS DONE playing!
	     FirstTimeFuncCalled = TRUE;              // Reset for next file.
	     return(ReturnStatus);                    // RETURN!
	    }
	  ProgramDSP(Count, &FileHdrInfo, AUTO_INIT);  // Begin audio.
	}
	else
	{
	  Count = (int) BlockByteCount;
	  _dos_read(File,DMABuf, Count, NULL);
	  ProgramDSP(Count, &FileHdrInfo, SINGLE_CYCLE);  // Begin audio.
	}
  } // End: else if (BlockByteCount >= 0)

BlockByteCount -= (long) Count;  // Update No. of bytes left in block.

return(ReturnStatus);
}

/*************************************************************************
*
* FUNCTION: ProgramDMA()
*
* DESCRIPTION:  This function programs the DMA chip to use a single
*               8-bit or 16-bit DMA channel (specified by the BLASTER
*               environment string) for audio transfer.  It also programs
*               the size of the DMA transfer and the DMA buffer address
*               used for the audio transfer.
*
*************************************************************************/
void ProgramDMA(unsigned long int DMABufPhysAddr, FILEINFO *FileHdrInfo,
		unsigned short int Count)
{
  short int Command,
	    DMAAddr,
	    DMACount,
	    DMAPage,
	    Offset,
		 Page,
	    Temp;

  Page   = (short int) (DMABufPhysAddr >> 16);
  Offset = (short int) (DMABufPhysAddr & 0xFFFF);

  if (FileHdrInfo->FileFormat < 4)  // 8-BIT FILE
  {
    switch(gBlastSet.DMAChan8Bit)
    {
      case 0:
	DMAAddr  = 0x0000;
	DMACount = 0x0001;
	DMAPage  = 0x0087;
      break;

      case 1:
	DMAAddr  = 0x0002;
	DMACount = 0x0003;
	DMAPage  = 0x0083;
      break;

      case 3:
	DMAAddr  = 0x0006;
	DMACount = 0x0007;
	DMAPage  = 0x0082;
		break;
    }

    outp(DMA8_MASK_REG, gBlastSet.DMAChan8Bit | 4);    // Disable DMA
    outp(DMA8_FF_REG, 0x0000);                         // Clear F-F
    outp(DMA8_MODE_REG, gBlastSet.DMAChan8Bit | 0x58); // 8-bit AI
    outp(DMACount, ((Count - 1) & 0xFF));              // LO byte
    outp(DMACount, ((Count - 1) >> 8));                // HI byte
  }
  else  // 16-BIT FILE
  {
    switch(gBlastSet.DMAChan16Bit)
    {
		case 5:
	DMAAddr  = 0x00C4;
	DMACount = 0x00C6;
	DMAPage  = 0x008B;
      break;

      case 6:
	DMAAddr  = 0x00C8;
	DMACount = 0x00CA;
	DMAPage  = 0x0089;
      break;

      case 7:
	DMAAddr  = 0x00CC;
	DMACount = 0x00CE;
	DMAPage  = 0x008A;
      break;
    }

    // Offset for 16-bit DMA must be calculated different than 8-bit.
    // Shift Offset 1 bit right.  Then copy LSBit of Page to MSBit of Offset.
    Temp = Page & 0x0001;  // Get LSBit of Page and...
    Temp <<= 15;           // move it to MSBit of Temp.
    Offset >>= 1;          // Divide Offset by 2.
    Offset &= 0x7FFF;      // Clear MSBit of Offset.
    Offset |= Temp;        // Put LSBit of Page into MSBit of Offset.

    outp(DMA16_MASK_REG, (gBlastSet.DMAChan16Bit - 4) | 4);    // Disable DMA
    outp(DMA16_FF_REG, 0x0000) ;                               // Clear F-F
    outp(DMA16_MODE_REG, (gBlastSet.DMAChan16Bit - 4) | 0x58); // 16-bit AI
    outp(DMACount, ((Count/2 - 1) & 0xFF));                    // LO byte
    outp(DMACount, ((Count/2 - 1) >> 8));                      // HI byte
  }

  // Program the starting address of the DMA buffer.
  outp(DMAPage, Page);             // Page number of DMA buffer.
  outp(DMAAddr, Offset & 0x00FF);  // LO byte offset address of DMA buffer.
  outp(DMAAddr, (Offset >> 8));    // HI byte offset address of DMA buffer.

  // Reenable 8-bit or 16-bit DMA.
  if (FileHdrInfo->FileFormat < 4)
    outp(DMA8_MASK_REG,  gBlastSet.DMAChan8Bit);
  else
    outp(DMA16_MASK_REG, gBlastSet.DMAChan16Bit - 4);

  return;
}


/*************************************************************************
*
* FUNCTION: ProgramDSP()
*
* DESCRIPTION: This function programs the DSP chip on the Sound Blaster
*              card.  The card type is identified by the DSP version
*              number.  Each type of Sound Blaster card is programmed
*              differently unless an 8-bit ADPCM file is played.  In that
*              case, all SB cards are programmed identically.
*
*************************************************************************/
void ProgramDSP(unsigned short int Count, FILEINFO *FileHdrInfo,
		unsigned char DMAMode)
{
  short int     Command,
		Mode;

  if (gHighSpeed == TRUE)  // Once in high-speed mode, DSP can only be reset!
    return;

  // Make sure Count is >= 2, so when the DSP is programmed for a block
  // tranfer, Count doesn't wrap around to a large number when 1 is
  // subtracted from it.
  if (Count <= 1)
    Count = 2;


  /*--- DETERMINE SOUND BLASTER CARD TYPE ----------------------------*/
  /*------------------------------------------------------------------*/
  if (gBlastSet.DSPVersion >= 0x0400)       // DSP version >= 4.00
    gCardType = SB16;
  else if (gBlastSet.DSPVersion >= 0x0300)  // DSP version = 3.xx
  {
    // Set SBPRO mixer register default to MONO, Output filter to OFF.
    outp(gBlastSet.BaseIOPort + 4, 0x000E);  // Select mixer reg. 0x0E.
    outp(gBlastSet.BaseIOPort + 5, 0x0000);  // MONO, Output filter off.
    gCardType = SBPRO;
  }
  else if (gBlastSet.DSPVersion >= 0x0201)  // 2.01 <= DSP version < 3.00
    gCardType = SB2_HI;
  else if (gBlastSet.DSPVersion == 0x0200)  // DSP version = 2.00
    gCardType = SB2_LO;


  /*--- FILE IS 8-BIT ADPCM PLAYBACK. IN THIS CASE, ALL SB CARDS  ------*/
  /*--- ARE PROGRAMMED IDENTICALLY.  DETERMINE THE COMMAND.       ------*/
  /*--------------------------------------------------------------------*/
  if (FileHdrInfo->FileFormat > 0 && FileHdrInfo->FileFormat < 4)
  {
    switch(FileHdrInfo->FileFormat)
    {
      case 1:  // 8-bit 2:1 compression (4-bit ADPCM)
	Command = 0x0074;  // default to single-cycle
      break;

		case 2:  // 8-bit 8:3 compression (2.6-bit ADPCM)
	Command = 0x0076;  // default to single-cycle
      break;

      case 3:  // 8-bit 4:1 compression (2-bit ADPCM)
	Command = 0x0016;  // default to single-cycle
      break;
    }

    if (DMAMode == AUTO_INIT)
      Command |= 0x0009;   // Set to auto-init mode.
  }
  else
  {
    /*--- FILE IS 8-BIT OR 16-BIT UNCOMPRESSED AUDIO.  PROGRAM EACH ----*/
    /*--- SOUND BLASTER CARD DIFFERENTLY.                           ----*/
    /*------------------------------------------------------------------*/
    switch(gCardType)
    {
      case SB16:
	// Program sample rate HI and LO byte.
	DSPWrite(gBlastSet.BaseIOPort, 0x0041);
	DSPWrite(gBlastSet.BaseIOPort, (FileHdrInfo->SampPerSec & 0xFF00) >> 8);
	DSPWrite(gBlastSet.BaseIOPort, (FileHdrInfo->SampPerSec & 0xFF));

	/*--- DETERMINE 8-bit OR 16-bit, MONO OR STEREO ----------------*/
	/*--------------------------------------------------------------*/
	if (FileHdrInfo->BitsPerSample == 8)
	{
	  Command = 0x00C0;  // 8-bit transfer (default: single-cycle, D/A)

	  if (FileHdrInfo->Channels == 1)
	    Mode = 0x0000;   // MONO, unsigned PCM data
	  else
	    Mode = 0x0020;   // STEREO, unsigned PCM data
	}
	else  // 16-BIT AUDIO
	{
	  Command = 0x00B0;  // 16-bit transfer (default: single-cycle, D/A)
	  Count  /= 2;       // Set Count to transfer 16-bit words.

	  if (FileHdrInfo->Channels == 1)
	    Mode = 0x0010;   // MONO, signed PCM data
	  else
	    Mode = 0x0030;   // STEREO, signed PCM data
	}

	/*--- CHANGE COMMAND TO AUTO-INIT, IF NEEDED. ------------------*/
	/*--------------------------------------------------------------*/
	if (DMAMode == AUTO_INIT)
	  Command |= 0x0004;     // Auto-init


	/*--- PROGRAM THE DSP CHIP (BEGIN DMA TRANSFER) AND RETURN! ----*/
	/*--------------------------------------------------------------*/
	DSPWrite(gBlastSet.BaseIOPort, Command);
	DSPWrite(gBlastSet.BaseIOPort, Mode);
	DSPWrite(gBlastSet.BaseIOPort, (Count - 1) & 0xFF);  // LO byte
	DSPWrite(gBlastSet.BaseIOPort, (Count - 1) >> 8);    // HI byte
      return;    // RETURN!


      case SBPRO:
	if (FileHdrInfo->Channels == 2)
	{
	  // HI-SPEED, STEREO
	  gHighSpeed = TRUE;

	  DSPWrite(gBlastSet.BaseIOPort, 0x00A8);  // STEREO MODE
	  outp(gBlastSet.BaseIOPort + 4, 0x000E);  // Select mixer reg. 0x0E.
	  outp(gBlastSet.BaseIOPort + 5, 0x0002);  // STEREO, output filter off.

	  if (DMAMode == AUTO_INIT)
	  {
	    Command = 0x0090;  // HIGH-SPEED, AUTO-INIT MODE
	  }
	  else
	  {
		 Command = 0x0091;  // HIGH-SPEED, SINGLE-CYCLE MODE
	  }
	}
	else if (FileHdrInfo->SampPerSec >= 23000)
	{
	  // HI-SPEED, MONO
	  gHighSpeed = TRUE;

	  if (DMAMode == AUTO_INIT)
	  {
	    Command = 0x0090;  // HIGH-SPEED, AUTO-INIT MODE
	  }
	  else
	  {
	    Command = 0x0091;  // HIGH-SPEED, SINGLE-CYCLE MODE
	  }
	}
	else if (DMAMode == AUTO_INIT)
	{
	  Command = 0x001C;   // NORMAL, AUTO-INIT
	}
	else
	{
	  Command = 0x0014;   // NORMAL, SINGLE-CYCLE
	}
      break;


      case SB2_HI:
	if (FileHdrInfo->SampPerSec > 13000 || FileHdrInfo->Channels == 2)
	{
	  // HI-SPEED
	  gHighSpeed = TRUE;

	  if (DMAMode == AUTO_INIT)
	    Command = 0x0090;  // HIGH-SPEED, AUTO-INIT MODE
	  else
	    Command = 0x0091;  // HIGH-SPEED, SINGLE-CYCLE MODE
	}
	else if (DMAMode == AUTO_INIT)
	  Command = 0x001C;  // NORMAL, MONO, AUTO-INIT
	else
	  Command = 0x0014;  // NORMAL, MONO, SINGLE-CYCLE
      break;


      case SB2_LO:  // DSP VERSION == 2.00.  HIGH-SPEED MODE NOT AVAILABLE.
	if (DMAMode == AUTO_INIT)
	  Command = 0x001C;  // NORMAL, MONO, AUTO-INIT
	else
	  Command = 0x0014;  // NORMAL, MONO, SINGLE-CYCLE
      break;
	 }
  }


  /*--- IF FILE IS 8-BIT ADPCM (REGARDLESS OF CARD TYPE), OR CARD IS ---*/
  /*--- AN 8-BIT AUDIO CARD (DSP VERSION < 4.xx), BEGIN DMA TRANFER. ---*/
  /*--------------------------------------------------------------------*/
  DSPWrite(gBlastSet.BaseIOPort, 0x00D1);  // Turn speaker on.
  DSPWrite(gBlastSet.BaseIOPort, 0x0040);  // Program Time Constant
  DSPWrite(gBlastSet.BaseIOPort, FileHdrInfo->TimeConstant);

  /*--- NOTE: If in high-speed mode, single-cycle DMA is programmed ----*/
  /*--- using the same initial DSP command as auto-init (0x0048).   ----*/
  /*--------------------------------------------------------------------*/
  if (DMAMode == AUTO_INIT || gHighSpeed == TRUE)
  {
    // Program block tranfer size LO and HI byte and begin tranfer.
    DSPWrite(gBlastSet.BaseIOPort, 0x0048);
    DSPWrite(gBlastSet.BaseIOPort, (Count - 1) & 0x00FF);  // LO byte
    DSPWrite(gBlastSet.BaseIOPort, (Count - 1) >> 8);      // HI byte
    DSPWrite(gBlastSet.BaseIOPort, Command);               // Begin Xfer
  }
  else  // DMAMode == SINGLE_CYCLE  If mode is high-speed, execute above code.
  {
    // Program size of last block and begin transfer.
    DSPWrite(gBlastSet.BaseIOPort, Command);
	 DSPWrite(gBlastSet.BaseIOPort, (Count - 1) & 0x00FF);  // LO byte
    DSPWrite(gBlastSet.BaseIOPort, (Count - 1) >> 8);      // HI byte
  }

  return;
}



/*************************************************************************
*
* FUNCTION:  DmaISR()
*
* DESCRIPTION: If the interrupt was a DMA generated by the Sound Blaster,
*              acknowledge the interrupt, update the global variables,
*              and send the end-of-interrupt command(s).
*
*              If the interrupt was NOT generated by the Sound Blaster,
*              call the ISR (gOldISRFuncPtr) saved by SetDMAISR() or
*              return.
*
*************************************************************************/
void interrupt DmaISR(void)
{
  unsigned char InterruptStatus;

  // Read Sound Blaster mixer interrupt register to determine interrupt type.
  if (gCardType != SB16)
  {
    inp(gBlastSet.BaseIOPort + 0x000E);  // Acknowledge the interrupt.
  }
  else  // Card is 16-bit SB card.
  {
    outp(gBlastSet.BaseIOPort + 4, 0x0082);          // Select interrupt reg.
    InterruptStatus = inp(gBlastSet.BaseIOPort + 5); // Read interrupt reg.

    if (InterruptStatus & 0x01)            // Interrupt is from 8-bit DMA.
      inp(gBlastSet.BaseIOPort + 0x000E);  // Acknowledge the interrupt.
    else if (InterruptStatus & 0x02)       // Interrupt is from 16-bit DMA.
		inp(gBlastSet.BaseIOPort + 0x000F);  // Acknowledge the interrupt.
    else
    {
      // Interrupt is NOT SB DMA.  Call ISR saved by SetDmaISR() or return.
      if (gOldISRFuncPtr)
	(*gOldISRFuncPtr)();
      return;
    }
  }

  gBytesLeftToPlay  -= (long) (gDMABufSize / 2);
  gDMABufNowPlaying ^= 1;  // Keep track of which 1/2 DMA buffer is playing.

  // Send end-of-interrupt command(s).
  if (gBlastSet.IRQNumber > 7)
    outp(PIC1_COMMAND_REG, END_OF_INTERRUPT);

  outp(PIC0_COMMAND_REG, END_OF_INTERRUPT);

  return;
}


/*************************************************************************
*
* FUNCTION: RestoreOldISR()
*
* DESCRIPTION:  1) Disable all interrupts.
*               2) Restore IRQ mask for IRQs 0 to 7.
*               3) If necessary, restore IRQ mask for IRQs 8 to 15.
*               4) Restore the original ISR (which was saved to a global
*                  varaiable function pointer in SetDmaISR()) for the
*                  interrupt vector number associated with
*                  BlastSet->IRQNumber.
*               5) Enable all interrupts.
*
*************************************************************************/
void RestoreOldISR(BLASTER *BlastSet)
{
  short int IntVectorNumber;

  disable();  // Temporarily disable interrupts

  outp(PIC0_MASK_REG, gIRQMaskSave0);    // Restore IRQ mask for IRQs 0 to 7.

  if (BlastSet->IRQNumber > 7)
  {
    outp(PIC1_MASK_REG, gIRQMaskSave1);  // Restore IRQ mask for IRQs 8 to 15.
    IntVectorNumber = BlastSet->IRQNumber - 8 + 0x70;
  }
  else  // BlastSet->IRQNumber is 0 to 7.
    IntVectorNumber = BlastSet->IRQNumber + 8;

  // Restore the old ISR to the interrupt vector number.
  setvect(IntVectorNumber,
	  gOldISRFuncPtr);

  enable();   // Enable interrupts
  return;
}


/*************************************************************************
*
* FUNCTION: SetDmaISR()
*
* DESCRIPTION:  1) Disable all interrupts.
*               2) Save current interrupt mask(s) to global variable(s).
*               3) Set new interrupt mask(s).
*               4) Save ISR associated with BlastSet->IRQNumber to
*                  a global variable function pointer.
*               5) Set the new ISR associated with BlastSet->IRQNumber.
*               6) Enable all interrupts.
*
*************************************************************************/
void SetDmaISR(BLASTER *BlastSet)
{
  short int IntVectorNum,
		 IRQMaskNew;

  disable();  // Temporarily disable interrupts.

  /*--- Save current interrupt masks and set the new ones. -------------*/
  gIRQMaskSave0 = inp(PIC0_MASK_REG);    // Save IRQ 0 to 7 mask.
  if (BlastSet->IRQNumber > 7)
  {
    IntVectorNum  = BlastSet->IRQNumber - 8 + 0x70;
    gIRQMaskSave1 = inp(PIC1_MASK_REG);  // Save IRQ 8 to 15 mask.

    // Set new IRQ mask for IRQs 8 to 15.
    IRQMaskNew = ~(((short int) 0x0001) << (BlastSet->IRQNumber - 8));
	 outp(PIC1_MASK_REG, gIRQMaskSave1 & IRQMaskNew);

    // Setting IRQ mask 2 on PIC 0 enables IRQs 8 to 15 on PIC 1.
    outp(PIC0_MASK_REG, gIRQMaskSave0 & ~0x0004);
  }
  else  // BlastSet->IRQNumber is 0 to 7.
  {
    IntVectorNum = BlastSet->IRQNumber + 8;

    // Set new IRQ mask for IRQs 0 to 7.
    IRQMaskNew = ~(((short int) 0x0001) << BlastSet->IRQNumber);
    outp(PIC0_MASK_REG, gIRQMaskSave0 & IRQMaskNew);
  }

  /*--- Save current ISR and set the new one. --------------------------*/
  gOldISRFuncPtr = getvect(IntVectorNum);
  setvect(IntVectorNum, DmaISR);

  enable();  // Enable interrupts.

  return;
}


/************************************************************************
*
* FUNCTION: GetBlastEnv()
*
* DESCRIPTION: Search the BLASTER environment string for:
*
*              A) Base IO Port Address
*              B) LO (8 bit)  DMA Channel
*              C) HI (16 bit) DMA Channel
*              D) IRQ Number
*              E) MIDI Port address
*
*              The Base I/O port address and the MIDI address are stored
*              in the environment string in hex--convert them to integer.
*              These numbers from the environment string are then placed
*              in the BLASTER struct passed to this function as a pointer.
*              The BLASTER struct is defined as:
*
*              typedef struct _BLASTER
*              {
*                short int BaseIOPort,
*                          DMAChan8Bit,
*                          DMAChan16Bit,
*                          DSPVersion,
*                          IRQNumber,
*                          MIDIPort;
*              } BLASTER;
*
*              Then, get the DSP version of the Sound Blaster DSP chip.
*              This is used to determine the Sound Blaster's capabilities.
*
* RETURN: FAIL - BLASTER environment string is not found or any of
*                the BLASTER structure members aren't found.
*
*         SUCCESS - All 5 members of BLASTER struct are found in the
*                   BLASTER environment string.
*
************************************************************************/
char GetBlastEnv(BLASTER *Blast)
{
  char  Buffer[5],
	DMAChannelNotFound = TRUE,
       *EnvString,
	IOPortNotFound     = TRUE,
	IRQNotFound        = TRUE,
	SaveChar;

  short int digit,
	    i,
	    Major,
	    Minor,
	    multiplier;


  EnvString = getenv("BLASTER");

  if (EnvString == NULL)
    return(FAIL);  // BLASTER environment variable not found.

  do
  {
    switch(*EnvString)
    {
      case 'A':  // I/O base port address found
      case 'a':
	EnvString++;
	for (i = 0; i < 3; i++)  // Grab the digits
	{
	  Buffer[i] = *EnvString;
	  EnvString++;
	}

	// The string is in ASCII HEX, convert it to decimal
	multiplier = 1;
	Blast->BaseIOPort = 0;
	for (i -= 1; i >= 0; i--)
	{
	  // Convert to HEX
	  if (Buffer[i] >= '0' && Buffer[i] <= '9')
	    digit = Buffer[i] - '0';
	  else if (Buffer[i] >= 'A' && Buffer[i] <= 'F')
	    digit = Buffer[i] - 'A' + 10;
	  else if (Buffer[i] >= 'a' && Buffer[i] <= 'f')
	    digit = Buffer[i] - 'a' + 10;

	  Blast->BaseIOPort += digit * multiplier;
	  multiplier *= 16;
	}

	IOPortNotFound = FALSE;
      break;


		case 'D': // 8-bit DMA channel
      case 'd':
      case 'H': // 16-bit DMA channel
      case 'h':
	SaveChar = *EnvString;
	EnvString++;
	Buffer[0] = *EnvString;
	EnvString++;

	if (*EnvString >= '0' && *EnvString <= '9')
	{
	  Buffer[1] = *EnvString; // DMA Channel No. is 2 digits
	  Buffer[2] = NULL;
	  EnvString++;
	}
	else
	  Buffer[1] = NULL;       // DMA Channel No. is 1 digit

	if (SaveChar == 'D' || SaveChar == 'd')
	  Blast->DMAChan8Bit  = atoi(Buffer);  // 8-Bit DMA channel
	else
	  Blast->DMAChan16Bit = atoi(Buffer);  // 16-bit DMA channel

	DMAChannelNotFound = FALSE;
      break;


      case 'I':  // IRQ number
      case 'i':
	EnvString++;
	Buffer[0] = *EnvString;
	EnvString++;

	if (*EnvString >= '0' && *EnvString <= '9')
	{
	  Buffer[1] = *EnvString; // IRQ No. is 2 digits
	  Buffer[2] = NULL;
	  EnvString++;
	}
	else
	  Buffer[1] = NULL;       // IRQ No. is 1 digit

	Blast->IRQNumber  = atoi(Buffer);
	IRQNotFound = FALSE;
      break;


      default:
	EnvString++;
      break;
    }

  } while (*EnvString != NULL);

  if (DMAChannelNotFound || IOPortNotFound || IRQNotFound)
    return(FAIL);

  /*--- Get the DSP version number.  The next read from the DSP will ---*/
  /*--- return the major version number.  The following read will    ---*/
  /*--- return the minor version number.                             ---*/
  ResetDSP(gBlastSet.BaseIOPort);
  DSPWrite(Blast->BaseIOPort, DSP_GET_VERSION);
  Major = DSPRead(Blast->BaseIOPort);  /* Read Major DSP version no. */
  Minor = DSPRead(Blast->BaseIOPort);  /* Read Minor DSP version no. */
  Blast->DSPVersion = (Major << 8) | Minor;

  return(SUCCESS);
}


/*************************************************************************
*
* FUNCTION: DSPRead()
*
* DESCRIPTION: Reads a value from the DSP Read Port.
*
* Entry: BaseIOPort - The Sound Blaster's base I/O address.
*
*************************************************************************/
short int DSPRead(short int BaseIOPort)
{
  /* Wait until DSP is ready before reading from the DSP. */
  while ((inp(BaseIOPort + DSP_DATA_AVAIL) & 0x80) == 0);

  /* Return value read from the Read Port. */
  return(inp(BaseIOPort + DSP_READ_PORT));
}


/*************************************************************************
*
* FUNCTION: DSPWrite()
*
* DESCRIPTION: Writes the value passed to this function to the DSP Write
*              Port.
*
* Entry: BaseIOAddr - The Sound Blaster's base I/O address.
*
*************************************************************************/
void DSPWrite(short int BaseIOAddr, short int WriteValue)
{
  /* Wait until DSP is ready before writing to the DSP. */
  while ((inp(BaseIOAddr + DSP_WRITE_PORT) & 0x80) != 0);

  outp(BaseIOAddr + DSP_WRITE_PORT, WriteValue);
  return;
}


/*************************************************************************
*
* FUNCTION: ResetDSP()
*
* DESCRIPTION: Self explanatory
*
* Entry: BaseIOAddr - The Sound Blaster's base I/O address.
*
*************************************************************************/
char ResetDSP(short int IOBasePort)
{
  outp(IOBasePort + DSP_RESET_PORT, 0x0001);  /* Write "1" to Reset Port. */
  delay(10);                                  /* Wait 10 mS.              */
  outp(IOBasePort + DSP_RESET_PORT, 0x0000);  /* Write "0" to Reset port. */

  /* Wait until data is available. (Wait while BIT-7 == 0.) */
  while ((inp(IOBasePort + DSP_DATA_AVAIL) & 0x80) == 0);

  if (inp(IOBasePort + DSP_READ_PORT) == DSP_READY)
    return(SUCCESS);
  return(FAIL);
}

/*************************************************************************
*
* FUNCTION: AllocateDMABuffer()
*
* DESCRIPTION : Allocate memory for the DMA buffer.  After memory is
*               allocated for the buffer, call OnSamePage() to verify
*               that the entire buffer is located on the same page.
*               If the buffer crosses a page boundary, allocate another
*               buffer. Continue this process until the DMA buffer resides
*               entirely within the same page.
*
*               For every malloc() called, save a pointer that points to
*               the block of memory allocated.  Deallocate ALL memory blocks
*               allocated that cross a page boundary.  Once a memory block
*               is allocated that does NOT cross a page boudary, this block
*               will be used for the DMA buffer--any previously allocated
*               memory blocks will be deallocated.
*
* ENTRY: **DMABuffer is the address of the pointer that will point to
*        the memory allocated.
*
* EXIT: If a buffer is succesfully allocated, *DMABuffer will point to
*       the buffer and the physical address of the buffer pointer will
*       be returned.
*
*       If a buffer is NOT successfully allocated, return FAIL.
*
*************************************************************************/

unsigned long int AllocateDMABuffer(unsigned char **DMABuffer,
				    unsigned short int *DMABufSize)
{
  unsigned long int PhysAddress;



  *DMABuffer = (unsigned char *) REC_BUF_A;
  PhysAddress = OnSamePage(*DMABuffer, *DMABufSize);
  if (PhysAddress==FAIL)
      {
	*DMABuffer=(unsigned char *) REC_BUF_B;
	PhysAddress=OnSamePage(*DMABuffer, *DMABufSize);
      }


  return(PhysAddress);
}


/**************************************************************************
*
* FUNCTION: OnSamePage()
*
* DESCRIPTION: Check the memory block pointed to by the parameter
*              passed to make sure the entire block of memory is on the
*              same page.  If a buffer DOES cross a page boundary,
*              return FAIL. Otherwise, return the physical address
*              of the beginning of the DMA buffer.
*
*              A page corresponds to the following addresses:
*
*              PAGE NO.   SEG:OFF ADDRESS          PHYSICAL ADDRESS
*              --------   ----------------------   ----------------
*                 0       0000:0000 to 0000:FFFF   00000 to 0FFFF
*                 1       1000:0000 to 1000:FFFF   10000 to 1FFFF
*                 .                 .                    .
*                 .                 .                    .
*                 E       E000:0000 to E000:FFFF   E0000 to EFFFF
*                 F       F000:0000 to F000:FFFF   F0000 to FFFFF
*
*              NOTE: The upper nibble of the physical address is the
*                    same as the page number!
*
* ENTRY: *DMABuffer - Points to beginning of DMA buffer.
*
* EXIT: If the buffer is located entirely within one page, return the
*       physical address of the buffer pointer.  Otherwise return FAIL.
*
**************************************************************************/
unsigned long int OnSamePage(unsigned char *DMABuffer,
			     unsigned short int DMABufSize)
{
  unsigned long int BegBuffer,
		    EndBuffer,
		    PhysAddress;

  /*----- Obtain the physical address of DMABuffer -----*/
  BegBuffer = ((unsigned long) (FP_SEG(DMABuffer)) << 4) +
	       (unsigned long) FP_OFF(DMABuffer);
  EndBuffer   = BegBuffer + DMABufSize - 1;
  PhysAddress = BegBuffer;

  /*-- Get page numbers for start and end of DMA buffer. --*/
  BegBuffer >>= 16;
  EndBuffer >>= 16;

  if (BegBuffer == EndBuffer)
    return(PhysAddress);  // Entire buffer IS on same page!
  return(FAIL); // Entire buffer NOT on same page.  Thanks Intel!
}


/*************************************************************************
*
* FUNCTION: KillVolume()
*
*************************************************************************/
void KillVolume(void)
{
  // Only SB 2 with CD interface has a mixer chip.
  if (gBlastSet.DSPVersion < 0x0300)         // Select master volume reg.
    outp(gBlastSet.BaseIOPort + 4, 0x0002);
  else  // SB Pro or SB16/AWE32 mixer.
    outp(gBlastSet.BaseIOPort + 4, 0x0022);  // Select master volume reg.

  outp(gBlastSet.BaseIOPort + 5, 0x0000);    // KILL the volume.

  return;
}


/*************************************************************************
*
* FUNCTION: RestoreOrRememberVolume()
*
*************************************************************************/
void RestoreOrRememberVolume(char Command)
{
  static short int MasterVolume = 0;


  if (gBlastSet.DSPVersion < 0x0300)         // Only SB 2 with CD has a mixer.
    outp(gBlastSet.BaseIOPort + 4, 0x0002);  // Select master volume reg.
  else // SB Pro or SB16/AWE32 mixer.
    outp(gBlastSet.BaseIOPort + 4, 0x0022);  // Select master volume reg.


  if (Command == REMEMBER_VOLUME)
       {
	MasterVolume = inp(gBlastSet.BaseIOPort + 5);  // Save master volume.
	printf("Master Volume = %i",MasterVolume);
	getch();
       }
       else if (Command == RESTORE_VOLUME)
       {
	outp(gBlastSet.BaseIOPort + 5,MasterVolume);  // Restore master volume.
	printf("restoreing volume\n\r");
	getch();
       }
  return;
}


/*************************************************************************
*
* FUNCTION: ContinueDMA()
*
* DESCRIPTION: Continues the DMA transfer that was halted.  The DMA
*              tranfer can be halted by calling PauseDMA().
*
* SEE ALSO: PauseDMA()
*
*************************************************************************/
void ContinueDMA(unsigned char BitsPerSample)
{

  if (gBlastSet.DSPVersion >= 0x0200 && gBlastSet.DSPVersion < 0x0400)
	 RestoreOrRememberVolume(RESTORE_VOLUME);

  /*--- IF IN HIGH-SPEED MODE, CAN'T REPROGRAM DSP CHIP--RETURN! -------*/
  /*--- OTHERWISE, RESUME THE DMA.                               -------*/
  /*--------------------------------------------------------------------*/
  if (gHighSpeed == TRUE)
    return;
  else if (BitsPerSample == 8)
    DSPWrite(gBlastSet.BaseIOPort, 0x00D4);  // Continue SB 8-bit DMA xfer.
  else  // BitsPerSample == 16
    DSPWrite(gBlastSet.BaseIOPort, 0x00D6);  // Continue SB 16-bit DMA xfer.

  return;
}


/*************************************************************************
*
* FUNCTION: PauseDMA()
*
* DESCRIPTION: Halts the DMA tranfer.  The DMA tranfer can be resumed by
*              calling ContinueDMA().
*
* SEE ALSO: ContinueDMA()
*
*************************************************************************/
void PauseDMA(unsigned char BitsPerSample)
{

  if (gBlastSet.DSPVersion >= 0x0200 && gBlastSet.DSPVersion < 0x0400)
  {
    RestoreOrRememberVolume(REMEMBER_VOLUME);
    KillVolume();
  }

  /*--- IF IN HIGH-SPEED MODE, CAN'T REPROGRAM DSP CHIP--RETURN. -------*/
  /*--- OTHERWISE, HALT THE DMA.                                 -------*/
  /*--------------------------------------------------------------------*/
  if (gHighSpeed == TRUE)
	 return;
  else if (BitsPerSample == 8)
    DSPWrite(gBlastSet.BaseIOPort, 0x00D0);  // Pause SB 8-bit DMA xfer.
  else  // BitsPerSample == 16
    DSPWrite(gBlastSet.BaseIOPort, 0x00D5);  // Pause SB 16-bit DMA xfer.

  return;
}


/*************************************************************************
*
* FUNCTION: SetMixer()
*
* DESCRIPTION: Sets mixer to maximum volume.
*
*************************************************************************/
void SetMixer(void)
{

  GetBlastEnv(&gBlastSet);
  outp(gBlastSet.BaseIOPort + 4, (int) 0x02); //  Master Volume SBPRO
  outp(gBlastSet.BaseIOPort + 5, (int) 0xFF);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x22); // MASTER VOL PRO
  outp(gBlastSet.BaseIOPort + 5, (int) 0xFF);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x38); // LINE IN RIGHT SB16
  outp(gBlastSet.BaseIOPort + 5, (int) 0x00);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x39); // LINE IN RIGHT SB16
  outp(gBlastSet.BaseIOPort + 5, (int) 0x00);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x43); // AGC OFF SB16
  outp(gBlastSet.BaseIOPort + 5, (int) 0x01);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x04); // DAC LEVEL SBPRO
  outp(gBlastSet.BaseIOPort + 5, (int) 0xFF);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x0A); // MIC LEVEL SBPRO
  outp(gBlastSet.BaseIOPort + 5, (int) 0x07);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x0C); // MIC LEVEL SBPRO
  outp(gBlastSet.BaseIOPort + 5, (int) 0x01);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x32); // MASTER DAC LEFT SB16
  outp(gBlastSet.BaseIOPort + 5, (int) 0xF8);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x33); // MASTER DAC RIGHT SB16
  outp(gBlastSet.BaseIOPort + 5, (int) 0xF8);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x3A); // MASTER MIC SB16
  outp(gBlastSet.BaseIOPort + 5, (int) 0xF8);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x3C); // MASTER MIC SB16
  outp(gBlastSet.BaseIOPort + 5, (int) 0x00);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x3D); // SELECT INPUT MIC Right
  outp(gBlastSet.BaseIOPort + 5, (int) 0x01);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x3E); // SELECT INPUT MIC Left
  outp(gBlastSet.BaseIOPort + 5, (int) 0x01);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x3F); // GAIN IN LEFT  SB16
  outp(gBlastSet.BaseIOPort + 5, (int) 0xC0);

  outp(gBlastSet.BaseIOPort + 4, (int) 0x40); // GAIN IN RIGHT SB16
  outp(gBlastSet.BaseIOPort + 5, (int) 0xC0);

  return;
}

