/* set_8563_clock_and_alarm_N5IB.ino
 *  v2.00   22 January 2020
 *  
 *  Rtc_Pcf8563 Set Time and Alarm
 * Sets a PCF8563 clock chip to a time that is input from the serial monitor,
 * Also set an alarm time
 * then loop continuously reading time and 
 * printing the time and date, and alarm setting to the serial monitor
 * Indicates whether the alrm is currently active (i.e.  ringing) or not
 * Then clears the alarm and sets it forward a few minutes for testing
 *
 * Pin connections to Arduino are (see Pcf8563 data sheet)
 * Vcc can be either 3.3 or 5 V
 * SCK -> A5,   SDA -> A4,   optionally INT -> D3/INT1 (needs ~10 K pullup)
 * No pullups needed on SCK or SDA (Arduino's I2C internal pullups used)
 * 
 * After loading and starting the sketch, use the serial monitor
 * to type in the time and date settings, then to see the clock operating
 * 
 * Joe Robertson, jmr
 * modified by Jim Giammanco October 2019, January 2020
 */

#include <Wire.h>         // included library for I2C protocol
#include <Rtc_Pcf8563.h>  // custom library for PCF8563 chip

// Initialize the real time clock object. Within the library the
// I2C 8 bit slave addresses default to 0xA2 (write) and 0xA3 (read)
// so the 7 bit slave address used to communicate on the I2C bus is 0x51
Rtc_Pcf8563 rtc;

//global variables  
byte I2C_slave = 0x51;   // 7-bit slave address of 8563 chip
byte CLKOUT_cntl = 0x0D; // 8563 register address that controls clock output pulse
byte freqSelect = 0x80;  //pulse control byte: 0x80 - 32.768 kHz, 0x81 - 1024 Hz, 0x82 - 32 Hz,  0x83 - 1 Hz
byte error;              // acknowledgement return from I2C device
int setMonth;
int setDay, setAlarmDay;
int setWeekday, setAlarmWeekday;  // note: Sun=0 Mon=1 .... Sat=6
int setYear;    // only last two digits used
int setHour, setAlarmHour;
int setMinute, setAlarmMinute;
int setSecond;
byte alarm_minute;

//***********************************************************************
void setup()
{
  Serial.begin(9600);
  Serial.setTimeout(30000);
  while (!Serial) {
    ; // wait for serial port to connect. Needed for native USB port only
  }
  Serial.println("Setting 8563 Real Time Clock, by N5IB v1.0");
  delay(3000);
  
  //clear out the RTC registers
  rtc.initClock();

  // enable an output pulse on clock output pin - open drain requires a pullup resistor  ~10K
  // control byte's MSB=1 enables, MSB=0 disables, LS2bits set frequency (0x00 to disable)
  Wire.beginTransmission(I2C_slave);    // Issue I2C start signal to slave address (using default 0x51)
  Wire.write(CLKOUT_cntl);  // 8563 register address that controls clock output pulse
  Wire.write(freqSelect);  // control byte
  error=Wire.endTransmission();
  if (error != 0) Serial.print("Unknown I2C error");   // check for slave acknowledge   

  while (Serial.available() > 0) Serial.read();   // clear the serial buffer
  Serial.print("Month: ");
  setMonth = Serial.parseInt();
  if (setMonth==0) 
  {
   while (setMonth==0) {}
  }
  
  Serial.println(setMonth);

  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Day: ");
  setDay = Serial.parseInt();
  if (setDay==0) setDay=1;
  Serial.println(setDay);

  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Day of week [Sunday=0]: ");
  setWeekday = Serial.parseInt();
  if (setWeekday==0) setWeekday=1;
  Serial.println(setWeekday);
    
  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Year -20yy:  ");
  setYear = Serial.parseInt();
  if (setYear==0) setYear=19;
  Serial.print("20");
  Serial.println(setYear);
  
  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Hour: ");
  setHour = Serial.parseInt();
  if (setHour==0) setHour=0;
  Serial.println(setHour);

  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Minute: ");
  setMinute = Serial.parseInt();
  if (setMinute==0) setMinute=0;
  Serial.println(setMinute);
  
  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Second: ");
  setSecond = Serial.parseInt();
  if (setSecond==0) setSecond=0;
  Serial.println(setSecond);
  Serial.println(); //send CR+LF
  rtc.setDateTime(setDay,setWeekday,setMonth,0,setYear,setHour,setMinute,setSecond);

  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Alarm day of month (0 or 99 = no alarm set): ");
  setAlarmDay = Serial.parseInt();
  if ((setAlarmDay==0) || (setAlarmDay>31)) setAlarmDay=99;  // no alarm set if day out of range
  Serial.println(setAlarmDay);
  Serial.println(); //send CR+LF

  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Alarm weekday (0 = Sunday,  or else 99 = no alarm set): ");
  setAlarmWeekday = Serial.parseInt();
  if (setAlarmWeekday>6) setAlarmWeekday=99;  // if anything past 6 (Saturday), don't set an alarm 
  Serial.println(setAlarmWeekday);
  Serial.println(); //send CR+LF

  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Alarm hour(99 = no alarm set): ");
  setAlarmHour = Serial.parseInt();
  if (setAlarmHour>23) setAlarmHour=99;  // no alarm set if anything over 23 entered
  Serial.println(setAlarmHour);
  Serial.println(); //send CR+LF

  while (Serial.available() > 0) Serial.read();  // clear the serial buffer
  Serial.print("Alarm minute (99 = no alarm set): ");
  setAlarmMinute = Serial.parseInt();
  if (setAlarmMinute>59) setAlarmDay=99;  // no alarm set if  anything over 59 entered
  Serial.println(setAlarmMinute);
  Serial.println(); //send CR+LF

  rtc.setAlarm(setAlarmMinute, setAlarmHour, setAlarmDay, setAlarmWeekday);
}

//****************************************************************
void loop()  // now display a clock 
{
  //both format functions call the internal getTime() so that the 
  //formatted strings are at the current time/date.
  Serial.print(rtc.formatTime());
  Serial.print("    ");
  Serial.println(rtc.formatDate());

  // now show what the alarm is set for
  rtc.getAlarm();
  Serial.print(rtc.getAlarmHour()); Serial.print(":"); Serial.print(rtc.getAlarmMinute());
  if (rtc.alarmActive()) 
  {
    Serial.println("     *****Alarm Ringing****");
    rtc.resetAlarm(); // clear the alarm flag and make the alarm active again at the same time
                      // alarm will not trigger again until time makes a transition into the set alarm time
                      // even if current time is still the same as the set alarm time.

    setAlarmMinute=setAlarmMinute+5;  // but now, bump alarm 5 minutes into the future for test purposes
    if(setAlarmMinute>59) {
      setAlarmMinute=setAlarmMinute-60; setAlarmHour=setAlarmHour+1; // roll over the hour if needed
    }
    if (setAlarmHour>23) {
      setAlarmHour=setAlarmHour-24;
    }
    rtc.setAlarm(setAlarmMinute, setAlarmHour, setAlarmDay, setAlarmWeekday);   
  }
  else {
    Serial.println();
  }
    
  delay(1000);
//  Serial.print("testing...");  // for debugging
}    // end of main program loop




