/*
 Dual BME-280 sensors & SD card weather data logger v3.1
 Jim Giammanco, N5IB  21 September 2019
 logs data from analog, digital, and I2C sensors
 to an SD card (SPI interface) using the SD library.

 Local data display on an I2C, either 2-line x 16-char, or 4-line x 20-char

 The circuit:
 Bosch BME-280 pressure, temperature, humidity sensor on I2C addr 0x76
 LCD at I2C 7-bit slave address 0x27
 
 Tipping bucket rain gauge contact closure to GND 
 on digital pin D17 (A3), normally HIGH
 rainfall = 0.02" per bucket tip
 
 Micro SD card attached to SPI bus as follows:
   MOSI - pin D11
   MISO - pin D12
   CLK - pin D13
   CS - pin D9 (SDsel)

BME280 breakout board connections to Arduino NANO:
  GND -> GND
  Vin -> +5V
  SDA -> A4
  SCL -> A5 
 adapted from original code of 9 Apr 2012 by Tom Igoe
 This example code is in the public domain.
 Compiles using Arduino IDE 1.8.5
 */
 
#include <EEPROM.h> // library to support on-chip EEPROM read/write
#include <SPI.h>  // library to support SPI bus communications
#include <SD.h>  // library to support SD card file operations
#include <Rtc_Pcf8563.h>  // library to supprt real time clock and calendar operations
#include <Wire.h>  //  library to support I2C bus communications
#include <LiquidCrystal_I2C.h> //  library to support I2C based LCD display operations
#include <SparkFunBME280.h>  //  library to support the BME-280 environmental sensor module

// create the real time PCF8563 clock object, 
// assume the I2C 8-bit addresses are the default 0xA2 for WRITE and 0xA3 for READ (that is, the 7-bit slave address is 0x51)
// These address are pre-set by #define statements within the Rtc_Pcf8563.h library file
Rtc_Pcf8563 rtc;

//create two sensor objects, assumes "A" I2C address is default 0x76, "B" address is 0x77
//these address will be invoked later, for example the  "mySensorA.setI2CAddress(0x77)" statement
BME280 mySensorA;
//BME280 mySensorB;  // don't need the second sensor right now

// now invoke the LCD_I2C library and create an LCD object
// assumes the default I2C bus 7-bit slave address is 0x27
// other displays may use other addresses. Use I2CScanner to discover them
// a later "lcd.begin(16, 2);"  statement will set the number of characters (e.g. 16) and rows (e.g. 2) 
LiquidCrystal_I2C lcd(0x27, 2, 1, 0, 4, 5, 6, 7, 3, POSITIVE);    //

/*specify the pin connections from the LCD to the PCF8574 interface chip
  user defines the address of their I2C enabled display (use I2CScanner to determine the address)
  arguments for use with PCF8574A I2C I/O Expander chip on LCD I2C interface module:
  LiquidCrystal_I2C lcd(I2C, EN, RW, RST, D4, D5, D6, D7, Backlite, Backlite sense)  I2C 7-bit slave address in hexadecimal, the least significant three bits are user configurable on the interface module
    I2C: 7-bit slave address in hexadecimal, the least significant three bits are user configurable on the interface module  LCD EN pin to PCF8574A I/O pin P2
    EN: LCD EN pin to PCF8574A I/O pin P2 
    RW: LCD RW pin to PCF8574A I/O pin P1
    RST: LCD RST pin to PCF8574A I/O pin P0
    D4: LCD D4 pin to PCF8574A I/O pin P4
    D5: LCD D5 pin to PCF8574A I/O pin P5
    D6: LCD D6 pin to PCF8574A I/O pin P6
    D7: LCD D7 pin to PCF8574A I/O pin P7
    Backlight: LCD backlite pin to PCF8574A I/O pin P3  (optional parameter)
    Backlight sense: set backlite pin POSITIVE ("1" = ON) or NEGATIVE ("0" = ON) logic, (optional parameter)
*/

//  ********* user selected parameters **********************
const int logInterval = 5;   // number of minutes between data records if no tip event occurs
const float rain_tip = 0.02; // inches of rainfall represented by a single bucket tip event
const float baro_fix = 0.3;  // fudge factor for barometer, in hPa, can be plus or minus
byte fudgeClock = 3;         // fudge factor in seconds per 24 hours for real time clock:   -9 <= factor <= +9
#define LCDsize 20,4         // LCD display size: number of characters, then number of rows

//  ********** hardware determined parameters ***************
const int SDsel = 9; //Arduino digital output pin 9 on v3.2 PCB is used as chip select for the micro-SD card interface
const int rain = 17; // Arduino digital input pin D17 (also known as A3) on v3.1 PC board is used for the rain gauge

//  globally defined variables
String monthNames = "JanFebMarAprMayJunJulAugSepOctNovDec";
String WxDataFile = "Nul_File.txt";
String dataString = ""; // initialize an empty string
byte nextMin;
byte thisMin;
int setMonth;
int setDay;
int setHour;
int setMinute;
int rain_count; // accumulator for rain:  <rain_tip> inches of rainfall per count
//float Temp_A;
//float Press_A;
//float RH_A;
//float Dew_pt;
unsigned long prev_rain;  // to keep track of last time the rain bucket tipped
unsigned long rain_dt;  // to calculate the elapsed time since last rain tip
float rain_rate;
bool R_S = true; // flag to indicate a power-on, or other reset has occurred

void setup() {
  //  analogReference(EXTERNAL);  // optionally use a precision voltage reference for the ADC reference
                                  // CAUTION - MUST be uncommented if EXTERNAL reference source is connected
                                  
  pinMode(rain,INPUT_PULLUP); // configure digital input pin to read tipping bucket rain gauge
  pinMode(10, OUTPUT); // D10 must be set to output and reserved for SPI driver, even if not used

  // Open serial communications and wait for port to open:
  // all serial data is transmitted via USB based COM: port and is echoed to the RX/TX pins of the NANO, and BlueTooth (if connected)
  Serial.begin(9600);
  Serial.setTimeout(30000); // wait up to 30 seconds for keyboard input
  while (!Serial) {
    ; // wait for serial port to connect. Needed for native USB port only
  }
  
  lcd.begin(LCDsize);  // start LCD driver and configure for selected display

  Wire.begin(); // start I2C bus driver
  
  mySensorA.setI2CAddress(0x76); //set BME-280(A) I2C address to 0x76, note: Sparkfun BME280 hardware defaults to 0x77
  if (mySensorA.beginI2C() == false) //Begin communication over I2C,
  {
    Serial.println("BME280A ERR");
    while(1); //Freeze if I2C comms to sensor A are not working
  }
  delay(100);
  
//  don't need second sensor right now
//  mySensorB.setI2CAddress(0x77); //set BME-280(B) I2C address to default 0x77, note: Sparkfun BME280 hardware defaults to 0x77
//  if (mySensorB.beginI2C() == false) //Begin communication over I2C,
//  {
//    Serial.println("BME280B ERR");
//    while(1); //Freeze if I2C comms to sensor B are not working
//  }
   Serial.println("BME280 OK");
  
  delay(1500);

  Serial.println("v3.2 24JAN2020"); // Start-up splash screen
  lcd.setCursor(0,0);    // (char pos, 0-15, line 0-1)
  lcd.print("v3.2 24JAN20"); //more sign on splash
  
  delay(1500);
  showTime(); // display time retrieved from battery backed up RTC for confirmation
  delay(3000);

  generateFilname(); //create a filename based on the current month
  lcd.clear();
  lcd.setCursor(0,0);
  lcd.print("Data file:");
  lcd.setCursor(0,1); // display data filename
  lcd.print(WxDataFile);
  delay(1000);

  // see if the card is present and can be initialized:
  if (!SD.begin(SDsel)) {
    Serial.println("*CARD ERR*");
    lcd.clear();
    lcd.setCursor(0,0);
    lcd.print("uSD ERR");  // local error notice
    // if card won't open, hang here, don't do anything more
    while (1);
  }
  Serial.print("uSD OK, "); // otherwise confirm card is OK 
  Serial.print("File: ");
  Serial.println(WxDataFile);

// to initialize the rainfall counter, perform the recommended Method 1, 
// or else: 
// uncomment one of the following program segments in Methods 2 and 3
// compile, upload and run the program one time. 
// Then re-comment the segment, compile, upload, and run again to restore normal operation

//    (Method 1)  ***recommended***
//    Compile, load, and run the program "EEPROM_write_and_read_tester.ino"
//    enter "0" for the EEPROM address, and enter a value = (accumulated rainfal in inches) / (rain_tip)
//    for the data value

//    (Method 2) uncomment the next 5 lines to allow for manual input of initial rain count, if not saved in EEPROM
//    Serial.println(); //send CR+LF
//    Serial.print("Starting rain count: "); // count = (accmulated rainfall in inches)/(rain_tip)
//    int init_rain = Serial.parseInt();  // get initial count from serial monitor input
//    Serial.println(init_rain); // echo the count for confirmation
//    rain_count = init_rain;

//    (Method 3) uncomment the following 2 lines
//    rain_count = 2400;        // force a starting count. the count = (accumulated rainfall in inches)/(0.02)
//    EEPROM.put(0,rain_count); // the                         n save the rain count to EEPROM for retrieval at next normal startup

      EEPROM.get(0,rain_count); // retrieve last saved rain count from EEPROM 
      prev_rain=millis();  //start timing the interval between rain bucket tips

//    set an alarm on the real time clock to activate each midnight
//    at each alarm event the clock will be given a slight correction to account for its inaccuracy
//    in main loop, the alarm flag     rtc.alarmActive()    will be polled to see if it has been set
      rtc.setAlarm(0, 0, 99, 99);  // minute, hour, day of month, day of week   --->  00:00 time, '99' means no alarm set for these  
      
}  //  end of setup sequence

//******************************************************************************************
// Monitor the digital input connected to the tipping bucket rain gauge. 
// Each time the bucket tips, write a data record to the uSD card and serial port. 
// If no tip occurs during the pre-determined time interval, write a record anyway.
void loop() {
  
  buildData(); //create a data log record with date/time stamp and sensor readings
  generateFilname(); // file names are of the form <mmmyyyy.txt>  where mmm is month abbreviated, yyyy is 4 digit year
  logData(); // open the data file, write a record, then close the file

// check if the alarm flag nas been set since the last time a record was written
  if (rtc.alarmActive()) 
  {
     rtc.getDateTime();                             // update the time registers
     while (abs(rtc.getSecond() - 30) < 20) {       // want to do this in the middle of a minute to avoid rollover problem     
        rtc.getDateTime();                          // update the time registers
        delay(1500);                                // wait a bit if necessary to get in the middle of a minute
     }
     rtc.getDateTime();                    // update the time registers, now that it's the middle of a minute
     byte newMin = rtc.getMinute();
     byte newHr = rtc.getHour();
     byte newSec = rtc.getSecond(); 
     newSec = newSec + fudgeClock;        // fudge can be plus or minus, up to 9 seconds
     rtc.setTime(newHr, newMin, newSec);  // bump the clock back or forward by up to 9 seconds
     rtc.resetAlarm();                    // reactivate the alarm for the same time as before
     Serial.println("Clock fudged after alarm flag set");   // for debugging         
  }
  
// wait for the correct moment
   while (rtc.getMinute()!= nextMin) {
      rtc.getDateTime();
      
      //showTime();                         // debugging
      //Serial.print(nextMin);              // debugging
      //Serial.print(" $ ");                // debugging
      //Serial.println(rtc.getMinute());    // debugging
      //delay(1000);                        // debugging
      
      // while waiting for time to to match up
      // see how long it's been since the last rain gauge tip event
      rain_dt = millis()-prev_rain;  // determine elapsed time since last tip (0.02") of rain bucket
      if (rain_dt > 3e6) // if more than an hour since last tip event
      {
        rain_rate = 0; // force the rate parameter to zero
      }
      
      // poll the tipping bucket rain gauge for a pulse:  H -> L -> H   
      if (digitalRead(rain)==LOW) {
         rain_count=rain_count + 1; // increment the rain counter
         EEPROM.put(0,rain_count); //save rain count to EEPROM for retrieval at next startup
         rain_dt = millis()-prev_rain;  // determine elapsed time since last tip (0.02") of rain bucket
         rain_rate = ((rain_tip) * (3.6E6)) / (rain_dt);  // calculate rainfall rate in inches per hour
         if (rain_rate < 0.02) rain_rate = 0; // force a zero display if extremely low rate
         prev_rain=millis(); // reset the rain timer to the current time

         buildData(); //create a data log record with time/date stamp and sensor readings
         generateFilname(); // file names are of the form <mmmyyyy.txt>  where mmm is month abbreviated, yyyy is 4 digit year
         logData(); // write a data record every time the rain bucket tips

         while (digitalRead(rain) == LOW) { // wait for the pin to return HIGH before resuming polling
         }      
      }
   }    
   // Serial.println("Looping");  //debugging
}

//**********************************************************************************
// routine to dynamically create a data file name based on the current month
// of the form <mmmyyyy.txt> where mmm is the 3-letter month abbreviation
// and yyyy is the year as a four digit integer
void generateFilname() {  
   WxDataFile= monthNames.substring(3*(rtc.getMonth()-1),3*(rtc.getMonth()-1)+3) + String(rtc.getYear()+2000 ) + ".txt";
   // Serial.println(WxDataFile); // for debugging
}

//***********************************************************************************
// routine to build the data string to be logged to uSD and written to serial port
void buildData() {
   // make a string for assembling the data to log:
  dataString = ""; // initialize a string

  // assemble a date-time stamp
  dataString += rtc.formatDate();
  dataString += " ";
  //byte thisSec=rtc.getSecond(); //  set up to wait a preset interval in seconds ..... not used any more
  //byte nextSec=thisSec+30;
  //if (nextSec>59) nextSec=nextSec-60;

  thisMin=rtc.getMinute(); //  set up to wait a preset interval in minutes
  nextMin=thisMin+logInterval;
  if (nextMin>59) nextMin=nextMin-60;
  //Serial.print(thisMin, DEC);   //debug
  //Serial.print(" * ");            //debug
  //Serial.println(nextMin, DEC); //debug
 
  //Serial.print(thisSec, DEC);  //debug
  //Serial.print(" ");  //debug
  //Serial.println(nextSec, DEC); //debug
  
  dataString += rtc.formatTime();
  dataString += " ";
  lcd.begin(16, 2);  // configure for 16 X 2 display
  lcd.clear();
  lcd.setCursor(0,0);

  // read temperature and append to string and to LCD
  // should read trmperature first to update compensators for other data
  float X = mySensorA.readTempF(); //read thermometer
  dataString += String(X,1);
  dataString += "F, ";
    lcd.print(String(X,1)+"F ");
  
  // read pressure sensor and append to the string and to LCD
  X = mySensorA.readFloatPressure()/100 + baro_fix; // read barometer, correct for HASL
  dataString += String(X,1);
  dataString += " hPa, ";
     lcd.print(String(X,1)+"hPa");
  
  // read relative humidity and append to string and to LCD
  X = mySensorA.readFloatHumidity(); //read humidity sensor
  dataString += String(X,1);
  dataString += "%, ";
    lcd.setCursor(0,1);
    lcd.print(String(X,0)+"% ");

  // read dew point and append to string
  X= mySensorA.dewPointF(); //calculate dew point in deg F
  dataString += String(X,1);
  dataString += "F, ";

  dataString += String(rain_count * 0.02);
  dataString += " inch";
  lcd.print(String(rain_count * 0.02)+'"');

  dataString += ", "; // append rainfall rate to data record and to LCD
  dataString += String(rain_rate,1);
  dataString += " in/hr";
  lcd.setCursor(11,1);
  lcd.print(String(rain_rate,1)+"/h"); // display the most recent rainfall rate
  if (R_S) {                             // check if a reset has occurred
    dataString += " *RESET*";            // write a reset indicator to the log
    R_S = false;                         // and clear the flag
  }
}

//******************************************************************************
// routine to write a data record to the uSD card and also to the serial port
void logData() {
    File dataFile = SD.open(WxDataFile, FILE_WRITE);
//  if the file is available, write to it, then close it
    if (dataFile) {
       dataFile.println(dataString);
       dataFile.close();
//     echo print to the serial port too: 
       Serial.println(dataString);
    }
//  if the file isn't open, pop up an error:
    else {
       Serial.println("*FILE ERR*");
    }
}

//*********************************************************
// routine to display time and date on LCD
void showTime() {
  lcd.clear();
  lcd.setCursor(0,0);
  lcd.print(rtc.formatDate());  //show recovered RTC date
  lcd.setCursor(0,1);
  lcd.print(rtc.formatTime());  // show recovered RTC time
}

