;**************************************************************************
; DipMeter.ASM

PROC_TYPE	   equ	  23	  ; must be 23(18F23K22) or 24(18F24K22)
							  ;	 need to modify project properties!!!!

CPU_MHZ		   equ   .4		  ; must be either .16 or .4
							  ;		only if OSC_INT_EXT = INTIO67

#define OSC_INT_EXT ECMPIO6	  ; INTIO67 or ECMPIO6
							  ;		INTIO67 = internal R/C oscillator
							  ;		   frequency determined by CPU_MHZ value
							  ;		   use RC6,7 as I/O
							  ;		ECMPIO6 = external crystal (500KHz-16MHz)
							  ;		   use RC6 as I/O

   if 0
	  This program requires a PIC18F24K22 or PIC18F23K22.
	  hardware usage notes
		 Port A: segment driver for the 3 digit LED display
		 Port B: digit driver bits for the display and the gate input for Timer 1
		 Port C: switches, output of EUSART 1, input frequency
		 Interrupt priority is enabled
		 Timer 1: count the input signal and is programmed to use its gate feature
			high priority interrupt on overflow enabled
			gate not available on the 4520
			The gate is enabled as LOW true
		 Timer 2: generate a low priority interrupt about every 0.4msec to update the display
			The divisor for Timer 2 was derived experimentally by increasing the
			frequency until there was no noticible flicker.
		 EUSART 1: synchronous mode, gate for timer 1
			uses high priority interrupt for overflow
			The baud rate divisor is 20,000 which yields a bit time of 5msec
			IMPORTANT - data for the 1st byte activates BEFORE the first clock edge
			   so, a 0 byte should be transmitted first.
			   Do not use bit 7 of the last byte because the idle state is 1

	  some software information
		 FSR2: used ONLY to point to the digit being displayed

	  The Watch Dog Timer is enabled for a count of 64 which is approx. 256ms.
	  It is cleared in the function Gate.1Byte.  One byte takes 40ms.  There
	  should be no timeouts since the unit is always taking a frequency measurement.

	  LED Brightness is controlled by the potentiometer connected to A/D input 11.
	  The low priority interrupt has a 10ms period.  The ISR is used to point to
	  the digits.  It also updates an aribtrary period counter which is a single
	  byte value so it has a count range of 256.  Only the upper eight bits of
	  the A/D reading is used.

	  All measurements start with the divide-by-10 circuit and a 10ms gate time
	  in order to determine if the frequency is within range.
	  A second reading is taken based on the following:
		 Dip Meter: use 100ms gate still
		 External frequency input: gate time and divide-by-10 circuit usage
		 is determined by the initial frequency reading.

	  Port assignments
PortA
   0  output:  segment A
   1  output:  segment B
   2  output:  segment C
   3  output:  segment D
   4  output:  segment E
   5  output:  segment F
   6  output:  segment G
   7  input:   external clock
PortB
   0  output:  enable digit D3 - LSdigit
   1  output:  enable digit D2
   2  output:  enable digit D1 - MSdigit
   3  output:  unused
   4  input:   A/D: Display Brightness
   5  input:   gate input for Timer 1 - from EUSART Tx
   6  input:   PGC
   7  input:   PGD
PortC
   0  input:   Timer 1 input - frequency being measured
   1  output:  control for Divide-by-10
   2  output:  segment decimal point
   3  input:   LSDigit switch
   4  output:  not used
   5  input:   /DIP_SEL switch: 0=DipMeter, 1=External Input
   6  output:  EUSART 1 clock - not used
   7  output:  EUSART 1 Tx data = gate time for Timer 1
   endif

   if CPU_MHZ == .16
OSCON_VALUE	   equ	 b'01111000'
GATE_BAUD	   equ	 .19999		 ; BRG divides by n+1 = 20000 = 5ms/bit
T2CON_VALUE	   equ	 b'01001110' ; postscaler = /10, enable timer, prescaler = /16
   messg "CPU speed = 16MHz"
   else
	  if CPU_MHZ == .4
OSCON_VALUE	   equ	 b'01011000'
GATE_BAUD	   equ	 .4999		 ; BRG divides by n+1 = 5000 = 5ms/bit
T2CON_VALUE	   equ	 b'01001101' ; postscaler = /10, enable timer, prescaler = /4
      messg "CPU speed = 4MHz"
	  else
	  error "Invalid value for  CPU_MHZ"
	  endif
   endif

delay macro delay_time
   movlw	delay_time
   call		Delay.Wmsec
   endm

;******************************************************************************
;************************* CPU_TYPE == 2422 ***********************************
;******************************************************************************

   if PROC_TYPE == 23
	  #include p18f23K22.inc
      LIST P=18f23k22, R=DEC
	  messg "Processor = 18F_23_K22"
   else
      if PROC_TYPE == 24
		 #include p18f24K22.inc
	     LIST P=18f24k22, R=DEC
	  messg "Processor = 18F_24_K22"
	  else
		 error "Invalid value for  CPU_MHZ"
	  endif
   endif


; CONFIG1H
  CONFIG  FOSC = OSC_INT_EXT	; Oscillator Selection bits
  CONFIG  PLLCFG = OFF          ; 4X PLL Enable (Oscillator used directly)
  CONFIG  PRICLKEN = ON         ; Primary clock enable bit (Primary clock enabled)
  CONFIG  FCMEN = OFF           ; Fail-Safe Clock Monitor Enable bit (Fail-Safe Clock Monitor disabled)
  CONFIG  IESO = OFF            ; Internal/External Oscillator Switchover bit (Oscillator Switchover mode disabled)

; CONFIG2L
  CONFIG  PWRTEN = OFF          ; Power-up Timer Enable bit (Power up timer disabled)
  CONFIG  BOREN = SBORDIS       ; Brown-out Reset Enable bits (Brown-out Reset enabled in hardware only (SBOREN is disabled))
  CONFIG  BORV = 285            ; Brown Out Reset Voltage bits (VBOR set to 2.85 V nominal)

; CONFIG2H
  CONFIG  WDTEN = ON			; Watchdog Timer Enable bits (WDT is always enabled. SWDTEN bit has no effect)
  CONFIG  WDTPS = 64            ; Watchdog Timer Postscale Select bits (1:32768)

; CONFIG3H
  CONFIG  CCP2MX = PORTB3       ; CCP2 MUX bit (CCP2 input/output is multiplexed with RC1)
  CONFIG  PBADEN = OFF          ; PORTB A/D Enable bit (PORTB<5:0> pins are configured as digital I/O on Reset)
  CONFIG  CCP3MX = PORTB5       ; P3A/CCP3 Mux bit (P3A/CCP3 input/output is multiplexed with RB5)
  CONFIG  HFOFST = ON           ; HFINTOSC Fast Start-up (HFINTOSC output and ready status are not delayed by the oscillator stable status)
  CONFIG  T3CMX = PORTC0        ; Timer3 Clock input mux bit (T3CKI is on RC0)
  CONFIG  P2BMX = PORTB5        ; ECCP2 B output mux bit (P2B is on RB5)
  CONFIG  MCLRE = EXTMCLR       ; MCLR Pin Enable bit (MCLR pin enabled, RE3 input pin disabled)

; CONFIG4L
  CONFIG  STVREN = ON           ; Stack Full/Underflow Reset Enable bit (Stack full/underflow will cause Reset)
  CONFIG  LVP = OFF             ; Single-Supply ICSP Enable bit (Single-Supply ICSP disabled)
  CONFIG  XINST = OFF           ; Extended Instruction Set Enable bit (Instruction set extension and Indexed Addressing mode disabled (Legacy mode))

; CONFIG5L
  CONFIG  CP0 = OFF             ; Code Protection Block 0 (Block 0 (000800-001FFFh) not code-protected)
  CONFIG  CP1 = OFF             ; Code Protection Block 1 (Block 1 (002000-003FFFh) not code-protected)

; CONFIG5H
  CONFIG  CPB = OFF             ; Boot Block Code Protection bit (Boot block (000000-0007FFh) not code-protected)
  CONFIG  CPD = OFF             ; Data EEPROM Code Protection bit (Data EEPROM not code-protected)

; CONFIG6L
  CONFIG  WRT0 = OFF            ; Write Protection Block 0 (Block 0 (000800-001FFFh) not write-protected)
  CONFIG  WRT1 = OFF            ; Write Protection Block 1 (Block 1 (002000-003FFFh) not write-protected)

; CONFIG6H
  CONFIG  WRTC = OFF            ; Configuration Register Write Protection bit (Configuration registers (300000-3000FFh) not write-protected)
  CONFIG  WRTB = OFF            ; Boot Block Write Protection bit (Boot Block (000000-0007FFh) not write-protected)
  CONFIG  WRTD = OFF            ; Data EEPROM Write Protection bit (Data EEPROM not write-protected)

; CONFIG7L
  CONFIG  EBTR0 = OFF           ; Table Read Protection Block 0 (Block 0 (000800-001FFFh) not protected from table reads executed in other blocks)
  CONFIG  EBTR1 = OFF           ; Table Read Protection Block 1 (Block 1 (002000-003FFFh) not protected from table reads executed in other blocks)

; CONFIG7H
  CONFIG  EBTRB = OFF           ; Boot Block Table Read Protection bit (Boot Block (000000-0007FFh) not protected from table reads executed in other blocks)
; *************************************************************************************************

SEGMENT_PORT   equ	 LATA
DCML_PT_FLAG   equ	 7
PORTA_TRIS	   equ   0x80
PORTA_INIT	   equ	 0

DIGITS_PORT	   equ   LATB
DIGIT_0		   equ   0
DIGIT_1		   equ   1
DIGIT_2		   equ   2
BRIGHTNESS	   equ	 4		  ; A/D input
mDIGITS		   equ	 7
PORTB_TRIS	   equ   b'11110000'
PORTB_INIT	   equ	 0

SWITCH_PORT	   equ   PORTC
DIV_BY_10	   equ   1
#define	 DECIMAL_POINT	LATC,2
SW_DISP_LSDs   equ   3		  ; switches are low true
SW_DIP_SEL	   equ	 5
PORTC_TRIS	   equ   b'11101001'
mSWITCHES	   equ	 b'00101000'
PORTC_INIT	   equ	 (1<<DIV_BY_10)

; bit mask values
mBIT0	 equ   0x01
mBIT1	 equ   0x02
mBIT2	 equ   0x04
mBIT3	 equ   0x08
mBIT4	 equ   0x10
mBIT5	 equ   0x20
mBIT6	 equ   0x40
mBIT7	 equ   0x80

ScratchPadRam   udata_acs   0x10
; the Access RAM ends at 0x5F ==> .96 bytes
;  for ISR
W_save		   res	 1
STATUS_save	   res	 1

;  display LED
BRIGHTNESS_PERIOD equ 45   ; Timer 2 interrupts for LED period
mLSDigit	   equ	 0x01  ; mask for bit to drive LSDigit
Frequency	   res	 3	   ; measured frequency goes here, LSbyte first
mCurrent_Digit res	 1	   ; 001, 010, 100, repeat
Digits		   res	 6	   ; LSDigit first
Bright_Period  res	 1	   ; Timer 2 interrupt count - used for LED brightness
ADvalue        res   1     ; upper 4 bits of A/D reading
BrightCount    res   1     ; working copy of A/D value for LED brightness

DcmlPt_Digit   res	 1	   ; address within Digits for the decimal point

GateTime	   res	 1	   ; token value for selected gate time
GATE_10MS	   equ	 1
GATE_100MS	   equ	 2
GATE_1S	 	   equ	 4

;  Delay functions
DelayCount_1   res	 1
DelayCount_2   res	 1
DelayCount_3   res	 1
Delay_100val   res	 1

;  FXD2416U
;Dividend	   res	 3	   ; use Frequency as the Dividend
Divisor		   res	 2
Remainder	   res	 2
DivLoopCount   res	 1

;  misc
ByteCount	   res	 1	   ; used by Gate time
LoopCount      res   1     ; general purpose counter
Switches	   res	 1	   ; switch values - high true - see SW_...
Flags		   res	 1	   ; misc flags

#define	 FREQ2HI  Flags, 7 ; indicator that input frequency is too high

;T1GCON_values  res	 2


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						BSR = 0
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	udata  0x060  ; remaining bytes of the 0th RAM block

; NOTE: these values must always be accessed via either BSR=0 or an FSR
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						Interrupt Vectors
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  org   0
  goto  Main

  org   0x08
  goto  HiPriInt

  org   0x18
  goto  LoPriInt

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						Constants in low memory
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PackedData   code_pack	0x20	  ; this data in flash - for constants

;  this data MUST be within the 1st 256 bytes of Flash
;					0     1     2     3     4     5     6     7     8     9
TABLE_7_SEG	db	  0x3F, 0x06, 0x5B, 0x4F, 0x66, 0x6D, 0x7C, 0x07, 0x7F, 0x67, 0
BRG_VALUE	db	  LOW GATE_BAUD, HIGH GATE_BAUD
;					 I	   H	 2
TOO_HIGH	db	  0x30, 0x76, 0x5B

   code
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	  interrupt routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

LoPriInt ; Lo pri ints cannot use fast retfie if interrupt priority is enabled
; Timer2 is the only valid Low Priority interrupt
; update the display digits
; Most of the interrupts either continue driving the "current" digit (~25 cycles) or no digits (~22 cycles)

   movff	STATUS, STATUS_save
   movwf	W_save

   btfss	PIR1, TMR2IF	  ; Timer 2 interrupt?
   goto		LoPriInt.Exit	  ; exit if no
   bcf		PIR1, TMR2IF	  ; clear the interrupt flag

; test for Brightness = 0
   movf		ADvalue, w		  ; get brigntness
   bnz		LoPriInt.10		  ; br if not 0
   clrf		SEGMENT_PORT	  ; turn off all segments
   bra		LoPriInt.Exit

LoPriInt.10
   dcfsnz	Bright_Period	  ; update brightness period value
   bra		LoPriInt.100	  ; start cycle over
; continue current cycle
   movf		mCurrent_Digit, w ; still lighting a digit?
   bz		LoPriInt.Exit	  ; br if no
; 16(?) cycles to here

; current digit still lit
   decfsz	BrightCount		  ; update counter for current digit
   bra		LoPriInt.Exit	  ; exit if digit to remain on
; +4 = branch to LoPriInt.Exit

; start next digit (w = mask for current digit)
   movff	PREINC2, SEGMENT_PORT ; get new digit segments value and set output port
   rlncf	WREG			  ; mask for new digit
   andlw	mDIGITS			  ; leave only digit bits
LoPriInt.20
   nop						  ; for debugging
   nop						  ; for debugging
   bcf		DECIMAL_POINT	  ; assume no decimal point for this digit
   btfsc	INDF2, DCML_PT_FLAG ; decimal point for this digit? - skip if no
   bsf		DECIMAL_POINT
   movwf	mCurrent_Digit	  ; save digit mask for new current digit
   movf	 	DIGITS_PORT, w	  ; get current port value
   andlw	~mDIGITS		  ; clear digit bits
   iorwf	mCurrent_Digit, w ; insert current digit bit
   movwf	DIGITS_PORT		  ; turn on new current digit
   movff	ADvalue, BrightCount ; on-time for new current digit

LoPriInt.Exit
   movff	STATUS_save, STATUS
   movf		W_save, w
   retfie
; +4

; start display over again with 1st digit to display
LoPriInt.100
   movlw	BRIGHTNESS_PERIOD ; get brightness period
   movwf	Bright_Period	  ; save it for ISR
   movlw	Digits-1		  ; point to LSDigit
   btfss	Switches, SW_DISP_LSDs ; skip if displaying the LSDigits
   addlw	3				  ; point to digit 3
   movwf	FSR2L			  ; save for ISR processing
   movff	PREINC2, SEGMENT_PORT ; get current digit segments value and set output port
   movlw	mLSDigit		  ; mask for digits 0 and 3 position
   bra		LoPriInt.20
; LoPriInt


HiPriInt
; Timer1 is the only valid High Priority interrupt
   btfss	PIR1, TMR1IF		 ; skip if Timer 1 interrupt
   goto		HiPriInt.Exit
   bcf		PIR1, TMR1IF		 ; clear the interrupt
   incf		Frequency+2			 ; update MSbyte of frequency
HiPriInt.Exit
   retfie FAST
; HiPriInt

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						Main Program
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Main
; set some system registers
   movlw	OSCON_VALUE		  ; 16 or 4MHz, fosc<3:0>
   movwf	OSCCON			  ; set up oscillator control to 4MHz
   clrf		RCON			  ; clear some interrupt flags, IPEN=0 <7>

; clear the first RAM bank
   lfsr		0, 0			  ; start with address 0
   movlw	0				  ; clear 256 bytes
Main.10
   clrf		POSTINC0		  ; clear a byte - using FSR precludes using BSR
   decf		WREG			  ; update bytecount
   bnz		Main.10			  ; br if not done
; some initialization
   movlb	0				  ; init BSR to page 0
   call		Init.Ports		  ; initialize the I/O ports

; initialize values for display
   movlw	mBIT0			  ; mask for LSDigit - digit 0
   movwf	mCurrent_Digit
   lfsr		2, Digits+2		  ; point to digit 2 for Dip Meter (Freq/10 and 10msec gate)

; main program loop
Main.100
   call     ReadA2D           ; read the A/D to get brightness level
   nop						  ; for debugging

   movlw	.100
   call		Delay.Wmsec

   call		Meas.Freq
   nop						  ; for debugging
   call		Check.Switches
   nop
   call		Display.Update
   nop						  ; for debugging
   goto		Main.100

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						End Main Program
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


; Read the switches and set up the display parameters
; inputs: Switches:DIV_BY_10, GateTime
; outputs: Switches, DcmlPt_Digit
; NOTES: If divide-by-10 is enabled, the only valid gate time is 100ms
;			and the display is: xx.x xxx
;		 Decimal point is to right of digit
Check.Switches
; read the switches
   movf	 	SWITCH_PORT, w	  ; read the switches port
   xorlw	mSWITCHES		  ; invert because hardware is low true
   movwf	Switches		  ; save
; initial values are for: displaying MSDigits and 100ms gate
   movlw	Digits+4
   movwf	DcmlPt_Digit
   btfsc	Switches, SW_DISP_LSDs ; skip if displaying the MSDigits
   return					  ; no further processing for LSDigits
   btfsc	Switches, DIV_BY_10
   return					  ; no further processing if divide-by-10 enabled

; here if Divide-by-10 NOT enabled
; if gate time = 1s, display is xxx xxx.
; if gate time = 100ms, display is x.xx xxx
   movlw	Digits	 		  ; assume 1s gate
   movwf	DcmlPt_Digit
   movlw	GATE_100MS 		  ; test for 100ms gate
   cpfseq	GateTime		  ; skip if it is 100ms
   return
   movlw	Digits+5
   movwf	DcmlPt_Digit
   return
; Check.Switches


; Dip Meter always uses the divide-by-10 and 100ms gate
Meas.Freq
   bsf		SWITCH_PORT, DIV_BY_10 ; enable the divide-by-10 circuit
   call		Gate.10ms		  ; do initial read with a 10msec gate
   tstfsz	Frequency+2		  ; skip next if frequency is OK
   goto		Meas.Freq.Too.High ; show error

   btfsc	Switches, SW_DIP_SEL ; skip if not Dip Meter
   goto		Gate.100ms		  ; read again with more resolution and return

; IMPORTANT: frequency comparisons must take into account that
;  the initial frequency measurement uses the divide-by-10 circuit
;  and a 10ms gate time.
;	  Timer count = 9728 for Frequency = 9.728Mhz
; here if not Dip Meter
   movlw	HIGH .9728		  ; high byte for freq = 9.728MHz
   cpfslt	Frequency+1		  ; skip if frequency < 9.728MHz
   goto		Gate.100ms		  ; read the frequency and return to caller
							  ;		keep divide-by-10
; here if frequency <= 9.728MHz
Meas.Freq.10
   bcf		SWITCH_PORT, DIV_BY_10 ; disable the divide-by-10 circuit
   movlw	HIGH .950		  ; high byte for freq = 950kHz
   cpfsgt	Frequency+1		  ; skip if frequency > 950kHz
   bra		Meas.Freq.20
; here if frequency > 950kHz
   goto		Gate.100ms		  ; read the frequency and return to caller

; here if frequency <= 950kHz
Meas.Freq.20
   cpfseq	Frequency+1		  ; skip if frequency NOT < 950kHz
   goto		Gate.1s			  ; frequency must be < 950kHz
; here if need to check the low byte
   movlw	LOW .950
   cpfsgt	Frequency		  ; skip if frequency > 950kHz
   goto		Gate.1s	 		  ; read the frequency and return to caller
   goto		Gate.100ms		  ; read the frequency and return to caller

Meas.Freq.Too.High
   bsf		FREQ2HI			  ; show frequency too high
   return
; Meas.Freq


; For these gate times to be accurate, the bit time MUST be 5msec.
; Bit 0 of the initial byte is not used to start the gate because the processor
; will set the bit 1/2 clock early
; Note that bit 0 is transmitted first!
Gate.10ms
; execute a gate time of 10msec = 2 bits
   nop						  ; debugging
   movlw	GATE_10MS		  ; show 10ms gate time
   movwf	GateTime
   call		Gate.Init
   movlw	~0x06	 		  ; two bits = 10msec
   goto		Gate.Exit
; Gate.10msec

Gate.100ms
; execute a gate time of 100msec = 20 bits
   movlw	GATE_100MS		  ; show 100ms gate time
   movwf	GateTime
   call		Gate.Init
   movlw	~0xFE  			  ; start pulse for 100msec gate: 7 bits = 35msec
   call		Gate.1Byte		  ; send the byte
   movlw	0	  			  ; insert 8 more
   call		Gate.1Byte		  ; send the byte
   movlw	~0x1F			  ; last 5
   goto		Gate.Exit
; Gate.100msec

Gate.1s
; execute a gate time of 1sec = 200 bits
   movlw	GATE_1S	 		  ; show 1s gate time
   movwf	GateTime
   call		Gate.Init
   movlw	~0xFE  			  ; start pulse for 1sec gate: 7 bits = 35msec
   call		Gate.1Byte		  ; send the byte
   movlw	.24	  			  ; 24 bytes = 192 bits = 960msec
   movwf	ByteCount		  ; copy for loop
   movlw	0
Gate.1s.10
   call		Gate.1Byte		  ; send the byte
   decfsz	ByteCount		  ; update byte count
   goto		Gate.1s.10		  ; goto if not done
   movlw	~1				  ; last byte value = last 5msec
;  goto		Gate.Exit
; Gate.1sec

Gate.Exit
; enter with W = byte to transmit
; send the last byte, wait for it to complete, disable the transmitter
   nop
   call		Gate.1Byte		  ; send the last byte

Gate.Exit.10
   btfsc	PORTB, 5
   bra		Gate.Exit.10
;   movff	T1GCON, T1GCON_values
Gate.Exit.15
   btfss	PORTB, 5
   bra		Gate.Exit.15
   nop						  ; for debugging
   bcf		TXSTA1, TXEN	  ; disable the transmitter
   movff	TMR1L, Frequency  ; get low byte
   movff	TMR1H, Frequency+1 ; and high byte
   return

Gate.Init
; send the leading dummy byte
   bsf		TXSTA1, TXEN	  ; enable the transmitter
   clrf		Frequency+2		  ; init the MSByte of the frequency - updated via ISR
   clrf		TMR1H			  ; clear the timer
   clrf		TMR1L			  ;		counter registers
   movlw	0xFF			  ; leading byte - see note in file header info
   call		Gate.1Byte		  ; send the byte
   return
; Gate.Init

Gate.1Byte
; enter with W = byte to transmit
   clrwdt					  ; clear the watch dog timer
   btfss	TXSTA1, TRMT	  ; skip next if TSR is empty
   goto		Gate.1Byte		  ; wait for TSR empty
   movwf	TXREG1			  ; send the byte
   return
; Gate.1Byte


Display.Update
; convert Frequency to BCD then 7 segment pattern using FXD2416U
; each call to FXD2416U yields Frequency = Frequency/10 and Remainder
; INDF0 points to the digit values
   btfsc	FREQ2HI			  ; skip next if frequency is OK
   goto		Display.2High
   lfsr		0, Digits		  ; point to digit values
   movlw	6				  ; nbr of digits
   movwf	ByteCount		  ; save for loop below
   clrf		TBLPTRU			  ; clear upper bytes
   clrf		TBLPTRH			  ;		of the ROM table pointer
Display.Update.20
   call		FXD2416U		  ; remainder = value of current LSDigit
   movff	Remainder, INDF0  ; store the remainder
; convert BCD value to 7 segment bit pattern
   movlw	TABLE_7_SEG		  ; get address of table
   addwf	INDF0, w		  ; add the BCD value
   movwf	TBLPTRL
   TBLRD*					  ; get the pattern for the digit
   movff	TABLAT, POSTINC0  ; save the pattern and point to next digit
; update counter, test for done
   decfsz	ByteCount		  ; update count and skip next when done
   goto		Display.Update.20
; insert decimal point
   movff	DcmlPt_Digit, FSR0L ; get address of decimal point digit
   bsf		INDF0, DCML_PT_FLAG ; enable the decimal point
Display.Update.30
   return

Display.2High
   lfsr		0, Digits		  ; point to LSDigits
   movlw	TOO_HIGH		  ; addr of 2HI text
   movwf	TBLPTRL
   movlw	3				  ; loop counter
Display.2High.10
   TBLRD*+					  ; get the pattern
   movff	TABLAT, POSTINC0  ; save to LSDigit
   movff	TABLAT, POSTINC1  ; save to MSDigit
   decfsz	WREG
   bra		Display.2High.10
   return
; Display.Update


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	  Initialize the I/O ports
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Init.Ports

; Port A
   movlw	PORTA_INIT
   movwf	PORTA
   movlw	PORTA_TRIS
   movwf	TRISA

; Port B
   movlw	PORTB_INIT
   movwf	PORTB
   movlw	PORTB_TRIS
   movwf	TRISB

; Port C
   movlw	PORTC_INIT
   movwf	PORTC
   movlw	PORTC_TRIS
   movwf 	TRISC

   movlb	0x0F			  ; the ANSEL registers are NOT in the Access page
   clrf		ANSELA, BANKED	  ; do not use PORTA bits for the A/D
   movlw    mBIT4             ; use RB4
   movwf	ANSELB, BANKED	  ;     = AN11 = LED brightness
   clrf		ANSELC, BANKED	  ; do not use PORTC bits for the A/D
   movlb	0				  ; back to the Access page

; A/D
   clrf     ADCON1            ; use AVdd and AVss for A/D references
   movlw    (0<<ADFM)|(0<<ACQT0)|(5<<ADCS0) ; conv clk = Fosc/16 = 1us per bit,
   movwf    ADCON2            ;      left justified data, ACQT = 0
   movlw    (.11<<CHS0)|(1<<ADON) ; AN11 and turn on A/D
   movwf    ADCON0
   bcf      PIE1, ADIE        ; do not use A/D interrupt

; EUSART 1
   movlw	(1<<CSRC)|(1<<SYNC) ; master, synchronous
   movwf	TXSTA1
   movlw	1<<SPEN			  ; serial port enabled - sets I/O pins
   movwf	RCSTA1
   movlw	1<<BRG16		  ; use 16 bits of BRG, clock idle low
   movwf	BAUDCON1
   movlw	LOW GATE_BAUD	  ; low byte for 10msec gate time
   movwf	SPBRG1
   movlw	HIGH GATE_BAUD	  ; high byte
   movwf	SPBRGH1

; Timer 1
   movlw	(2<<TMR1CS0)|(0<<T1CKPS0)|(0<<T1SOSCEN)|(1<<T1SYNC)|(0<<T1RD16)|(1<<TMR1ON)
							  ; source=T1CKI, prescale=1:1, 2nd osc disabled,
							  ;	do not sync ext clk, rd/wr using 2 bytes, enable timer
   movwf	T1CON
   movlw	(1<<TMR1GE)|(0<<T1GPOL)|(0<<T1GTM)|(0<<T1GSPM)|(0<<T1GGO)|(0<<T1GSS0)
							  ; enable gate function, gate is active low, disable toggle mode,
							  ; single pulse disabled), do not GO yet, gate source = pin
   movwf	T1GCON
   bsf		IPR1, TMR1IP	  ; set TMR1 overflow to high priority interrupt
   bsf	 	PIE1, TMR1IE	  ; timer 1 interrupt enable

; Timer 2
   movlw	T2CON_VALUE		  ; post = /10, enable timer, pre = /16 or 4
   movwf	T2CON
   movlw	.10	  			  ; Fcpu/4 /16 /10 /.10 = 2.5KHz => 0.4msec
   movwf	PR2
   bcf		IPR1, TMR2IP	  ; set TMR2 match to low priority interrupt
   bsf	 	PIE1, TMR2IE	  ; timer 2 interrupt enable

; Power saving
   movlw	b'10111100'		  ; enable UART1, Timer2 and Timer1
   movwf	PMD0
   movlw	0xFF			  ; disable MSSP 1&2 and all CCPs
   movwf	PMD1
   movlw	0xFF			  ; disable CTMU, CMPx, ADC
   movwf	PMD2

   clrf	 	PIE3			  ;	disable other peripheral interrupts
   clrf	 	PIE4			  ;	disable other peripheral interrupts
   clrf	 	PIE5			  ;	disable other peripheral interrupts

; interrupts
   movlw	1<<IPEN			  ; enable interrupt
   movwf	RCON			  ;		priority levels
   movlw	(1<<GIE)|(1<<PEIE) ; enable interrupts
   movwf	INTCON
   clrf	 	INTCON2			  ; no external interrupts
   clrf	 	INTCON3			  ; disable other ext interrupt inputs
   clrf	 	PIE2			  ;	disable other peripheral interrupts

   return
; Init.Ports



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						Math functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;		 divide 24 bit integer by 16 bit integer
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; the following divide function was copied from:
;		 http://techref.massmind.org/techref/microchip/math/div/24by16.htm
; customized to divide only by 10
; operation: Frequency = Frequency/Divisor,  Remainder
;Inputs:
;   Dividend:  Frequency[0] = LSByte
;Temporary:
;   DivLoopCount
;Output:
;   Quotient:  Frequency
;   Remainder: Remainder[0] = LSByte

; estimated maximum execution is 563 instructions = 140usec @16MHz clock

FXD2416U
   movlw	.10				  ; initialize the Divisor to 10
   movwf	Divisor
   clrf		Divisor+1
   CLRF		Remainder+1
   CLRF		Remainder
   MOVLW	24
   MOVWF	DivLoopCount
LOOPU2416
   RLcF		Frequency, F	  ;shift dividend left to move next bit to remainder
   RLcF		Frequency+1, F	  ;and shift in next bit of result
   RLcF		Frequency+2, F	  ;

   RLcF		Remainder, F	  ;shift carry (next dividend bit) into remainder
   RLcF		Remainder+1, F

   RLcF		DivLoopCount, F	  ;save carry in counter, since remainder
							  ;can be 17 bit long in some cases (e.g.
							  ;0x800000/0xFFFF)

   MOVF		Divisor, W		  ;substract divisor from 16-bit remainder
   SUBWF	Remainder, F	  ;
   MOVF		Divisor+1, W	  ;
   BTFSS	STATUS, C		  ;
   INCFSZ	Divisor+1, W	  ;
   SUBWF	Remainder+1, F	  ;

;here we also need to take into account the 17th bit of remainder, which
;is in DivLoopCount.0. If we don't have a borrow after subtracting from lower
;16 bits of remainder, then there is no borrow regardless of 17th bit
;value. But, if we have the borrow, then that will depend on 17th bit
;value. If it is 1, then no final borrow will occur. If it is 0, borrow
;will occur.

   bnc	    FXD2416U.1
   BSF		DivLoopCount, 0	  ;then no no borrow in result. Overwrite
							  ;DivLoopCount.0 with 1 to indicate no
							  ;borrow.
							  ;if borrow did occur, DivLoopCount.0 will
							  ;hold the eventual borrow value (0-borrow,
							  ;1-no borrow)
FXD2416U.1
   BTFSC	DivLoopCount, 0	  ;if no borrow after 17-bit subtraction
   GOTO		UOK46LL			  ;skip remainder restoration.

   ADDWF	Remainder+1, F	  ;restore higher byte of remainder. (w
							  ;contains the value subtracted from it
							  ;previously)
   MOVF		Divisor, W		  ;restore lower byte of remainder
   ADDWF	Remainder, F	  ;

UOK46LL
   bcf		STATUS, C		  ;copy bit DivLoopCount.0 to carry
   RRcF		DivLoopCount, F	  ;and restore counter

   DECFSZ	DivLoopCount, f	  ;decrement counter
   bra		LOOPU2416		  ;and repeat loop if not zero. carry
							  ;contains next quotient bit (if borrow,
							  ;it is 0, if not, it is 1).

   RLcF		Frequency, F	  ;shift in last bit of quotient
   RLcF		Frequency+1, F
   RLcF		Frequency+2, F
   RETURN
; FXD2416U


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	  Delay functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Delay.W100msec	   ; delay nbr of 100 msec intervals in W - up to 256 (w=0)
; uses W, DelayCount_3

   movwf	DelayCount_3	  ; save nbr of 100 msec intervals
Delay.W100msec.1
   delay	100
   decf		DelayCount_3	  ; done all 100 msec intervals?
   bnz		Delay.W100msec.1  ; br if no
   return


Delay.Wmsec	   ; delay nbr of msec in W - up to 255
; uses W, DelayCount_2

   movwf	DelayCount_2	  ; get nbr of msec
   andwf	DelayCount_2	  ; is delay 0?
   bz		Delay.Wmsec.2	  ; br if yes
Delay.Wmsec.1
   call		Delay.Msec		  ; delay 1msec
   decf		DelayCount_2	  ; update msec counter
   bnz		Delay.Wmsec.1	  ; br if not done
Delay.Wmsec.2
   return
; Delay.Wmsec


Delay.Msec	   ; delay one millisecond
; uses W and DelayCount_1

   movlw	9				  ; 2
   movwf	DelayCount_1	  ; 2
Delay.Msec.1
   call		Delay.100usec	  ; Exactly 100usec
   decf		DelayCount_1	  ; 1
   bnz		Delay.Msec.1	  ; 2(1)
; loop time = (n*100usec) + ( (n-1)*3 + 2 )cycles = 900usec + 26cycles

 if CPU_MHZ == .16
   movlw	.90				  ; 1 attempted adjustment for loop execution
 endif
 if CPU_MHZ == .4
   movlw	.22				  ; 1 attempted adjustment for loop execution
 endif

   call		Delay.100usec.1	  ; 2+[4(n-1)+3+5] = 2+4n-4+8 = 4n+6
   return					  ; 2
; time of complete function - including the calling statement =
;	  900usec +[2 + 4 + 26 + (4n+6) + 2]cycles = 900usec + (4n+40) cycles
;	  therefore: 4n+38 == 400,  n = (400-40)/4 = 90
; Delay.Msec


 if CPU_MHZ == .16
Delay.100usec	  ; delay one hundred micro seconds - exactly!
; this routine requires a 16MHz clock - instruction cycle time - 250ns
; requires 400 instruction cycles
; uses W, Delay_100val
; total execution time = (2) + 2 + 4(n-1) + 3 + 5 = 4+4(97)+8 = 388+12 = 400
; the (2) accounts for the CALL instruction in the calling function
   movlw	98				  ; 1
Delay.100usec.1				  ; enter here with a different value for W
   movwf	Delay_100val	  ; 1
Delay.100usec.2				  ; total loop time is 4(n-1) + 3
   decf	 	Delay_100val	  ; 1 update inner loop count
   nop						  ; 1 - to make loop time a "nice" value == 4
   bnz		Delay.100usec.2	  ; 2(1) br if not done
   nop						  ; 1 to make
   nop						  ; 1	exactly
   nop						  ; 1	   	400 cycles
   return					  ; 2
; Delay.100usec
 endif

 if CPU_MHZ == .4
Delay.100usec	  ; delay one hundred micro seconds - exactly!
; this routine requires a 4MHz clock - instruction cycle time - 1us
; requires 100 instruction cycles
; uses W, Delay_100val
; total execution time = (2)+2+[4(n-1)+3]+5 = 4+[4(22)+3]+5 = 4+91+5 = 100
; the (2) accounts for the CALL instruction in the calling function
   movlw	.23				  ; 1
Delay.100usec.1				  ; enter here with a different value for W
   movwf	Delay_100val	  ; 1
Delay.100usec.2				  ; total loop time is 4(n-1) + 3
   decf	 	Delay_100val	  ; 1 update inner loop count
   nop						  ; 1 - to make loop time a "nice" value == 4
   bnz		Delay.100usec.2	  ; 2(1) br if not done
   nop						  ; 1 to make
   nop						  ; 1	exactly
   nop						  ; 1	   	100 cycles
   return					  ; 2
; Delay.100usec
 endif

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	  A/D functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Read the A/D to get brightness level.  Use only 4 Msbits
; A/D readings are left justified so that the 8 MSbits are in ADRESH
ReadA2D
   bsf      ADCON0, GO        ; start the conversion
   nop
ReadA2D.10
   btfsc    ADCON0, GO        ; skip next if done
   bra      ReadA2D.10
   movf		ADRESH, w	      ; get the reading
   rrncf	WREG			  ; keep
   rrncf	WREG			  ;		only
   rrncf	WREG			  ;		   4
   rrncf	WREG			  ;			  LSbits
   andlw	0x0F			  ; keep only 4 LSbits
;   bnz		ReadA2D.20
;   movlw	1				  ; force minimum brigntness of 1
ReadA2D.20
   movwf	ADvalue			  ; store the reading for the ISR
   return
; ReadA2D


   END
