;*******************************************************************************
;
;	Filename:		NCO-1.asm
;	Date:				24 Oct 2016
;	File Version:	1.0
;	Author:			Larry Cicchinelli
;	Description:
;		This program uses a PIC 16F18313 and its Numerically Controlled Oscillator
;			It allows for variable frequency operation via a potentiometer
;			connected to RA2
;
;		PIC12F1501 I/O:
;			pin 1: VDD
;			pin 2: RA5						Output
;			pin 3: RA4/AN3					On-Off control: 0=on
;			pin 4: RA3/~MCLR
;			pin 5: RA2/ANA2				A/D input for frequency control
;			pin 6: RA1/ANA1/ICSPCLK		Range (when not debugging)
;			pin 7: RA0/ANA0/ISCPDAT
;			pin 8: VSS
;
; The frequency values are based on the CPU clock = 32MHz divided by 4.
; The min frequency is also the frequency step size.
; The max frequency is 1023 times the min frequency.
; There are several ranges available based on the voltage at pin 6: RA1/AN1
;		Range		A/D shifts		Min Freq
;			0				0			7.6Hz (most appropriate for audio applications)
;			1				2			31Hz
;			2				4			123Hz
;			3				6			490Hz
;			4				8			1.96KHz
;			5				10			7.85KHz
; Attempting to use ranges 6 or 7 will lose the most significant bits
; There is no test for ranges 6 or 7.
;
; The ranges may be increased by a factor of 4 by dividing the CPU clock by 1.
;
; Definitions used by the program which are associated with specific I/O pins:
;	O_PULSE			- output pulse pin
;	AV_FREQUENCY	- pin for analog voltage for frequency
;	AV_RANGE			- pin for analog voltage for range
;
;*******************************************************************************
;*******************************************************************************

		#include p16F18313.inc

; CONFIG1
; __config 0xDF8F
 __CONFIG _CONFIG1, _FEXTOSC_OFF & _RSTOSC_HFINT32 & _CLKOUTEN_OFF & _CSWEN_ON & _FCMEN_OFF
; CONFIG2
; __config 0xF732
 __CONFIG _CONFIG2, _MCLRE_OFF & _PWRTE_OFF & _WDTE_OFF & _LPBOREN_OFF & _BOREN_OFF & _BORV_LOW & _PPS1WAY_OFF & _STVREN_ON & _DEBUG_OFF
; CONFIG3
; __config 0x3
 __CONFIG _CONFIG3, _WRT_OFF & _LVP_OFF
; CONFIG4
; __config 0x3
 __CONFIG _CONFIG4, _CP_OFF & _CPD_OFF


O_PULSE			equ	RA5
O_PPS				equ	RA5PPS
I_ON_OFF			equ	RA4
AV_FREQUENCY	equ	RA2
AV_RANGE			equ	RA1

TRISA_VAL		equ	(1<<AV_FREQUENCY) | (1<<AV_RANGE) | (1<<I_ON_OFF)
ADCH_FREQ		equ	( AV_FREQUENCY<<CHS0 ) | (1<< ADON )
ADCH_RANGE		equ	( AV_RANGE<<CHS0 ) | (1<< ADON )

RIGHT_JUSTIFY	equ	0x80 | (b'010'<<ADCS0)
LEFT_JUSTIFY	equ	0 | (b'010'<<ADCS0)		; not used by this program

DEBUG				equ	0
; values to use when DEBUG = 1
DBG_RANGE_VALUE	equ	0

;*******************************************************************************
; RAM
;*******************************************************************************

Access_RAM		udata_shr	0x70		; 16 bytes starting at address 0x70
NCO_Accum		res	3
Range				res	1
Dbg_Range		res	1

;*******************************************************************************
; Vectors
;*******************************************************************************

	org	0
	goto	Main

	org	0x04								; interrupt vector
ISR.Exit
	retfie


;*******************************************************************************
; MAIN PROGRAM
;*******************************************************************************

MAIN_PROG CODE                      ; let linker place main program

Main
	call		init							; initialize the system hardware
#if DEBUG == 1						; if debugging
	movlw		DBG_RANGE_VALUE
	movwf		Dbg_Range
#endif

	BANKSEL	ADCON0
	movlw		ADCH_FREQ					; set the A/D channel and keep A/D on
	movwf		ADCON0
	bsf		ADCON0, ADGO				; start first conversion

;*******************************************************************************
; Main Program Loop
;*******************************************************************************
Main.10
	call		Check_On_Off
	call		Get.Range					; needs to be done before getting Frequency
	call		Get.Frequency
	goto		Main.10


Check_On_Off
	BANKSEL	PORTA
	btfsc		PORTA, I_ON_OFF			; skip next if on
	goto		Turn_Off
; here if switch is on
	BANKSEL	O_PPS
	movlw		b'11101'						; NCO output
	movwf		O_PPS
	return
Turn_Off
	BANKSEL	O_PPS
	clrf		O_PPS
	BANKSEL	LATA
	clrf		LATA							; make output low
	return
; Check_On_Off


Get.Frequency
	movlw		ADCH_FREQ
	call		AD.Read
; test for 0
	movfw		ADRESL
	iorwf		ADRESH, w
	btfsc		STATUS, Z
	incf		ADRESL, f
;
	movfw		ADRESL
	movwf		NCO_Accum					; store Low byte for processing
	movfw		ADRESH
	movwf		NCO_Accum+1					; store High byte for processing
	clrf		NCO_Accum+2					; ensure U byte is clear
; shift the data as necessary based on the range
	movfw		Range							; get Range value
	lslf		Range, w						; two shifts per range value
Get.Frequency.10
	btfsc		STATUS, Z					; skip next if not done
	goto		Get.Frequency.20
	lslf		NCO_Accum, f				; L
	rlf		NCO_Accum+1, f				; H
	rlf		NCO_Accum+2, f				; U
	decf		WREG, f						; update shift counter
	goto		Get.Frequency.10
; put the results into the increment registers
Get.Frequency.20
	BANKSEL	NCO1INCU
	movfw		NCO_Accum+2					; get U byte
	movwf		NCO1INCU
	movfw		NCO_Accum+1					; get H byte
	movwf		NCO1INCH
	movfw		NCO_Accum+0					; get L byte
	movwf		NCO1INCL						; this writes all 3 bytes to the register
	return
; Get.Frequency


Get.Range
; exit with
;	Range set using bits 7-9 of A/D reading
#if DEBUG == 0						; if normal operation
	movlw		ADCH_RANGE					; get the Range channel
	call		AD.Read						; read the A/D
	lslf		ADRESL, w					; get low byte - bit 7 => C
	rlf		ADRESH, w					; w = bits 9..7 = range
#else									; if debugging
	movfw		Dbg_Range					; get the debug range
#endif ; DEBUG
	movwf		Range							; store it
	return
; Get.Range


AD.Read
; read the 10 bit A/D
; enter with w = A/D channel in appropriate bits and ADON bit set
; exit with 10 bit value in AD registers
;		w = low byte value
;		BANKSEL = ADCON0

	BANKSEL	ADCON0
	movwf		ADCON0						; set the channel and keep A/D on
; delay for acquisition time - 20 instruction sycles
	movlw		.7
AD.Read.05
	decfsz	WREG, f						; 1 update count, skip next if done
	goto		AD.Read.05					; 2
	bsf		ADCON0, ADGO				; start the conversion
AD.Read.10
	btfsc		ADCON0, ADGO				; skip next if done conversion
	goto		AD.Read.10					; wait for done
	return
; AD.Read

;===============================================================================
;									Initialize Hardware
;===============================================================================

init	; initialize the system hardware

; Oscillator - set to 32MHz using 8MHz clock and PLL
	BANKSEL	OSCCON1
	movlw		(b'000'<<NOSC0) | (b'0010'<<NDIV0)
	movwf		OSCCON1						; HFINTOSC with 2x PLL (32 MHz), DIV = 4
												; OSCCON2 is read-only
	clrf		OSCCON3
	movlw		1<<HFOEN						; enable the HF internal oscillator
	movwf		OSCEN
	movlw		7								; select 32MHz
	movwf		OSCFRQ
init.10
	btfss		OSCCON3, NOSCR				; wait for oscillator ready
	goto		init.10

; Port A
	BANKSEL	PORTA
	clrf		PORTA
	BANKSEL	TRISA
	movlw		TRISA_VAL
	movwf		TRISA							; set analog and digital input pins
	BANKSEL	ANSELA
	movlw		TRISA							; set analog input pins
	movwf		ANSELA
	BANKSEL	INLVLA
	movlw		0xFF							; set up to
	movwf		INLVLA						;		make all inputs Schmitt Trigger CMOS

; Peripheral Pin Select
	BANKSEL	RA5PPS
	movlw		b'11101'						; NCO output
	movwf		RA5PPS

; A/D convertor
	BANKSEL	ADCON0
	movlw		(1<<ADON)					; A/D on, used later to select A/D channel
	movwf		ADCON0
	movlw		RIGHT_JUSTIFY				; clk=Fosc/32, VRPOS=VDD, right justify data
	movwf		ADCON1
	clrf		ADACT							; no auto conversion trigger

; Interrupts
	clrf		INTCON						; clear global interrupt enable
	BANKSEL	PIE0
	clrf		PIE0							; clear peripheral interrupt enables
	clrf		PIE1							; clear peripheral interrupt enables

; NCO
	BANKSEL	NCO1ACCL
	movlw		b'01'							; use Fosc for clock
	movwf		NCO1CLK
	movlw		(1<<N1EN) | (0<<N1PFM)	; enable NCO1, set to Fixed Duty Cycle (50%)
	movwf		NCO1CON
	return
; init

	END
