;*******************************************************************************
;
;	Filename:		NCO_3 StateToneGen.asm
;	Date:				3 Apr 2017
;	File Version:	1.0
;	Author:			Larry Cicchinelli
;	Description:
;		This program uses a PIC 16F18313 to emulate the	3 StateToneGen in the Mims
;		book.  The NCO is used to develop a gated astable multivigrator with
;		voltage controlled frequency.  Timer 2 interrupts are used as a time base
;		(10KHz, 100us) to emulate a mono-stable multivibrator with independant,
;		voltage controlled, pulse on and off times.
;
;		PIC12F1501 I/O:
;			pin 1: VDD
;			pin 2: RA5						Tone Output
;			pin 3: RA4/AN3					Tone Frequency
;			pin 4: RA3/~MCLR				Tone Gate
;			pin 5: RA2/ANA2				Control PW
;			pin 6: RA1/ANA1/ICSPCLK		Control Period
;			pin 7: RA0/ANA0/ISCPDAT		Control Output
;			pin 8: VSS
;
;
; Definitions used by the program which are associated with specific I/O pins:
;	DO_TONE			- output tone
;	AI_TONE_FREQ	- analog voltage for tone frequency
;	DI_TONE_GATE	- digital input for tone gate: 1 = on
;	AI_CTRL_PW		- control signal pulse width
;	AI_CTRL_PER		- control signal Period
;	DO_CTRL			- control signal output
;
;	Set DEBUG == 0 no debug: full application
;	Set DEBUG == 1 test the NCO
;	Set DEBUG == 2 test the Control I/O
;
;		DEBUG == 1: PIC12F1501 I/O:
;			pin 1: VDD
;			pin 2: RA5						Tone Output
;			pin 3: RA4/AN3					Tone Frequency
;			pin 4: RA3/~MCLR
;			pin 5: RA2/ANA2				Tone Gate
;			pin 6: RA1/ANA1/ICSPCLK
;			pin 7: RA0/ANA0/ISCPDAT
;			pin 8: VSS
;
;		DEBUG == 2: PIC12F1501 I/O:
;			pin 1: VDD
;			pin 2: RA5						Control Output
;			pin 3: RA4/AN3					Control Period
;			pin 4: RA3/~MCLR
;			pin 5: RA2/ANA2				Control PW
;			pin 6: RA1/ANA1/ICSPCLK
;			pin 7: RA0/ANA0/ISCPDAT
;			pin 8: VSS
;
;
		
		
		
;
;
;*******************************************************************************
;*******************************************************************************

DEBUG				equ	0

		#include p16F18313.inc

; CONFIG1
; __config 0xDF8F
 __CONFIG _CONFIG1, _FEXTOSC_OFF & _RSTOSC_HFINT32 & _CLKOUTEN_OFF & _CSWEN_ON & _FCMEN_OFF
; CONFIG2
; __config 0xF732
 __CONFIG _CONFIG2, _MCLRE_OFF & _PWRTE_OFF & _WDTE_OFF & _LPBOREN_OFF & _BOREN_OFF & _BORV_LOW & _PPS1WAY_OFF & _STVREN_ON & _DEBUG_OFF
; CONFIG3
; __config 0x3
 __CONFIG _CONFIG3, _WRT_OFF & _LVP_OFF
; CONFIG4
; __config 0x3
 __CONFIG _CONFIG4, _CP_OFF & _CPD_OFF

#if DEBUG == 0					; full application
DO_TONE			equ	RA5
AI_TONE_FREQ	equ	RA4
DI_TONE_GATE	equ	RA3
AI_CTRL_PW		equ	RA2
AI_CTRL_PER		equ	RA1
DO_CTRL			equ	RA0

DO_TONE_PPS		equ	RA5PPS
DO_CTRL_PPS		equ	RA0PPS
#endif

#if DEBUG == 1					; debug NCO
DO_TONE			equ	RA5
AI_TONE_FREQ	equ	RA4
DI_TONE_GATE	equ	RA2
;
DO_TONE_PPS		equ	RA5PPS
DO_CTRL_PPS		equ	RA0PPS
;
AI_CTRL_PW		equ	RA0	; dummy definitions
AI_CTRL_PER		equ	RA1
DO_CTRL			equ	RA0
#endif

#if DEBUG == 2					; debug Control
DO_CTRL			equ	RA5
AI_CTRL_PER		equ	RA4
AI_CTRL_PW		equ	RA2
;
DO_TONE_PPS		equ	RA0PPS
DO_CTRL_PPS		equ	RA5PPS
;
DO_TONE			equ	RA0	; dummy definitions
AI_TONE_FREQ	equ	RA1
DI_TONE_GATE	equ	RA3
#endif

AI_MASK			equ	( 1<<AI_TONE_FREQ ) | ( 1<<AI_CTRL_PW ) | ( 1<<AI_CTRL_PER )
DI_MASK			equ	( 1 << DI_TONE_GATE )
TRISA_VAL		equ	( AI_MASK | DI_MASK )

ADCH_TONE_FREQ	equ	( AI_TONE_FREQ << CHS0 )  | ( 1 << ADON )
ADCH_CTRL_PW	equ	( AI_CTRL_PW << CHS0 )  | ( 1 << ADON )
ADCH_CTRL_PER	equ	( AI_CTRL_PER << CHS0 )   | ( 1 << ADON )

RIGHT_JUSTIFY	equ	( 1 << ADFM ) | ( b'111' << ADCS0 ) ; use ADCRC clock
LEFT_JUSTIFY	equ	( 0 << ADFM ) | ( b'111' << ADCS0 )

TONE_FREQ_OFFSET	equ	.1

;*******************************************************************************
; RAM
;*******************************************************************************

Access_RAM		udata_shr	0x70		; 16 bytes starting at address 0x70
NCO_Accum		res	3
Ctrl_Off			res	2
Ctrl_PW			res	2
ISR_Counter		res	2
Temp_Off			res	2			; used to hold Off time temporarily in order
										; to reduce the time interrupts are disabled

;*******************************************************************************
; Vectors
;*******************************************************************************

	org	0
	goto	Main

	org	0x04								; interrupt vector
ISR											; label is not really needed
; only Timer 2 interrupts are enabled
	BANKSEL	PIR1
	bcf		PIR1, TMR2IF				; clear Timer 2 interrupt flag
; decrement the ISR counter
	movlw		1								; set up to update counter
	subwf		ISR_Counter, f
	clrf		WREG
	subwfb	ISR_Counter+1, f
; check for done: ISR_Counter < 0
	btfss		ISR_Counter+1, 7			; ISR_Counter+1:7 = 1 if negative
	retfie									; exit if counter not done
; handle counter done
	BANKSEL	PORTA
	btfsc		PORTA, DO_CTRL				; skip next if Control is 0
	goto		ISR.20
; here if Control is low
	BANKSEL	LATA
	bsf		LATA, DO_CTRL				; DO_CTRL = 1
	movfw		Ctrl_PW						; get low byte of PW
	movwf		ISR_Counter
	movfw		Ctrl_PW+1					; get high byte of PW
	movwf		ISR_Counter+1
	retfie
; here if Control is high
ISR.20
	BANKSEL	LATA
	bcf		LATA, DO_CTRL				; DO_CTRL = 0
	movfw		Ctrl_Off						; get low byte of off time
	movwf		ISR_Counter
	movfw		Ctrl_Off+1					; get high byte of off time
	movwf		ISR_Counter+1
	retfie
; ISR

;*******************************************************************************
; MAIN PROGRAM
;*******************************************************************************

MAIN_PROG CODE                      ; let linker place main program

Main
	call		init							; initialize the system hardware
	movlw		2								; set up for
	movwf		ISR_Counter					;		1st interrupt
	clrf		ISR_Counter+1

;*******************************************************************************
; Main Program Loop
;*******************************************************************************
Main.10
#if DEBUG == 0  ||  DEBUG == 1
	call		Set.NCO
#endif
#if DEBUG == 0  ||  DEBUG == 2
	call		Set.Control
#endif
	goto		Main.10


Set.NCO
; check gate and turn off output if gate is low
	BANKSEL	PORTA
	btfsc		PORTA, DI_TONE_GATE		; skip next if gate is low
	goto		Set.NCO.05					; gate is high - set NCO frequency
; turn off NCO output
	BANKSEL	NCO1CON
Set.NCO.03
	btfsc		NCO1CON, N1OUT				; skip next if output is low
	goto		Set.NCO.03					; loop if output is high
	bcf		NCO1CON, N1EN				; disable NCO 
	return
;
Set.NCO.05
	movlw		ADCH_TONE_FREQ				; set up to
	call		AD.Read.R					;		read the frequency A/D input
	movfw		ADRESL
	movwf		NCO_Accum					; store Low byte for processing
	movfw		ADRESH
	movwf		NCO_Accum+1					; store High byte for processing
	clrf		NCO_Accum+2					; ensure U byte is clear
;
; add TONE_FREQ_OFFSET to the value to define a minimum frequency
	movlw		TONE_FREQ_OFFSET
	addwf		NCO_Accum, f
	clrf		WREG
	addwfc	NCO_Accum+1, f
	addwfc	NCO_Accum+2, f
;
Set.NCO.20
; put the results into the increment registers
	BANKSEL	NCO1INCU
	bsf		NCO1CON, N1EN				; enable NCO 
	movfw		NCO_Accum+2					; get U byte
	movwf		NCO1INCU
	movfw		NCO_Accum+1					; get H byte
	movwf		NCO1INCH
	movfw		NCO_Accum+0					; get L byte
	movwf		NCO1INCL						; this writes all 3 bytes to the register
	return
; Set.NCO


Set.Control
; Peripheral interrupts are disabled, for a total of 8 instructions, when
; updating the Ctrl_x values to prevent the ISR from using partial values.
; Interrupts should NOT be disabled when calling the AD read functions due to
; the time it takes to allow the A/D to acquire its new input value.
; Values of 0 are OK since the ISR looks for a negative value to determine
; when the event has occured.
;
; get the PW
	movlw		ADCH_CTRL_PW				; set up to
	call		AD.Read.R					;		read the PW value
	movfw		ADRESL						; get low byte
	bcf		INTCON, PEIE			; disable peripheral interrupts
	movwf		Ctrl_PW						; save it
	movfw		ADRESH						; get high byte
	movwf		Ctrl_PW+1					; save it
	bsf		INTCON, PEIE			; enable peripheral interrupts
; get the off time
	movlw		ADCH_CTRL_PER				; set up to
	call		AD.Read.R					;		read the period value
	movfw		ADRESL						; get low byte
	bcf		INTCON, PEIE			; disable peripheral interrupts
	movwf		Ctrl_Off						; save it
	movfw		ADRESH						; get high byte
	movwf		Ctrl_Off+1					; save it
	bsf		INTCON, PEIE			; enable peripheral interrupts
	return
; Set.Control


; read the 10 bit A/D
; enter with w = A/D channel in appropriate bits and ADON bit set
; exit with 10 bit value in AD registers
;		BANKSEL = ADCON0
;
AD.Read.L
	BANKSEL	ADCON0						; ADRESL @ 0x09B
	movwf		ADCON0						; set the channel and keep A/D on
	movlw		LEFT_JUSTIFY				; set to left justify data
	movwf		ADCON1
	goto		AD.Read

AD.Read.R
	BANKSEL	ADCON0						; ADRESL @ 0x09B
	movwf		ADCON0						; set the channel and keep A/D on
	movlw		RIGHT_JUSTIFY				; set to right justify data
	movwf		ADCON1
	
AD.Read
; delay for acquisition time - 20 instruction sycles
	movlw		.10
AD.Read.05
	decfsz	WREG, f						; 1 update count, skip next if done
	goto		AD.Read.05					; 2
	bsf		ADCON0, ADGO				; start the conversion
AD.Read.10
	btfsc		ADCON0, ADGO				; skip next if done conversion
	goto		AD.Read.10					; wait for done
	return
; AD.Read

;===============================================================================
;									Initialize Hardware
;===============================================================================

init	; initialize the system hardware

; Oscillator - set to 1MHz
	BANKSEL	OSCCON1
	movlw		(b'000'<<NOSC0) | (b'0001'<<NDIV0)
	movwf		OSCCON1						; HFINTOSC, DIV = 2^1
												; OSCCON2 is read-only
	clrf		OSCCON3
	movlw		1<<HFOEN						; enable the HF internal oscillator
	movwf		OSCEN
	movlw		b'0110'						; HFINTOSC
	movwf		OSCFRQ						;		= 32MHz
init.10
	btfss		OSCCON3, ORDY				; wait for oscillator ready
	goto		init.10

; Port A
	BANKSEL	PORTA
	clrf		PORTA
	BANKSEL	TRISA
	movlw		TRISA_VAL
	movwf		TRISA							; set analog and digital input pins
	BANKSEL	ANSELA
	movlw		AI_MASK						; set analog input pins
	movwf		ANSELA
	BANKSEL	INLVLA
	movlw		0xFF							; set up to
	movwf		INLVLA						;		make all inputs Schmitt Trigger CMOS
	clrf		DO_CTRL_PPS					; Control pin is PortA digital I/O

; A/D convertor
	BANKSEL	ADCON0
	movlw		(1<<ADON)					; A/D on, used later to select A/D channel
	movwf		ADCON0
	clrf		ADACT							; no auto conversion trigger

; Timer 2 interrupt for the PWM system
; TMR2IF frequency = Fosc/4 / Prescale / PR / Postscale = 
;							16MHz / 4 / 2 / 64 / 156 = 200.3Hz = 5ms resolution
	BANKSEL	T2CON
	movlw		(b'0001' << T2OUTPS0 ) | ( 1 << TMR2ON ) | ( b'11' << T2CKPS0 )
	movwf		T2CON							; Postscaler=2, T2 on, Prescale = 64
	movlw		.156							; period = 156
	movwf		PR2

; NCO
	BANKSEL	NCO1ACCL
	movlw		b'01'							; use Fosc for clock
	movwf		NCO1CLK
	movlw		(1<<N1EN) | (0<<N1PFM)	; enable NCO1, set to Fixed Duty Cycle (50%)
	movwf		NCO1CON
#if DEBUG == 0  ||  DEBUG == 1
	BANKSEL	DO_TONE_PPS
	movlw		b'11101'						; send output to selected I/O pin
	movwf		DO_TONE_PPS
#endif
	
; Interrupts
	movlw		( 1 << GIE ) | ( 1 << PEIE )
	movwf		INTCON						; enable global and peripheral interrupts
	BANKSEL	PIE0
	clrf		PIE0							; clear some peripheral interrupt enables
	movlw		( 1 << TMR2IE )
	movwf		PIE1							; enable Timer2 interrupt

	return
; init

	END
