;*******************************************************************************
;
;	Filename:		PIC 555-16F.asm
;	Date:				21 Mar 2017
;	Author:			Larry Cicchinelli
;	Description:
;		This program uses a PIC 16F18313 to (somewhat) emulate a 555 timer.
;		It is derived from the program PIC 555 v1.4.asm
;
;		PIC I/O:
;			pin 1: VDD
;			pin 2: RA5							Pulse output
;			pin 3: RA4, AN3					Mode divider
;			pin 4: RA3, ~MCLR					Trigger/Gate
;			pin 5: RA2, AN2					Pulse Width potentiometer
;			pin 6: RA1, AN1, ICSPCLK		Delay/Period/Off Time potentiometer
;			pin 7: RA0, AN0, ISCPDAT		Range potentiometer
;			pin 8: VSS
;
;		PIC I/O - DEBUG!!
;			pin 1: VDD
;			pin 2: RA5							Pulse output
;			pin 3: RA4, AN3					Trigger or test output
;			pin 4: ~MCLR
;			pin 5: RA2, AN2					Pulse Width potentiometer
;			pin 6: ICSPCLK
;			pin 7: ISCPDAT
;			pin 8: VSS
;
; Metronome only - uses Timer 0 to generate 400Hz and 800Hz tones
;			pin 1: VDD
;			pin 2: RA5							Tone output
;			pin 3: RA4, AN3					Mode divider and downbeat LED drive
;			pin 4: RA3, ~MCLR					gate
;			pin 5: RA2, AN2					beats per measure potentiometer
;			pin 6: RA1, AN1, ICSPCLK		tempo potentiometer
;			pin 7: RA0, AN0, ISCPDAT		Range input and Downbeat output
;			pin 8: VSS
;
; IMPORTANT
;	Both Mode and Range are read only once - when the program starts.
;	Both the Pulse Width and Pulse Delay are read immdediately preceding the
;		generation of each output occurance.
;	Response time can be improved by reading the Pulse Width and Pulse Delay
;		values only one time.  See the code which executes if FAST is defined.
;	If a function changes the BSR, it MUST set it back to PORTA before returning.
; NOTE: the T1 registers are in the same bank (0) as PORTA
;
; Definitions used by the program which are associated with specific I/O pins:
;	AI_PW			- analog voltage which defines the output pulse width
;	AI_DELAY		- analog voltage which defines the delay preceding the output pulse
;						as well as define the "0" time of the astable multivibrator
;	AI_MODE		- analog voltage defining operating mode - most sig 3 bits
;	AI_RANGE		- analog voltage defining pulse width range/delay - most sig 3 bits
;	DO_PULSE		- output pulse pin
;	DI_TRIGGER	- input trigger pin
;	DO_DOWNBEAT	- Metronome only (mode 6) output pin for the downbeat LED
;
;	Programming notes:
;		The CPU clock is the internal 8MHz oscillator: (see Get.Range)
;			Ranges 0 - 4 =- PLL on: Fosc = 32MHz
;			Ranges 5 - 7 = PLL off: Fosc = 8MHz
;			The A/D sample clock is always 1MHz
;		All functions that change the BSR return with BSR = 0
;		The Mode and Range/Resolution values are read only once - at startup
;
;	Timer 1 is used as a programmable timer for Ranges 1-7:
;		source = Fosc/4, prescale = 8: T1 clock is always Fcpu/32
;		Range 0 does not use the timer.  It uses timing loops based on instruction
;		execution time.
;		
;							T1			T1			Software
;		Resolution		Clk		Count		Divisor		Max PW
; PLL on: Fcpu = 8MHz*4 = 32MHz
;		0 - 1us										A/D		1ms
;		1 - 10us			1Mhz		10				A/D		10ms
;		2 - 100us		1Mhz		100			A/D		100ms
;		3 - 1ms			1Mhz		1000			A/D		1s
;		4 - 10ms			1Mhz		10000			A/D		10s
; PLL off: Fcpu = 8MHz
;		5 - 100ms		250KHz	25000			A/D		100s
; PLL off: Fcpu = 2MHz
;		6 - 1s			62.5KHz	62500			A/D		1000s (16.6 mins)
;		7 - 10s			62.5KHz	62500			A/D*10	10,000s (166 mins)
;
;	Modes:
;		0 = One Shot
;		1 = Delayed One Shot
;		2 = Retriggerable One Shot
;		3 = Delayed Retriggerable One Shot
;				note for modes 2 & 3:
;					The pulse width cannot be changed once the one shot fires
;					until the trigger is removed or the trigger period exceeds the
;					pulse width.
;		4 = Gated Multivibrator using Pulse Width and Off-Time
;		5 = Gated Multivibrator using Pulse Width and Period
;		6 = metronome
;		7 = force to mode = 0
;
;*******************************************************************************
DEBUG		equ	0				; 0 = no debugging
FAST		equ	0				; 1 = bypass some code to speed up the response time
									;		by reading the PW and Delay values once only.
;*******************************************************************************

		#include p16F18313.inc

; CONFIG1
; __config 0xDF8F
 __CONFIG _CONFIG1, _FEXTOSC_OFF & _RSTOSC_HFINT32 & _CLKOUTEN_OFF & _CSWEN_ON & _FCMEN_OFF
; CONFIG2
; __config 0xF732
 __CONFIG _CONFIG2, _MCLRE_OFF & _PWRTE_ON & _WDTE_OFF & _LPBOREN_OFF & _BOREN_OFF & _BORV_LOW & _PPS1WAY_OFF & _STVREN_ON & _DEBUG_OFF
; CONFIG3
; __config 0x3
 __CONFIG _CONFIG3, _WRT_OFF & _LVP_OFF
; CONFIG4
; __config 0x3
 __CONFIG _CONFIG4, _CP_OFF & _CPD_OFF

#if DEBUG == 0	;========= hardware definitions when =NOT= debugging ============
;																		 =====
AI_RANGE		equ	RA0
AI_DELAY		equ	RA1
AI_PW			equ	RA2
DI_TRIGGER	equ	RA3
AI_MODE		equ	RA4
DO_PULSE		equ	RA5
DO_PULSE_PPS equ	RA5PPS

AI_MODE_VALUE equ 0		; needed because of conditional assembly nr line 365

AI_MASK			equ	(1<<AI_RANGE) | (1<<AI_DELAY) | (1<<AI_PW) | (1<<AI_MODE)
DI_MASK			equ	( 1 << DI_TRIGGER )
TRISA_VAL		equ	( AI_MASK | DI_MASK )

ANSEL_VAL	equ	AI_MASK
ADCH_RANGE	equ	( AI_RANGE<<CHS0 ) | (1<< ADON )
ADCH_DELAY	equ	( AI_DELAY<<CHS0 ) | (1<< ADON )
ADCH_PW		equ	( AI_PW<<CHS0 )	 | (1<< ADON )
ADCH_MODE	equ	( AI_MODE<<CHS0 )  | (1<< ADON )
	
DO_DOWNBEAT	equ	RA0
DO_DOWNBEAT_PPS equ RA0PPS
		
#else ;====================== hardware definitions when debugging ==============

AI_PW			equ	RA2
DI_TRIGGER	equ	RA4
DO_PULSE		equ	RA5		; do not change from RA5 if Tone burst is required
DO_PULSE_PPS equ	RA5PPS

; the following 3 values are modified as necessary to test the software
AI_DELAY_VALUE	equ .10			; also the Tempo (beats/minute) for mode 6
AI_RANGE_VALUE	equ .4
AI_MODE_VALUE	equ .6

AI_MASK			equ	( 1 << AI_PW )
DI_MASK			equ	( 1 << DI_TRIGGER )
TRISA_VAL		equ	( AI_MASK | DI_MASK )

ANSEL_VAL	equ	AI_MASK
ADCH_PW		equ	( AI_PW<<CHS0 ) | (1<< ADON )
		
DO_DOWNBEAT	equ	RA4
DO_DOWNBEAT_PPS equ RA4PPS

#if AI_RANGE_VALUE == 3
MET_TEMPO = .100
#else
MET_TEMPO = .10
#endif

#endif ; DEBUG == 0 ============================================================

;*******************************************************************************
AD_NO_BITS_0_1	equ	0		; 1 = clear A/D bits 0 & 1 for PW and Delay.
									;		This is done to make it easier to adjust the
									;		potentiometers for the full range of values.
									;		It also forces the resolution to be 4 times the
									;		listed value; ie, 4us, 40us, etc
AD_USE_8_BITS equ		0		; 1 = use upper 8 bits for PW and delay.  This is
									;		done to make it easier to adjust the
									;		potentiometers leaving the resolution unchanged.
									;		However, the full range of values is decreased
									;		by a factor of 4; ie, 250us, 2.5ms, etc
; both AD_NO_BITS_0_1 and AD_USE_8_BITS may be enabled simultaneously, this will
; result essentially in a 6 bit A/D reading.  It will force both the resolution
; and full range to be reduced.

EDGE_REG equ	IOCAP			; IOCAP = trigger on positive edge of input
;EDGE_REG equ	IOCAN			; IOCAN = trigger on negative edge of input
;
;*******************************************************************************

; Common RAM allocations: 16 bytes
Access_RAM		udata_shr	0x70		; 16 bytes starting at address 0x70

; Since the total number of bytes of RAM required are 14, the program does not
; need to make use of banked RAM

ADvalue			res	2
Mode				res	1
Range				res	1

PW_Counter		res	2
Delay_Counter	res	2
T1_Preset		res	2
temp0				res	2
BPM				res	1		; used by Metronome
Flags				res	1
FLAGS_RETRIGGER	equ	0

;*******************************************************************************
; Vectors
;*******************************************************************************

	org	0
	goto	Main

	org	0x04								; interrupt vector
ISR.Exit
	retfie

BPM_Table
	dw		0, 2, 3, 4, 6, .8, .12, .12 ; max of 8 values

;*******************************************************************************
; MAIN PROGRAM
;*******************************************************************************

MAIN_PROG CODE                      ; let linker place main program

Main
	call		init							; initialize the system hardware
	clrf		Flags

#if AD_USE_8_BITS == 1
	BANKSEL	ADCON1
	bcf		ADCON1, ADFM				; left justify the data
	BANKSEL	PORTA
#endif
	BANKSEL	EDGE_REG
	bcf		EDGE_REG, DI_TRIGGER		; ensure edge detect disabled
	call		Get.Range					; value saved in Range
	call		Get.Mode						; value in W and saved in Mode
; FSR0 is used to point to Common RAM so it upper byte is ALWAYS 0
	clrf		FSR0H							; clear upper byte of pointer

; determine Mode of operation by successive decrementing W
	andlw		0xFF							; Z=1 iff W=0
	btfsc		STATUS, Z					; skip next if Mode <> 0
	goto		One.Shot

	addlw		-1								; W = Mode - 1
	btfsc		STATUS, Z					; skip next if Mode <> 1
	goto		Delay.One.Shot

	addlw		-1								; W = Mode - 2
	btfsc		STATUS, Z					; skip next if Mode <> 2
	goto		Retrig.One.Shot

	addlw		-1								; W = Mode - 3
	btfsc		STATUS, Z					; skip next if Mode <> 3
	goto		Delay.Retrig.One.Shot

	addlw		-1								; W = Mode - 4
	btfsc		STATUS, Z					; skip next if Mode <> 4
	goto		Gated.Multivibrator

	addlw		-1								; W = Mode - 5
	btfsc		STATUS, Z					; skip next if Mode <> 5
	goto		Gated.Multivibrator

	addlw		-1								; W = Mode - 6
	btfsc		STATUS, Z					; skip next if Mode <> 6
	goto		Metronome					; W = 6 = metronome

; mode 7: drop thru to One.Shot		


; ==============================================================================
; ================================== One Shot ==================================
; ==============================================================================

One.Shot										; mode 0 = one-shot
	call		Get.PW
	movlw		LOW PW_Counter				; get address of value for software counter
	movwf		FSR0L							;		used by Run.T1

One.Shot.10
; wait for trigger
	BANKSEL	EDGE_REG
	bsf		EDGE_REG, DI_TRIGGER		; enable + edge detect
One.Shot.30									; wait for trigger
	btfss		IOCAF, DI_TRIGGER			; skip next if + edge detected
	goto		One.Shot.30
	btfss		Flags, FLAGS_RETRIGGER	; skip next if retrigger needed
	bcf		EDGE_REG, DI_TRIGGER		; disable + edge detect
	bcf		IOCAF, DI_TRIGGER			; clear the edge detect flag
	BANKSEL	PORTA
	bsf		PORTA, DO_PULSE			; turn on output pulse
	call		Run.T1						; execute pulse width
	bcf		PORTA, DO_PULSE			; turn off output pulse

#if FAST == 0
	goto		One.Shot
#else
	goto		One.Shot.10					; do not read A/D again
#endif
; One.Shot

; ==============================================================================
; ============================== Delayed One Shot ==============================
; ==============================================================================

Delay.One.Shot								; mode 1 = delayed one-shot
	call		Get.PW
	call		Get.Delay
Delay.One.Shot.10
; set up for delay
	movlw		LOW Delay_Counter			; get address of value for software counter
	movwf		FSR0L							;		used by Run.T1
; wait for trigger
	BANKSEL	EDGE_REG
	bsf		EDGE_REG, DI_TRIGGER		; enable edge detect
Delay.One.Shot.30							; wait for trigger
	btfss		IOCAF, DI_TRIGGER			; skip next if + edge detected
	goto		Delay.One.Shot.30
	btfss		Flags, FLAGS_RETRIGGER	; skip next if retrigger needed
	bcf		EDGE_REG, DI_TRIGGER		; disable edge detect
	bcf		IOCAF, DI_TRIGGER			; clear the edge detect flag
	BANKSEL	PORTA
	call		Run.T1						; execute delay time
; set up for PW
	movlw		LOW PW_Counter				; point to the PW counter value
	movwf		FSR0L							;		used by Run.T1
	bsf		PORTA, DO_PULSE			; turn on output pulse
	call		Run.T1						; execute pulse width
	bcf		PORTA, DO_PULSE			; turn off output pulse

#if FAST == 0
	goto		Delay.One.Shot
#else
	goto		Delay.One.Shot.10			; do not read A/D again
#endif
; Delay.One.Shot

; ==============================================================================
; ============================ Retriggered One Shot ============================
; ==============================================================================

Retrig.One.Shot							; mode 2 = retriggerable one-shot
	bsf		Flags, FLAGS_RETRIGGER
	goto		One.Shot

; ==============================================================================
; ======================= Delayed & Retriggered One Shot =======================
; ==============================================================================

Delay.Retrig.One.Shot					; mode 3 = delayed, retriggerable one-shot

	bsf		Flags, FLAGS_RETRIGGER
	goto		Delay.One.Shot


; ==============================================================================
; ============================ Gated Multivibrator =============================
; ==============================================================================

Gated.Multivibrator
#if FAST == 1								; get values one time
	call		Get.Delay
	call		Get.PW
#endif
Gated.Multivibrator.10
	btfsc		PORTA, DI_TRIGGER			; skip next if the gate is off
	call		Do.Period
	goto		Gated.Multivibrator.10
; Gated.Multivibrator


Do.Period
; set up timer for PW
; the output pulse is turned on "early" in order to enable a square wave closer
; to 50% when the Pulse Width and Off-Time inputs are tied together.

#if FAST == 0
	call		Get.PW
#endif
	bsf		PORTA, DO_PULSE			; turn on output pulse
	movlw		LOW PW_Counter				; point to the PW counter value
	movwf		FSR0L							;		used by Run.T1
	call		Run.T1						; execute pulse width
;	bcf		PORTA, O_PULSE				; turn off output pulse			1.1

; set up timer for off time
#if FAST == 0
	call		Get.Delay
#endif
	bcf		PORTA, DO_PULSE			; turn off output pulse			1.1
	movlw		LOW Delay_Counter			; get address of value for software counter
	movwf		FSR0L							;		used by Run.T1
	call		Run.T1						; execute off time and return
	return
; Do.Period


; ==============================================================================
; ================================= Metronome ==================================
; ==============================================================================

Metronome
; This feature uses timer 2 to generate an interrupt at twice the desired tone
;	frequency.  The ISR inverts the DO_PULSE output at each interrupt which
;	generates a 50% duty cycle.  This turns out to be easier than using the PWM.

; set up output pins
	BANKSEL	TRISA
	bcf		TRISA, DO_DOWNBEAT		; enable output for downbeat LED
	BANKSEL	RA5PPS
	movlw		b'11100'						; DO_PULSE = TIMER0 output
	movwf		DO_PULSE_PPS
	clrf		DO_DOWNBEAT_PPS
; initialize Timer 0 registers
	BANKSEL	T0CON0						; Timer 0 registers
	movlw		(0<<T0EN) | (0 << T016BIT) | (0 << T0OUTPS0) ; Timer disabled,
	movwf		T0CON0						; 8 bit operation, Postscale=1
	movlw		( b'010' << T0CS0 ) | ( b'0110' << T0CKPS0 ) ; clock = FOSC/4,
	movwf		T0CON1						; Prescale = 64
Metronome.10
; get off time
#if DEBUG == 0
	call		Get.Delay					; get Tempo
#else
	movlw		LOW MET_TEMPO				; set tempo
	movwf		Delay_Counter
	movlw		HIGH MET_TEMPO
	movwf		Delay_Counter+1
#endif
; get Beats Per Measure
	movlw		LOW BPM_Table				; get adddress of BPM table
	movwf		FSR1L
	clrf		FSR1H
	bsf		FSR1H, 7						; bit 7 = 1 for program memory access
	call		Get.PW						; values in PW_Counter and PW_Counter+1
	lslf		PW_Counter, f
	rlf		PW_Counter+1, W			; W = MS 3 bits = BPM value
	addwf		FSR1L, f						; FSR0 = address of BPM table entry
	movfw		INDF1							; get entry
	movwf		BPM							; save BPM for loop
	decf		BPM, f						; make 0 relative for loop

Metronome.20
; wait for gate
#if DEBUG == 0
	BANKSEL	PORTA
	btfss		PORTA, DI_TRIGGER			; skip next if the gate is on
	goto		Metronome.20
#endif
; set period to 400HZ, duty cycle is 50%
	clrf		TMR0L							; clears timing registers
	BANKSEL	TMR0H							; Timer 0 registers
	movlw		.156							; set up for 400Hz output
	movwf		TMR0H							; freq = Fosc/4/64/156 /2 = 400.6Hz
												; see Timer0 block diagram for final /2
Metronome.22
	BANKSEL	T0CON0						; Timer 0 registers
	bsf		T0CON0, T0EN				; enable the timer
; there should now be tone on DO_PULSE

; set up Tone burst time
	movlw		.100							; set pulse width using range 3 to 100ms
	movwf		PW_Counter
	movfw		Range							; test for Range = 3
	sublw		3
	btfsc		STATUS, Z					; skip next if Range <> 3
	goto		Metronome.25
	movlw		.10							; assume range = 4
	movwf		PW_Counter

Metronome.25
	clrf		PW_Counter+1
	movlw		LOW PW_Counter				; point to the burst time counter
	movwf		FSR0L							;		used by Run.T1
	call		Run.T1						; execute on time
	BANKSEL	LATA
	bcf		LATA, DO_DOWNBEAT			; turn off downbeat LED
	
Metronome.30
	btfsc		PORTA, DO_PULSE			; wait for pulse low
	goto		Metronome.30
	BANKSEL	T0CON0						; Timer 0 registers
	bcf		T0CON0, T0EN				; disable the timer

; set up timer for off time
	movlw		LOW Delay_Counter			; get address of value for software counter
	movwf		FSR0L							;		used by Run.T1
	call		Run.T1						; execute off time

; now do testing to determine if time for downbeat
	decf		BPM, f						; update beat counter
	btfsc		STATUS, Z					; skip if not emphasized beat yet
	goto		Metronome.40				; do downbeat
	btfsc		BPM, 7
	goto		Metronome.10				; bit 7=1 if BPM<0: completed downbeat
	goto		Metronome.20				; do regular beat

; set downbeat frequency
Metronome.40
	BANKSEL	LATA
	bsf		LATA, DO_DOWNBEAT			; turn on downbeat LED
	BANKSEL	TMR0H							; Timer 0 registers
	movlw		.78							; set up for 800Hz output
	movwf		TMR0H							; freq = Fosc/4/64/78 /2 = 800.6Hz
	goto		Metronome.22
; Metronome


;===============================================================================
;									Main Utility Functions
;===============================================================================

; Start Timer 1 and wait for it to complete
; Enter with:
;		T1_Preset value set
;		FSR0L = address of low byte of x_Counter value
;			value of x_Counter not modified
;		FSR1 = used as 16 bit counter - not a pointer
; I_TRIGGER bit indicates retrigger requested

Run.T1
	BANKSEL	PORTA							; ensure Bank 0
; check for Range = 0
	movfw		Range							; get the range
	andlw		0xFF							; w = 0 iff Range = 0
	btfsc		STATUS, Z					; skip if w <> 0
	goto		Run.Range0					; goto if Range = 0
;
; Ranges 1-7
;
Run.T1.10
; get software counter value
	moviw		FSR0++						; low byte and point to high byte
	movwf		FSR1L
	moviw		FSR0--						; high byte and point back to low byte
	movwf		FSR1H

Run.T1.20
; check if done software counter
	moviw		FSR1--						; update software counter
	btfsc		FSR1H, 7						; bit 7 = 1 if decremented to -1
	return
; set up Timer 1
	BANKSEL	PORTA
	bcf		T1CON, TMR1ON				; turn timer off
	bcf		PIR1, TMR1IF				; reset the overflow flag
; set Timer 1 preset and turn it on
	movf		T1_Preset, w				; get low byte of Timer preset count
	movwf		TMR1L
	movf		T1_Preset+1, w				; get high byte
	movwf		TMR1H
	bsf		T1CON, TMR1ON				; turn timer on
;
;  11 I-cycles from Run.T1.20
;
; Timer 1 loop
Run.T1.30
; test for re-trigger
	BANKSEL	EDGE_REG
	btfss		IOCAF, DI_TRIGGER			; skip next if re-trigger
	goto		Run.T1.40
	bcf		IOCAF, DI_TRIGGER
	goto		Run.T1.10					; start timing (hardware & software) over
; test for Timer 1 done
Run.T1.40
	BANKSEL	PORTA
	btfss		PIR1, TMR1IF				; skip next if Timer 1 done
	goto		Run.T1.30
	goto		Run.T1.20					; update software counter, restart Timer 1
; Run.T1
	

; these two lines of code are here in order to remove the necessity for a goto
; and are executed when a retrigger is detected.
Run.Range0.30
	bcf		IOCAF, DI_TRIGGER			; clear the edge detect flag
	BANKSEL	PORTA

Run.Range0									; Range = 0 only!
; This functions does not use Timer 1 - it uses instruction execution only!!
; Enter with:
;		FSR0L = address of low byte of x_Counter value
; get the count value into FSR1 used as 16 bit counter - not a pointer
	moviw		FSR0++						; low byte
	movwf		FSR1L
	moviw		FSR0--						; high byte (and point back to low byte)
	movwf		FSR1H
; values of 0-3 will all yield the same PW: about 3usec
	moviw		FSR1--						; adjust
	moviw		FSR1--						;		for fixed
	moviw		FSR1--						;				delays
	
; the following loop executes in exactly 8 instruction cycles = 1us
Run.Range0.10
	; test for x_Counter == 0
	moviw		FSR1--						; update the count value
	btfsc		FSR1H, 7						; skip if new count >= 0
	goto		Run.Range0.20				; exit loop when counter < 0
	nop										; needed to make 8 instruction cycles
	nop
	nop
	goto		Run.Range0.10

Run.Range0.20
; test for retrigger
	BANKSEL	EDGE_REG
	btfsc		IOCAF, DI_TRIGGER			; skip next if edge NOT detected
	goto		Run.Range0.30				; goto if retrigger detected
	BANKSEL	PORTA
	return
; Run.T1.Range0


;===============================================================================
;									Main Operating Functions
;===============================================================================

Get.PW
; Read the A/D
; determine if the value needs to be multiplied by 10 (range 7)
; store value into PW_Counter
	movlw		ADCH_PW						; set up to
	call		AD.Read						;		get Pulse width
; if Range = 7 multiply ADvalue by 10
	movlw		7
	subwf		Range, w						; Z = 1 if Range = 7
	btfsc		STATUS, Z					; skip next if range <> 7
	call		AD.X10
;
	movfw		ADvalue						; need to save the pulse width
	movwf		PW_Counter					;		so it can be used
	movfw		ADvalue+1					;			repeatitively
	movwf		PW_Counter+1
	return
; Get.PW


Get.Delay
; Read the A/D
; determine if the value needs to be multiplied by 10 (range 7)
; store value into Delay_Counter

#if DEBUG == 0						; if normal operation
	movlw		ADCH_DELAY			; set up to
	call		AD.Read				;		get delay
#else									; if debugging
#if AI_MODE_VALUE == 4
	movfw		PW_Counter
	movwf		ADvalue
	movfw		PW_Counter+1
	movwf		ADvalue+1
#else
	movlw		LOW AI_DELAY_VALUE
	movwf		ADvalue
	movlw		HIGH AI_DELAY_VALUE
	movwf		ADvalue+1
#endif
#endif
; if Range = 7 multiply ADvalue by 10
	movlw		7
	subwf		Range, w						; Z = 1 if Range = 7
	btfsc		STATUS, Z					; skip next if range <> 7
	call		AD.X10
;
	movfw		ADvalue						; need to save the pulse width
	movwf		Delay_Counter				;		so it can be used
	movfw		ADvalue+1					;			repeatitively
	movwf		Delay_Counter+1

	movlw		5								; set up to test for mode = 5		1.2
	subwf		Mode, w						; Z = 0 if mode = 5
	btfss		STATUS, Z
	return									; return if mode <> 5
; here if mode = 5: change delay to period
	movfw		PW_Counter					; low byte of pulse width
	subwf		Delay_Counter, f			; Delay_Counter = Delay_Counter - PW_Counter
	movfw		PW_Counter+1				; high byte of pulse width
	subwfb	Delay_Counter+1, f
	return
; Get.Delay


; compare PW and Period: set Period = PW+1 if Period <= PW
Compare_PW_Per
; compare high bytes
	movfw		PW_Counter+1				; get PW high byte
	subwf		Delay_Counter+1, w		; w = Per - PW
	btfsc		STATUS, Z					; Z=0 if high bytes are NOT equal
	goto		Compare_PW_Per.50			; Z=1: need to compare low bytes
	btfss		STATUS, C					; C=1 if Period > PW
	goto		Compare_PW_Per.200		; must compare low bytes
	return
; here if high bytes are equal - need to compare low bytes
Compare_PW_Per.50
	movfw		PW_Counter					; get low byte
	subwf		Delay_Counter, w			; w = Per - PW
	btfsc		STATUS, Z					; Z=0 if bytes are NOT equql
	goto		Compare_PW_Per.200		; Z=1: goto if Period = PW
	btfsc		STATUS, C					; C=0 if Period < PW
	return
;set Period = PW+1
Compare_PW_Per.200						; here if Period <= PW
	movlw		1
	addwf		PW_Counter, w				; set low byte
	movwf		Delay_Counter				; store it
	clrf		WREG							; set up operate on high byte
	addwfc	PW_Counter+1, w
	movwf		Delay_Counter+1
	return
; Compare_PW_Per


Get.Mode
; exit with the mode in W and Mode
#if DEBUG == 0						; if normal operation
	movlw		ADCH_MODE					; get the Mode channel
	call		Get.Range.Mode				; read the A/D
#else									; if debugging
	movlw		AI_MODE_VALUE				; get the debug mode
#endif ; DEBUG
	movwf		Mode							; store it
	return
; Get.Mode


Get.Range
; exit with
;	Range set using bits 7-9 of A/D reading
;	T1_Preset initialized from the table
;	OSCCON set based on Range
; uses
;	INDF1
#if DEBUG == 0						; if normal operation
	movlw		ADCH_RANGE					; get the Range channel
	call		Get.Range.Mode				; read the A/D
#else									; if debugging
	movlw		AI_RANGE_VALUE				; get the debug mode
#endif ; DEBUG
	movwf		Range							; store it

; Get Timer 1 preset value from T1PresetTable
; get address of the T1 Preset Table into FSR1
	movlw		LOW T1PresetTable
	movwf		FSR1L
	movlw		HIGH T1PresetTable
	movwf		FSR1H
; calculate offset into table
	movfw		Range							; get range value
	lslf		WREG, w						; make table offset value
; calculate table entry address
	addwf		FSR1L, f
	clrf		WREG
	addwfc	FSR1H, f
	bsf		FSR1H, 7						; tell INDF1 to access program memory
; get and save the table value
	moviw		INDF1++						; get the low byte
	movwf		T1_Preset					; save it
	moviw		INDF1++						; get the high byte
	movwf		T1_Preset+1					; save it
; check if CPU frequency & A/D clock need to change
	movlw		5								; ranges 5-7 do not use PLL
	subwf		Range, w						; w = range - 5
	btfsc		WREG, 7						; w is negative if range < 5
	return									; exit for Ranges 0-4
; handle ranges 5-7
	BANKSEL	OSCCON1
	movlw		(0<<NOSC0) | (b'0010'<<NDIV0) ; 32MHZ / 4 = 8MHz
	movwf		OSCCON1
	BANKSEL	ADCON1
	movlw		(1<<ADFM) | (b'001'<<ADCS0) | (0<<ADPREF0)
	movwf		ADCON1						; right justify data, clk=Fosc/8, VRPOS=VDD
	btfsc		STATUS, Z					; Z=1 if range = 5 (leftover from subwf)
	goto		Get.Range.100
; ranges 6-7
	BANKSEL	OSCCON1
	movlw		(0<<NOSC0) | (b'0100'<<NDIV0) ; 32MHZ / 16 = 2MHz
	movwf		OSCCON1
	BANKSEL	ADCON1
	movlw		(1<<ADFM) | (b'000'<<ADCS0) | (0<<ADPREF0)
	movwf		ADCON1						; right justify data, clk=Fosc/2, VRPOS=VDD
Get.Range.100
	BANKSEL	PORTA
	return

T1PresetTable
; values must be negative since T1 only counts up
; values for ranges 1..4 are 1 count less than expected due to instruction
; execution time in Run.T1
; note: value for Range=0 is not used
	dw	0, 0				; 0 PLL on, Fcpu = 32MHz, T1 clock = 1MHz
	dw	LOW -.9,		 HIGH -.9			; 1
	dw	LOW -.99,	 HIGH -.99			; 2
	dw	LOW -.999,	 HIGH -.999			; 3
	dw	LOW -.9999,	 HIGH -.9999		; 4
; Fcpu = 8MHz
	dw	LOW -.25000, HIGH -.25000		; 5 PLL off, Fcpu=8MHz, T1 clock = 250KHz
; Fcpu = 2MHz
	dw	LOW -.62500, HIGH -.62500		; 6 PLL off, Fcpu=2MHz, T1 clock = 62.5KHz
	dw	LOW -.62500, HIGH -.62500		; 7
; Get.Range


Get.Range.Mode
; read the A/D
;	enter with W = A/D channel in appropriate bits and ADON bit set
;	exit with W = 3 MSbits of reading in bits 2..0
	call		AD.Read						; read the A/D
#if AD_USE_8_BITS == 0				; if using Right justified data
	rlf		ADvalue, f					; bit 7 into C
	rlf		ADvalue+1, f				; C into bit 8
	movf		ADvalue+1, w
#else										; if using left justified data
	swapf		ADvalue, w					; swap nibbles
	rrf		WREG							; bits 7..5 are now in 2..0
#endif ; AD_USE_8_BITS
	andlw		0x07							; keep only Mode bits
	return
; Get.Range.Mode


AD.Read
; read the 10 bit A/D
; enter with w = A/D channel in appropriate bits and ADON bit set
; exit with 10 bit value in ADvalue and ADvalue+1
;		and w = low byte value
; about 28 instruction cycles + about 12usec with 32MHz clock
;	total of about 16usec with 32MHz clock
	BANKSEL	ADCON0
	movwf		ADCON0						; set the channel and keep A/D on
; delay for acquisition time - 20 instruction sycles
	movlw		.7
AD.Read.05
	decfsz	WREG, f						; 1 update count, skip next if done
	goto		AD.Read.05					; 2
;
	bsf		ADCON0, ADGO				; start the conversion
AD.Read.10
	btfsc		ADCON0, ADGO				; skip next if done conversion
	goto		AD.Read.10					; wait for done

#if AD_USE_8_BITS == 0				; if using Right justified data
	movf		ADRESH, w					; get the high byte
	andlw		0x03							; clear upper 6 bits
	movwf		ADvalue+1					; save it
	movf		ADRESL, w					; get the low byte
	movwf		ADvalue						; save it
#else										; if using left justified data
	movfw		ADRESH						; get the high byte
	movwf		ADvalue						; save it
	clrf		ADvalue+1
#endif

#if AD_NO_BITS_0_1 == 1
	movlw		~3								; clear bits 0 & 1
	andwf		ADvalue, f
#endif

	BANKSEL	PORTA
	return
; AD.Read


AD.X10
; multiply  ADvalue by 10
;	enter with:
;		value in ADvalue
;	exit with: ADvalue multiplied by 10
;	uses: temp0

; save original value
	movfw		ADvalue						; get low byte
	movwf		temp0							; save it
	movfw		ADvalue+1					; get high byte
	movwf		temp0+1						; save it
; now, multiply by 10
;	*2
	lslf		ADvalue, f
	rlf		ADvalue+1, f
;	*4
	lslf		ADvalue, f
	rlf		ADvalue+1, f
;	*5
	movfw		temp0							; retrieve original low byte
	addwf		ADvalue, f					; update accumulated low byte value
	movfw		temp0+1						; retrieve original high byte value
	addwfc	ADvalue+1, f				; update accumulated high byte value
;	*10
	lslf		ADvalue, f
	rlf		ADvalue+1, f
	return
; AD.X10


;===============================================================================
;									Initialize Hardware
;===============================================================================

init	; initialize the system hardware

; Oscillator - set to 32MHz using 8MHz clock and PLL
	BANKSEL	OSCCON1
	movlw		(0<<NOSC0) | (b'0000'<<NDIV0) ; 32MHZ / 1 = 32MHz
	movwf		OSCCON1
init.10
	btfss		OSCSTAT1, HFOR				; skip next if HF osc ready
	goto		init.10
init.20
	btfss		OSCSTAT1, PLLR				; skip next if PLL ready
	goto		init.20

; Port A
	BANKSEL	PORTA
	clrf		PORTA
	BANKSEL	TRISA
	movlw		TRISA_VAL
	movwf		TRISA
	BANKSEL	ANSELA
	movlw		ANSEL_VAL
	movwf		ANSELA

; A/D convertor
	BANKSEL	ADCON0
	movlw		(1<<ADON)					; A/D on, used later to select A/D channel
	movwf		ADCON0
	movlw		(1<<ADFM) | (b'010'<<ADCS0) | (0<<ADPREF0)
	movwf		ADCON1						; right justify data, clk=Fosc/32, VRPOS=VDD

; Timer 1
	BANKSEL	T1CON
	movlw		(b'00'<<TMR1CS0) | (b'11'<<T1CKPS0) ; source = FOSC/4	
	movwf		T1CON							; PRESCALE = 8:1, Timer off
	clrf		T1GCON						; do not use gate

; Interrupts
	clrf		INTCON						; clear global interrupt enable
	BANKSEL	PIE1
	clrf		PIE1							; clear peripheral interrupt enables
	BANKSEL	PORTA
	return
; init

	END
