;*******************************************************************************
;
;	Filename:		NCO_SteppedToneGen.asm
;	Date:				25 Mar 2017
;	File Version:	1.0
;	Author:			Larry Cicchinelli
;	Description:
;		This program uses a PIC 16F18313 and its Numerically Controlled Oscillator
;			and Timer0 to emulate the Stepped Tone Generator circuit in the Mims book
;		The program emulates two independent one-shot circuits:
;			1) astable-multivibrator
;			2) monostable-multivibrator
;
; IMPORTANT
;	There will be lots of jitter between the triggering edge of the monostable
;	and its output due to its output being turned on via programmed I/O which is
;	NOT in an ISR.
;
;		The following tables shows pin usage
;
;		16F18313 I/O: DEBUG == 0
;			pin 1: VDD
;			pin 2: RA5/AN5					Astable: Output
;			pin 3: RA4/AN4					Astable: A/D input for frquency adjustment
;			pin 4: RA3/~MCLR				Astable: Gate
;			pin 5: RA2/ANA2				Monostable: Trigger
;			pin 6: RA1/ANA1/ICSPCLK		Monostable: A/D input for pulse width
;			pin 7: RA0/ANA0/ISCPDAT		Monostable: Output
;			pin 8: VSS
;
;		16F18313 I/O: DEBUG == 1
;			pin 1: VDD
;			pin 2: RA5/AN5					Monostable: Output
;			pin 3: RA4/AN4					Monostable: A/D input for frquency adjustment
;			pin 4: RA3/~MCLR
;			pin 5: RA2/ANA2				Monostable: Trigger
;			pin 6: RA1/ANA1/ICSPCLK
;			pin 7: RA0/ANA0/ISCPDAT
;			pin 8: VSS
;
; The NCO clock is derived via the CLC1 subsystem in order to get a clock
;	frequency of 8MHz.
; The output frequency of the NCO = (NCO CLock x Increment value / 2^20) /2
;	The "/2" is due to requiring two complete cycles per output period.
;	The formula reduces to FNCO = 3.81 * Increment value with Fosc = 8MHz
;
; Definitions used by the program which are associated with specific I/O pins:
;	DO_FREQ			- astable output
;	AI_FREQ			- astable analog voltage for frequency
;	DI_GATE			- astable gate
;	AI_PW				- monostable analog voltage for pulse width
;	DI_TRIGGER		- monostable trigger
;	DO_PULSE			- monostable output
;
;*******************************************************************************
;*******************************************************************************

; When DEBUG == 1, the monostable pins are moved to the astable pins and there
;	is  no astable operation
; When DEBUG == 0 AND you are debugging, there will be no monostable output
;	since its pins are used for communication with the debugger.
DEBUG				equ	0

FIXED_PW			equ	.0		; used to verify Timer0 clock, DEBUG == 1
									; set to 0 for normal debugging
;
NCO_OFFSET		equ	.14	; 14 counts @ 3.81Hz resolution = 53Hz min NCO freq.

		#include p16F18313.inc

; CONFIG1
; __config 0xDF8F
 __CONFIG _CONFIG1, _FEXTOSC_OFF & _RSTOSC_HFINT32 & _CLKOUTEN_OFF & _CSWEN_ON & _FCMEN_OFF
; CONFIG2
; __config 0xF732
 __CONFIG _CONFIG2, _MCLRE_OFF & _PWRTE_OFF & _WDTE_OFF & _LPBOREN_OFF & _BOREN_OFF & _BORV_LOW & _PPS1WAY_OFF & _STVREN_ON & _DEBUG_OFF
; CONFIG3
; __config 0x3
 __CONFIG _CONFIG3, _WRT_OFF & _LVP_OFF
; CONFIG4
; __config 0x3
 __CONFIG _CONFIG4, _CP_OFF & _CPD_OFF


#if DEBUG == 0
DO_FREQ			equ	RA5
AI_FREQ			equ	RA4
DI_GATE			equ	RA3
DI_TRIGGER		equ	RA2
AI_PW				equ	RA1
DO_PULSE			equ	RA0

DO_FREQ_PPS		equ	RA5PPS
DO_PULSE_PPS	equ	RA0PPS

AI_MASK			equ	( 1<<AI_FREQ ) | ( 1<<AI_PW )
DI_MASK			equ	( 1<<DI_TRIGGER ) | ( 1<<DI_GATE )
TRISA_VAL		equ	( AI_MASK | DI_MASK )
ANSEL_VAL		equ	AI_MASK

ADCH_FREQ		equ	( AI_FREQ<<CHS0 ) | (1<<ADON )
ADCH_PW			equ	( AI_PW<<CHS0 ) | (1<<ADON )

EDGE_REG equ	IOCAP			; IOCAP = trigger on positive edge of input
;EDGE_REG equ	IOCAN			; IOCAN = trigger on negative edge of input

FIXED_PW			equ	0		; just in case FIXED_PW was left != 0

#else

DO_PULSE			equ	RA5
AI_PW				equ	RA4
DI_TRIGGER		equ	RA2

DO_PULSE_PPS	equ	RA5PPS

AI_MASK			equ	( 1<<AI_PW )
DI_MASK			equ	( 1<<DI_TRIGGER )
TRISA_VAL		equ	( AI_MASK | DI_MASK )
ANSEL_VAL		equ	AI_MASK

ADCH_FREQ		equ	0		; needed when Debug = 1
ADCH_PW			equ	( AI_PW<<CHS0 ) | (1<<ADON )

EDGE_REG equ	IOCAP			; IOCAP = trigger on positive edge of input
;EDGE_REG equ	IOCAN			; IOCAN = trigger on negative edge of input

#endif

;*******************************************************************************
; RAM
;*******************************************************************************

Access_RAM		udata_shr	0x70		; 16 bytes starting at address 0x70
NCO_Accum		res	3
PWvalue			res	2
TempPW			res	2					; used only when FIXED_PW != 0
												; allows value to be changed while debugging

;*******************************************************************************
; Vectors
;*******************************************************************************

	org	0
	goto	Main

	org	0x04								; interrupt vector
; check for monostable trigger interrupt
	BANKSEL	PIR0
	btfss		PIR0, IOCIF
	goto		ISR_50
; here if monostable trigger interrupt
	BANKSEL	LATA
	bsf		LATA, DO_PULSE				; turn on output pulse
	BANKSEL	IOCAF
	bcf		IOCAF, DI_TRIGGER			; clear the edge detect flag
	bcf		EDGE_REG, DI_TRIGGER		; disable edge detect while pulse is active
	BANKSEL	T0CON0
	movfw		PWvalue+1					; get high byte of PW
	movwf		TMR0H							; store it in buffer
	movfw		PWvalue						; get low byte
	movwf		TMR0L							; store it - also write high byte to register
												; also clears the prescale counter
	bsf		T0CON0, T0EN				; enable the timer
	retfie

ISR_50
; Timer 0 overflow
	BANKSEL	LATA
	bcf		LATA, DO_PULSE				; turn off output pulse
	BANKSEL	T0CON0
	bcf		T0CON0, T0EN				; disable Timer 0
	bcf		PIR0, TMR0IF				; clear interrupt flag
	BANKSEL	EDGE_REG
	bsf		EDGE_REG, DI_TRIGGER		; enable edge detect
	retfie


;*******************************************************************************
; MAIN PROGRAM
;*******************************************************************************

MAIN_PROG CODE                      ; let linker place main program

Main
	call		init							; initialize the system hardware
	BANKSEL	EDGE_REG
	bsf		EDGE_REG, DI_TRIGGER		; enable edge detect
#if FIXED_PW != 0
	movlw		LOW FIXED_PW
	movwf		TempPW
	movlw		HIGH FIXED_PW
	movwf		TempPW+1
#endif


;*******************************************************************************
; Main Program Loop
;*******************************************************************************
Main.10
;	BANKSEL	TRISA
;	bcf		TRISA, RA2
;	BANKSEL	LATA
;	movlw		( 1<<RA2 )
Main.11
;	xorwf		LATA
;	goto		Main.11
	
	call		Set.NCO						; set the NCO frequency
	call		Get.PW						; get the pulse width
	goto		Main.10


Set.NCO
	movlw		ADCH_FREQ
	call		AD.Read_R
; add offset value
	movlw		NCO_OFFSET					; min NCO frequency
	addwf		ADRESL, f
	clrf		WREG
	addwfc	ADRESH, f
; store A/D value into NCO accumulator registers
	movfw		ADRESL
	movwf		NCO_Accum					; store Low byte
	movfw		ADRESH
	movwf		NCO_Accum+1					; store High byte
	clrf		NCO_Accum+2					; ensure U byte is clear
; test Gate signal
	BANKSEL	NCO1CON
	movfw		NCO1CON						; get current NCO control state
	bsf		WREG, N1EN 					; assume NCO enabled
#if DEBUG == 0
	BANKSEL	PORTA
	btfss		PORTA, DI_GATE				; skip next if Gate enabled
	bcf		WREG, N1EN					; disable NCO
	BANKSEL	NCO1CON
#endif
	movwf		NCO1CON						; use Gate value to enable or disable NCO
; put the results into the increment registers
	movfw		NCO_Accum+2					; get U byte
	movwf		NCO1INCU
	movfw		NCO_Accum+1					; get H byte
	movwf		NCO1INCH
	movfw		NCO_Accum+0					; get L byte
	movwf		NCO1INCL						; this writes all 3 bytes to the register
	return
; Get.Frequency

Get.PW
#if FIXED_PW == 0				; get reading from A/D
	movlw		ADCH_PW
	call		AD.Read_L
	movfw		ADRESL
	movwf		TempPW						; save low byte
	movfw		ADRESH
	movwf		TempPW+1						; and high byte
#else								; use TempPW from FIXED_PW
	movfw		TempPW
	movwf		PWvalue
	movfw		TempPW+1
	movwf		PWvalue+1
#endif
; negate the reading by complementing and incrementing
	comf		TempPW, f
	comf		TempPW+1, f
	movlw		1								; must use add instruction
	addwf		TempPW, f					;	to generate carry if appropriate
	clrf		WREG							; must use W: only way to add with carry
	addwfc	TempPW+1, f
; now copy to PWvalue
	BANKSEL	PIE0
	bcf		PIE0, IOCIE					; disable IOC interrupts
	movfw		TempPW
	movwf		PWvalue
	movfw		TempPW+1
	movwf		PWvalue+1
	bsf		PIE0, IOCIE					; disable IOC interrupts
	return
; Get.PW


AD.Read_R
	BANKSEL	ADCON1
	bsf		ADCON1, ADFM				; right justified data
	goto		AD.Read
AD.Read_L
	BANKSEL	ADCON1
	bcf		ADCON1, ADFM				; left justified data

AD.Read
; read the 10 bit A/D
; enter with w = A/D channel in appropriate bits and ADON bit set
; exit with 10 bit value in AD registers
;		w = low byte value
;		BANKSEL = ADCON0
; low byte will be forced to 1 if both bytes = 0

	movwf		ADCON0						; set the channel and keep A/D on
; delay for acquisition time - 3 instruction cycles/loop
	movlw		.50
AD.Read.05
	decfsz	WREG, f						; 1 update count, skip next if done
	goto		AD.Read.05					; 2
	bsf		ADCON0, ADGO				; start the conversion
AD.Read.10
	btfsc		ADCON0, ADGO				; skip next if done conversion
	goto		AD.Read.10					; wait for done
; make value = 1 if reading is 0
	movfw		ADRESL						; get the low byte
	iorwf		ADRESH, w					; Z = 1 if both registers are 0
	btfsc		STATUS, Z					; skip next if not 0
	incf		ADRESL,f						; force value = 1
	return
; AD.Read

;===============================================================================
;									Initialize Hardware
;===============================================================================

init	; initialize the system hardware

; Oscillator - set to 32MHz using 16MHz clock with 2xPLL
;	= 8MHz instruction rate = 125ns instruction execution time
	BANKSEL	OSCCON1
	movlw		1<<HFOEN						; enable the HF internal oscillator
	movwf		OSCEN
	movlw		7								; select 32MHz
	movwf		OSCFRQ
	movlw		(b'000'<<NOSC0) | (b'0000'<<NDIV0)
	movwf		OSCCON1						; HFINTOSC with 2x PLL (32 MHz), DIV = 1
												;		Fosc = 32MHz
												; OSCCON2 is read-only
	clrf		OSCCON3
init.10
	btfss		OSCCON3, ORDY				; wait for oscillator ready
	goto		init.10

; Port A
	BANKSEL	LATA
	clrf		LATA
	BANKSEL	TRISA
	movlw		TRISA_VAL
	movwf		TRISA							; set analog and digital input pins
	BANKSEL	ANSELA
	movlw		ANSEL_VAL					; set analog input pin(s)
	movwf		ANSELA
	BANKSEL	INLVLA
	movlw		0xFF							; set up to
	movwf		INLVLA						;		make all inputs Schmitt Trigger CMOS

; Peripheral Pin Select
	BANKSEL	DO_PULSE_PPS
#if DEBUG == 0
	movlw		b'11101'						; NCO output
	movwf		DO_FREQ_PPS
#endif
	clrf		DO_PULSE_PPS				; PORTA monostable output

; A/D convertor
	BANKSEL	ADCON0
	movlw		(1<<ADON)					; A/D on, used later to select A/D channel
	movwf		ADCON0
	movlw		(b'111'<<ADCS0 )
	movwf		ADCON1						; clk = ADCRC - about 1MHz
	clrf		ADACT							; no auto conversion trigger

; Timer 0
	BANKSEL	T0CON0
	movlw		( 0<<T0EN ) | ( 1<<T016BIT )
	movwf		T0CON0						; timer disabled, 16 bit, postscale = /1
		; since T0 is used as a programmable timer, you CANNOT use the postscaler
		; unless you want to divide by values > 65536
	movlw		( b'011'<<T0CS0 ) | ( 4<<T0CKPS0 )
	movwf		T0CON1						; clk source = HFINTSC (32MHz),
												; prescale = 2^4 = 16, T0clk = 2MHz
; Timer 2		; used to drive CLC0
	BANKSEL	T2CON
	movlw		( 1<<TMR2ON )				; Timer 2 on, Postscaler = 1, Prescaler = 1
	movwf		T2CON
	movlw		.1								; set period to Fosc/4 = 8MHz
	movwf		PR2

; CLC 1
	BANKSEL	CLC1CON
	movlw		( 1<<LC1EN ) | ( b'010'<<LC1MODE0 )
	movwf		CLC1CON						; enable CLC1 as 4 input AND
	movlw		0x0F							; all four Gate outputs inverted
	movwf		CLC1POL						;		output of logic cell not inverted
	movlw		.26							; input 0 is timer 2 period match
	movwf		CLC1SEL0
	movwf		CLC1SEL1
	movwf		CLC1SEL2
	movwf		CLC1SEL3
	movlw		0x55							; use inverted output from each DATAy
	movwf		CLC1GLS0
	clrf		CLC1GLS1
	clrf		CLC1GLS2
	clrf		CLC1GLS3

; Interrupts
	movlw		( 1<<GIE ) | ( 1<<PEIE )
	movwf		INTCON						; enable interrupts and peripheral interrupts
	BANKSEL	PIE0
	movlw		( 1<<TMR0IE ) | ( 1<<IOCIE ) ; enable Timer 1 & IOC interrupts
	movwf		PIE0

; NCO
	BANKSEL	NCO1ACCL
	movlw		b'10'							; use CLC1OUT for clock
	movwf		NCO1CLK
	movlw		(1<<N1EN) | (0<<N1PFM)	; enable NCO1, set to Fixed Duty Cycle (50%)
	movwf		NCO1CON
	return
; init

	END
