;*******************************************************************************
;
;	Filename:		NCO_SoundEffects.asm
;	Date:				8 Mar 2017
;	File Version:	1.0
;	Author:			Larry Cicchinelli
;	Description:
;		This program uses a PIC 16F18313 and its Numerically Controlled Oscillator
;			It simulates the Sound Effects Generator in the Mimms book
;		The program is based on NCO-3.asm of Part 3 of the 555 series.			
;
;		16F18313 I/O:
;			pin 1: VDD
;			pin 2: RA5/ANA5				Output
;			pin 3: RA4/ANA4				Ramp mode
;			pin 4: RA3/~MCLR				On-Off control: 0=on
;			pin 5: RA2/ANA2				Low frequency limit
;			pin 6: RA1/ANA1/ICSPCLK		High frequency limit
;			pin 7: RA0/ANA0/ISCPDAT		Ramp speed
;			pin 8: VSS
;
; Programming Notes:
;		An offset value is added to the low frequency A/D reading in order to
;			establish a minimum value for the low frequency.
;		The low frequency (after adjustment) is added to the high frequency
;			A/D reading.  This makes the High Frequency an offset from the Low
;			Frequency.
;		Both frequencies use only the upper 8 bits of the A/D reading in order to
;			limit the maximum values to <4KHz
;		Parameter changes are only applied at the completion of a complete ramp
;			cycle.
;		If Ramp Type is set to 3 it will be forced to 2.  The logic does not
;			allow for starting a bidirectional ramp by ramping down.
;			This can be changed but requires some more logic.
;		Due to the way in which frequencies are compared to determine the end of a
;			ramp, setting the high frequency to 0 will cause a constant tone at the
;			low frequency independant of Ramp mode.
;
; The frequency values and delay times are based on the CPU clock = 32MHz.
;
; The square wave frequency = (NCO Clock Frequency * Increment value /2^20)/2
;		The final /2 is due to the output being fixed as a square wave - it takes
;		two complete cycles of the NCO Accumulater to generate one period.
;	Since the program uses HFINTSC as the NCO clock, the formula reduces to:
;		Freq = ( 32MHz * Increment value / 2^20) / 2 = 15.26 * Increment value
;	The Increment value is initialized to either the Low or High Frequency value
;		and then incremented or decremented by the selected Ramp_Incr_Table value.
;
; Ramp Type is derived from an analog value that uses only the upper two bits of
;		the A/D reading: up(0), down(1), up-down(2,3)
;
; Ramp speed is comprised of two componants:
;		NCO accumulator increment (Ramp_Incr_Table) value
;		Delay time after each frequency change
;	These two values are determined from a single analog input: ANA0
;		The program uses only the six most significant bits of the A/D reading.
;		Bits 9..7 are the accumulator increment value
;		Bits 6..4 are the delay value
;		Both sets of three bits are used to access values in look-up tables.
;		As the pot is rotated, the delay value is changing the fastest.  The
;			voltage step between each pair of values is about 78mv with a 5V supply.
;			The center of each step is at an offset of 39mv.
;		To calculate the voltage for a pair of table entries:
;			Let delay table entry nbr = 3 and increment table entry nbr = 5
;				V = (3*.078) + (5*.625) +.039 = 3.398V
;
; Definitions used by the program which are associated with specific I/O pins:
;	DO_PULSE			- digital output pulse pin
;	DI_ENABLE		- digital input for enabling the output
;	AI_LOW_FREQ		- analog input pin for low ramp frequency
;	AI_HIGH_FREQ	- analog input pin for high ramp frequency
;	AI_RAMP_SPEED	- analog input pin for ramp speed
;	AI_RAMP_MODE	- analog input pin for ramp mode
;
;*******************************************************************************
;*******************************************************************************

		#include p16F18313.inc

DEBUG				equ	0
DEBUG_RAMP_SPEED equ	b'100000'
								; Ramp_Incr_Table entry nbr = (DEBUG_RAMP_SPEED>>3) & 7
								; Ramp_Delay_Table entry nbr = DEBUG_RAMP_SPEED & 7
DEBUG_HIGH_FREQ equ	.50 ;  may not excede 255 (about 3.8KHz)


; CONFIG1
; __config 0xDF8F
 __CONFIG _CONFIG1, _FEXTOSC_OFF & _RSTOSC_HFINT32 & _CLKOUTEN_OFF & _CSWEN_ON & _FCMEN_OFF
; CONFIG2
; __config 0xF732
 __CONFIG _CONFIG2, _MCLRE_OFF & _PWRTE_ON & _WDTE_OFF & _LPBOREN_OFF & _BOREN_OFF & _BORV_LOW & _PPS1WAY_OFF & _STVREN_ON & _DEBUG_OFF
; CONFIG3
; __config 0x3
 __CONFIG _CONFIG3, _WRT_OFF & _LVP_OFF
; CONFIG4
; __config 0x3
 __CONFIG _CONFIG4, _CP_OFF & _CPD_OFF


DO_PULSE			equ	RA5
DO_PPS			equ	RA5PPS
AI_RAMP_MODE	equ	RA4
DI_ENABLE		equ	RA3
AI_LOW_FREQ		equ	RA2
AI_HIGH_FREQ	equ	RA1
AI_RAMP_SPEED	equ	RA0
			
AI_MASK			equ	(1<<AI_RAMP_MODE) | (1<<AI_LOW_FREQ) | (1<<AI_HIGH_FREQ) | (1<<AI_RAMP_SPEED)
DI_MASK			equ	( 1 << DI_ENABLE )
TRISA_VAL		equ	( AI_MASK | DI_MASK )

ADCH_RAMP_MODE	equ	( AI_RAMP_MODE<<CHS0 )  | ( 1<<ADON )
ADCH_HIGH_FREQ	equ	( AI_HIGH_FREQ<<CHS0 )  | ( 1<<ADON )
ADCH_LOW_FREQ	equ	( AI_LOW_FREQ<<CHS0 )   | ( 1<<ADON )
ADCH_RAMP_SPEED equ	( AI_RAMP_SPEED<<CHS0 ) | ( 1<<ADON )

RIGHT_JUSTIFY	equ	( 1<<ADFM ) | ( b'010'<<ADCS0 )
LEFT_JUSTIFY	equ	( 0<<ADFM ) | ( b'010'<<ADCS0 )

LOW_FREQ_OFFSET equ	3		; limit lowest frequency to 3*15 = 45Hz

; these three definitions are used in Ramp_mode
RAMP_DIR			equ	0		; bit 0: 0=up, 1=down
RAMP_UP_DOWN	equ	1		; bit 1: 0=single direction ramp, 1=bidirectional ramp
RAMP_STATE		equ	2		; bit 2: 0=not ramping, 1 = ramping

;*******************************************************************************
; RAM
;*******************************************************************************

Access_RAM		udata_shr	0x70		; 16 bytes starting at address 0x70
NCO_Inc			res	3
Low_Freq			res	2
High_Freq		res	2
Ramp_Incr		res	1			; bits 9..7 of A/D reading
Ramp_Delay		res	1			; bits 6..4 of A/D reading
Ramp_Mode		res	1
MS_timer			res	2

#if DEBUG == 1						; so values can be changed during debug
Dbg_Ramp_speed	res	1
Dbg_High_Freq	res	1
#endif

;*******************************************************************************
; Startup, Interrupt Vectors, and constant tables
;*******************************************************************************

	org	0
	goto	Main

	org	0x04								; interrupt vector
ISR.Exit
	retfie

; These tables are WORD values (14 bits) but the program uses only 8 bits.
;	In order to access values >8 bits the table method must be changed as well as
;	the program code which accesses them.
; Both tables must contain exactly 8 values.
; Both tables are set up from the slowest ramp to the fastest.
; The tables must be accessed via INDFn where FSRnH must have bit 7 set in order
;	to read from program memory.
Ramp_Delay_Table
	dw		.100, .50, .35, .20, .10, 5, 2, 1
Ramp_Incr_Table
	dw		1, 2, 4, 8, .10, .15, .20, .25	; step size = value * 15Hz

;*******************************************************************************
; MAIN PROGRAM
;*******************************************************************************

MAIN_PROG CODE                      ; let linker place main program

Main
	call		init							; initialize the system hardware
	clrf		Ramp_Mode

#if DEBUG == 1
	movlw		DEBUG_RAMP_SPEED
	movwf		Dbg_Ramp_speed
	movlw		DEBUG_HIGH_FREQ
	movwf		Dbg_High_Freq
#endif

;*******************************************************************************
; Main Program Loop
;*******************************************************************************

Main.10
	btfsc		Ramp_Mode, RAMP_STATE	; skip next if not ramping
	goto		Main.20
; here only if not ramping
	call		Check_On_Off
	call		Get.RampType
	call		Get.Freqs
	call		Get.RampSpeed
;
Main.20
	call		Set.NCO						; operate on NCO
	goto		Main.10


Check_On_Off
#if DEBUG == 0
	BANKSEL	PORTA
	btfsc		PORTA, DI_ENABLE			; skip next if on
	goto		Turn_Off
#endif
; here if switch is on
	BANKSEL	DO_PPS
	movlw		b'11101'						; NCO output
	movwf		DO_PPS
	return
Turn_Off
	BANKSEL	DO_PPS
	clrf		DO_PPS
	BANKSEL	LATA
	clrf		LATA							; make output low
	return
; Check_On_Off


Get.RampType
	movlw		ADCH_RAMP_MODE
	call		AD.Read.R
; use only two MS digits
	movfw		ADRESH
	movwf		Ramp_Mode
	btfsc		Ramp_Mode, RAMP_UP_DOWN	; skip nexit if not ramping up and down
	bcf		Ramp_Mode, RAMP_DIR		; force start with ramp down
	return
; Get.RampType


Get.Freqs
; use only upper 8 bits to limit frequency value to <4KHz
; Low Frequency
	movlw		ADCH_LOW_FREQ
	call		AD.Read.L
;
	movfw		ADRESH						; get upper 8 bits
	movwf		Low_Freq
	clrf		Low_Freq+1
; add offset to force minimum low frequency
	movlw		LOW_FREQ_OFFSET			; get offset
	addwf		Low_Freq, f
	clrf		WREG
	addwfc	Low_Freq+1, f
; High Frequency
#if DEBUG == 0
	movlw		ADCH_HIGH_FREQ
	call		AD.Read.L
	movfw		ADRESH
	movwf		High_Freq
#else
	movfw		Dbg_High_Freq
	movwf		High_Freq
#endif
; add Low Frequency so that High Frequency is an offset from Low Frequency
	clrf		High_Freq+1
	movfw		Low_Freq
	addwf		High_Freq, f
	movfw		Low_Freq+1
	addwfc	High_Freq+1, f
	return
; Get.Freqs


Get.RampSpeed
; Dbg_Ramp_Speed is 1 byte
#if DEBUG == 0
	movlw		ADCH_RAMP_SPEED
	call		AD.Read.L
	movfw		ADRESH						; get upper 8 bits
	lsrf		ADRESH, f					; keep only
	lsrf		ADRESH, w					;		upper 6 bits
#else
	movfw		Dbg_Ramp_speed
#endif
; A/D bits 0-2 are Ramp_Delay
	movwf		Ramp_Incr					; save
	andlw		0x07							; set up to access Ramp_Delay_Table
	addlw		LOW Ramp_Delay_Table
	movwf		FSR0L
	movlw		0x80							; tables are
	movwf		FSR0H							;		in low memory
	movfw		INDF0							; get the table entry
	movwf		Ramp_Delay					; save it
; A/D bits 3-5 are Ramp_Incr
	lsrf		Ramp_Incr, w				; move
	lsrf		WREG, f						;	into
	lsrf		WREG, f						;		bits 0-2
	addlw		LOW Ramp_Incr_Table
	movwf		FSR0L							; set up to access Ramp_Incr_Table
	movfw		INDF0							; get the table entry
	movwf		Ramp_Incr					; save it
	return
; Get.RampSpeed


Set.NCO
	btfsc		Ramp_Mode, RAMP_STATE	; skip next if not ramping
	goto		Set.NCO.100					; do ramp

Set.NCO.05 
; here if state = 0 = not ramping (yet)
	clrf		NCO_Inc+2					; ensure initial Upper byte is 0
	btfss		Ramp_Mode, RAMP_DIR		; skip next RAMP_DIR = 1 = ramp down
	goto		Set.NCO.10
; here if Ramp down: set the high frequency
	movfw		High_Freq
	movwf		NCO_Inc						; store Low byte for processing
	movfw		High_Freq+1
	movwf		NCO_Inc+1					; store High byte for processing
	goto		Set.NCO.20

Set.NCO.10
; here if Ramp up: set the low frequency
	movfw		Low_Freq
	movwf		NCO_Inc						; store Low byte for processing
	movfw		Low_Freq+1
	movwf		NCO_Inc+1					; store High byte for processing

; put the results into the increment registers
Set.NCO.20
	bsf		Ramp_Mode, RAMP_STATE	; show ramp started
Set.NCO.25
	BANKSEL	NCO1INCU						; NCO1ACCL @ 0x498
	movfw		NCO_Inc+2					; get U byte
	movwf		NCO1INCU
	movfw		NCO_Inc+1					; get H byte
	movwf		NCO1INCH
	movfw		NCO_Inc+0					; get L byte
	movwf		NCO1INCL						; this writes all 3 bytes to the register

; delay a little after setting NCO register
	movfw		Ramp_Delay					; delay value in milliseconds
	call		Delay.Wms
	return

; here if ramping
Set.NCO.100
	btfsc		Ramp_Mode, RAMP_DIR		; skip next RAMP_DIR = 0 = ramp up
	goto		Set.NCO.120
; here if ramping up
	movfw		Ramp_Incr					; update to new frequency
	addwf		NCO_Inc+0, f
	clrw
	addwfc	NCO_Inc+1, f
	addwfc	NCO_Inc+2, f
; check if new frequency is still < High_Freq
	movlw		High_Freq
	movwf		FSR0L
	call		Compare_Frequencies
	btfss		STATUS, C
	goto		Set.NCO.200					; goto if done ramp up
	goto		Set.NCO.25

Set.NCO.120
; here if ramping down
	movfw		Ramp_Incr
	subwf		NCO_Inc+0, f
	clrw										; Ramp_Incr is one byte
	subwfb	NCO_Inc+1, f
	subwfb	NCO_Inc+2, f
	btfsc		NCO_Inc+2, 7				; skip next if NCO_Inc is positive
	goto		Set.NCO.200
; check if new frequency is still > Low_Freq
	movlw		Low_Freq
	movwf		FSR0L
	call		Compare_Frequencies
	btfss		STATUS, C
	goto		Set.NCO.25					; goto if NOT done ramp down

Set.NCO.200
; here if done ramping
	bcf		Ramp_Mode, RAMP_STATE	; show ramp done
	btfss		Ramp_Mode, RAMP_UP_DOWN	; skip next if ramping up and down
	return
	movlw		(1<<RAMP_DIR)				; set up to
	xorwf		Ramp_Mode, f				; change ramp direction
	btfss		Ramp_Mode, RAMP_DIR		; skip next if done only 1st half of ramp
	return
	goto		Set.NCO						; do 2nd half of ramp up-down
; Set.NCO


Compare_Frequencies
; compare the value pointed to by FSR0 with NCO_Inc
; the algorithm performs (FSR0) - NCO_Inc
; exit with C = 1 if (FSR0) >= NCO_Inc,  else C = 0
	clrf		FSR0H							; force Bank 0 RAM and Common RAM
	movfw		NCO_Inc						; subtract low bytes
	subwf		INDF0, w
	incf		FSR0, f							; point to high byte
	movfw		NCO_Inc+1					; subtract high bytes
	subwfb	INDF0, w
	return
; Compare_Frequencies


; read the 10 bit A/D
; enter with w = A/D channel in appropriate bits and ADON bit set
; exit with 10 bit value in AD registers
;		BANKSEL = ADCON0
;
AD.Read.R
	BANKSEL	ADCON0						; ADRESL @ 0x09B
	movwf		ADCON0						; set the channel and keep A/D on
	movlw		RIGHT_JUSTIFY				; set to right justify data
	movwf		ADCON1
	goto		AD.Read
AD.Read.L
	BANKSEL	ADCON0						; ADRESL @ 0x09B
	movwf		ADCON0						; set the channel and keep A/D on
	movlw		LEFT_JUSTIFY				; set to left justify data
	movwf		ADCON1
	
AD.Read
; delay for acquisition time - 20 instruction sycles
	movlw		.10
AD.Read.05
	decfsz	WREG, f						; 1 update count, skip next if done
	goto		AD.Read.05					; 2
	bsf		ADCON0, ADGO				; start the conversion
AD.Read.10
	btfsc		ADCON0, ADGO				; skip next if done conversion
	goto		AD.Read.10					; wait for done
	return
; AD.Read


Delay.Wms
; Execute a delay of W milliseconds by executing a specific number of instructions.
; The execution time is predicated on a 32MHz CPU clock.
; register and memory usege:
;	enter: W = number of ms: 1 <= W <= 255 (W is not tested for 0!)
;	uses: MS_timer
;	exit: W = 0
; notes:
;	the first number in each comment is the number of I-cycles for the instruction
; since each I-cycle takes 125ns ==> there are 8000 I-cycles in 1ms

; NOTE: this function does NOT take into account the set up time of any of the
; loops.  The set up times will add about <2us to the total execution time.

	movwf		MS_timer						; 1 save for outer loop
; loop 1
Delay.Wms.10
	movlw		.100							; 100 * 80 = 8000
	movwf		MS_timer+1

; loop 2
Delay.Wms.20

; loop 3
; this loop takes 1 + (N-1)*8 + 7 = N*8 I-cycles = 80
	movlw		.10							; 1 inner loop counter
Delay.Wms.50
	call		Delay.Wms.90				; 4 call + return
	decf		WREG, w						; 1 update counter
	btfss		STATUS, Z					; 1(2) skip next if W=0
	goto		Delay.Wms.50				; 2
; end loop 3 - 80 I-cycles

	decf		MS_timer+1, f
	btfss		STATUS, Z
	goto		Delay.Wms.20
; end loop 2 - 8000 I-cycles = 1ms

	decf		MS_timer, f
	btfss		STATUS, Z
	goto		Delay.Wms.10
; end loop 1 - W milliseconds

Delay.Wms.90
	return
; Delay.Wus


;===============================================================================
;									Initialize Hardware
;===============================================================================

init	; initialize the system hardware

; Oscillator - set to 32MHz using 8MHz clock and PLL
	BANKSEL	OSCCON1
	movlw		(b'000'<<NOSC0) | (b'0000'<<NDIV0)
	movwf		OSCCON1						; HFINTOSC with 2x PLL (32 MHz), DIV = 1
												; OSCCON2 is read-only
	clrf		OSCCON3
	movlw		1<<HFOEN						; enable the HF internal oscillator
	movwf		OSCEN
	movlw		7								; select 32MHz
	movwf		OSCFRQ
init.10
	btfss		OSCCON3, NOSCR				; wait for oscillator ready
	goto		init.10

; Port A
	BANKSEL	PORTA
	clrf		PORTA
	BANKSEL	TRISA
	movlw		TRISA_VAL
	movwf		TRISA							; set analog and digital input pins
	BANKSEL	ANSELA
	movlw		TRISA							; set analog input pins
	movwf		ANSELA
	BANKSEL	INLVLA
	movlw		0xFF							; set up to
	movwf		INLVLA						;		make all inputs Schmitt Trigger CMOS

; Peripheral Pin Select
	BANKSEL	RA5PPS
	movlw		b'11101'						; NCO output to RA5
	movwf		RA5PPS

; A/D convertor
	BANKSEL	ADCON0
	movlw		(1<<ADON)					; A/D on, used later to select A/D channel
	movwf		ADCON0
	movlw		RIGHT_JUSTIFY				; clk=Fosc/32, VRPOS=VDD, right justify data
	movwf		ADCON1
	clrf		ADACT							; no auto conversion trigger

; Interrupts
	clrf		INTCON						; clear global interrupt enable
	BANKSEL	PIE0
	clrf		PIE0							; clear peripheral interrupt enables
	clrf		PIE1							; clear peripheral interrupt enables

; Timer 1
	BANKSEL	T1CON
	movlw		(b'00'<<TMR1CS0) | (b'11'<<T1CKPS0) | (1<<T1SYNC) ; source = FOSC/4	
	movwf		T1CON							; PRESCALE = 8:1, no sync, Timer off
	clrf		T1GCON						; do not use gate

; NCO
	BANKSEL	NCO1ACCL
	movlw		b'00'							; use HFINTOSC (32MHz) for clock
	movwf		NCO1CLK
	movlw		(1<<N1EN) | (0<<N1PFM)	; enable NCO1, set to Fixed Duty Cycle (50%)
	movwf		NCO1CON
	return
; init

	END
