/* Exercise_7.asm

Task: Transmit a message from Serial Port 0 and receive a response terminated
with a carriage return.  Use the receive (Sample 3) and transmit ISRs from the
ISR chapter.

First it would be a good idea to outline the entire process:
 1.Define the memory configuration values
 2.Define any other constants that may be needed for this task
(a)Serial port values
(b)Other values
 3.Interrupt Vectors
 4.Any required constants
 5.Any required RAM
(a)pointers
(b)buffers
(c)counters
 6.Program code
(a)Set the startup vector
(b)Initialize the Stack Pointer
(c)Set up the serial port registers
(d)Transmit a request message
 i.write a separate function for transmitting a message
(e)Get the response
 i.write a separate function for receiving a response
(f)wait forever

The outline statements above will be inserted into the program code below as
comments.  The highest level outline sections will be indicated with a full
line of asterisks.  The additional descriptions for the initialization
statements will not be repeated here.
*/

#nprint	PR_PROC_DEFS     ;print everything BUT the standard processor definitions

;******************************* Memory Definitions *********************************

;	define the RAM values
#define  RAMSTART    0x40000				; starting address of RAM
#define  RAMSIZE     0x40000				;		and its size
#define  RAMEND      RAMSTART+ RAMSIZE	; calculate the end address of the RAM

;	define the stack size and  location
#define  STACKSIZE   0x1000       ; size of stack
#define  STACK_TOP   RAMEND - STACKSIZE ; top of the stack

;	define the values for the Address Pointer
#address NEXT_ROM    0x400					; starting address to put any constants
#address NEXT_PC     0x1000	     ; starting address of the program
#address NEXT_RAM    RAMSTART    ; start of the RAM

;********************************************************************************
; Define any other constants that may be needed for this task
; ***	Serial port values
#define	RXIPL_MASK     0x07				; mask for the receive IPL
#define	RX0_IPL        0x01				; IPL for SP0 Rx = 1
#define	TXIPL_MASK     0x70				; mask for the transmit IPL
#define	TX0_IPL        0x10				; IPL for SP0 Tx = 1
#define	TX_BUSY        0x80				; transmitter busy bit
#define	SP0_BAUD       19200			; baud rate for SP0
#define	SP0RXBUF_SIZE  100   			; define the number of bytes
                                 ;		for the serial port Rx buffer

#define	ASCII_CR	      0x0D				; ASCII CR value		

;********************************************************************************
; Any required constants

#address	NEXT_ROM
Message_1:	#alloc.s "Enter a numeric integer string terminated with a carriage return\n\r\0"

;********************************************************************************
; Any required RAM

#address	 NEXT_RAM								   ; set the Address Pointer to the next available RAM

;	pointers, buffers and counters

; set up the RAM required for Serial Port 0

BaudConstant:	#alloc.4							; CPU_FREQUENCY/16 needed for calculating USART
                                 ; divisor

SP0RxBuffer:      #allocb.1	SP0RXBUF_SIZE		; allocate the serial port 0 Rx buffer
SP0RxByteCount:   #alloc.2       ; serial port 0 Rx byte count
SP0RxBufferAddr:  #alloc.4					; serial port 0 Rx buffer pointer

SP0TxByteCount:   #alloc.2					; serial port 0 Tx byte count
SP0TxBufferAddr:  #alloc.4					; serial port 0 Tx buffer pointer

;********************************************************************************
; Program code

#address	NEXT_PC              ; set the Address Pointer to the start of the program

#vector     VECTOR_RTC        ; dummy ISR required by the assembler
   reti

#vector	VECTOR_START         ; set up the start up vector to point to the first
                             ; instruction

;****************************** begin initialization code ******************************
; first set up the stack pointer
   copy.4		#RAMEND, SP	      ; set up the stack pointer to the top of the RAM

; calculate the base frequency value needed by the UARTs
   iocopy.4 CPU_FREQUENCY, r0 ; get the cpu frequency
   div.4    #16, r0           ; base divisor for the USART baud rate calculations
   copy.4   r0, BaudConstant  ; save for other USARTs if necessary

; set up Parallel Port 0.0 for Serial Port 0 Tx and Parallel Port 0.1 as input.
   iocopy.1 #4, PP0.0CFG	     ; PP0.0 = SP0 Tx
   clr.1    PP0.1CFG          ; PP0.1 = input

; Set up the serial port registers.  Since it is possible for the program to
;  restart without a reset, it is usually a good idea to set up all the registers. 

   div.4    #SP0_BAUD, r0     ; calculate baud divisor for SP0
   iocopy.2 r0, SP0DIV        ; set the baud rate
   iocopy.1 #1, SP0RPORT      ; SP0 Rx to use PP0.1
   iocopy.1 #0x10, SP0CFG     ; Asynchronous, 1 stop bit, no parity, 8 data bits
   clr.1    SP0CLK            ; use internal clock
   clr.1    SP0INT            ; disable serial port interrupts
   setipl		0     								; enable interrupts
;******************************* end initialization code *******************************

; now send a message and receive a message
   copy.4   #Message_1, r0    ; the transmit function requires the message address in r0
   callr    Send_Message_w    ; transmit the message and wait for it to complete 
   copy.4   #SP0RxBuffer, r0  ; the receive function requires the buffer address in r0
   callr    Rcv_Message_w	   ; receive a message and wait for the carriage return
Loop:
   br			Loop								   ; wait forever

;************************** Receive and Transmit Functions ****************************

/* A few comments about the following functions (which should be true about all functions):
   comments at the beginning describe:
      what is necessary for calling the function
      what the function has to do for the ISR - if applicable
      how the function exits
      all status return values - if applicable
      a description of all register and RAM usage
   each statement has a meaningful comment based on context
   a comment describing each block of statements
   the format of the internal labels insures no duplication
   the function name at the end allows you to know what function it is when scrolling from below
   the comments line up (as much as is practical) making it easier to read
*/

/******************************** Send_Message_w *********************************
Send_Message_w - transmit a message via serial port 0 and wait for the message to complete.

Enter with:
	R0 = the address of the message terminated with a 0 - the 0 byte will not be transmitted.

Uses:			R0, R1, R2
Initializes:		SP0TxBufferAddr and SP0TxByteCount
Enables SP0 Tx interrupt and sends the first byte of the message
   The remaining bytes are sent by the ISR

The ISR expects:
   SP0TxBufferAddr = address of the message to be sent
   SP0TxByteCount = the number of bytes to be transmitted
   the interrupt to be enabled
   the first byte to be sent
The ISR will disable the interrupt when done

note: since this is not an ISR, the number of clocks is not indicated.
*/
Send_Message_w:

; count the number of bytes in the message
   copy.4   r0, r1            ; get a working copy of the message address
   clr.2    r2                ; initialize the byte count
Send_Message_w_1:
   test.1   (r1)              ; end of message?
   beq      Send_Message_w_2  ; br if yes
   inc.2    r2                ; update byte count
   incp.1   r1                ; update pointer to the message
   br       Send_Message_w_1  ; loop until done

Send_Message_w_2:
; adjust for the first byte and set up the values required by the ISR
   copy.1   (r0), r1          ; get first byte to be transmitted
   dec.2    r2                ; adjust for
   incp.1   r0                ;		 first byte
   copy.2   r2, SP0TxByteCount   ; copy the byte count for the ISR
   copy.4   r0, SP0TxBufferAddr  ; point to the message to be sent

; enable interrupt and send the first byte - must be done in this order.
; If not, the interrupt will be missed since the Tx Shift Register is empty
; the byte will be transferred immediately from the Data register to the TSR.

   ioor.1   #TX0_IPL, SP0INT  ; set Tx IPL - assume it was previously disabled
   iocopy.1 r1, SP0DATA       ; send the first byte

;wait for the transmitter to get done
Send_Message_w_3:
   test.2   SP0TxByteCount    ; have all bytes have been transmitted?
   bne      Send_Message_w_3  ; br if no
Send_Message_w_4:
   iotst.1  #TX_BUSY, SP0STAT ; has last byte has completed?
   bcs      Send_Message_w_4  ; br if no
   rets
; Send_Message_w


/********************************* Rcv_Message_w *********************************

Rcv_Message_w  receive a message via serial port 0 and wait for a carriage return

Enter with:
	R0 = the address to put the message
Return with:
	success:             R0 = address of terminating null
	buffer overflow:     R0 = 0
note: This function could simply use the value of SP0RxByteCount (= 0)
to indicate overflow but that places the burden on the caller to know
ISR details.  The purpose of this "driver" is to isolate the main program
from having to know those details.

Uses: R0
Initializes:  SP0RxBufferAddr and SP0RxByteCount
Enables SP0 Rx interrupt and then disables it upon receiving a CR

The ISR expects:
   SP0RxBufferAddr   = address of the message to be sent
   SP0RxByteCount    = the number of bytes to be received
   the interrupt to be enabled

The ISR will disable the interrupt when done based on byte count.
Since this function is looking for a specific character it loads the
byte count value with a larger number than what is expected and this
function will disable the interrupt.

note: since this is not an ISR, the number of clocks is not indicated.
*/

Rcv_Message_w:
; set up values for ISR
   copy.4   r0, SP0RxBufferAddr ; tell ISR where to put the characters
   copy.2   #SP0RXBUF_SIZE-1, SP0RxByteCount ; set byte count to buffer size
   ioor.1   #RX0_IPL, SP0INT	; set Rx IPL - assumes it was previously disabled

; wait for a CR character: r0 still points to the buffer starting address
Rcv_Message_w_1:
   test.2   SP0RxByteCount    ; is buffer full?
   beq      Rcv_Message_w_2   ; br if yes - overflow!
   comp.4   r0, SP0RxBufferAddr ; has buffer address changed?
   bne      Rcv_Message_w_1   ; br if no
   comp.1   (r0), #ASCII_CR   ; is the new byte a CR?
   beq      Rcv_Message_w_3   ; br if yes
   incp.1   r0                ; point to next byte location in buffer
   br       Rcv_Message_w_1   ; wait for next byte

; handle overflow
Rcv_Message_w_2:
   clr.4	   r0                ; show overflow
   br       Rcv_Message_w_4

; handle the CR
Rcv_Message_w_3:
   clr.1	   (r0)              ; replace CR with null
Rcv_Message_w_4:
   ioand.1  #~RXIPL_MASK, SP0INT ; disable the Rx interrupt
   rets
; Rcv_Message_w


; the ISRs are duplicated here for completeness
;******************************** VECTOR_SP0RX **********************************

/*	SP0RxBufferAddr must be initialized to the start of the receive buffer
   - incremented by the ISR.
SP0RxByteCount must be initialized to the number of bytes to be received
   - decremented by the ISR.  The interrupt is disabled when this count reaches 0.
The main program must enable SP0 interrupts - this should be done last.
*/
#vector  	VECTOR_SP0RX            ; tell the assembler the interrupt vector address
SP0RxISR:	; serial port 0 Rx Interrupt Service Routine

; get the address to store the byte
   copy.4   SP0RxBufferAddr, r0  ; B = 6, T = 5  get the buffer pointer

; clear the interrupt, read and store the byte
   iocopy.1 SP0DATA, (r0)        ; B = 4, T = 4  read and store the byte
                                 ;					- also clears the interrupt

; update the buffer pointer
   incp.1   r0                   ; B = 2, T = 2 point to next byte location
   copy.4   r0, SP0RxBufferAddr  ; B = 6, T = 5 save it

; the following instruction could replace the previous two except that it takes
; 1 more clock	even though it is 2 less bytes.  Incrementing a register is much
; faster than incrementing a pointer in	memory.
;			incp.1		SP0RxBufferAddr			; B = 6, T = 8

; update the byte count - a 2-byte value
   dec.2    SP0RxByteCount       ; B = 6, T = 6
                                 ;			B = 24, T = 22 to this point
; disable the interrupt if done
   bne      SP0RxISRexit         ; B = 2, T = 2/3  branch if not done receiving all bytes
   ioand.1  #~RXIPL_MASK, SP0INT ; B = 6, T = 6    disable the interrupt
	
; restore the registers and exit the ISR
SP0RxISRexit:
   reti                          ; B = 2, T = 6  return from the interrupt

			; nbr of instructions = 8		total length = 34 bytes		total time = 36/31 clocks

;********************************* VECTOR_SP0TX *********************************

/*	SP0TxBufferAddr must be initialized to the start of the message to be sent -
      incremented by the ISR.
SP0TxByteCount must be initialized to the number of bytes to be transmitted -
      decremented by the ISR.  The interrupt is disabled when this count reaches 0.
*/

#vector  VECTOR_SP0TX				; tell the assembler the interrupt vector address
SP0TxISR:	; serial port 0 Tx Interrupt Service Routine

   copy.4   SP0TxBufferAddr, r0  ; B = 6, T = 5  get the address of the next byte

   iocopy.1 (r0), SP0DATA        ; B = 4, T = 4  get and transmit the byte,
                                 ;					also clears the interrupt

   incp.1   r0                   ; B = 2, T = 2 point to next byte location
   copy.4   r0, SP0TxBufferAddr  ; B = 6, T = 5 save it

   dec.2    SP0TxByteCount       ; B = 6, T = 6  update the byte count  a 2-byte value
                                 ;			B = 24, T = 22 to this point

   bne      SP0TxISRexit         ; B = 2, T = 2/3  branch if not done receiving all bytes
   ioand.1  #~TXIPL_MASK, SP0INT ; B = 6, T = 6 disable the interrupt
	
SP0TxISRexit:
   reti                          ; B = 2, T = 6  return from the interrupt

			; nbr of instructions = 8		total length = 34 bytes		total time = 36/31 clocks

; the two numbers for the time are due to the bne instruction -
;     most of the time the branch will be executed.

; if you are really concerned about ISR time you should also account for
; interrupt latency - for this processor it is 8 + nbr of clocks remaining
; in the instruction being executed when the interrupt occurs.
; A good average for instruction length is about 6.
