#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>

/* These are some of the variables that a person may want to change */
/* NOTE: all times are in millseconds */
#define HANGTIME 3000
#define SHORTKEY 2000
#define TIMEOUT 300000

/* Port functions */
#define MUTE_INVERTED

/* Courtesy beep */
#define BEEP_FM
#define BEEP_SCRIPT "courtesy_tone"

int verbose = 0;
char buf[40];

/* Make a timestamp */
char *timestamp()
{
    struct timeval tv;
    if( gettimeofday(&tv, NULL) < 0 ) return (0);

    strftime(buf, sizeof(buf), "%b %d %H:%M:%S", localtime(&tv.tv_sec));
    return(buf);
}

/* Beep using the FM Syntesizer */
void beep(void)
{
    system(BEEP_SCRIPT);
}

/* state == 1 for mute, 0 for unmute */
int do_mute(int irlpdev, int state) {
    /* A static variable will keep its value between calls.  If this
     * program ever becomes multi-threaded this might cause issues.
     */

    char output = 0;
    char c[2] = "";

    /* Read the current status */
    if(read(irlpdev, c, 2) != 2) return state;

    /* Set the appropriate bits */
    #ifndef MUTE_INVERTED
    if(state == 1) output = (c[1] | 0x08);
    if(state == 0) output = (c[1] & 0xf7);
    #endif
    #ifdef MUTE_INVERTED
    if(state == 1) output = (c[1] & 0xf7);
    if(state == 0) output = (c[1] | 0x08);
    #endif

    /* Write the new state to the hardware */
    if(write(irlpdev, &output, 1) != 1) return state;

    /* The change took, so update our current state and return */
    return state;
}

int mute(int irlpdev) {
    if (verbose) fprintf(stdout,"[%s] Mute: ON\n", timestamp());
    return do_mute(irlpdev, 1);
}

int unmute(int irlpdev) {
    if (verbose) fprintf(stdout,"[%s] Mute: OFF\n", timestamp());
    return do_mute(irlpdev, 0);
}

int do_fan(int irlpdev, int state) {
    char output = 0;
    char c[2] = "";

    /* Read the current status */
    if(read(irlpdev, c, 2) != 2) return state;

    /* Set the appropriate bits */
    if(state == 1) output = (c[1] | 0x20);
    if(state == 0) output = (c[1] & ~0x20);
    
    /* Write the new state to the hardware */
    if(write(irlpdev, &output, 1) != 1) return state;

    /* The change took, so update our current state and return */
    return state;
}

int fanon(int irlpdev) {
    if (verbose) fprintf(stdout,"[%s] Fan:  ON\n", timestamp());
    return do_fan(irlpdev, 1);
}
    
int fanoff(int irlpdev) {
    if (verbose) fprintf(stdout,"[%s] Fan:  OFF\n", timestamp());
    return do_fan(irlpdev, 0);
}

/* state == 1 for key, 0 for unkey */
int do_key(int irlpdev, int state) {
    /* A static variable will keep its value between calls.  If this
     * program ever becomes multi-threaded this might cause issues.
     */
    char output = 0;
    char c[2] = "";

    /* Read the current status */
    if(read(irlpdev, c, 2) != 2) return state;
    /* Set the appropriate bits */
    if(state == 1) output = (c[1] | 0x04);
    if(state == 0) output = (c[1] & 0xfb);
    /* Write the new state to the hardware */
    if(write(irlpdev, &output, 1) != 1) return state;
    return state;
}

int keyup(int irlpdev) {
    if (verbose) fprintf(stdout,"[%s] Key:  ON\n", timestamp());
    return do_key(irlpdev, 1);
}

int unkey(int irlpdev) {
    if (verbose) fprintf(stdout,"[%s] Key:  OFF\n", timestamp());
    return do_key(irlpdev, 0);
}

/* This is a super-accurate millisecond clock that does not roll over 
 * at 32 bits. It returns the current time in milliseconds 
 */
double dnow()
{
    struct timeval tv;
    if( gettimeofday(&tv, NULL) < 0 ) return (0);
    else return(1000*((double)tv.tv_sec + 1.e-6 * (double)tv.tv_usec));
}

int main(int argc, char *argv[])
/* Main function */
  {  
    int irlpdev = 0;
    unsigned char c[2];          /* Returned string from parallel port */

    int ctflag = 0;              /* Flag when the Courtesy tone has played */
    int muteflag = 0;            /* Flag when the muter is on */
    int keyflag = 0;             /* Flag when the system (AUX1) is keyed */
    int shortkeyflag = 0;        /* Flag when the shortkey feature is active */
    int fanflag = 0;             /* Flag when the fan is active */
    int irlpflag = 0;            /* Flag when IRLP keyed and is active */

    unsigned char COS = '1';     /* Character which determines the state of the 
                                    COS. Capitals used to avoid confusion with 
                                    the cosine function */
    unsigned char dtmf = '0';    /* Character which determines when a valid 
                                    DTMF tone is recieved */
    unsigned char irlpkey;       /* Character which determines when IRLP 
                                    software has the key triggered */
    struct timespec tim;         /* Timespec for the loop timer function */
    double mutetimer = 0;        /* Definition of the timer to measure time 
                                    bewteen mute on and mute off */
    double hangtimer = 0;        /* Definition of the timer to measure time 
                                    bewteen start of key and unkey */
    double cttimer = 0;          /* Definition of the timer to measure time 
                                    bewteen unkey and the playing of the 
                                    courtesy tone */
    double shortkeytimer = 0;    /* Definition of the timer to measure time 
                                    bewteen mute on and mute off */
    double fantimer = 0;         /* Definition of the time to measure time
                                    from transmit drop to fan off */

    /* Look for the command line arg we know of */
    while (argc > 1) {
        if (!strcmp(argv[1], "-v")) {
            verbose = 1;
        }
        argc -= 1;
        argv += 1;
    }
        
    /* Opens the /dev/irlp-port device, read/write. This is the communication
     * Channel to the IRLP hardware from the software 
     */
    if( (irlpdev = open("/dev/irlp-port", O_RDWR)) < 0 ) { 
        perror("Can't open /dev/irlp-port"); 
        exit(errno); 
    } 

    /* Sets default settings for the main variables */
    tim.tv_sec = 0;
    tim.tv_nsec = 5000000;

    keyflag = unkey(irlpdev);

    while (1) {
        /* Reads the input and output bit from the port */
        read(irlpdev, c, 2);

        /* Determines the status of various inputs and outputs from the port */
        COS = (c[0] >> 7) & 0x01;
        dtmf = (c[0] >> 3) & 0x0f;
        irlpkey = c[1] & 0x02;

        /*
         * FAN control
         */
        
        /* This controles the fan. If the radio has been keyed we turn on
         * the fan.
         */
        if (keyflag && !fanflag) {
            fanflag = fanon(irlpdev);
        }
        /* Unkey 5 minutes (300,000 ms) after transmitter dropped */
        if (!keyflag && fanflag && dnow() - fantimer > 300000) {
            fanflag = fanoff(irlpdev);
        }
        /* fantimer is being set as long as we are keyed */
        if (keyflag) {
            fantimer = dnow();
        }
        
        /*
         * MUTE control
         */
        
        /* This sets the muter, muteflag, and mutetimer when DTMF is detected
         * It also resets the mute if no DTMF is present, and the timer has 
         * elapsed.
         */
        if (COS) {
            if (dtmf >= 1 && dtmf <= 17) {
              if (!muteflag)
                    muteflag = mute(irlpdev);
                mutetimer = dnow();
            } else {
                if (((dnow() - mutetimer) > 1000) && muteflag)
                    muteflag = unmute(irlpdev);
            }
        }

        /* This mutes repeated audio if there is no COS */
        if (!COS && !muteflag) {
            muteflag = mute(irlpdev);
        }
       
        /*
         * Events that KEY
         */
        
        /* This is the start of the hangtimer. It detects COS and determines 
         * whether to key up or reset the hangtimer, depending on keyflag. 
         * Embedded is also the timer for the shortkey system 
         */
        if (COS) {
            if (!keyflag) {
                keyflag = keyup(irlpdev);
                shortkeytimer = dnow();
            }
            hangtimer = dnow();
            cttimer = dnow();
            ctflag = 0;
        }

        /* Determines if the IRLP software has keyed the radio, and sets the 
         * hangtimer, and keys up the radio. 
         */
        if (irlpkey) {
            if (!keyflag) {
                keyflag = keyup(irlpdev);
                shortkeytimer = dnow();
            }
            hangtimer = dnow();
            cttimer = dnow();
            irlpflag = 1;
            ctflag = 0;
        }

        /*
         * Shortkey feature
         */

        /* Once the shortkey timer is exceeded, it stops the shortkey 
         * features from unkeying the radio.
         */
        if (COS || irlpkey) { 
            if (!shortkeyflag && dnow() - shortkeytimer > 10) {
                shortkeyflag = 1;
            }
        }

        /*
         * Play the courtesy tone
         */

        /* Once COS is dropped, and the cttimer is exceeded, we play the 
         * courtesy tone 
         */
        if (!COS && !irlpkey) {
            /* If IRLP was last to drop cttimer is shorter because IRLP
             * has a longer delay before unkey
             */
            if (!ctflag && irlpflag && dnow() - cttimer > 300) {
                /* Plays the courtesy tone */
                beep();
                ctflag = 1;
                irlpflag = 0;
            }
            /* If COS was last to drop cttimer is longer */
            if (!ctflag && dnow() - cttimer > 1000) {
                /* Plays the courtesy tone */
                beep();
                ctflag = 1;
            }
        }

        /*
         * Events that UNKEY
         */

        /* If the shortkey timer has not been exceeded, and COS is dropped, 
         * we drop the transmitter. It also makes sure there is no courtesy 
         * tones. 
         */
        if (!COS && !irlpkey && keyflag && !shortkeyflag) {
            keyflag = unkey(irlpdev);
            ctflag = 1;
            irlpflag = 0;
            shortkeyflag = 0;
        }

        /* When the hangtime is exceeded, the radio is unkeyed, and the 
         * shortkeytimer is reset. 
         */
        if (!COS && !irlpkey && keyflag && (dnow() - hangtimer > HANGTIME)) {
            keyflag = unkey(irlpdev);
            irlpflag = 0;
            shortkeyflag = 0;
        }

        fflush(stdout);

        /* This is a delay timer to keep this from sucking 100% processor, 
         * important in any loop. 
         */
        nanosleep(&tim, NULL);
    }
}

