/* $Id: fan_old.c,v 1.1 2004/01/05 00:47:00 adicvs Exp $
 *
 * -------------------------------------------------------------------------
 *
 * fan.c  Control the radio fan using PWM to increase fan life and reduce
 *        noise. Several different operating schemes are supported. The fan
 *        is started with 100% duty cycle and speed is reduced to the
 *        specified duty cycle after a specified timeout.
 *
 *        - Fan always on with specified duty cycle.
 *        - Fan follows port pin changes with optional hang time.
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>

#include "repeater.h"

#define DEFAULT_OUT     "AUX4"          /* Default output pin */
#define DEFAULT_DC      50              /* Default duty-cycle */
#define MIN_DC          15              /* Minimum duty-cycle */
#define MAX_DC          100             /* Maximum duty-cycle */
#define DEFAULT_START   10              /* Default time for startup */
#define MIN_START       0               /* Minimum startup time */
#define MAX_START       60              /* Maximum startup time */
#define DEFAULT_FREQ    50              /* Default pwm frequency */
#define MIN_FREQ        5               /* Minimum frequency */
#define MAX_FREQ        60              /* Maximum frequency */
#define DEFAULT_RUN     60              /* Default runimee in seconds */
#define MIN_RUN         0               /* Minimum runtime (0 is continous) */
#define MAX_RUN         14400           /* Maximum runtime (4hours) */

/* Usage */
void usage(void)
{
    fprintf(stdout, "Usage: fan [options] ...\n");
    fprintf(stdout, "Control a fan via an AUX port. The fan will startup with 100%% duty-cycle and\n");
    fprintf(stdout, "fall back to the specified duty-cycle after a few seconds.\n\n");
    fprintf(stdout, "  -d duty-cycle     duty-cycle in %d - %d, default %d.\n", MIN_DC, MAX_DC, DEFAULT_DC);
    fprintf(stdout, "  -f frequency      frequency in %d - %d Hz, default %d Hz.\n", MIN_FREQ, MAX_FREQ, DEFAULT_FREQ);
    fprintf(stdout, "  -o AUX[1-5]       output pin, default %s.\n", DEFAULT_OUT);
    fprintf(stdout, "  -r runtime        runtime in %d - %d seconds, default %d seconds.\n", MIN_RUN, MAX_RUN, DEFAULT_RUN);
    fprintf(stdout, "  -s startup        startup time in %d - %d seconds, default %d seconds.\n", MIN_START, MAX_START, DEFAULT_START);
    fprintf(stdout, "\nThe runtime specifies the total time the fan will run. Therefor the runtime\n");
    fprintf(stdout, "needs to be larger than the startup time.\n");
    fprintf(stdout, "\nReport bugs to <adi@adis.on.ca>.\n");
}

/* Turn port pin on */
void pin_on(int irlpdev, char out)
{
    unsigned char c[2];
    char output;

    read(irlpdev, c, 2);
    output = (c[1] | out);
    write(irlpdev, &output, 1);
}

/* Turn port pin off */
void pin_off(int irlpdev, char out)
{
    unsigned char c[2];
    char output;

    read(irlpdev, c, 2);
    output = (c[1] & ~out);
    write(irlpdev, &output, 1);
}

/* Create the pwm pulse train */
void do_pwm(int irlpdev, int out, int ton, int toff)
{
    /* Turn port pin on for ton */
    pin_on(irlpdev, out);
    usleep(ton);

    /* Turn port pin off for toff */
    pin_off(irlpdev, out);
    usleep(toff);
}

/* Main program */
int main(int argc, char *argv[])
{
    int irlpdev = -1;

    /* Runtime parameters and commandline args */
    char *cout  = DEFAULT_OUT;      /* output string */
    char out;                       /* output bit mask */ 
    int dc      = DEFAULT_DC;       /* duty-cycle */
    int run     = DEFAULT_RUN;      /* runtime in seconds */
    int start   = DEFAULT_START;    /* fast start time in seconds */
    int freq    = DEFAULT_FREQ;     /* frequency in Hz */

    /* PWM values */
    int ton;                        /* On time for output pulse in us */
    int toff;                       /* Off time for output pulse in us */
    int count;                      /* Number of pulses to generate */
    
    /* Get command line arguments */
    while (argc > 1) {
        if        (!strncmp(argv[1], "-d", 2)) {
            sscanf(argv[2], "%d", &dc);
            argc -= 2;
            argv += 2;
        } else if (!strncmp(argv[1], "-f", 2)) {
            sscanf(argv[2], "%d", &freq);
            argc -= 2;
            argv += 2;
        } else if (!strncmp(argv[1], "-o", 2)) {
            cout = argv[2];
            argc -= 2;
            argv += 2;
        } else if (!strncmp(argv[1], "-r", 2)) {
            sscanf(argv[2], "%d", &run);
            argc -= 2;
            argv += 2;
        } else if (!strncmp(argv[1], "-s", 2)) {
            sscanf(argv[2], "%d", &start);
            argc -= 2;
            argv += 2;
        } else if (!strncmp(argv[1], "-h", 2)) {
            usage();
            return(0);
        } else {
            fprintf(stderr, "Error: unrecognized option `%s'\n", argv[1]);
            fprintf(stderr, "Try `fan -h' for more information.\n");
            return (1);
        }
    }

    /* Get bit mask from output string */
    if        (!strcmp(cout, "AUX1")) {
        out = AUX1;
    } else if (!strcmp(cout, "AUX2")) {
        out = AUX2;
    } else if (!strcmp(cout, "AUX3")) {
        out = AUX3;
    } else if (!strcmp(cout, "AUX4")) {
        out = AUX4;
    } else if (!strcmp(cout, "AUX5")) {
        out = AUX5;
    } else {
        fprintf(stderr, "Error: unrecognized output option `%s'\n", cout);
        fprintf(stderr, "Try `fan -h' for more information.\n");
        return (1);
    }

    /* Check duty-cycle range */
    if (dc < MIN_DC || dc > MAX_DC) {
        fprintf(stderr, "Error: duty-cycle of `%d' is out of range.\n", dc);
        fprintf(stderr, "Try `fan -h' for more information.\n");
        return (1);
    }

    /* Check frequency range */
    if (freq < MIN_FREQ || freq > MAX_FREQ) {
        fprintf(stderr, "Error: frequency of `%d' is out of range.\n", freq);
        fprintf(stderr, "Try `fan -h' for more information.\n");
        return (1);
    }

    /* Check start time */
    if (start < MIN_START || start > MAX_START) {
        fprintf(stderr, "Error: startup time of `%d' is out of range.\n", freq);
        fprintf(stderr, "Try `fan -h' for more information.\n");
        return (1);
    }

    /* Check runtime range */
    if (run < MIN_RUN || run > MAX_RUN) {
        fprintf(stderr, "Error: runtime of `%d' is out of range.\n", run);
        fprintf(stderr, "Try `fan -h' for more information.\n");
        return (1);
    }
    if (run != 0 && run < start) {
        fprintf(stderr, "Error: runtime is less than startup time.\n");
        fprintf(stderr, "Try `fan -h' for more information.\n");
        return (1);
    }

    /* Open irlp-port */
    if ( (irlpdev = open("/dev/irlp-port", O_RDWR)) < 0 ) {
        perror("Can't open /dev/irlp-port");
        return(errno);
    }

    /* Calculate the on and off times for the pwm pulses.
     * 
     * We are using micro second timing and integer math. The inaccuracies
     * shouldn't be relevant for this application. Also ignoring the execution
     * time of the pwm generating loop.
     *
     * t(us) = 1,000,000 / freq
     * ton   = t(us) * dc / 100
     * 
     * ton   = 10,000 * dc / freq
     */
    ton   = (dc * 10000 / freq);
    toff  = ( (100 - dc) * 10000 / freq);
    count = freq * (run - start);

    fprintf(stdout, "Output on time:  %d us\n", ton);
    fprintf(stdout, "Output off time: %d us\n", toff);
    fprintf(stdout, "Pulse count:     %d\n", count);
    
    /* Turn on fans for 100% for the startup time */
    pin_on(irlpdev, out);
    usleep(start * 1000000);
    pin_off(irlpdev, out);

    /* Allow for continous running*/
    if (run == 0) {
        while (1) {
            do_pwm(irlpdev, out, ton, toff);
        }
    } else {
        while (count > 0) {
            do_pwm(irlpdev, out, ton, toff);
            count -= 1;
        }
    }
    
    close(irlpdev);
    return (0);
}
