/*
 * Grid Locator - calculator for distance/bearing for PalmOS devices.
 *
 * file: gps.h
 * Copyright (C) 2002, Rex Allers
 *
 * Definition for utilities to communicate from a Palm handheld  
 * to read position from an NMEA GPS receiver  
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <PalmOS.h>
#include <PalmCompatibility.h>
#include <SerialMgrOld.h>
#include "resource.h"
#include "gps.h"
#include "resource.h"


// Global variables
Boolean serIsOpen;

char bigbuf[BUFMAX];
Int16 datalen;

Int32 ticksPerSec;
Int32 GPSticks;
Int32 QuartSec;

UInt16 serRef;

// for debug
//Int16 count;


// GetField
// returns n'th (0-based) comma-delimeted field within buffer
// true if field found, false otherwise
Boolean 
GetField(char *sen, UInt n, char *result)
{
	UInt	i;
	
	// skip n commas
	for (i = 0; i < n; i++) {
		while (*sen && *sen != ',')
			sen++;
		if (*sen == '\0')
			return false;
		sen++;
	}
	while (*sen && *sen != ',')
		*result++ = *sen++;
	if (*sen != ',')
		return false;
	*result = '\0';
	return true;
}


// SplitDegMin
// add a space to separate deg from min
// inp must be long enough to allow string 1 larger
void 
SplitDegMin(char *inp)
{
	Int16 i, j;

	for(i=0; i<StrLen(inp); i++) if(inp[i] == '.') break;
	for(j=StrLen(inp)+1; j>i-2; j--) inp[j] = inp[j-1];
	inp[j] = ' ';
}


// verify checksum
Err
VerifyChecksum(char *sen)
{
	Int16		i, j, errcnt, cksum;
	char		ckstr[10];

	errcnt = 0;
	i = StrLen(sen);
	if(sen[i-3] != '*') return 1;

	j=i-4;			// last char to sum
	ckstr[0] = '\0';
	if(!errcnt) {
		cksum = 0;
		j = i-3;
		for(i=1; i<j; i++) cksum ^= sen[i];
		cksum &= 0xff;
		StrIToH(ckstr, cksum); // convert to string
		// here we test checksum value
		if(StrCompare(&ckstr[6], &sen[j+1])) errcnt = 1;
	}
	if(errcnt) {
		// FrmCustomAlert(a_Warning, "Checksum error\n", "on $GPRMC\n", "Sentance");
		return 1;
	}
	return 0;
}


Err
GetGPSLatLon(char *sen, char *LatStr, char *LonStr, char *Name)
{
	char	s[20];
	DateTimeType now;
	Int16 i, julian;

	if (GetField(sen, 2, s)) {
		// See if the return is a valid fix
		if (s[0] != 'A') return 1;
		
		// update Lat & Lon
		// 4 is N or S for Lat direction, 3 is lat
		if (GetField(sen, 3, s))
			SplitDegMin(s);
		StrCat(s, " ");	 
		if (GetField(sen, 4, s + StrLen(s))) { 
			//FrmCustomAlert(a_Debug, "do Lat", "", "");
			StrCopy(LatStr, s);
		}
		
		// 6 is E or W for Lat direction, 5 is lon
		if (GetField(sen, 5, s))
			SplitDegMin(s);
		StrCat(s, " ");	 
		if (GetField(sen, 6, s + StrLen(s))) { 
			//FrmCustomAlert(a_Debug, "do Lng", "", "");
			StrCopy(LonStr, s);
		}

		// make up a name using current time
		StrCopy(Name, "G");
		// get current date/time
		TimSecondsToDateTime(TimGetSeconds(), &now);
		// find julian date
		for(i=1, julian=0; i<now.month; i++) 
			julian += DaysInMonth (i, now.year);
		julian += now.day;
		// build the name from date
		StrIToA (s, (now.year % 100));
		if(StrLen(s) == 1) StrCat(Name, "0");
		StrCat(Name, s);
		StrIToA (s, julian);
		if(StrLen(s) == 2) StrCat(Name, "0");
		if(StrLen(s) == 1) StrCat(Name, "00");
		StrCat(Name, s);
		StrIToA (s, now.hour);
		if(StrLen(s) == 1) StrCat(Name, "0");
		StrCat(Name, s);
		StrIToA (s, now.minute);
		if(StrLen(s) == 1) StrCat(Name, "0");
		StrCat(Name, s);
		StrIToA (s, now.second);
		if(StrLen(s) == 1) StrCat(Name, "0");
		StrCat(Name, s);

		return 0;

	} else return 1;
}


// get time string from $GPSRMC sentance
Err
GetGPSTime(char *sen, char *Hrs, char *Mins, char *Secs)
{
	char	s[20];

	if (GetField(sen, 1, s)) {
//		FrmCustomAlert(a_Debug, "Time string", s, "");
		if(StrLen(s) != 6) return 1;

		// hour string
		Hrs[0] = s[0];
		Hrs[1] = s[1];
		Hrs[2] = '\0';

		// minute string
		Mins[0] = s[2];
		Mins[1] = s[3];
		Mins[2] = '\0';

		// second string
		Secs[0] = s[4];
		Secs[1] = s[5];
		Secs[2] = '\0';

		return 0;
	} else return 1;
}


Err
StartSerial()
{
	Err err;
	SerSettingsType serSettings;

	err = 0;
	SysLibFind("Serial Library", &serRef);
	err = SerOpen(serRef, 0, BAUD);
	if(err) {
		if(err == serErrAlreadyOpen) SerClose(serRef);
		return err;
	}
	SerGetSettings(serRef, &serSettings);
	serSettings.flags = serSettingsFlagBitsPerChar8 | serSettingsFlagStopBits1;
	SerSetSettings(serRef, &serSettings);

	serIsOpen = true;
	return 0;
}

void
StopSerial()
{
	if(serIsOpen) {
		SerClose(serRef);
		serIsOpen = false;
	}
}

/* Read serial stream searching for tag string.
 * If found place this sentance in dest, and return 0.
 *	shift working buffer putting next char at start and return 0.
 *
 * If fail return serErr, or 100 == not found.
 */

Err 
Get_Sentance(char *dest, char *tag)
{
	static Int16 datalen = 0;
	long nbytes;
	char *bp;
	Err err;
	Int16 n;
//	char		temp[16];


	// count = 0;
	dest[0] = '\0';

	for(n=0; n<16; n++) {
		// get at least two sentance lines of input
		// allow to loop on LineErr for intermittant problem
		// not sure why but sometimes immediate framing errors 
		// a few retries is not enough to eliminate framing error
		SerReceiveFlush (serRef, QuartSec);  // start clean
		err = SerReceiveWait (serRef, 160, GPSticks);
		if(err != serErrLineErr) break;  // the normal way to complete
	}
	if (err) return err;

	// count = 1;
	/* see how many bytes are waiting for us */
	err = SerReceiveCheck(serRef, &nbytes);
	if (err) return err;
	if (nbytes == 0) return serErrTimeOut;
	
	// count = 2;
	/* check for buffer overflow */
	if (nbytes + datalen > BUFMAX) nbytes = BUFMAX - datalen;
	
	/* receive bytes, if there's room for them. */ 
	if (0 < nbytes) SerReceive(serRef, &bigbuf[datalen], nbytes, QuartSec, &err);
	if(err) return err;
	
	datalen += nbytes;

//	FrmCustomAlert(a_Debug, "Serial read 1", "Bytes:", StrIToA(temp, datalen));


	n = 0;
	while(n<8) {
		/* shift so that start of message is at start of buffer */
		if (bigbuf[0] != '$') 
		{
			bp = bigbuf;
			while (datalen && *bp != '$') bp++, datalen--;

			if (datalen > 0)
				MemMove((void *)bigbuf, (void *)bp, datalen);
		}

	// count = 3;
		/* if not enough data, wait for some */
		if (datalen < StrLen(tag)+2) {
			nbytes = SerReceive(serRef, &bigbuf[datalen], 80, GPSticks, &err);
			if(err) return err;
			datalen += nbytes;
			n++;
		}

		if( StrNCompare(&bigbuf[1], tag, StrLen(tag)) ) {
			bigbuf[0] = ' ';  // mask the '$' for this wrong tag
		} else break;  // we found it
	}

	if(n == 8)	return NOTFOUND;

	bp = bigbuf;

	// count = 4;
	// hunt for CR (end of message)
	bp++;
	datalen--;
	while(1) 
	{
		if (*bp == '\r') {
			// found the whole sentance
			*bp = '\0';				 // make it a string
			if(StrLen(bigbuf) < 83) {
				StrCopy(dest, bigbuf);	 //and copy it
				err = 0;
			} else err = NOTFOUND;
			// shift buffer for a new start
			datalen--;
			if(datalen) MemMove((void *)bigbuf, (void *)bp, datalen);
			return err;
		}

		if (*bp == '$') 
		{
			// oops found start of message before we 
			// found the end of the last one
			MemMove((void *)bigbuf, (void *)bp, datalen);
			return NOTFOUND;
		}
		bp++;
		datalen--;
		if(datalen == 0) {
			// get more data
			nbytes = SerReceive(serRef, bp, 80, QuartSec, &err);
			if(err) return err;
			datalen += nbytes;
		}
	}
}

void
FlushSerial() {
	SerReceiveFlush (serRef, 1);
}


void
InitForGPS()
{
	serIsOpen = false;
	datalen = 0;
	ticksPerSec = SysTicksPerSecond();
	GPSticks =  2 * ticksPerSec + (ticksPerSec >> 1);	// set to 2.5 seconds
	QuartSec = 	ticksPerSec >> 2;
}