/*
 * Grid Locator - calculator for distance/bearing for PalmOS devices.
 *
 * file: gl_db.c
 * Copyright (C) 2001, Rex Allers
 *
 *   Some program structure based on: 
 *   Astro Info (Astro.c) Copyright (C) 2000, Michael Heinz
 *
 *	  Dist/bear modules based on code from PROJ.4 1995 by 
 *		Gerald I. Evenden
 *		USGS, Woods Hole, MA 02543
 *
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <PalmOS.h>
#include "PalmUtil.h"
#include "resource.h"
#include "gl.h"
#include "gl_db.h"

extern GlPrefsType glPrefs;

// Global variables
GLLstRec	GLLst[MAXLST];
GLLocRec	GLLoc[MAXLOC];
UInt16		GLLstSz, GLLocSz;
char		s_WkMh[WMH+1];
char		s_WkLt[WLT+1];
char		s_WkLn[WLN+1];


//
// Write a new MemoPad DB record
//
Err  WriteMemoRecord(Char * data)
{

	char		*pRec;
	MemHandle	memoRec;
	Err			error;
	UInt16		mode;
	DmOpenRef	MemoPadDB;
	UInt32		memoPadDBType = 'DATA';
//	UInt32		memoPadDBCreator = 'memo';
	UInt16		index;

	mode = dmModeReadWrite;

	MemoPadDB =  DmOpenDatabaseByTypeCreator(memoPadDBType, 
	    						sysFileCMemo, mode);

	if(!MemoPadDB) {
		FrmCustomAlert(a_Debug, "MemoDB", "Open", "failed");
		return 1;
	}

	// Get index of new memo record to be added.
	index = DmNumRecords (MemoPadDB);

	// Create a new record in the database
	memoRec = DmNewRecord(MemoPadDB, &index, StrLen(data) + 1);

	// Lock down the block containing the new record.
	pRec = (Char *)MemHandleLock(memoRec);

	// Write record.
	error = DmWrite(pRec, 0, data, StrLen(data) + 1);
	if (error != 0)
	        return error;

	// Unlock the block of the new record.
	error = MemPtrUnlock(pRec);
	if (error != 0)
	        return error;

	error = DmReleaseRecord (MemoPadDB, index, true);
	if (error != 0)
	        return error;

	// Close the application's database.
	DmCloseDatabase(MemoPadDB);

	return 0;

}


//
// Find MemoPad DB records starting GL-
//  fill in table GLLst
//
UInt16  FindGLRecord(void)
{

	char		*pRec;
	MemHandle	memoRec;
	UInt16		mode;
	DmOpenRef	MemoPadDB;
	UInt32		memoPadDBType = 'DATA';
	UInt16		index, recMax;
	LocalID		dbID;
	UInt16		cardNo;
	UInt32		dbSz;
//	char		temp[16];

//	FrmCustomAlert(a_Debug, "FindGLRecord", "reached", "");
	mode = dmModeReadOnly;

	GLLstSz = 0;
	MemoPadDB =  DmOpenDatabaseByTypeCreator(memoPadDBType, 
	    						sysFileCMemo, mode);

	if(!MemoPadDB) {
		return 1;
	}

	// get size of all DB data
	DmOpenDatabaseInfo(MemoPadDB, &dbID, NULL, NULL, &cardNo, NULL);
	DmDatabaseSize(cardNo, dbID, NULL, NULL, &dbSz);
	if(dbSz != glPrefs.memoDbSize) {
		// Memo DB was never used or has changed
		// be sure we do not use a saved index for the table
		glPrefs.lstIdx = LSTEMPTY;
		glPrefs.memoDbSize = dbSz;
	}

	// Get number of memos (records).
	recMax = DmNumRecords (MemoPadDB);
//	FrmCustomAlert(a_Debug, "FindGL", "record count", StrIToA(temp, recMax));

	// Find and save GL memo pointers
	for(index=0; index<recMax; index++) {

		if(GLLstSz == MAXLST) break;	 // skip if too many
		// open the record
		memoRec = DmGetRecord(MemoPadDB, index);
		if(memoRec == NULL) {
			// deleted record
			DmReleaseRecord(MemoPadDB, index, 0);
			continue;
		}

		// get allocated size of this record
		GLLst[GLLstSz].RecSz = MemHandleSize(memoRec);

		// Lock down the block containing this record.
		pRec = (Char *)MemHandleLock(memoRec);

		if(!StrNCompare(pRec, "GL-", 3) || !StrNCompare(pRec, "gl-", 3)) {
			// we found one with our tag
			// save index into table
			GLLst[GLLstSz].RecIdx = index;
			// save name from this record
			strNcopy(GLLst[GLLstSz].name, pRec+3, MAXNME);
			// if it looks good, move to next in table
			if(StrLen(GLLst[GLLstSz].name))   GLLstSz++;
		}
		// release handle and record
		MemHandleUnlock(memoRec);
		DmReleaseRecord(MemoPadDB, index, 0);
	}

	// Close the application's database.
	DmCloseDatabase(MemoPadDB);

//	FrmCustomAlert(a_Debug, "FindGLRecord", "exit", "");
	return 0;

}

//
// Open a MemoPad GL record from a
// GLLst entry by glPrefs.lstIdx
//  fill in table GLLoc
//
UInt16 FindGLLocs(void)
{
	char		*pRec;
	MemHandle	memoRec;
	UInt16		mode;
	DmOpenRef	MemoPadDB;
	UInt32		memoPadDBType = 'DATA';
	UInt16		index, locIdx;
	Int16		j;
	UInt32		recSz;
	LocalID		dbID;
	UInt16		cardNo;
	UInt32		dbSz;

//	char		temp[16];
//	char		temp2[16];

//	FrmCustomAlert(a_Debug, "FindGLLocs", "reached", "");
	mode = dmModeReadOnly;

	// if not a valid index, there is nothing to do
	if(glPrefs.lstIdx == LSTEMPTY) return 1; 

	GLLocSz = 0;	// start with no locations
	MemoPadDB =  DmOpenDatabaseByTypeCreator(memoPadDBType, 
	    						sysFileCMemo, mode);

	if(!MemoPadDB) {
		return 1;
	}

	// get size of all DB data
	DmOpenDatabaseInfo(MemoPadDB, &dbID, NULL, NULL, &cardNo, NULL);
	DmDatabaseSize(cardNo, dbID, NULL, NULL, &dbSz);
	if(dbSz != glPrefs.memoDbSize) {
		// Memo DB was never used or has changed
		// be sure we do not use a saved index for the table
		glPrefs.lstIdx = LSTEMPTY;
		return 1;
	}

	// Get number of memos (records).
	index = DmNumRecords (MemoPadDB);
//	FrmCustomAlert(a_Debug, "FindGLLocs", "record count", StrIToA(temp, index));
	if(GLLst[glPrefs.lstIdx].RecIdx >= index) {
		DmCloseDatabase(MemoPadDB);
		return 1;   // we point beyond the record length
	}


	// open the record
	memoRec = DmGetRecord(MemoPadDB, GLLst[glPrefs.lstIdx].RecIdx);
	if(memoRec == NULL) {
		// deleted record
		FrmCustomAlert(a_Debug, "FindGLLocs", "record failed to open", "");
		DmReleaseRecord(MemoPadDB, GLLst[glPrefs.lstIdx].RecIdx, 0);
		DmCloseDatabase(MemoPadDB);
		return 1;
	}

	// get allocated size of this record and check
	recSz = MemHandleSize(memoRec);
//	FrmCustomAlert(a_Debug, "FindGLLocs", "LstSz", StrIToA(temp, GLLstSz));
//	FrmCustomAlert(a_Debug, "recSz", StrIToA(temp, recSz), 
//								StrIToA(temp2, GLLst[glPrefs.lstIdx].RecSz));
	if(GLLst[glPrefs.lstIdx].RecSz != recSz) {
//		FrmCustomAlert(a_Debug, "FindGLLocs", "record has changed", "");
		DmReleaseRecord(MemoPadDB, GLLst[glPrefs.lstIdx].RecIdx, 0);
		DmCloseDatabase(MemoPadDB);
		return 1;
	}


	// Lock down the block containing this record.
	pRec = (Char *)MemHandleLock(memoRec);

	// check that record name still matches list
	for(j = StrLen(GLLst[glPrefs.lstIdx].name); j>0; j--) {
		if(GLLst[glPrefs.lstIdx].name[j-1] != pRec[j+2]) break;
	}
	if(j != 0) {	// if j != 0, strings did not match
//		FrmCustomAlert(a_Debug, "FindGLLocs", "record has changed", "");
		// release handle and record
		MemHandleUnlock(memoRec);
		DmReleaseRecord(MemoPadDB, GLLst[glPrefs.lstIdx].RecIdx, 0);
		// Close the application's database.
		DmCloseDatabase(MemoPadDB);

		return 1;
	}

	// find location names in this record
	locIdx = 0;		// index in location table
	index = StrLen(GLLst[glPrefs.lstIdx].name) + 3; // index into record
	for( ; locIdx < MAXLOC; locIdx++) {
		while(pRec[index] != '$') { // skip to next name
			if(pRec[index] == '\0') {
				break;
			}
			index++;
		}
		if(index == (recSz - 1)) break;
		index++;

		// we found a loc name
		j = 0;
		while((pRec[index] != '/') && (pRec[index] != '$') && 
				(pRec[index] != '\0')  && (j < MAXNME)) {
			if(pRec[index] < ' ') index++;	 // skip white
			else
			// copy the name
			GLLoc[locIdx].name[j++] = pRec[index++];
		}
		GLLoc[locIdx].name[j] = '\0';		// end name string
		if(index == (recSz - 1)) break;

		// if there is a string we found a loc name
		if(StrLen(GLLoc[locIdx].name)) 	GLLocSz++;
		else continue;
		
		// if we stopped before /, find one
		if(j >= MAXNME) while((pRec[index] != '/') && (pRec[index] != '\0') &&
								(pRec[index] != '$')) index++;

		if(pRec[index] == '/')	GLLoc[locIdx].RecIdx = index;
		else GLLocSz--;	// bad entry -- drop it

		if(pRec[index] == '$') index--;
	}

	// release handle and record
	MemHandleUnlock(memoRec);
	DmReleaseRecord(MemoPadDB, GLLst[glPrefs.lstIdx].RecIdx, 0);

	// Close the application's database.
	DmCloseDatabase(MemoPadDB);

	return 0;

}

UInt16 StrBySlash(char *dest, char *src, Int16 strSz)
{
	Int16 Idx, j;

	Idx = 0;
	// skip leading slash
	if(src[Idx] == '/') Idx++;
	// skip any leading white
	while((src[Idx] != '\0') && (src[Idx] <= ' ')) Idx++;
	j = 0;
	while((src[Idx] != '/') && (src[Idx] != '$') && 
			(src[Idx] != '\0')  && (j < strSz)) {
		if(src[Idx] < ' ') Idx++;	 // skip white
		else
		// copy the name
		dest[j++] = src[Idx++];
	}
	dest[j] = '\0';		// end name string

	// if we stopped before /, find one
	if(j >= strSz) while((src[Idx] != '/') && (src[Idx] != '\0') &&
							(src[Idx] != '$')) Idx++;

	if(src[Idx] == '$') Idx--;

	return Idx;
}

// extract the location fields from Loc record
void GetLocData(UInt16 LocIdx)
{
	char		*pRec;
	MemHandle	memoRec;
	UInt16		mode;
	DmOpenRef	MemoPadDB;
	UInt32		memoPadDBType = 'DATA';
	Int16		j;
	UInt16		index;
//	char		temp[16];

//	FrmCustomAlert(a_Debug, "GetLocData", "reached", "");
	mode = dmModeReadOnly;

	MemoPadDB =  DmOpenDatabaseByTypeCreator(memoPadDBType, 
	    						sysFileCMemo, mode);

	if(!MemoPadDB) {
		return;
	}

	// Get number of memos (records).
	index = DmNumRecords (MemoPadDB);
	if(GLLst[glPrefs.lstIdx].RecIdx >= index) {
		DmCloseDatabase(MemoPadDB);
		return;   // we point beyond the record length
	}

	// open the record
	memoRec = DmGetRecord(MemoPadDB, GLLst[glPrefs.lstIdx].RecIdx);
	if(memoRec == NULL) {
		// deleted record
		DmReleaseRecord(MemoPadDB, GLLst[glPrefs.lstIdx].RecIdx, 0);
		DmCloseDatabase(MemoPadDB);
		return;
	}

	// Lock down the block containing this record.
	pRec = (Char *)MemHandleLock(memoRec);

	// check that record still matches list
	for(j = StrLen(GLLst[glPrefs.lstIdx].name); j>0; j--) {
		if(GLLst[glPrefs.lstIdx].name[j-1] != pRec[j+2]) break;
	}
	if(j != 0) {	// if j == 0, strings matched
		MemHandleUnlock(memoRec);
		DmReleaseRecord(MemoPadDB, GLLst[glPrefs.lstIdx].RecIdx, 0);
		DmCloseDatabase(MemoPadDB);
		return;
	}

//	FrmCustomAlert(a_Debug, "GetLocData", "open OK", "");

	// get pointer to data string part
	if(GLLoc[LocIdx].RecIdx) pRec += GLLoc[LocIdx].RecIdx;

//	strNcopy(temp, pRec, 8);
//	FrmCustomAlert(a_Debug, "GetLocData", "string", temp);

	// get mh
	index = StrBySlash(s_WkMh, pRec, WMH);
	pRec += index; 

	// get Lat
	index = StrBySlash(s_WkLt, pRec, WLT);
	pRec += index; 

	// get Lon
	index = StrBySlash(s_WkLn, pRec, WLN);


	// release handle and record
	MemHandleUnlock(memoRec);
	DmReleaseRecord(MemoPadDB, GLLst[glPrefs.lstIdx].RecIdx, 0);

	// Close the application's database.
	DmCloseDatabase(MemoPadDB);

//	FrmCustomAlert(a_Debug, "FindGLLocs", "done", "");
	return;
}

//
// SaveToGL
// Saves a record string into the Memo 
// name defined as SAVEMEMO for saving
//
UInt16  SaveToGL(char *recStr)
{

	char		*pRec, *pTmp;
	MemHandle	memoRec;
	UInt16		mode, ret;
	DmOpenRef	MemoPadDB;
	UInt32		memoPadDBType = 'DATA';
	UInt16		index, recMax, found;
	UInt16		taglen, newstrlen;
	UInt32		recSz;
	Err			error;
//	char		temp[16];

//	FrmCustomAlert(a_Debug, "SaveToGL", "reached", "");

	// following 3 lines to avoid compiler warnings
	pRec = 0;
	memoRec = 0;
	recSz = 0;

	taglen = StrLen(SAVEMEMO);
	newstrlen = StrLen(recStr);

	mode = dmModeReadWrite;

	pTmp = 0;
	MemoPadDB =  DmOpenDatabaseByTypeCreator(memoPadDBType, 
	    						sysFileCMemo, mode);

	if(!MemoPadDB) {
		return 1;
	}

	// Get number of memos (records).
	recMax = DmNumRecords (MemoPadDB);
//	FrmCustomAlert(a_Debug, "SaveToGL", "record count", StrIToA(temp, recMax));

	// Find a GL- <Save> memo
	found = 0;
	for(index=0; index<recMax; index++) {
		// open the record
		memoRec = DmGetRecord(MemoPadDB, index);
		if(memoRec == NULL) {
			// deleted record
			DmReleaseRecord(MemoPadDB, index, 0);
			continue;
		}

		// get allocated size of this record
		recSz = MemHandleSize(memoRec);

		// Lock down the block containing this record.
		pRec = (Char *)MemHandleLock(memoRec);

		if(!StrNCompare(pRec, SAVEMEMO, taglen) ) {
			// we found the record
			found = 1;
			break;
		}
		// release handle and record
		MemHandleUnlock(memoRec);
		DmReleaseRecord(MemoPadDB, index, 0);
	}

	ret = 1;
	if(!found) {
		// no record - create one
		// FrmCustomAlert(a_Debug, "SaveToGL", "<Save> rec", "not found");
		index = 0;
		memoRec = DmNewRecord(MemoPadDB, &index, taglen+2);
		if(memoRec == NULL) {
			// failed on create
			FrmCustomAlert(a_Debug, "SaveToGL", "could not create", "GL- <Save>");
			// Close the application's database.
			DmCloseDatabase(MemoPadDB);
			return 1;
		}
		// get allocated size of this record
		recSz = MemHandleSize(memoRec);

		// Lock down the block containing this record.
		pRec = (Char *)MemHandleLock(memoRec);

		error = DmWrite(pRec, 0, SAVEMEMO"\n", taglen+2);
		if (error != 0)	goto SvExit;

		FrmCustomAlert(a_Warning, "A new Memo\n", SAVEMEMO"\n", "was created as Unfiled");

	} else {
		// there is existing record
		// Copy the current Record
		if(recSz) {
			pTmp = MemPtrNew(recSz);
			if(!pTmp) {
				FrmCustomAlert(a_Debug, "SaveToGL", "could not alloc", "temp memory");
				if(memoRec) MemHandleUnlock(memoRec);
				DmCloseDatabase(MemoPadDB);
				return 1;
			}
			// copy the record
			MemMove(pTmp, pRec, recSz);
		}
	}

	MemHandleUnlock(memoRec);
	DmReleaseRecord(MemoPadDB, index, 0);
	memoRec = DmResizeRecord(MemoPadDB, index, recSz+newstrlen+1);
	// get allocated size of this record
	// recSz = MemHandleSize(memoRec);

	if(memoRec != 0) {
		pRec = (Char *)MemHandleLock(memoRec);
		// got resized record - now write it
		// 1st the record name
		    //  FrmCustomAlert(a_Debug, "SaveToGL", "DmWrite", "1");
		error = DmWrite(pRec, 0, SAVEMEMO, taglen+1);
		if (error != 0)	goto SvExit;
		error = DmWrite(pRec, taglen, "\n", 2);
		if (error != 0)	goto SvExit;
		// then the new record
		    //  FrmCustomAlert(a_Debug, "SaveToGL", "DmWrite", "2");
		error = DmWrite(pRec, taglen+1, recStr, newstrlen+1);
		if (error != 0)	goto SvExit;
		error = DmWrite(pRec, taglen+newstrlen+1, "\n", 2);
		if (error != 0)	goto SvExit;
		// then the new record
		if(pTmp && (recSz > (taglen+2)) ) {
			    //  FrmCustomAlert(a_Debug, "SaveToGL", "DmWrite", "3");
			    //  FrmCustomAlert(a_Debug, "SaveToGL", "pTmp = ", pTmp);
			error = DmWrite(pRec, (taglen+newstrlen+2), (pTmp+taglen+1), (recSz-taglen-1));
			if (error != 0)	goto SvExit;
		}

		ret = 0;
	}

SvExit:
	if(pTmp) MemPtrFree(pTmp);
	if(pRec) MemHandleUnlock(memoRec);
	DmReleaseRecord(MemoPadDB, index, 1);
	// Close the application's database.
	DmCloseDatabase(MemoPadDB);

	return ret;

}


