/*
 * Grid Locator - calculator for distance/bearing for PalmOS devices.
 *
 * file: gl.c
 * Copyright (C) 2001, Rex Allers
 *
 *   Some program structure based on: 
 *   Astro Info (Astro.c) Copyright (C) 2000, Michael Heinz
 *
 *	  Dist/bear modules based on code from PROJ.4 1995 by 
 *		Gerald I. Evenden
 *		USGS, Woods Hole, MA 02543
 *
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <PalmOS.h>
#include "PalmUtil.h"
#include "PalmMath.h"
#include "resource.h"
#include "MathLib.h"
#include "gl.h"
#include "gl_db.h"
#include "gps.h"
//#include "glLib.h"
#include "constants.h"


SystemPreferencesType sysPrefs;
GlPrefsType           glPrefs;
UInt16                gptSize = sizeof(GlPrefsType);

// minimum Palm OS version to run this prog
#define RomRunVers	0x02003000	// Minimum required PalmOS version number

extern GLLstRec	GLLst[MAXLST];		// table of list records we found
extern GLLocRec	GLLoc[MAXLOC];		// table of locations in current list
extern UInt16		GLLstSz, GLLocSz;
extern char		s_WkMh[WMH+1];
extern char		s_WkLt[WLT+1];
extern char		s_WkLn[WLN+1];
extern UInt16		mathError;

/*
 * This is used to remember the previous form when a dialog is up.
 * It's not needed generally, but we use it so glPrefs doesn't get
 * out of sync.
 */
UInt16                 saveFormId;

// strings for main button labels
char	s_Bmh[10];
char  s_Bng[12];
char	s_Blt[18];
char	s_Bln[18];
char	s_Tmh[10];
char  s_Tng[12];
char	s_Tlt[18];
char	s_Tln[18];

// Global variables
Int16 locOff;

// Bearing/Distance values
double Az;
double Baz;
double Dist;

// sharable variables
char *s;
int decimal;


// utils for lat/lon parsing
void skip_to_num(void)
{
	for( ; (*s!='\0')&&(*s!='.')&&((*s<'0')||(*s>'9')); s++);
}

void to_num_end(void)
{
	decimal = 0;
	for( ; ((*s>='0')&&(*s<='9'))||(*s=='.'); s++) {
		if(*s == '.') decimal = 1;
	}
}

// parse lat or lon string into double
// dolat = 1 for latitude
// return 0 = ok; -1 = bad and dout not changed
int parseLL(double *dout, char * sin, int dolat)
{
	char	sWrk[21], *tok1, *tok2, *tok3;
	int tcnt, len1, len2, len3, noforce; 
	double	dResult, dWrk;
	unsigned char sign;
//	char	stmp[10];


	if (StrLen(sin) < 1) return -1;

	// next two lines just to avoid compiler warnings
	tok2 = tok3 = sin;
	len2 = len3 = 0;

	// skip initial blanks
	for(s = sin; (*s != '\0')&&(*s < '+'); s++);

	// save sign if we have one
	sign = 0;
	noforce = 0;
	if(*s == '-') {
		sign = 1;
		s++;
	} else if(*s == '+') {
		noforce = 1;
		s++;
	}

	// find first number
	skip_to_num();
	tok1 = s;

	to_num_end();
	len1 = (int) (s - tok1);

	tcnt = 0;

	// do min, sec if we have them
	// if there is a decimal we stop there
	if(len1 > 0) {
		tcnt = 1;
		if(decimal) goto skip1;	
		skip_to_num();
		tok2 = s;

		to_num_end();
		len2 = (int) (s - tok2);
		if(len2 > 0) {
			tcnt++;	
			if(decimal) goto skip1;	
			skip_to_num();
			tok3 = s;

			to_num_end();
			len3 = (int) (s - tok3);
			if(len3 > 0) {
				tcnt++;
			} else s = tok2+len2;
		} else s = tok1+len1;
	}
skip1:

	if(tcnt == 0) return -1;

	// look for ordinal letter if we didn't have a sign
	for( ; !sign && !noforce && (*s!='\0'); s++) {
		if(dolat){
			if((*s=='s')||(*s=='S')) {
				sign = 1;
				break;
			}
			if((*s=='n')||(*s=='N')) {
				noforce = 1;
				break;
			}
		} else {
			if((*s=='w')||(*s=='W')) {
				sign = 1;
				break;
			}
			if((*s=='e')||(*s=='E')) {
				noforce = 1;
				break;
			}
		}
	}

	// set hemisphere default if no explicit
	if(!dolat && !noforce && glPrefs.forceWest) sign = 1;
	if(dolat && !noforce && glPrefs.forceSouth) sign = 1;


	dResult = 0.0;
	if(tcnt > 2) {
		strNcopy(sWrk, tok3, len3);
		dWrk = strToDouble(sWrk);
		if(mathError) return -1;
		if(dWrk >= 60.0) return -1;
		dResult += dWrk * 0.000277778;
	
	}
	if(tcnt > 1) {
		strNcopy(sWrk, tok2, len2);
		dWrk = strToDouble(sWrk);
		if(mathError) return -1;
		if(dWrk >= 60.0) return -1;
		dResult += dWrk * 0.016666666;
	}

	strNcopy(sWrk, tok1, len1);
	dWrk = strToDouble(sWrk);
	if(mathError) return -1;
	if(dolat && (dWrk > 90.0)) return -1;
	if(!dolat && (dWrk > 180.0)) return -1;
	dResult += dWrk;


	if(dolat && (dResult > 90.0)) return -1;
	if(!dolat && (dResult > 180.0)) return -1;
	if(sign) dResult *= -1;


	*dout = dResult;
	return 0;

}


//----------------------------------------
// reduce argument to range +/- PI */
	double
adjlon (double lon) {
	while ( fabs(lon) > SPI )
		lon += lon < 0. ? TWOPI : -TWOPI;
	return( lon );
}

// working vals for geodinv
// these defaults are WGS 84
double a = 6378137.0;
double onef = 0.996647189335;
double f2 = 0.0016764053324;
double f4 = 0.0008382026662;
double f64 = 0.0000001756459274;

// calculate working value for 
// current ellipsoid
void setEllipse(void)
{
	double  f;

	switch (glPrefs.datum) {
		case 1:
			// WGS 84
			a = 6378137.0;
			f = .00335281066475;
			break;
		case 2:
			// NAD27
			a = 6378206.4;
			f = .00339007530;
			break;
		case 3:
			// sphere
			a = 6373000.0;
			f = 0.0;
			break;
		default:
			// WGS 84
			glPrefs.datum = 1;
			a = 6378137.0;
			f = .00335281066475;
			break;
	}

	// calc working values
	onef = 1.0 - f;
	f2	= f / 2.0;
	f4	= f / 4.0;
	f64	= (f * f)/ 64.0;

	return;
}

// -----------------------------------
//  do inverse geodesic computations
//    take 2 lat/long points and compute azimuth/distance
//      all angles in radians, S in meters
//  input:    
//      phi = input geodetic latitude, (sub 1&2 for 2 points)
//		lam = input geodetic longitude 
//  output:
//      az12, az21 = azimuths,  S = distance (m)
//		  az > 2pi means az is undefined (coincident or antipode)
//  global ellipsoid parms for elliptic adjustment:
//      ellipse, a, onef, f2, f4, f64
//		if ellipse=0 model is spherical
//     sphere if onef = 1.0
//
//	this version by Rex Allers

# define DTOL	1e-16       // tolerance limit

	void
geod_inv(double phi1, double lam1, double phi2, double lam2,
		 double *az12, double *az21, double *S) 
{
	int	antip = 0;
	int ellipse;
	double	th1, th2, thm, dthm, dlamm, dlam, 
		sindlamm, costhm, sinthm, cosdthm, sindthm, 
		L, E, cosd, d, X, Y, T, sind, tandlammp, u, v, D, A, B;

	//FrmCustomAlert(a_Debug, "Geode", "entered", "");
	if(onef == 1.000000) {
		// spherical model
		ellipse = 0;
	} else {
		ellipse = 1;
	}

	// get geocentric latitude th means theta	
	if (ellipse) {
		th1 = atan(onef * tan(phi1));
		th2 = atan(onef * tan(phi2));
	} else {
		th1 = phi1;
		th2 = phi2;
	}
	thm = .5 * (th1 + th2);
	dthm = .5 * (th2 - th1);
    dlam = adjlon(lam2 - lam1);
	dlamm = .5 * dlam;

	sindlamm = sin(dlamm);
	costhm = cos(thm);	
	sinthm = sin(thm);
	cosdthm = cos(dthm);	
	sindthm = sin(dthm);
	
	L = sindthm * sindthm + (cosdthm * cosdthm - sinthm * sinthm)
		* sindlamm * sindlamm;

	if((1 - L) < DTOL) {
		// antipod that will crash -- force not quite 180 apart
		L = 1 - DTOL;
	}

   cosd = 1 - L - L;

	// test for coincidence
	if(cosd > .9999999883) {	   // minimum distance ~ 1 km
		// loc1 and loc2 coincide
		*S = 0.;
		// return azimuths > 2pi as undefined flag
		*az12 = *az21 = 9.69530;
		return;
	}

	// test for anitpodes
	if(cosd < -.999999)  antip = 1;

	// d is the Azimuth angle of the great circle arc
	d = acos(cosd);

	// compute distance (S) and
    // adjustment for azimuth (gives tandlammp)
	if (ellipse) {
    	// adjust for elliptic geod
		E = cosd + cosd;
		sind = sin( d );
		Y = sinthm * cosdthm;
		Y *= (Y + Y) / (1. - L);
		T = sindthm * costhm;
		T *= (T + T) / L;
		X = Y + T;
		Y -= T;
		T = d / sind;
		D = 4. * T * T;
		A = D * E;
		B = D + D;

		*S = a * sind * (T - f4 * (T * X - Y) +
			f64 * (X * (A + (T - .5 * (A - E)) * X) -
			Y * (B + E * Y) + D * X * Y));

		tandlammp = tan(.5 * (dlam - .25 * (Y + Y - E * (4. - X)) *
			(f2 * T + f64 * (32. * T - (20. * T - A)
			* X - (B + 4.) * Y)) * tan(dlam)));
	} else {
    	// results for spherical earth
		*S = a * d;
		tandlammp = tan(dlamm);
	}

	if(antip) {
		// for antipod return azimuths > 2pi as undefined flag
		*az12 = *az21 = 9.69530;
		return;
	}

	u = atan2(sindthm , (tandlammp * costhm));
	v = atan2(cosdthm , (tandlammp * sinthm));

	// get direct and reverse azimuth
	*az12 = adjlon(TWOPI + v - u);
	*az21 = adjlon(TWOPI - v - u);
}


//-----------------------------------------------
// compute maidenhead from position (long/lat)
//      negative is west long. or south lat.
//      return 0 == success
//
void make_mh(
    char *mh,  	// pointer to maidenhead output string
    double lat,	// location in degrees to convert
    double lng
    )
{
    double lg1, lt1;
    double wrk;
    int i;

    for(i=0; i<8; i++) mh[i] = ' ';
    mh[9] = '\0';

	 // adjust lt/lng for mh with special for wrapping boundry cases
	 if(lng >= 180) lg1 = 359.999999999;
    else lg1 = lng + 180.0;  /* convert degree longitude for maidenhead */
	 if(lat >= 90) lt1 = 179.999999999;
    else lt1 = lat +  90.0;  /* convert degree latitude for maidenhead  */

    mh[0] = 'A' + (int)(lg1/20.0);
    wrk = fmod(lg1,20.0);
    mh[2] = '0' + (int)(wrk/2.0);
    wrk = fmod(wrk,2.0);
    mh[4] = 'A' + (int)(wrk/(5.0/60.0));
    wrk = fmod(wrk,5.0/60.0);
    mh[6] = '0' + (int)(wrk/(5.0/600.0));

    mh[1] = 'A' + (int)(lt1/10.0);
    wrk = fmod(lt1,10.0);
    mh[3] = '0' + (int)(wrk/1.0);
    wrk = fmod(wrk,1.0);
    mh[5] = 'A' + (int)(wrk/(2.5/60.0));
    wrk = fmod(wrk,2.5/60.0);
    mh[7] = '0' + (int)(wrk/(2.5/600.0));

    return;

}

//-----------------------------------------------
// compute NGR from position (long/lat)
//      negative is west long. or south lat.
//      return 0 == success
//
void make_ngr(
    char *ngr,  	// pointer to maidenhead output string
    double lat,	// location in degrees to convert
    double lon
    )
{
    double e, n;
    double e1, e2, e3, n1;
    Int32 eastings, northings;
    int s500, s100;
    char buffer[10];

    if (lat < 49.0 || lat > 61.0 || lon < -8.0 || lon > 1.8) {
        StrCopy(ngr, "N/A");
        return;
    }

    e = (lon + 2.0) * D2R;
    n = lat * D2R;

    e1 = cos(n) * sin(e);
    e2 = PI / 4.0 + atan2(e1 / sqrt(1.0 - e1 * e1), 1.0) / 2.0;
    e3 = 6389.7 * log(sin(e2) / cos(e2)) + 400.0;
    n1 = 6371.28 * atan2(sin(n) / (cos(n) * cos(e)), 1.0) - 5548.79;

    eastings  = (Int32)(e3 * 1000.0 + 0.5);
    northings = (Int32)(n1 * 1000.0 + 0.5);

    eastings  += 1000000;
    northings += 500000;

    if (eastings  < 0 || eastings  >= 2500000 ||
        northings < 0 || northings >= 2500000) {
        StrCopy(ngr, "N/A");
        return;
    }

    s500 = eastings / 500000 + 5 * (4 - northings / 500000);
    if (s500 > 7)
         s500++;

    eastings %= 500000;
    northings %= 500000;

    s100 = eastings / 100000 + 5 * (4 - northings / 100000);
    if (s100 > 7)
          s100++;

    eastings %= 100000;
    northings %= 100000;

    ngr[0] = s500 + 'A';
    ngr[1] = s100 + 'A';
    ngr[2] = '\0';

    StrIToA(buffer, eastings / 10);

    switch (StrLen(buffer)) {
        case 1:
            StrCat(ngr, "000");
            break;
        case 2:
            StrCat(ngr, "00");
            break;
        case 3:
            StrCat(ngr, "0");
            break;
    }

    StrCat(ngr, buffer);

    StrIToA(buffer, northings / 10);

    switch (StrLen(buffer)) {
        case 1:
            StrCat(ngr, "000");
            break;
        case 2:
            StrCat(ngr, "00");
            break;
        case 3:
            StrCat(ngr, "0");
            break;
    }

    StrCat(ngr, buffer);
}

//---------------------------------------------
// position from maidenhead
// compute long/lat for center of maidenhead
//      negative is west long. or south lat.
//      return 0 = success
//      return -1 = bad and dout not changed
int p_mh(
    double *lat,    // pointer to lat (deg) for result
    double *lon,    // pointer to lng (deg) for result
    char *mh    // pointer to maidenhead input string
)
{
        int mhsz;
        char mhl[9];
        double lt, ln;

        mhsz = StrLen(mh);
		 // remove trailing blanks
		 if(mh[mhsz-1] == ' ') mhsz--;
		 if(mh[mhsz-1] == ' ') mhsz--;

        if(mhsz != 6 && mhsz != 4 && mhsz != 8)  return(-1);
        StrToLower(mhl, mh);               // ensure lower case
        if((mhl[0] < 'a') || (mhl[0] > 'r'))  return(-1);
        if((mhl[1] < 'a') || (mhl[1] > 's'))  return(-1);
        if((mhl[2] < '0') || (mhl[2] > '9'))  return(-1);
        if((mhl[3] < '0') || (mhl[3] > '9'))  return(-1);
        if(mhsz == 6 || mhsz == 8) {
            if((mhl[4] < 'a') || (mhl[4] > 'x'))  return(-1);
            if((mhl[5] < 'a') || (mhl[5] > 'x'))  return(-1);
        }
        if (mhsz == 8) {
            if (mhl[6] < '0' || mhl[6] > '9') return -1;
            if (mhl[7] < '0' || mhl[7] > '9') return -1;
        }

        ln = 20 * (double)(mhl[0] - 'a');
        lt = 10 * (double)(mhl[1] - 'a');
        ln += 2 * (double)(mhl[2] - '0');
        lt += 1 * (double)(mhl[3] - '0');
        if(mhsz == 6 || mhsz == 8) {
            ln += 0.08333333333 * (double)(mhl[4] - 'a');
            lt += 0.04166666667 * (double)(mhl[5] - 'a');
        }
        if (mhsz == 8) {
            ln += 0.00833333333 * (double)(mhl[6] - '0');
            lt += 0.00416666667 * (double)(mhl[7] - '0');
       }

        // adjust for center of grid
        if (mhsz == 8) {
            // its a 8 char grid
            ln += 0.00416666667;
            lt += 0.00208333333;
        } else if (mhsz == 6) {
            // its a 6 char grid
            ln += 0.04166666667;
            lt += 0.02083333333;
        } else {
            // its a 4 char grid (bigger grid square)
            ln += 1.0;
            lt += 0.5;
        }

        // convert from maidenhead degrees to actual lng/lat
        *lon = ln - 180;
        *lat = lt - 90;
        return(0);
}

//---------------------------------------------
// position from ngr
// compute long/lat for center of ngr
//      negative is west long. or south lat.
//      return 0 = success
//      return -1 = bad and dout not changed
int p_ngr(
    double *lat,    // pointer to lat (deg) for result
    double *lon,    // pointer to lng (deg) for result
    char *ngr    // pointer to NGR input string
)
{
    char ngrl[12], buffer[6];
    char first, second, c;
    int ngrsz, numsz, i, s500, s100, scale;
    Int32 northings, eastings, n, e;
    double t1, t2, nf;

    ngrsz = StrLen(ngr);

    StrToLower(ngrl, ngr);

    if (ngrsz != 2 && ngrsz != 4 && ngrsz != 6 && ngrsz != 8 && ngrsz != 10)
        return -1;

    first  = ngrl[0];
    second = ngrl[1];

    if (first != 's' && first != 't' && first != 'n' && first != 'h' && first != 'o')
        return -1;

    if (second < 'a' || second > 'z' || second == 'i')
        return -1;

    for (i = 0; i < (ngrsz - 2); i++) {
        c = ngrl[i + 2];

        if (c < '0' || c > '9')
             return -1;
    }

    s500 = first - 'a';
    if (s500 > 8) s500--;

    northings = 500000 * (3 - s500 / 5);
    eastings  = 500000 * (s500 % 5 - 2);

    s100 = second - 'a';
    if (s100 > 8) s100--;

    northings += 100000 * (4 - s100 / 5);
    eastings  += 100000 * (s100 % 5);

    if (ngrsz > 2) {
        scale = 0;

        switch (ngrsz) {
            case 4:
                scale = 10000;
                break;
            case 6:
                scale = 1000;
                break;
            case 8:
                scale = 100;
                 break;
            case 10:
                scale = 10;
                break;
        }

        numsz = (ngrsz - 2) / 2;

	  StrNCopy(buffer, ngrl + 2, numsz);
        e = StrAToI(buffer);

	  StrNCopy(buffer, ngrl + numsz + 2, numsz);
        n = StrAToI(buffer);

        eastings  += e * scale;
        northings += n * scale;
    }

    t1 = ((double)northings / 1000.0 + 5548.79) / 6371.28;
    t2 = 2.0 * atan2(exp(((double)eastings / 1000.0 - 400.0) / 6389.70), 1.0);

    *lon = atan2(-cos(t2) / (cos(t1) * sin(t2)), 1.0) * R2D - 2.0;

    nf = sin(t2) * sin(t1);
    *lat = atan2(nf / sqrt(1.0 - nf * nf), 1.0) * R2D;

    return 0;
}

/*
 * double degree to string in one of 3 formats
 */
static void
DmsToStr(char *sval, double dg, Int16 Fmt, Int16 lat)
{
	double deg, min, sec;
   char quad[2];
	char temp[12];

	deg = fabs(dg);

    if(lat == 0) {
    	if(dg >= 0.0)	StrCopy(quad, "E");
        else			StrCopy(quad, "W");
    } else if(lat == 1) {
    	if(dg >= 0.0)	StrCopy(quad, "N");
        else			StrCopy(quad, "S");
    } else {
		StrCopy(quad, "");
    }
    
	if(Fmt == 1) {
		// dd.dddd
		if(deg >= 360) deg -=360;
		if(lat<2) {
			StrDToA(sval, deg, 6, 12);
			StrCat(sval, " ");
	 		StrCat(sval, quad);
		} else {
			StrDToA(sval, deg, 2, 7);
			// xb0 is the degree symbol
			StrCat(sval, " \xb0");
		}
		return;    
    } 			  

    min = modf(deg, &deg) * 60.0;
    
    if(Fmt == 2) {
		// dd mm.mmmm
		if(lat<2) {
			if(min >= 59.99995) {
				min = 0.0;
				deg += 1;
				if(deg >= 360) deg -=360;
			}
			StrDToA(sval, deg, 0, 4);
			StrDToA(temp, min, 4, 7);
			StrCat(sval, "\xb0 ");			  
	 		StrCat(sval, temp);
			StrCat(sval, "'");
	 		StrCat(sval, quad);
		} else {
			if(min >= 59.95) {
				min = 0.0;
				deg += 1;
				if(deg >= 360) deg -=360;
			}
			StrDToA(sval, deg, 0, 4);
			StrDToA(temp, min, 1, 5);
			StrCat(sval, "\xb0 ");
	 		StrCat(sval, temp);
	 		StrCat(sval, "'");
		}
//	FrmCustomAlert(a_Debug, "DmsToStr", "minutes", sval);
 		return;    
    }

    sec = modf(min, &min) * 60.0;
    
	if(lat<2) {
		// dd mm ss.ss
		if(sec >= 59.995) {
			sec = 0.0;
			min += 1;
			if(min == 60) {min=0.0; deg++;}
			if(deg >= 360) deg -=360;
		}
		StrDToA(sval, deg, 0, 4);
		StrDToA(temp, min, 0, 4);
		StrCat(sval, "\xb0 ");
	 	StrCat(sval, temp);
		StrDToA(temp, sec, 2, 5);
		StrCat(sval, "' ");
	 	StrCat(sval, temp);
		StrCat(sval, "\"");
		StrCat(sval, quad);
	} else {
		if(sec >= 59.5) {
			sec = 0.0;
			min += 1;
			if(min == 60) {min=0.0; deg++;}
			if(deg >= 360) deg -=360;
		}
		StrDToA(sval, deg, 0, 4);
		StrDToA(temp, min, 0, 4);
		StrCat(sval, "\xb0 ");
	 	StrCat(sval, temp);
		StrDToA(temp, sec, 0, 4);
		StrCat(sval, "' ");
	 	StrCat(sval, temp);
		StrCat(sval, "\"");
	}
   return;

}


/*
 * Generate main working values
 */
static void
Calc_main()
{

	// build strings from current values
	// base lat
	DmsToStr(s_Blt, glPrefs.blat, glPrefs.degfmt, 1);
	// base lon
	DmsToStr(s_Bln, glPrefs.blng, glPrefs.degfmt, 0);
	// target lat
	DmsToStr(s_Tlt, glPrefs.tlat, glPrefs.degfmt, 1);
	// target lon
	DmsToStr(s_Tln, glPrefs.tlng, glPrefs.degfmt, 0);

	// MH calc for Tgt and Base from lat/lng
	make_mh(s_Bmh, glPrefs.blat, glPrefs.blng);
	make_mh(s_Tmh, glPrefs.tlat, glPrefs.tlng);

      make_ngr(s_Bng, glPrefs.blat, glPrefs.blng);
      make_ngr(s_Tng, glPrefs.tlat, glPrefs.tlng);

	return;

}

/*
 * Show the values on the main page
 */
static void
Show_main()
{
	// following holds grid centers for Grd2Grd
	double blat_c, blng_c, tlat_c, tlng_c;
	char dbstr[12];

	FormPtr  formPtr;

//	FrmCustomAlert(a_Debug, "Show Main", "entered", "");

	formPtr = FrmGetFormPtr(f_Main);

	// calc bearing/dist for Tgt & Base Lat/Lon
	geod_inv(glPrefs.blat*D2R, glPrefs.blng*D2R, glPrefs.tlat*D2R, glPrefs.tlng*D2R,
				&Az, &Baz, &Dist);

	if (Az < 0.) Az += TWOPI;
	if (Baz < 0.) Baz += TWOPI;

	// fill base and target name fields
   UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, f_BNm1)),
										glPrefs.bnam);

	UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, f_TNm1)),
										glPrefs.tnam);


	// fill the buttons from strings
	CtlSetLabel((ControlType *)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, b_BMh)), s_Bmh);
	CtlSetLabel((ControlType *)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, b_BNg)), s_Bng);
	CtlSetLabel((ControlType *)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, b_BLt)), s_Blt);
	CtlSetLabel((ControlType *)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, b_BLn)), s_Bln);
	CtlSetLabel((ControlType *)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, b_TMh)), s_Tmh);
	CtlSetLabel((ControlType *)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, b_TNg)), s_Tng);
	CtlSetLabel((ControlType *)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, b_TLt)), s_Tlt);
	CtlSetLabel((ControlType *)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, b_TLn)), s_Tln);

	// fill the calculated results
	// Bearing
	if((Az*R2D) > 360.0) StrCopy(dbstr, "Undefined");
	else DmsToStr(dbstr, Az*R2D, glPrefs.brgfmt, 2);
   UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr,
                   FrmGetObjectIndex(formPtr, f_Brg)),
                   dbstr);
	// Reverse
	if((Baz*R2D) > 360.0) StrCopy(dbstr, "Undefined");
	else DmsToStr(dbstr, Baz*R2D, glPrefs.brgfmt, 2);
   UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr,
                   FrmGetObjectIndex(formPtr, f_Rev)),
                   dbstr);

	// Dist km
   UpdateDblField((FieldPtr)FrmGetObjectPtr(formPtr,
                   FrmGetObjectIndex(formPtr, f_Dkm)),
                   Dist/1000, 1);

	// Do Grd2Grd
	// first get lat/lon for centers of MH
	if(p_mh(&blat_c, &blng_c, s_Bmh) ||
		p_mh(&tlat_c, &tlng_c, s_Tmh) ) {
		// an error converting mh to lat/lon
	   UpdateDblField((FieldPtr)FrmGetObjectPtr(formPtr,
	                   FrmGetObjectIndex(formPtr, f_Grd)),
	                   999999.9, 1);
	} else {    
		// calc bear/dist for center of MH to get Grd2Grd dist.
		geod_inv(blat_c*D2R, blng_c*D2R, tlat_c*D2R, tlng_c*D2R,
					&Az, &Baz, &Dist);

	   // display Grid to Grid	km
	   UpdateDblField((FieldPtr)FrmGetObjectPtr(formPtr,
	                   FrmGetObjectIndex(formPtr, f_Grd)),
	                   Dist/1000, 1);
	}

	return;

}

/*
 * Check the name field for validity
 *  (change any "/" to "\" and set return to 1 if changed
 */

UInt16
ChkNameStr(char *n)
{
	Int16 idx, didit;

	didit = 0;
	for(idx=0; idx<StrLen(n); idx++)	{
		if(n[idx] == '/') {
			n[idx] = '\\';
			didit = 1;
		}
		if(n[idx] == '$') {
			n[idx] = '&';
			didit = 1;
		}
	}
	return didit;

}

/*
 * Events for the BaseLL form
 */
static Boolean
BaseLLFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	char *s;
	FormPtr  formPtr = FrmGetActiveForm();
	FieldPtr wkflp;

	switch (event->eType) {
	    case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_BllCanc) {   
			    FrmGotoForm(f_Main);

	            handled = true;
	        }
	        if (event->data.ctlSelect.controlID == b_BllDn) {
				// Done button

				// do the latitude
				wkflp = (FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_BllLat));
				s = FldGetTextPtr(wkflp); 

				if(FldDirty(wkflp) && parseLL(&(glPrefs.blat), s, 1)) {
					FrmCustomAlert(a_Warning, "New Latitude value\n", "has invalid format.\n", 
									"Please reenter.");
					FrmGotoForm(f_BllIn);
					handled = true;
					break;
				}

				// do the longitude
				wkflp = (FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_BllLng));
				s = FldGetTextPtr(wkflp); 

				if(FldDirty(wkflp) && parseLL(&(glPrefs.blng), s, 0)) {
					FrmCustomAlert(a_Warning, "New Longitude value\n", "has invalid format.\n", 
									"Please reenter.");
					FrmGotoForm(f_BllIn);
					handled = true;
					break;
				} 

				// update name
				s = FldGetTextPtr((FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_BllNam)));

				if(s && ChkNameStr(s)) {
					FrmCustomAlert(a_Warning, "Name contains \"/\" or \"$\".\n", 
									"Changed this to \"\\\" or \"&\"\n", 
									"to avoid problems with Memo format.");
				}

				StrCopy(glPrefs.bnam, s ? s : "");
				
				FrmGotoForm(f_Main);
	            handled = true;
	        }
	        break;
	     
	    case frmOpenEvent:

	        FrmSetMenu(formPtr,m_Main);
	         
	        FrmDrawForm(formPtr);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_BllNam)),
							glPrefs.bnam);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_BllLat)),
							s_Blt);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_BllLng)),
							s_Bln);

	        FrmDrawForm(formPtr);

	        handled = true;
	        break;

	    case menuEvent:
	        handled = DoMenu(event->data.menu.itemID);
	        break;

	    default:
	        handled = false;

	}

	return handled;
}

/*
 * Events for the BaseMH form
 */
static Boolean
BaseMHFormHandleEvent(EventPtr event)
{
// add local variables for working on MH to LL conversion

	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	FieldPtr wkflp;


	switch (event->eType) {
	    case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_BmhCanc) {   
			    FrmGotoForm(f_Main);

	            handled = true;
	        }
	        if (event->data.ctlSelect.controlID == b_BmhDn) {
				// Done button

	        	// get MH string to convert   
				wkflp = (FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_BmhMh));
				s = FldGetTextPtr(wkflp);

//				FrmCustomAlert(a_Debug, "Base Locator", s, s_Bmh);
//				if(StrCompare(s, s_Bmh)) {
				if(FldDirty(wkflp)) {
					// only if the MH string changed...
					// calc new lat/lon for this mh
					if(!((s != NULL) && p_mh(&(glPrefs.blat), &(glPrefs.blng), s) == 0)) {
			            // problem on maidenhead
						FrmCustomAlert(a_Warning, "New Locator value\n", "has invalid format.\n", 
										"Please reenter.");
						FrmGotoForm(f_BmhIn);
						handled = true;
						break;
					}
				} 

				// do the name
				s = FldGetTextPtr((FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_BmhNam)));

				if(s && ChkNameStr(s)) {
					FrmCustomAlert(a_Warning, "Name contains \"/\" or \"$\".\n", 
									"Changed this to \"\\\" or \"&\"\n", 
									"to avoid problems with Memo format.");
				}

				StrCopy(glPrefs.bnam, s ? s : "");

				FrmGotoForm(f_Main);
	            handled = true;
	        }
	        break;
	     
	    case frmOpenEvent:

	        FrmSetMenu(formPtr,m_Main); 

	        FrmDrawForm(formPtr);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_BmhNam)),
							glPrefs.bnam);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_BmhMh)),
							s_Bmh);

	        FrmDrawForm(formPtr);

	        handled = true;
	        break;

	    case menuEvent:
	        handled = DoMenu(event->data.menu.itemID);
	        break;

	    default:
	        handled = false;

	}

	return handled;
}

/*
 * Events for the BaseNG form
 */
static Boolean
BaseNGFormHandleEvent(EventPtr event)
{
// add local variables for working on NGR to LL conversion

	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	FieldPtr wkflp;


	switch (event->eType) {
	    case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_BngCanc) {   
			    FrmGotoForm(f_Main);

	            handled = true;
	        }
	        if (event->data.ctlSelect.controlID == b_BngDn) {
				// Done button

	        	// get MH string to convert   
				wkflp = (FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_BngNg));
				s = FldGetTextPtr(wkflp);

//				FrmCustomAlert(a_Debug, "Base NGR", s, s_Bng);
//				if(StrCompare(s, s_Bng)) {
				if(FldDirty(wkflp)) {
					// only if the NGR string changed...
					// calc new lat/lon for this mh
					if(!((s != NULL) && p_ngr(&(glPrefs.blat), &(glPrefs.blng), s) == 0)) {
			            // problem on NGR
						FrmCustomAlert(a_Warning, "New NGR value\n", "has invalid format.\n", 
										"Please reenter.");
						FrmGotoForm(f_BngIn);
						handled = true;
						break;
					}
				} 

				// do the name
				s = FldGetTextPtr((FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_BngNam)));

				if(s && ChkNameStr(s)) {
					FrmCustomAlert(a_Warning, "Name contains \"/\" or \"$\".\n", 
									"Changed this to \"\\\" or \"&\"\n", 
									"to avoid problems with Memo format.");
				}

				StrCopy(glPrefs.bnam, s ? s : "");

				FrmGotoForm(f_Main);
	            handled = true;
	        }
	        break;
	     
	    case frmOpenEvent:

	        FrmSetMenu(formPtr,m_Main); 

	        FrmDrawForm(formPtr);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_BngNam)),
							glPrefs.bnam);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_BngNg)),
							s_Bng);

	        FrmDrawForm(formPtr);

	        handled = true;
	        break;

	    case menuEvent:
	        handled = DoMenu(event->data.menu.itemID);
	        break;

	    default:
	        handled = false;

	}

	return handled;
}

/*
 * Events for the TargetLL form
 */
static Boolean
TargetLLFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	char *s;
	FormPtr  formPtr = FrmGetActiveForm();
	FieldPtr wkflp;

	switch (event->eType) {
	    case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_TllCanc) {   
			    FrmGotoForm(f_Main);

	            handled = true;
	        }
	        if (event->data.ctlSelect.controlID == b_TllDn) {

				// Done button

				// do the latitude
				wkflp = (FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_TllLat));
				s = FldGetTextPtr(wkflp); 

				if(FldDirty(wkflp) && parseLL(&(glPrefs.tlat), s, 1)) {
					FrmCustomAlert(a_Warning, "New Latitude value\n", "has invalid format.\n", 
									"Please reenter.");
					FrmGotoForm(f_TllIn);
					handled = true;
					break;
				}

				// do the longitude
				wkflp = (FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_TllLng));
				s = FldGetTextPtr(wkflp); 

				if(FldDirty(wkflp) && parseLL(&(glPrefs.tlng), s, 0)) {
					FrmCustomAlert(a_Warning, "New Longitude value\n", "has invalid format.\n", 
									"Please reenter.");
					FrmGotoForm(f_TllIn);
					handled = true;
					break;
				} 

				// do the name
				s = FldGetTextPtr((FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_TllNam)));

				if(s && ChkNameStr(s)) {
					FrmCustomAlert(a_Warning, "Name contains \"/\" or \"$\".\n", 
									"Changed this to \"\\\" or \"&\"\n", 
									"to avoid problems with Memo format.");
				}

				StrCopy(glPrefs.tnam, s ? s : "");

				FrmGotoForm(f_Main);
	            handled = true;
	        }
	        break;
	     
	    case frmOpenEvent:

	        FrmSetMenu(formPtr,m_Main); 

	        FrmDrawForm(formPtr);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_TllNam)),
							glPrefs.tnam);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_TllLat)),
							s_Tlt);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_TllLng)),
							s_Tln);

	        FrmDrawForm(formPtr);

	        handled = true;
	        break;

	    case menuEvent:
	        handled = DoMenu(event->data.menu.itemID);
	        break;

	    default:
	        handled = false;

	}

	return handled;
}

/*
 * Events for the TargetMH form
 */
static Boolean
TargetMHFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	FieldPtr wkflp;

	switch (event->eType) {
		case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_TmhCanc) {   
			    FrmGotoForm(f_Main);

	            handled = true;
	        }
	        if (event->data.ctlSelect.controlID == b_TmhDn) {
				// Done button

	        	// get MH string to convert   
				wkflp = (FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_TmhMh));
				s = FldGetTextPtr(wkflp); 

				if(FldDirty(wkflp)) {
					// only if the MH string changed...
					// calc new lat/lon for this mh
					if(!((s != NULL) && p_mh(&(glPrefs.tlat), &(glPrefs.tlng), s) == 0)) {
						FrmCustomAlert(a_Warning, "New Locator value\n", "has invalid format.\n", 
										"Please reenter.");
						FrmGotoForm(f_TmhIn);
						handled = true;
						break;
					}
				}

				// do name
				s = FldGetTextPtr((FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_TmhNam)));

				if(s && ChkNameStr(s)) {
					FrmCustomAlert(a_Warning, "Name contains \"/\" or \"$\".\n", 
									"Changed this to \"\\\" or \"&\"\n", 
									"to avoid problems with Memo format.");
				}

				StrCopy(glPrefs.tnam, s ? s : "");

				FrmGotoForm(f_Main);
	            handled = true;
	        }

		    break;
		 
		case frmOpenEvent:

		    FrmSetMenu(formPtr,m_Main); 

	        FrmDrawForm(formPtr);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_TmhNam)),
							glPrefs.tnam);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_TmhMh)),
							s_Tmh);

		    FrmDrawForm(formPtr);

		    handled = true;
		    break;

		case menuEvent:
		    handled = DoMenu(event->data.menu.itemID);
		    break;

		default:
		    handled = false;

	}

	return handled;
}

/*
 * Events for the TargetNG form
 */
static Boolean
TargetNGFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	FieldPtr wkflp;

	switch (event->eType) {
		case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_TngCanc) {   
			    FrmGotoForm(f_Main);

	            handled = true;
	        }
	        if (event->data.ctlSelect.controlID == b_TngDn) {
				// Done button

	        	// get NGR string to convert   
				wkflp = (FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_TngNg));
				s = FldGetTextPtr(wkflp); 

				if(FldDirty(wkflp)) {
					// only if the NGR string changed...
					// calc new lat/lon for this mh
					if(!((s != NULL) && p_ngr(&(glPrefs.tlat), &(glPrefs.tlng), s) == 0)) {
						FrmCustomAlert(a_Warning, "New NGR value\n", "has invalid format.\n", 
										"Please reenter.");
						FrmGotoForm(f_TngIn);
						handled = true;
						break;
					}
				}

				// do name
				s = FldGetTextPtr((FieldPtr)FrmGetObjectPtr(formPtr, 
									FrmGetObjectIndex(formPtr, f_TngNam)));

				if(s && ChkNameStr(s)) {
					FrmCustomAlert(a_Warning, "Name contains \"/\" or \"$\".\n", 
									"Changed this to \"\\\" or \"&\"\n", 
									"to avoid problems with Memo format.");
				}

				StrCopy(glPrefs.tnam, s ? s : "");

				FrmGotoForm(f_Main);
	            handled = true;
	        }

		    break;
		 
		case frmOpenEvent:

		    FrmSetMenu(formPtr,m_Main); 

	        FrmDrawForm(formPtr);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_TngNam)),
							glPrefs.tnam);

			UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, 
							FrmGetObjectIndex(formPtr, f_TngNg)),
							s_Tng);

		    FrmDrawForm(formPtr);

		    handled = true;
		    break;

		case menuEvent:
		    handled = DoMenu(event->data.menu.itemID);
		    break;

		default:
		    handled = false;

	}

	return handled;
}

// Draw callback func for list lines
void ListDraw(Int16 itemNumber, RectanglePtr bounds, 
	char **text __attribute__ ((unused)))
{
	char *s;
		
	if(itemNumber < MAXLST) {
		s = GLLst[itemNumber].name;		// string to put in list
		// in resource, choose vertical size carefully or you
		// can have text remnants displaied after scrolling
		//  -- is there a better way to prevent this?? --
		WinDrawChars(s, StrLen(s), bounds->topLeft.x, bounds->topLeft.y);
	}
}

// Draw callback func for location list lines
void LocDraw(Int16 itemNumber, RectanglePtr bounds, 
	char **text __attribute__ ((unused)))
{
	char *s;
		
	if(itemNumber < MAXLOC) {
		s = GLLoc[itemNumber].name;		// string to put in list
		// in resource, choose vertical size carefully or you
		// can have text remnants displaied after scrolling
		//  -- is there a better way to prevent this?? --
		WinDrawChars(s, StrLen(s), bounds->topLeft.x, bounds->topLeft.y);
	}
}

/*
 * Events for the GLList form
 */
Boolean
GLListFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	ListType *list;
	WinDirectionType direct;
//	char		temp[16];

	switch (event->eType) {
	    case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_Can) {   

			    FrmGotoForm(f_Main);

	            handled = true;
	        }
			break;
	     
	    case lstSelectEvent:
	        glPrefs.lstIdx = event->data.lstSelect.selection;
//	FrmCustomAlert(a_Debug, "GLLst selection", StrIToA(temp, glPrefs.lstIdx), 
//					GLLst[glPrefs.lstIdx].name);
			// clean up and go to location form
			FrmEraseForm(formPtr);
//			FrmDeleteForm(formPtr);
		    FrmGotoForm(f_GLLocs);

           handled = true;
	        break;
	     
		case keyDownEvent:
			if ((event->data.keyDown.chr == pageUpChr) ||
					(event->data.keyDown.chr == pageDownChr)) {
				
				if(event->data.keyDown.chr == pageUpChr) direct = winUp;
				else direct = winDown;

				list = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, ListGL));
				LstScrollList (list, direct, (LstGetVisibleItems(list) - 1) );
				handled = true;
			}
			break;

	    case frmOpenEvent:
			//	FrmCustomAlert(a_Debug, "Form open GLLst", "reached", "");

	        FrmSetMenu(formPtr,m_Main); 

			FindGLRecord();							  // list GL recs in memoDB
			if(GLLstSz == 0) {
				FrmCustomAlert(a_Warning, "No 'GL-' memos found\n", "in Memo database.\n", 
								"");

			    FrmGotoForm(f_Main);
	            handled = true;
			}

			//	FrmCustomAlert(a_Debug, "GLLst", "list setup", "");
			list = FrmGetObjectPtr(formPtr,
					FrmGetObjectIndex(formPtr, ListGL));
			LstSetListChoices(list, NULL, GLLstSz);	  // list for draw callback
			LstSetSelection(list, -1);				  // no active selection
			LstSetDrawFunction(list, ListDraw);		  // callback func for draw

	        FrmDrawForm(formPtr);					  // draw it


	        handled = true;
	        break;

#if 0
	    case frmCloseEvent:
//	FrmCustomAlert(a_Debug, "GLLst", "close form", "");
	        handled = true;
	        break;
#endif

	    case menuEvent:
	        handled = DoMenu(event->data.menu.itemID);
	        break;

	    default:
	        handled = false;

	}

	return handled;
}


/*
 * Events for the GL Location form
 */
Boolean
GLLocFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	ListType *list;
	Int16	selection;
	Int16	ret;
	WinDirectionType direct;
	double  lat_tmp;
//	char		temp[16];

	switch (event->eType) {
	    case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_Can) {   

			    FrmGotoForm(f_Main);

	            handled = true;
	        }
	        if (event->data.ctlSelect.controlID == b_Chg) {
			    FrmGotoForm(f_GLLists);

	            handled = true;
	        }
			break;
	     
	    case lstSelectEvent:
	        selection = event->data.lstSelect.selection;

			GetLocData(selection);

//	FrmCustomAlert(a_Debug, s_WkMh, s_WkLt, s_WkLn);

			// see if lat/lon is ok
			if(!parseLL(&lat_tmp, s_WkLt, 1)) 
				if(!parseLL((glPrefs.ToLoc ? &(glPrefs.tlng) : &(glPrefs.blng)), s_WkLn, 0)) {
					// both lat and lng look good
					if(glPrefs.ToLoc)	glPrefs.tlat = lat_tmp;
					else				glPrefs.blat = lat_tmp;

					if(glPrefs.ToLoc)	StrCopy(glPrefs.tnam, GLLoc[selection].name);
					else				StrCopy(glPrefs.bnam, GLLoc[selection].name);

				    FrmGotoForm(f_Main);
		            handled = true;
					break;
				}

			// lat/lon had a problem -- try maidenhead
			if(glPrefs.ToLoc) {
				ret = p_mh(&(glPrefs.tlat), &(glPrefs.tlng), s_WkMh);
			} else {
				ret = p_mh(&(glPrefs.blat), &(glPrefs.blng), s_WkMh);
			}

			if(ret == 0) {

				// MH was converted ok	
				if(glPrefs.ToLoc)	StrCopy(glPrefs.tnam, GLLoc[selection].name);
				else				StrCopy(glPrefs.bnam, GLLoc[selection].name);

				FrmGotoForm(f_Main);
				handled = true;
				break;
			}
			 
			FrmCustomAlert(a_Warning, "The Location record\n", "Could not be evaluated.", "");
           handled = true;
	        break;

		case keyDownEvent:
			if ((event->data.keyDown.chr == pageUpChr) ||
					(event->data.keyDown.chr == pageDownChr)) {
				
				if(event->data.keyDown.chr == pageUpChr) direct = winUp;
				else direct = winDown;

				list = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, LstLoc));
				LstScrollList (list, direct, (LstGetVisibleItems(list) - 1) );
				handled = true;
			}
			break;

	     
	    case frmOpenEvent:
			//	FrmCustomAlert(a_Debug, "Form open GLLoc", "reached", "");

			// see if program was restarted and rebuild list if so
			if(GLLstSz == 0) {
				//   FrmCustomAlert(a_Debug, "Form GLLoc", "LstSz is 0", "");
				FindGLRecord();	  			  // list GL recs in memoDB
				if(GLLstSz == 0) {
					// still bad -- back to main
				    FrmGotoForm(f_Main);
		            handled = true;
					break;
				}
			}

			//	   FrmCustomAlert(a_Debug, "Form GLLoc", "LstSz not 0", "");

			if(glPrefs.lstIdx == LSTEMPTY) {
			    FrmGotoForm(f_GLLists);
	            handled = true;
				break;
			} 

	        FrmSetMenu(formPtr,m_Main);

//	FrmCustomAlert(a_Debug, "Form open GLLoc", "before", GLLst[glPrefs.lstIdx].name);
	        UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, f_LstNam)), 
	        				GLLst[glPrefs.lstIdx].name); 
	        UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, f_LstTo)), 
	        				(glPrefs.ToLoc ? "[to Target]" : "[to Base]  ") ); 

			if(FindGLLocs()) {	   // list GL recs in memoDB
				// FrmCustomAlert(a_Debug, "GLLoc", "bad call to", "FindGLLocs");
			    FrmGotoForm(f_GLLists);
	            handled = true;
				break;
			}
			if(GLLocSz == 0) {
				FrmCustomAlert(a_Warning, "No location definitions\n", "in this record.\n", 
								"");

//				FrmEraseForm(formPtr);
//				FrmDeleteForm(formPtr);
			    FrmGotoForm(f_GLLists);
	            handled = true;
				break;
			}
//	FrmCustomAlert(a_Debug, "GLLoc", "list setup", "");
			list = FrmGetObjectPtr(formPtr,
					FrmGetObjectIndex(formPtr, LstLoc));
			LstSetListChoices(list, NULL, GLLocSz);	  // list for draw callback
			LstSetSelection(list, -1);				  // no active selection
			LstSetDrawFunction(list, LocDraw);		  // callback func for draw

	        FrmDrawForm(formPtr);					  // draw it


	        handled = true;
	        break;

#if 0
	    case frmCloseEvent:
//	FrmCustomAlert(a_Debug, "GLLoc", "close form", "");
	        handled = true;
	        break;
#endif

	    case menuEvent:
	        handled = DoMenu(event->data.menu.itemID);
	        break;

	    default:
	        handled = false;

	}

	return handled;
}

/*
 * Events for the Loc Choose Action form
 */
static Boolean
SelectFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	char	recStr[60];
	Int16 i;

	switch (event->eType) {
	    case ctlSelectEvent:
			if (event->data.ctlSelect.controlID == b_LsCanc) {   
				FrmGotoForm(f_Main);
				handled = true;
				break;
			}
																			
			if (event->data.ctlSelect.controlID == b_LsLoad) {				
				if(glPrefs.lstIdx < LSTEMPTY) {								
				    FrmGotoForm(f_GLLocs);									
				}else {														
				    FrmGotoForm(f_GLLists);									
				}
				handled = true;
				break;
			}
			if (event->data.ctlSelect.controlID == b_LsSave) {
				// build the record to save
				StrCopy(recStr, "$");
				glPrefs.ToLoc ? StrCat(recStr, glPrefs.tnam) : StrCat(recStr, glPrefs.bnam);
				StrCat(recStr, "/");
				glPrefs.ToLoc ? StrCat(recStr, s_Tmh) : StrCat(recStr, s_Bmh);
				StrCat(recStr, "/");
				glPrefs.ToLoc ? StrCat(recStr, s_Tlt) : StrCat(recStr, s_Blt);
				StrCat(recStr, "/");
				glPrefs.ToLoc ? StrCat(recStr, s_Tln) : StrCat(recStr, s_Bln);
				StrCat(recStr, "/");

				// change any degree symbols to 'd' for saving
				for(i=0; i<StrLen(recStr); i++) {
					if(recStr[i] == '\xb0') recStr[i] = 'd';
				}

				//  FrmCustomAlert(a_Debug, "Record save", "string = ", recStr);

				// save it to Memo record
				if(SaveToGL(recStr)) {
			 		FrmCustomAlert(a_Warning, "A Problem Occured", "while saving location", "");
				}   
				FrmGotoForm(f_Main);
				handled = true;
				break;
			}

			if (event->data.ctlSelect.controlID == b_LsGps) { 
  
				FrmGotoForm(f_GPS);
				handled = true;
				break;
			}

	        break;
	     
	    case frmOpenEvent:

		 //	FrmCustomAlert(a_Debug, "Form open select", "reached", "");
		    FrmSetMenu(formPtr,m_Main); 

	        FrmDrawForm(formPtr);

	        UpdateTextField((FieldPtr)FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, f_LstTo1)), 
	        				(glPrefs.ToLoc ? "[to Target]" : "[to Base]  ") ); 

		    FrmDrawForm(formPtr);

		    handled = true;
		    break;

	    case menuEvent:
	        handled = DoMenu(event->data.menu.itemID);
	        break;

	    default:
	        handled = false;

	}

	return handled;
}



/*
 * Events for the option form
 */
static Boolean
OptionFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	ControlPtr southChk, westChk;
	UInt16 datm, dms;

	southChk = FrmGetObjectPtr(formPtr, 
						FrmGetObjectIndex(formPtr, c_SLat));
	westChk = FrmGetObjectPtr(formPtr, 
						FrmGetObjectIndex(formPtr, c_WLon));
    
    switch (event->eType) {
        case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_OpCanc) {
	        	//Cancel button   
			    FrmGotoForm(f_Main);

	            handled = true;
	        }

	        if (event->data.ctlSelect.controlID == b_OpDn) {
				// Done button
				glPrefs.forceSouth = CtlGetValue(southChk);
				glPrefs.forceWest = CtlGetValue(westChk);
				datm = FrmGetObjectId(formPtr, 
							FrmGetControlGroupSelection(formPtr, 1));
				switch (datm) {
					case r_Ell1:
						// WGS 84 selected
						glPrefs.datum = 1;
						break;
					case r_Ell2:
						// NAD27 selected
						glPrefs.datum = 2;
						break;
					case r_Ell3:
						// Sphere selected
						glPrefs.datum = 3;
						break;
					default:
						// unknown - use WGS 84
						glPrefs.datum = 1;
						break;
				}
				setEllipse();

				dms = FrmGetObjectId(formPtr, 
							FrmGetControlGroupSelection(formPtr, 2));
				switch (dms) {
					case r_Dd1:
						// dd.ddd selected
						glPrefs.degfmt = 1;
						break;
					case r_Dd2:
						// dd mm.mmm selected
						glPrefs.degfmt = 2;
						break;
					case r_Dd3:
						// dd mm ss.sss selected
						glPrefs.degfmt = 3;
						break;
					default:
						// unknown - use dd.ddd
						glPrefs.degfmt = 1;
						break;
				}

				dms = FrmGetObjectId(formPtr, 
							FrmGetControlGroupSelection(formPtr, 3));
				switch (dms) {
					case r_Bd1:
						// dd.ddd selected
						glPrefs.brgfmt = 1;
						break;
					case r_Bd2:
						// dd mm.mmm selected
						glPrefs.brgfmt = 2;
						break;
					case r_Bd3:
						// dd mm ss.sss selected
						glPrefs.brgfmt = 3;
						break;
					default:
						// unknown - use dd.ddd
						glPrefs.brgfmt = 1;
						break;
				}
	           
			    FrmGotoForm(f_Main);

	            handled = true;
	        }

	        if (event->data.ctlSelect.controlID == b_OpTime) {
	        	// GPS Time button
	        	glPrefs.ToLoc = 2;     // tell GPS form what we want   
			    FrmGotoForm(f_UTC);

	            handled = true;
	        }


            
            break;
         
        case frmOpenEvent:
			CtlSetValue(southChk, glPrefs.forceSouth);
			CtlSetValue(westChk, glPrefs.forceWest);
			// valid datum values:
			// 1 = WGS 84
			// 2 = NAD27
			// 3 = sphere
			if((glPrefs.datum < 1) && (glPrefs.datum > 3)) {
				glPrefs.datum = 1;
			}
			switch(glPrefs.datum) {
				case 1:
					FrmSetControlGroupSelection(formPtr, 1, r_Ell1);
					break;
				case 2:
					FrmSetControlGroupSelection(formPtr, 1, r_Ell2);
					break;
				case 3:
					FrmSetControlGroupSelection(formPtr, 1, r_Ell3);
					break;
				default:
					// if none, force WGS 84
					FrmSetControlGroupSelection(formPtr, 1, r_Ell1);
					break;
			}
			// valid dms values:
			// 1 = dd.ddd
			// 2 = dd mm.mmm
			// 3 = dd mm ss.ss
			if((glPrefs.degfmt < 1) && (glPrefs.degfmt > 3)) {
				glPrefs.degfmt = 1;
			}
			switch(glPrefs.degfmt) {
				case 1:
					FrmSetControlGroupSelection(formPtr, 2, r_Dd1);
					break;
				case 2:
					FrmSetControlGroupSelection(formPtr, 2, r_Dd2);
					break;
				case 3:
					FrmSetControlGroupSelection(formPtr, 2, r_Dd3);
					break;
				default:
					// if none, force dd.ddd
					FrmSetControlGroupSelection(formPtr, 2, r_Dd1);
					break;
			}

			// valid brg values:
			// 1 = dd.ddd
			// 2 = dd mm.mmm
			// 3 = dd mm ss.ss
			if((glPrefs.brgfmt < 1) && (glPrefs.brgfmt > 3)) {
				glPrefs.brgfmt = 1;
			}
			switch(glPrefs.brgfmt) {
				case 1:
					FrmSetControlGroupSelection(formPtr, 3, r_Bd1);
					break;
				case 2:
					FrmSetControlGroupSelection(formPtr, 3, r_Bd2);
					break;
				case 3:
					FrmSetControlGroupSelection(formPtr, 3, r_Bd3);
					break;
				default:
					// if none, force dd.ddd
					FrmSetControlGroupSelection(formPtr, 3, r_Bd1);
					break;
			}

            FrmDrawForm(formPtr);

            handled = true;
            break;

        default:
            handled = false;

    }

    return handled;
}


/*
 * Events for the GPS Offset form
 * adjust the GPS offset value
 */
static Boolean
GPSOffFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	// uses global Int16 locOff;

	switch (event->eType) {
	    case ctlSelectEvent:
	        if (event->data.ctlSelect.controlID == b_OffCanc) {   
				FrmGotoForm(f_Main);
	            handled = true;
	        }
	        if (event->data.ctlSelect.controlID == b_OffDone) {
				// Done button
				glPrefs.GPSOffst = locOff;
				FrmGotoForm(f_GPS);
	            handled = true;
	        }
	        break;
	     
        case ctlRepeatEvent:
        	if (event->data.ctlSelect.controlID == btn_OffUp) {
				
				if(locOff < 12) locOff++;
				else locOff = -12;

				UpdateNumField((FieldPtr)FrmGetObjectPtr(formPtr,
					              FrmGetObjectIndex(formPtr, f_UOff)),
					              locOff);
				FldDrawField((FieldPtr)FrmGetObjectPtr(formPtr,
					              FrmGetObjectIndex(formPtr, f_UOff)));
				
           } else if (event->data.ctlSelect.controlID == btn_OffDn) {
				if(locOff > -12) locOff--;
				else locOff = 12;
					
				UpdateNumField((FieldPtr)FrmGetObjectPtr(formPtr,
					              FrmGetObjectIndex(formPtr, f_UOff)),
					              locOff);
				FldDrawField((FieldPtr)FrmGetObjectPtr(formPtr,
					              FrmGetObjectIndex(formPtr, f_UOff)));
				
           }
//			handled = true;  // to repeat, don't return handled
           break;

	    case frmOpenEvent:

	        FrmSetMenu(formPtr,m_Main);
	        locOff = glPrefs.GPSOffst; 

	        FrmDrawForm(formPtr);

			UpdateNumField((FieldPtr)FrmGetObjectPtr(formPtr,
				              FrmGetObjectIndex(formPtr, f_UOff)),
				              locOff);
			FldDrawField((FieldPtr)FrmGetObjectPtr(formPtr,
				              FrmGetObjectIndex(formPtr, f_UOff)));
//	        FrmDrawForm(formPtr);

	        handled = true;
	        break;

	    case menuEvent:
	        handled = DoMenu(event->data.menu.itemID);
	        break;

	    default:
	        handled = false;

	}

	return handled;
}


/*
 * Events for the GPS form
 * read location or time from GPS
 *  if glPrefs.ToLoc == 2, we want the time value
 *  otherwise it tells base vs target for location
 */
static Boolean
GPSFormHandleEvent(EventPtr event)
{
	Boolean handled = false;
	FormPtr  formPtr = FrmGetActiveForm();
	Err err;
	char Sentance[84];
	char s1[22];	// lat or hr
	char s2[22];	// lon or min
	char s3[16];	// nam or sec
	UInt32			timsecs;
	DateTimeType	now;

// for debug
//	char		temp[16];
//	char		temp2[16];
//	Boolean	bol1, bol2;


	switch (event->eType) {
    
	    case frmOpenEvent:

		 //	FrmCustomAlert(a_Debug, "Form GPS open", "reached", "");
		    FrmSetMenu(formPtr,f_GPS); 

	        FrmDrawForm(formPtr);

			// set up serial and read
			err = StartSerial();
			if(err) {
				StopSerial();
				FrmCustomAlert(a_Warning, "Can Not open\n", "the Serial Port.", "");
				FrmGotoForm(f_Main);									
				handled = true;
				break;
			}

			if(glPrefs.ToLoc == 2) {
				// doing UTC update, get current Palm time
				timsecs = TimGetSeconds();
				TimSecondsToDateTime(timsecs, &now);
			}

			FlushSerial();

			err = Get_Sentance(Sentance, "GPRMC");
			if(err) {

#if 0
	// debug stuff
	FrmCustomAlert(a_Debug, "Sentance error", StrIToH(temp, err), StrIToA(temp2, count));
	if(err == serErrLineErr) {
		err = SerGetStatus(serRef, &bol1, &bol2);
		FrmCustomAlert(a_Debug, "Line Error", "Status is:", StrIToH(temp, err));
	}
#endif

				StopSerial();
				FrmCustomAlert(a_Warning, "Can Not read from\n", "the GPS unit.", "");
				FrmGotoForm(f_Main);
				handled = true;
			    break;
			}

			StopSerial();

			//	FrmCustomAlert(a_Warning, "", "", Sentance);

			err = VerifyChecksum(Sentance);
			if(err) {
				FrmCustomAlert(a_Warning, "Checksum error\n", "on GPS\n", "Sentance");
				FrmGotoForm(f_Main);									
			    handled = true;
			    break;
			}


			if(glPrefs.ToLoc == 2) {
				// get the time from GPS sentance
				err = GetGPSTime(Sentance, s1, s2, s3);
				if(!err) {
					// update local time with UTC from GPS
						//hr  = s1
						//min = s2
						//sec = s3
						// adjust GPS hour to local
						now.hour = (Int16)StrAToI(s1);
						now.hour += glPrefs.GPSOffst;
						if(now.hour < 0) now.hour += 24;
						if(now.hour >= 24) now.hour -= 24;
						now.minute = (Int16)StrAToI(s2);
						now.second = (Int16)StrAToI(s3);

						// set the adjusted time/date to Palm clock
						timsecs = TimDateTimeToSeconds(&now);
						TimSetSeconds(timsecs);

//						FrmCustomAlert(a_Warning, s1, s2, s3);
				} else FrmCustomAlert(a_Warning, "The GPS time\n", "is not valid.", "");
			} else {
				// get the position from GPS sentance
				err = GetGPSLatLon(Sentance, s1, s2, s3);
				if(!err) {
					// update location with the values from GPS
					parseLL((glPrefs.ToLoc ? &(glPrefs.tlat) : &(glPrefs.blat)), s1, 1);
					parseLL((glPrefs.ToLoc ? &(glPrefs.tlng) : &(glPrefs.blng)), s2, 0);
					StrCopy((glPrefs.ToLoc ? glPrefs.tnam : glPrefs.bnam), s3);
				} else FrmCustomAlert(a_Warning, "The GPS position\n", "is not valid.", "");
			}

			FrmGotoForm(f_Main);									
		    handled = true;
		    break;

	    case menuEvent:
	        handled = DoMenu(event->data.menu.itemID);
	        break;

	    default:
	        handled = false;

	}

	return handled;
}

/*
 * Events for the main form
 */
static Boolean
MainFormHandleEvent(EventPtr event)
{
    Boolean handled = false;
    FormPtr  formPtr = FrmGetActiveForm();
    
    switch (event->eType) {
        case ctlSelectEvent:
            if (event->data.ctlSelect.controlID == b_Bhdr) {
		        glPrefs.ToLoc = 0;
				FrmGotoForm(f_LocSel);
               handled = true;
            }
            
            if (event->data.ctlSelect.controlID == b_BMh) {
	        
			    FrmGotoForm(f_BmhIn);
               handled = true;
            }
            
            if (event->data.ctlSelect.controlID == b_BNg) {
	        
			    FrmGotoForm(f_BngIn);
               handled = true;
            }
            
            if (event->data.ctlSelect.controlID == b_BLt ||
            	 event->data.ctlSelect.controlID == b_BLn) {
	        
			    FrmGotoForm(f_BllIn);
               handled = true;
            }
            
            if (event->data.ctlSelect.controlID == b_Thdr) {
		        glPrefs.ToLoc = 1;
				FrmGotoForm(f_LocSel);
               handled = true;
            }
            
            if (event->data.ctlSelect.controlID == b_TMh) {
	        
				FrmGotoForm(f_TmhIn);
				handled = true;
            }
            
            if (event->data.ctlSelect.controlID == b_TNg) {
	        
				FrmGotoForm(f_TngIn);
				handled = true;
            }
            
            if (event->data.ctlSelect.controlID == b_TLt ||
            	 event->data.ctlSelect.controlID == b_TLn) {
	        
			    FrmGotoForm(f_TllIn);
               handled = true;
            }

            break;
         
        case frmOpenEvent:

//	FrmCustomAlert(a_Debug, "Form open main", "reached", "");
            FrmSetMenu(formPtr,m_Main); 

            FrmDrawForm(formPtr);
        
			 Calc_main();
            Show_main();           

            FrmDrawForm(formPtr);
        
            handled = true;
            break;

        case menuEvent:
            handled = DoMenu(event->data.menu.itemID);
            break;

        default:
            handled = false;

    }

    return handled;
}


/*
 * Process all menu selections. This is possible because
 * all Astro forms use the same menu.
 */
Boolean
DoMenu(UInt16 itemID)
{
    Boolean handled = false;
    
    switch (itemID) {

        case mi_Help:
            FrmHelp(s_Mhlp);
            handled = true;
            break;

        case mi_About:
            FrmAlert(a_About);
            handled = true;
            break;

        case m_Copy :
            FldCopy( Id2Ptr( GetFocusID() ) );
            break;

        case m_Paste :
            FldPaste( Id2Ptr( GetFocusID() ) );
            break;

        case m_Opts :
		    FrmGotoForm(f_Options);
            handled = true;
            break;

    }

    return handled;
}

/*
 * Events for the main application (basically, setting form event
 * handlers.)
 */
static Boolean
ApplicationHandleEvent(EventPtr event)
{
    FormPtr form;
    Boolean handled = false;

    if (event->eType == frmLoadEvent) {
        glPrefs.activeFormId = event->data.frmLoad.formID;
        form = FrmInitForm(glPrefs.activeFormId);
        FrmSetActiveForm(form);
        
        switch (glPrefs.activeFormId) {
            case f_Main:
                FrmSetEventHandler(form,
                                   MainFormHandleEvent);
                handled = true;
                break;
            case f_LocSel:
                FrmSetEventHandler(form,
                                   SelectFormHandleEvent);
                handled = true;
                break;
            case f_BmhIn:
                FrmSetEventHandler(form,
                                   BaseMHFormHandleEvent);
                handled = true;
                break;
            case f_BngIn:
                FrmSetEventHandler(form,
                                   BaseNGFormHandleEvent);
                handled = true;
                break;
            case f_BllIn:
                FrmSetEventHandler(form,
                                   BaseLLFormHandleEvent);
                handled = true;
                break;
            case f_TmhIn:
                FrmSetEventHandler(form,
                                   TargetMHFormHandleEvent);
                handled = true;
                break;
            case f_TngIn:
                FrmSetEventHandler(form,
                                   TargetNGFormHandleEvent);
                handled = true;
                break;
            case f_TllIn:
                FrmSetEventHandler(form,
                                   TargetLLFormHandleEvent);
                handled = true;
                break;
           case f_GLLists:
               FrmSetEventHandler(form,
                                  GLListFormHandleEvent);
               handled = true;
               break;
           case f_GLLocs:
               FrmSetEventHandler(form,
                                  GLLocFormHandleEvent);
               handled = true;
               break;
            case f_Options:
                FrmSetEventHandler(form,
                                   OptionFormHandleEvent);
                handled = true;
                break;
            case f_GPS:
                FrmSetEventHandler(form,
                                   GPSFormHandleEvent);
                handled = true;
                break;
            case f_UTC:
                FrmSetEventHandler(form,
                                   GPSOffFormHandleEvent);
                handled = true;
                break;

         }
    }    return handled;
}

/*
 * Application event loop. duh.
 */
static void
EventLoop(void)
{
    EventType   event;
    UInt16      error;

    do {
        EvtGetEvent(&event,
                    evtWaitForever);
        if (!SysHandleEvent(&event))
            if (!MenuHandleEvent(NULL,
                                 &event,
                                 &error))
                if (!ApplicationHandleEvent(&event))
                    FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

/*
 * Initialize the math library; allocate the handles for
 * various text fields; load our previously saved preferences.
 */
static void
StartApplication(void)
{
    PrefGetPreferences(&sysPrefs);

    if ((PrefGetAppPreferences(APPID,
                              0,
                              &glPrefs,
                              &gptSize,
                              true) == noPreferenceFound) ||
        gptSize != sizeof(glPrefs) ||
        glPrefs.version != GLVERSION ) {	
        /*
         * New install or else preferences are old.  Use default
         * settings
         */
        glPrefs.version = GLVERSION;
        StrCopy(glPrefs.bnam, "Alport Heights");
        StrCopy(glPrefs.tnam, "Pfaffhausen"); 

		// original values (not debug !!)
        glPrefs.blat = 53.060417;
        glPrefs.blng = -1.545833;
        glPrefs.tlat = 47.364583;
        glPrefs.tlng = 8.620833;
        glPrefs.lstIdx = LSTEMPTY;
        glPrefs.ToLoc = 1;

//        glPrefs.dmode = 0;
		 glPrefs.forceWest = 1;
		 glPrefs.forceSouth = 0;
		 glPrefs.datum = 1;
		 glPrefs.degfmt = 1;
		 glPrefs.brgfmt = 1;
        glPrefs.activeFormId = f_Main;
        glPrefs.memoDbSize = 0;
        glPrefs.GPSOffst = 0;

		 setEllipse();
    }

	 // gen things some forms may need
	 Calc_main();
	 InitForGPS();

    saveFormId = glPrefs.activeFormId;
    FrmGotoForm(glPrefs.activeFormId);
}

/*
 * Releases all memory; closes libraries; saves preferences
 */
static void
StopApplication()
{
    Err         error;
    UInt16      usecount;

    FrmCloseAllForms();

    PrefSetAppPreferences(APPID,
                          0, 1,
                          &glPrefs,
                          sizeof(glPrefs),
                          true);
    
    error = MathLibClose(MathLibRef, &usecount);
    ErrFatalDisplayIf(error, "Can't close MathLib");
    if (usecount == 0)
        SysLibRemove(MathLibRef);

}

/***********************************************************************
 *
 * FUNCTION:    RomVersionCompatible
 *
 * DESCRIPTION: P4. Check that the ROM version meets your
 *              minimum requirement.  Warn if the app was switched to.
 *
 * PARAMETERS:  requiredVersion - minimum rom version required
 *                                (see sysFtrNumROMVersion in SystemMgr.h 
 *                                for format)
 *              launchFlags     - flags indicating how the application was
 *											 launched.  A warning is displayed only if
 *                                these flags indicate that the app is 
 *											 launched normally.
 *
 * RETURNED:    zero if rom is compatible else an error code
 *                             
 ***********************************************************************/
static Err RomVersionCompatible (UInt32 requiredVersion, UInt16 launchFlags)
{
	UInt32 romVersion;
	
	
	// See if we're on in minimum required version of the ROM or later.
	// The system records the version number in a feature.  A feature is a
	// piece of information which can be looked up by a creator and feature
	// number.
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (romVersion < requiredVersion)
		{
		// If the user launched the app from the launcher, explain
		// why the app shouldn't run.  If the app was contacted for something
		// else, like it was asked to find a string by the system find, then
		// don't bother the user with a warning dialog.  These flags tell how
		// the app was launched to decided if a warning should be displayed.
		if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
			{
				ErrDisplay("Your older Palm OS version can't run this program.");
		
			// Pilot 1.0 will continuously relaunch this app unless we switch to 
			// another safe one.  The sysFileCDefaultApp is considered "safe".
			if (romVersion < 0x02000000)
				{
				AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
				}
			}
		
		return (sysErrRomIncompatible);
		}

	return 0;
}



UInt32
PilotMain(UInt16 cmd,
          MemPtr cmdBPD __attribute__ ((unused)),
          UInt16 launchFlags)
{
    Err         error;
    
    if (cmd == sysAppLaunchCmdNormalLaunch)
    {        
		error = RomVersionCompatible (RomRunVers, launchFlags);
		if(error)	return error;
        
        error = SysLibFind(MathLibName, &MathLibRef);
        if (error)
            error = SysLibLoad(LibType, MathLibCreator, &MathLibRef);
        
        ErrFatalDisplayIf(error,
                          "MATHLIB NOT FOUND. You must install MathLib.prc To use this program.");
        
        error = MathLibOpen(MathLibRef, MathLibVersion);
        ErrFatalDisplayIf(error,
                          "MATHLIB NOT LOADED. MathLib.prc could not be opened.");
        
        // Main application
        StartApplication();
        EventLoop();
        StopApplication();
        // End of main application.
        
        return 0;
    }

    return 0;
}

