/*
 * Grid Locator - calculator for distance/bearing for PalmOS devices.
 *
 * file: PalmUtil.c
 * Copyright (C) 2001, Rex Allers
 *
 *   Some program structure based on: 
 *   Astro Info (Astro.c) Copyright (C) 2000, Michael Heinz
 *
 *	  Dist/bear modules based on code from PROJ.4 1995 by 
 *		Gerald I. Evenden
 *		USGS, Woods Hole, MA 02543
 *
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <PalmOS.h>
#include "PalmUtil.h"
#include "resource.h"

/* Max length of all text fields */
#define FIELDLENGTH     17

/*
 * Returns true if the ROM version is at least
 * minVersion.
 */
Boolean CheckMinRomVersion(UInt32 minVersion)
{
    UInt32 romVersion;
    
    FtrGet(sysFtrCreator,
           sysFtrNumROMVersion,
           &romVersion);

    return (minVersion <= romVersion);
}
// return ID of current focused object
//   returns noFocus if none focused
UInt16 GetFocusID( void )
{
    UInt16 wFocused = FrmGetFocus( FrmGetActiveForm() );

    if ( wFocused != noFocus )
    {
        wFocused = FrmGetObjectId( FrmGetActiveForm(), wFocused );
    }
    
    return( wFocused );
}

// return pointer to an object in a form
void *Id2Ptr( UInt16 wID )
{
    return( FrmGetObjectPtr( FrmGetActiveForm(), FrmGetObjectIndex( FrmGetActiveForm(), wID ) ) );
}


/*
 * Update a text field and redraw it.
 */
void
UpdateTextField(FieldPtr fld, char *t)
{
    MemHandle   h;
	 char		*s;
    UInt16      size, length;
    
    if (fld)
    {
        /*
         * Make sure the text will fit.
         */
        size = FldGetMaxChars(fld);
        length = StrLen(t);
        if (length>size)
            length = size;
        
        FldFreeMemory(fld);
        h = MemHandleNew(length+1);
        if (h) {
            s = MemHandleLock(h);
            strNcopy(s,t,length);
            MemHandleUnlock(h);
            FldSetTextHandle(fld,h); // Restore the handle.
        }
        else
        {
            ErrFatalDisplay("Out of Memory!");
        }
    }
    else
    {
        ErrFatalDisplay("Null Field Pointer!");
    }
}


/*
 * Update a numeric Int16 field and redraw it.
 */
void
UpdateNumField(FieldPtr fld, Int16 num)
{
    MemHandle   h;
	 char		*s;
    UInt16      size;

    if (fld) {
        FldFreeMemory(fld);
        size = FldGetMaxChars(fld);
        h = MemHandleNew(size+2);

        if (h) {
            s = MemHandleLock(h);
            StrIToA(s,num);
            MemHandleUnlock(h);
            FldSetTextHandle(fld,h);
        } else {
            ErrFatalDisplay("Out of Memory!");
        }
    } else {
        ErrFatalDisplay("Null Field Pointer!");
    }
}

/*
 * Update an editable integer field and redraw it.
 */
void
UpdateEditField(FieldPtr fld, long num)
{
    char s[FIELDLENGTH];
    
    FldFreeMemory(fld); 
    StrIToA(s,num); 
    FldInsert(fld,s,StrLen(s)); 
    FldSetDirty(fld,true);
}

// my string N copy 
// -- Palm version	doesn't append NULL
//	   max string out = n + 1 null
//    also stops at \n on input
//
void strNcopy(char *dst, char *src, UInt16 n)
{
	UInt16 x, y;

	for(x=n, y=0; x>0; x--, y++) {
		if((src[y] == '\0') || (src[y] == '\n')) break;
		dst[y] = src[y];
	}
	dst[y] = '\0';
}


/* Retrieve the current value of a field as an integer. */
int GetNumField(FormPtr formPtr, UInt16 id)
{ 
    FieldPtr fld = (FieldPtr)FrmGetObjectPtr(formPtr, 
                                             FrmGetObjectIndex(formPtr, 
                                                               id)); 
    char *s = FldGetTextPtr(fld);
    return StrAToI(s);
}

/* Retrieve the current value of a field as a string. */
char *GetTextField(FormPtr formPtr, UInt16 id)
{ 
    FieldPtr fld = (FieldPtr)FrmGetObjectPtr(formPtr, 
                                             FrmGetObjectIndex(formPtr, 
                                                               id)); 
    return FldGetTextPtr(fld);
}


// Database support

#define MAXDBRECORDS 16

DBRecord**
GetDBList(UInt32 type,
          UInt32 creator,
          int *count) 
{
    Err error;
    Boolean newSearch = 1;
    
    DmSearchStateType* state = (DmSearchStateType*)MemPtrNew(sizeof(DmSearchStateType));
    DBRecord** recordList = (DBRecord**)MemPtrNew(sizeof(DBRecord*)*MAXDBRECORDS);

    *count = 0;
    while (*count < MAXDBRECORDS) 
    {
        DBRecord* record = MemPtrNew(sizeof(DBRecord));
        recordList[*count] = record;
        
        error = DmGetNextDatabaseByTypeCreator(newSearch,
                                               state,
                                               type,
                                               creator,
                                               0,
                                               &(record->cardNo),
                                               &(record->dbID));
        if (error) 
        {
            MemPtrFree(record);
            break;
        }

        error = DmDatabaseInfo(record->cardNo,
                               record->dbID,
                               record->name,
                               NULL, NULL, NULL,NULL,
                               NULL, NULL, NULL, NULL,
                               NULL, NULL);
        if (error)
        {
            MemPtrFree(record);
            break;
        }
                               
        newSearch = 0;
        (*count)++;
    }
    
    MemPtrFree(state);
    
    if ((*count) == 0) 
    {
        FreeDBList(recordList,0);
        return NULL;
    }
    
    return recordList;
}

void
FreeDBList(DBRecord **recordList, int count)
{
    int i;
    for (i=0; i < count; i++) 
    {
        DBRecord *record = recordList[i];
        MemPtrFree(record);
    }
    MemPtrFree(recordList);
}

void
GotoApplication(UInt32 ApplId) 
{
    Err err;
    UInt16 cardNo;
    LocalID dbID;
    DmSearchStateType dss;

    /*
     * Bounce the application
     */
    err = DmGetNextDatabaseByTypeCreator(true,
                                         &dss,
                                         sysFileTApplication,
                                         ApplId,
                                         true,
                                         &cardNo,
                                         &dbID);
    ErrFatalDisplayIf(err != errNone,"Failed to launch application!");
    
    SysUIAppSwitch(cardNo,dbID,
                   sysAppLaunchCmdNormalLaunch,
                   NULL);
}
