/*
 * Grid Locator - calculator for distance/bearing for PalmOS devices.
 *
 * file: PalmMath.c
 * Copyright (C) 2001, Rex Allers
 *
 *   Some program structure based on: 
 *   Astro Info (Astro.c) Copyright (C) 2000, Michael Heinz
 *
 *	  Dist/bear modules based on code from PROJ.4 1995 by 
 *		Gerald I. Evenden
 *		USGS, Woods Hole, MA 02543
 *
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <PalmOS.h>
#include "PalmUtil.h"
#include "MathLib.h"

#include "resource.h"

UInt16		mathError;


/*
 * Convert a double to a string.
 * s = destination string
 * d = number to convert
 * p = # of digits to the right of the decimal.
 * l = the maximum length of s.
 */
void
StrDToA(char *s,
        double d,
        int p,	
        int l)
{
    int c;
    double i,f;

	// round the double at precision first
   d += 0.5/pow(10, p);	

    c = 0;

    if (d < 0.0)
        s[c++] = '-';

    d = fabs(d);
    i = floor(d);
    f = d - i;

    StrIToA(&s[c],
            i);

    if (p == 0 || c >= l) 
        return;
    
    c = StrLen(s);
    s[c++] = '.';
    
    while (p-- > 0 && c <= l) {
        f = f * 10.0;
        i = floor(f);
        f = f - i;
        s[c++] = 48 + i;
    }
    s[c] = 0;    	// string termination
}

/*
 * Update a numeric double field and redraw it.
 * p = # of digits to the right of the decimal.
 */
void
UpdateDblField(FieldPtr fld, double num, UInt16 p)
{
    MemHandle   h;
	 char		*s;
    UInt16      size;

    if (fld) {
        FldFreeMemory(fld);
        size = FldGetMaxChars(fld);
        h = MemHandleNew(size+2);

        if (h) {
            s = MemHandleLock(h);
            StrDToA(s,num,p,size);
            MemHandleUnlock(h);
            FldSetTextHandle(fld,h);
            FldDrawField(fld);
        } else {
            ErrFatalDisplay("Out of Memory!");
        }
    } else {
        ErrFatalDisplay("Null Field Pointer!");
    }
}

// Routine to convert a string to a double -- Allowed input is in fixed notation ddd.fff
// This does not use MathLib.
//
double strToDouble(char *str)
{ 
  int  i, start, length, punctPos;
  double  result, sign, fractPart;

//	FrmCustomAlert(a_Debug, "strToDoub", "in", "");

  mathError = 0;
  result = fractPart = 0.0;
  length = punctPos = StrLen(str);

	start = 0;
	sign = 1.0;
	if (str[0] == '-')
	{ 
		sign = -1.0;
		start = 1;
	}

	for (i=start; i<length; i++)				// parse the string from left to right converting the integer part
	{ 
		if (str[i] != '.')
		{ 
//			TxtChar requires OS 3.1
//			if (TxtCharIsDigit(str[i]))
			if ((str[i] >= '0')&&(str[i] <= '9'))
				result = result * 10.0 + (str[i] - 0x30);
//			TxtChar requires OS 3.1
//			else if (TxtCharIsSpace(str[i])) {
			else if ((str[i] == ' ')||(str[i] == '\t')) {
				break;
			} else { 
//				FrmCustomAlert(a_Debug, "Conversion error", "strToDouble", "1");
				mathError = 1;
				return 0.0;
			}
		} else { 
			punctPos = i;
			break;
		}
	 }

	if (str[punctPos] == '.')				// parse the string from the end to the '.' converting the fractional part
	{ 
		for (i=length-1; i>punctPos; i--)
			{ 
//				TxtChar requires OS 3.1
//				if (TxtCharIsDigit(str[i]))
				if ((str[i] >= '0')&&(str[i] <= '9'))
					fractPart = fractPart / 10.0 + (str[i] - 0x30);
				else { 
//					FrmCustomAlert(a_Debug, "Conversion error", "strToDouble", "2");
					mathError = 1;
					return 0.0;
				}
			}
		result += fractPart / 10.0;
	}

	return result * sign;				// correcting the sign
}

/* Retrieve the current value of a field as a double. */
double GetDblField(FormPtr formPtr, UInt16 id)
{ 
    FieldPtr fld = (FieldPtr)FrmGetObjectPtr(formPtr, 
                                             FrmGetObjectIndex(formPtr, 
                                                               id)); 
    char *s = FldGetTextPtr(fld);
    return strToDouble(s);
}


