#!/usr/bin/env python3
"""
G4AON Antenna Switch Controller
Web-enabled relay controller for Raspberry Pi
"""

from flask import Flask, render_template_string, jsonify, request
import RPi.GPIO as GPIO
import atexit

app = Flask(__name__)

# GPIO pin configuration edit to suit your relay board/hat
RELAY_PINS = {
    1: 5,
    2: 6,
    3: 13,
    4: 19,
    5: 26,
    6: 21
}

# GPIO LOW = OFF, HIGH = ON
RELAY_OFF = GPIO.LOW
RELAY_ON = GPIO.HIGH

# Initialize GPIO
GPIO.setmode(GPIO.BCM)
GPIO.setwarnings(False)

for pin in RELAY_PINS.values():
    GPIO.setup(pin, GPIO.OUT)
    GPIO.output(pin, RELAY_OFF)

# Track relay states
relay_states = {i: False for i in range(1, 7)}

def set_relay(relay_num, state):
    """Set a specific relay on or off"""
    if relay_num in RELAY_PINS:
        GPIO.output(RELAY_PINS[relay_num], RELAY_ON if state else RELAY_OFF)
        relay_states[relay_num] = state

def turn_on_relay(relay_num):
    """Turn on one relay and turn off all others"""
    for i in range(1, 7):
        set_relay(i, i == relay_num)

def turn_off_all():
    """Turn off all relays"""
    for i in range(1, 7):
        set_relay(i, False)

# HTML template
HTML_TEMPLATE = """
<!DOCTYPE html>
<html>
<head>
    <title>G4AON Antenna Switch</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
            min-height: 100vh;
            padding: 20px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .container {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            padding: 30px;
            max-width: 600px;
            width: 100%;
        }
        h1 {
            color: #1e3c72;
            text-align: center;
            margin-bottom: 10px;
            font-size: 28px;
        }
        .subtitle {
            text-align: center;
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }
        .status-bar {
            background: #f0f4f8;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 25px;
            text-align: center;
        }
        .status-text {
            font-weight: 600;
            color: #2a5298;
            font-size: 16px;
        }
        .relay-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-bottom: 25px;
        }
        .relay-btn {
            background: #e8ecf1;
            border: 2px solid #d0d7de;
            border-radius: 10px;
            padding: 20px;
            cursor: pointer;
            transition: all 0.3s ease;
            text-align: center;
            font-size: 16px;
            font-weight: 600;
            color: #444;
        }
        .relay-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        .relay-btn.active {
            background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
            border-color: #45a049;
            color: white;
            box-shadow: 0 5px 20px rgba(76, 175, 80, 0.4);
        }
        .relay-label {
            font-size: 18px;
            margin-bottom: 5px;
        }
        .relay-pin {
            font-size: 12px;
            opacity: 0.7;
        }
        .all-off-btn {
            width: 100%;
            background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);
            border: none;
            border-radius: 10px;
            padding: 18px;
            color: white;
            font-size: 18px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 5px 20px rgba(244, 67, 54, 0.3);
        }
        .all-off-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 7px 25px rgba(244, 67, 54, 0.4);
        }
        .all-off-btn:active {
            transform: translateY(0);
        }
        @media (max-width: 480px) {
            .relay-grid {
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>
<body>
# edit descriptions to match your antennas
    <div class="container">
        <h1>🛰️ G4AON Antenna Switch</h1>
        <div class="subtitle">When off antenna grounded</div>
        
        <div class="status-bar">
            <div class="status-text" id="status">All Relays OFF</div>
        </div>
        
        <div class="relay-grid">
            <div class="relay-btn" onclick="toggleRelay(1)" id="relay1">
                <div class="relay-label">Relay 1</div>
                <div class="relay-pin">Spare</div>
            </div>
            <div class="relay-btn" onclick="toggleRelay(2)" id="relay2">
                <div class="relay-label">Relay 2</div>
                <div class="relay-pin">15m EF vert</div>
            </div>
            <div class="relay-btn" onclick="toggleRelay(3)" id="relay3">
                <div class="relay-label">Relay 3</div>
                <div class="relay-pin">EFHW</div>
            </div>
            <div class="relay-btn" onclick="toggleRelay(4)" id="relay4">
                <div class="relay-label">Relay 4</div>
                <div class="relay-pin">30m/20m Vert</div>
            </div>
            <div class="relay-btn" onclick="toggleRelay(5)" id="relay5">
                <div class="relay-label">Relay 5</div>
                <div class="relay-pin">Spare</div>
            </div>
            <div class="relay-btn" onclick="toggleRelay(6)" id="relay6">
                <div class="relay-label">Relay 6</div>
                <div class="relay-pin">Spare</div>
            </div>
        </div>
        
        <button class="all-off-btn" onclick="allOff()">⏻ ALL OFF</button>
    </div>

    <script>
        function updateStatus(states) {
            for (let i = 1; i <= 6; i++) {
                const btn = document.getElementById('relay' + i);
                if (states[i]) {
                    btn.classList.add('active');
                } else {
                    btn.classList.remove('active');
                }
            }
            
            const activeRelay = Object.keys(states).find(key => states[key]);
            const statusText = document.getElementById('status');
            if (activeRelay) {
                statusText.textContent = `Relay ${activeRelay} ON`;
            } else {
                statusText.textContent = 'All Relays OFF';
            }
        }

        function toggleRelay(num) {
            fetch('/relay/' + num, {method: 'POST'})
                .then(r => r.json())
                .then(data => updateStatus(data.states));
        }

        function allOff() {
            fetch('/all_off', {method: 'POST'})
                .then(r => r.json())
                .then(data => updateStatus(data.states));
        }

        // Get initial status
        fetch('/status')
            .then(r => r.json())
            .then(data => updateStatus(data.states));
    </script>
</body>
</html>
"""

@app.route('/')
def index():
    return render_template_string(HTML_TEMPLATE)

@app.route('/status')
def status():
    return jsonify({'states': relay_states})

@app.route('/relay/<int:relay_num>', methods=['POST'])
def control_relay(relay_num):
    if 1 <= relay_num <= 6:
        turn_on_relay(relay_num)
    return jsonify({'states': relay_states})

@app.route('/all_off', methods=['POST'])
def all_off():
    turn_off_all()
    return jsonify({'states': relay_states})

def cleanup():
    """Clean up GPIO on exit"""
    turn_off_all()
    GPIO.cleanup()

atexit.register(cleanup)

if __name__ == '__main__':
    print("G4AON Antenna Switch Controller Starting...")
    print("Access the web interface at: http://[your-pi-ip]:5002")
    try:
        app.run(host='0.0.0.0', port=5002, debug=False)
    except KeyboardInterrupt:
        print("\nShutting down...")
        cleanup()
