#include "SPI.h"
#include ".\Adafruit_GFX_AS.h"
#include ".\Adafruit_ILI9341_STM.h"
#include "XPT2046_touch.h"

// TFT Pins plus SPI1
const int TFT_CS =    PA0;                  
const int TFT_DC =    PA2;                
const int TFT_RST =   PA1;
const int PEAK = 0;
const int AVE = 1;

// set this to PEAK or AVE to determine power meter mode
const int PowerMeterMode = PEAK;

// define the band switch pins
const int Band160M = PA9;
const int Band80M = PA10;
const int Band60M = PA14; // use the same filter for 60M was PA13
const int Band40M = PA14;
const int Band30M = PA15;
const int Band20M = PB3;
const int Band17M = PB4;
const int Band15M = PB5;
const int Band12M = PB6;
const int Band10M = PB7;
const int Band6M = PB8;
const int Band4M = PB9;

// define the analoge inputs for
// the forward and reflected
// power readings
const int ForwardPwr = PB0;
const int ReflectedPwr = PB1;

// touch screen is on SPI2
const int CS_PIN =    PA8;

// some global constants
const int KeyPadStarty = 110;
const int ButtonDepth = 53;
const int ButtonWidth = 50;

// global variables (always a bad idea)
boolean wastouched = true;
uint16_t xy[2];
int BandSelection = 20; // default band selection is 20M
int OldBandSelection = 160;

// variables used for power calcualtion and display
double Forward_Watts = 0.0;
double Reflected_Watts = 0.0;
double Calculated_SWR = 1.0; 
double Forward_dBm = 0.0;
double Reflected_dBm = 0.0; 

// declare the display on hardware SPI1
Adafruit_ILI9341_STM tft = Adafruit_ILI9341_STM(TFT_CS, TFT_DC, TFT_RST); // Use hardware SPI

// Create an SPI instance on SPI2 port.
// this is for the touch screen functionality
SPIClass mySPI(2); 
// now declare the touchsereen
// Chip Select pin, SPI port
XPT2046_touch ts(CS_PIN, mySPI); 


void DisplayScreen()
// displays the basic text and keypad buttons
{
  int BandList[12] = {160, 80, 60, 40, 30, 20, 17, 15, 12, 10, 6, 4};
  tft.setRotation(1);
  tft.fillScreen(ILI9341_BLACK);
  // fill rect startx, starty, width, depth, colour
  tft.fillRect(0, KeyPadStarty, 320, 140, ILI9341_RED);
  tft.setCursor(20, 10);
  tft.setTextColor(ILI9341_RED);    
  tft.setTextSize(4);
  tft.print("G0MGX LINEAR");
  tft.setTextColor(ILI9341_GREEN);  
  tft.setCursor(35, 50);
  tft.setTextSize(3);
  tft.print("FWD:   REF:");
  tft.setCursor(110, 80);
  tft.setTextSize (2);
  tft.setTextColor(ILI9341_YELLOW);
  tft.print("SWR:");

  for (int Bands = 0; Bands <= 11; Bands++)
  {
    DisplayButton(ILI9341_WHITE, BandList[Bands], ILI9341_BLACK);
  } 
}

void DisplayButton(int textcolour, int band, int boxcolour)
// displays a single button based on the band parameter
// you also specify text and button colour
{
  int textxcoord; int textycoord; int boxxcoord; int boxycoord;
  switch (band)
  {
    case 160:
      textxcoord = 10;
      textycoord = 140;
      boxxcoord = 3;
      boxycoord = KeyPadStarty+10;
      break;
    case 80:
      textxcoord = 70;
      textycoord = 140;
      boxxcoord = 56;
      boxycoord = KeyPadStarty+10;
      break; 
    case 60:
      textxcoord = 123;
      textycoord = 140;
      boxxcoord = 109;
      boxycoord = KeyPadStarty+10;
      break; 
    case 40:
      textxcoord = 175;
      textycoord = 140;
      boxxcoord = 162;
      boxycoord = KeyPadStarty+10;
      break; 
    case 30:
      textxcoord = 230;
      textycoord = 140;
      boxxcoord = 215;
      boxycoord = KeyPadStarty+10;
      break; 
    case 20:
      textxcoord = 280;
      textycoord = 140;
      boxxcoord = 268;
      boxycoord = KeyPadStarty+10;
      break; 
    case 17:
      textxcoord = 15;
      textycoord = 195;
      boxxcoord = 3;
      boxycoord = KeyPadStarty+66;
      break;
    case 15:
      textxcoord = 70;
      textycoord = 195;
      boxxcoord = 56;
      boxycoord = KeyPadStarty+66;
      break; 
    case 12:
      textxcoord = 123;
      textycoord = 195;
      boxxcoord = 109;
      boxycoord = KeyPadStarty+66;
      break; 
    case 10:
      textxcoord = 175;
      textycoord = 195;
      boxxcoord = 162;
      boxycoord = KeyPadStarty+66;
      break; 
    case 6:
      textxcoord = 235;
      textycoord = 195;
      boxxcoord = 215;
      boxycoord = KeyPadStarty+66;
      break; 
    case 4:
      textxcoord = 287;
      textycoord = 195;
      boxxcoord = 268;
      boxycoord = KeyPadStarty+66;
      break; 
    default:
      break;
  }  
  tft.fillRect(boxxcoord, boxycoord, ButtonWidth, ButtonDepth, boxcolour); 
  tft.setTextColor(textcolour);  
  tft.setTextSize(2); 
  tft.setCursor(textxcoord, textycoord);
  tft.print(band); 
}  

void setup() 
{

  // initialise the TFT and make it all black
  tft.begin();
  tft.fillScreen(ILI9341_BLACK);

  // initialise the touchscreen
  ts.begin();

  // initialise 12 digital outputs
  // used for band select
  // could simplify and use external BCD if this
  // gets out of hand....
  pinMode (Band160M, OUTPUT);
  digitalWrite (Band160M, LOW);
  pinMode (Band80M, OUTPUT);
  digitalWrite (Band80M, LOW);
  pinMode (Band60M, OUTPUT);
  digitalWrite (Band60M, LOW);
  pinMode (Band40M, OUTPUT);
  digitalWrite (Band40M, LOW);
  pinMode (Band30M, OUTPUT);
  digitalWrite (Band30M, LOW);
  pinMode (Band20M, OUTPUT);
  digitalWrite (Band20M, LOW);
  pinMode (Band17M, OUTPUT);
  digitalWrite (Band17M, LOW);
  pinMode (Band15M, OUTPUT);
  digitalWrite (Band15M, LOW);
  pinMode (Band12M, OUTPUT);
  digitalWrite (Band12M, LOW);
  pinMode (Band10M, OUTPUT);
  digitalWrite (Band10M, LOW);
  pinMode (Band6M, OUTPUT);
  digitalWrite (Band6M, LOW);
  pinMode (Band4M, OUTPUT);
  digitalWrite (Band4M, LOW);

  // Declare the inputs as INPUT_ANALOG:
  pinMode(ForwardPwr, INPUT_ANALOG);
  pinMode(ReflectedPwr, INPUT_ANALOG);

  // we need to setup the screen once
  DisplayScreen();

}

void CalculatePowerValues(int ForwardADCValue, int ReflectedADCValue)
// read in the external ADC values and calculate power and SWR
{
  double Pref_over_Pfwd = 0.0;
  double Root_Pref_over_Pfwd = 0.0;

  // calibration factors from spreadsheet
  const double Forward_Alpha = 0.0326; 
  const double Forward_Beta = -86.056;
  const double Reflected_Alpha = 0.0326;
  const double Reflected_Beta = -88.034;

  // this is the attenuation between the signal and the 
  // input to the AD8307
  const double ForwardSampleAttenuation = 52.7;
  const double ReflectedSampleAttenuation = 33.9;

  // We use the constants defined before to solve the equation 
  // dbM = Alpha ADC + Beta (where ADC = ADC value and Alpha and Beta are constants)
  // We originally determine the values of Alpha and Beta from the spreadsheet
  // We then also add on the Sample Attenuation values to get to the real world number

  // First lets do the Forward Power Channel

  Forward_dBm = ( ForwardADCValue * Forward_Alpha ) + Forward_Beta + ForwardSampleAttenuation;
  Forward_Watts = (pow(10,(Forward_dBm/10)) / 1000);

  // Now let's do it again but for the reflected power values

  Reflected_dBm = ( ReflectedADCValue * Reflected_Alpha ) + Reflected_Beta + ReflectedSampleAttenuation;
  Reflected_Watts = (pow(10,(Reflected_dBm/10)) / 1000);

  // if the reflected power is small dont bother calculating the SWR
  if (Reflected_Watts < 0.01) 
  {
    Pref_over_Pfwd = 0.0;
    Calculated_SWR = 1.0;
  }
  else
  {

    // if we determine that the reflected power is
    // greater than the forward power
    // then something is amiss; however
    // we dont want a -ve SWR
  
    if (Forward_Watts - Reflected_Watts < 0.0)
    {
      Calculated_SWR = 10.0;
    }
    else
    {
      Pref_over_Pfwd = Reflected_Watts / Forward_Watts;
      Root_Pref_over_Pfwd = sqrt(Pref_over_Pfwd);
      Calculated_SWR = ((1 + Root_Pref_over_Pfwd) / ( 1 - Root_Pref_over_Pfwd));
    }
    if (Calculated_SWR > 10.0)
      Calculated_SWR = 10.0;  
  }
}

void SetLPF (int BandSelection)
// select the appropriate digital output based on band selection
{
  int OutputPins[12] = {Band160M, Band80M, Band60M, Band40M, Band30M, Band20M, Band17M, Band15M, Band12M, Band10M, Band6M, Band4M};
  int OutputValues[12] = {0,0,0,0,0,0,0,0,0,0,0,0};

  switch (BandSelection)
  {
    case 160:
      OutputValues[0] = 1;
      break;
    case 80:
      OutputValues[1] = 1;
      break;
    case 60:
      OutputValues[2] = 1;
      break;
    case 40:
      OutputValues[3] = 1;
      break;
    case 30:
      OutputValues[4] = 1;
      break;  
    case 20:
      OutputValues[5] = 1;
      break; 
    case 17:
      OutputValues[6] = 1;
      break;  
    case 15:
      OutputValues[7] = 1;
      break;  
    case 12:
      OutputValues[8] = 1;
      break;  
    case 10:
      OutputValues[9] = 1;
      break;  
    case 6:
      OutputValues[10] = 1;
      break;  
    case 4:
      OutputValues[11] = 1;
      break;  
  }
  for (int writeOutputs = 0; writeOutputs <=11; writeOutputs++)
  {
    digitalWrite(OutputPins[writeOutputs],OutputValues[writeOutputs]);
  }
}

void CheckTouchScreen()
// this checks if the screen has been touched and takes action
{
  static unsigned long int CurrentTime = 0;
  static unsigned long int LastUpdateTime = 0;
  const unsigned long int UpdateInterval = 500.0; // ms between fastest allowable band change
  
  const int TFTWidth = 320;
  const int TFTDepth = 240;
  // the touch x and y coordinates are on a different
  // scale to the TFT 
  // so we need to convert between them

  // these are my cal factors and are the left most x coordinate and top most y
  const int CalFactorx = 258;
  const int TouchScreenWidth = 3650 - CalFactorx;
  const int CalFactory = 426;
  const int TouchScreenDepth = 3740 - CalFactory;
  
  const int XFactor = TouchScreenWidth / TFTWidth;
  const int YFactor = TouchScreenDepth / TFTDepth;
  
  const int TouchButtonWidth = ButtonWidth * XFactor;
  const int TouchButtonDepth = ButtonDepth * YFactor;
  
  // define my box x y start coords
  // the TFT is 320 accross and 250 deep
  // these numbers are relative to that baseline
  // row 1
  int Box160M[2] = {3 * XFactor + CalFactorx, 120 * YFactor + CalFactory};
  int Box80M[2] = {56 * XFactor + CalFactorx, 120 * YFactor + CalFactory};
  int Box60M[2] = {109 * XFactor + CalFactorx, 120 * YFactor + CalFactory};
  int Box40M[2] = {162 * XFactor + CalFactorx, 120 * YFactor + CalFactory};
  int Box30M[2] = {215 * XFactor + CalFactorx, 120 * YFactor + CalFactory};
  int Box20M[2] = {268 * XFactor + CalFactorx, 120 * YFactor + CalFactory};
  // row 2
  int Box17M[2] = {3 * XFactor + CalFactorx, 176 * YFactor + CalFactory};
  int Box15M[2] = {56 * XFactor + CalFactorx, 176 * YFactor + CalFactory};
  int Box12M[2] = {109 * XFactor + CalFactorx, 176 * YFactor + CalFactory};
  int Box10M[2] = {162 * XFactor + CalFactorx, 176 * YFactor + CalFactory};
  int Box6M[2] = {215 * XFactor + CalFactorx, 176 * YFactor + CalFactory};
  int Box4M[2] = {268 * XFactor + CalFactorx, 176 * YFactor + CalFactory};
  
  static uint16_t xy[2];
  boolean istouched = ts.read_XY(xy);
  
  if (istouched) 
  {
    // this is a crude debounce to only act once per touch
    if (!wastouched) 
    {
      // determine if the x and y coordinates are inside a button
      // this feels very messy and there must be a better way....
      
      if ((xy[0] >= Box160M[0]) && (xy[0] <= Box160M[0] + TouchButtonWidth) && 
          (xy[1] >= Box160M[1]) && (xy[1] <= Box160M[1] + TouchButtonDepth))
          {
            BandSelection = 160;
          }
       else if ((xy[0] >= Box80M[0]) && (xy[0] <= Box80M[0] + TouchButtonWidth) && 
          (xy[1] >= Box80M[1]) && (xy[1] <= Box80M[1] + TouchButtonDepth))
          {
            BandSelection = 80;
          }
       else if ((xy[0] >= Box60M[0]) && (xy[0] <= Box60M[0] + TouchButtonWidth) && 
          (xy[1] >= Box60M[1]) && (xy[1] <= Box60M[1] + TouchButtonDepth))
          {
            BandSelection = 60;
          }
       else if ((xy[0] >= Box40M[0]) && (xy[0] <= Box40M[0] + TouchButtonWidth) && 
          (xy[1] >= Box40M[1]) && (xy[1] <= Box40M[1] + TouchButtonDepth))
          {
            BandSelection = 40;
          }
       else if ((xy[0] >= Box30M[0]) && (xy[0] <= Box30M[0] + TouchButtonWidth) && 
          (xy[1] >= Box30M[1]) && (xy[1] <= Box30M[1] + TouchButtonDepth))
          {
            BandSelection = 30;
          }
       else if ((xy[0] >= Box20M[0]) && (xy[0] <= Box20M[0] + TouchButtonWidth) && 
          (xy[1] >= Box20M[1]) && (xy[1] <= Box20M[1] + TouchButtonDepth))
          {
            BandSelection = 20;
          }
       else if ((xy[0] >= Box17M[0]) && (xy[0] <= Box17M[0] + TouchButtonWidth) && 
          (xy[1] >= Box17M[1]) && (xy[1] <= Box17M[1] + TouchButtonDepth))
          {
            BandSelection = 17;
          } 
       else if ((xy[0] >= Box15M[0]) && (xy[0] <= Box15M[0] + TouchButtonWidth) && 
          (xy[1] >= Box15M[1]) && (xy[1] <= Box15M[1] + TouchButtonDepth))
          {
            BandSelection = 15;
          }
       else if ((xy[0] >= Box12M[0]) && (xy[0] <= Box12M[0] + TouchButtonWidth) && 
          (xy[1] >= Box12M[1]) && (xy[1] <= Box12M[1] + TouchButtonDepth))
          {
            BandSelection = 12;
          }
       else if ((xy[0] >= Box10M[0]) && (xy[0] <= Box10M[0] + TouchButtonWidth) && 
          (xy[1] >= Box10M[1]) && (xy[1] <= Box10M[1] + TouchButtonDepth))
          {
            BandSelection = 10;
          }
       else if ((xy[0] >= Box6M[0]) && (xy[0] <= Box6M[0] + TouchButtonWidth) && 
          (xy[1] >= Box6M[1]) && (xy[1] <= Box6M[1] + TouchButtonDepth))
          {
            BandSelection = 6;
          }
       else if ((xy[0] >= Box4M[0]) && (xy[0] <= Box4M[0] + TouchButtonWidth) && 
          (xy[1] >= Box4M[1]) && (xy[1] <= Box4M[1] + TouchButtonDepth))
          {
            BandSelection = 4;
          }
    }

  } 
  // if the band selection has changed act accordingly
  // but only at a maximum interval of UpdateInterval
  CurrentTime = millis();
  if ((OldBandSelection != BandSelection) && (CurrentTime - LastUpdateTime > UpdateInterval))
  {
    DisplayButton(ILI9341_RED, BandSelection, ILI9341_WHITE);
    DisplayButton(ILI9341_WHITE, OldBandSelection, ILI9341_BLACK);
    SetLPF(BandSelection);
    OldBandSelection = BandSelection;
    // as we dont want to keep switching 
    // record the time we last changed
    // and dont do it again until at least UpdateInterval has passed
    // neat - isn't it?
    // this is event driven programming at its best :-)
    LastUpdateTime = CurrentTime;
  }
  wastouched = istouched;
}

void UpdatePowerDisplay()
{
  int Forward_Watts_Int;
  int Reflected_Watts_Int;
  // this is the SWR reading
  tft.fillRect(160, 80, 60, 20, ILI9341_BLACK); 
  tft.setCursor(160, 80);
  tft.setTextSize (2);
  tft.setTextColor(ILI9341_YELLOW);
  if ((Calculated_SWR > 5.0) || (Calculated_SWR < 0.0))
    tft.print("Huge");
  else
    tft.print(Calculated_SWR,2);

  // this is the power
  tft.fillRect(110, 50, 50, 22, ILI9341_BLACK); 
  tft.setCursor(110, 50);
  tft.setTextSize(3);
  Forward_Watts_Int = round(Forward_Watts);
  if (Forward_Watts_Int > 10)
    tft.print("10");
  else
    tft.print(Forward_Watts_Int);  

  tft.fillRect(240, 50, 50, 22, ILI9341_BLACK); 
  tft.setCursor(240, 50);
  tft.setTextSize(3);
  Reflected_Watts_Int = round(Reflected_Watts);
  if (Reflected_Watts_Int > 10)
    tft.print("10");
  else
    tft.print(Reflected_Watts_Int);  
}

void MaintainPowerDisplay()
{
  // the update is event drive at this interval
  static const unsigned long TimeInterval = 1000; // ms
  static unsigned long lastRefreshTime = 0;
  
  if(millis() - lastRefreshTime >= TimeInterval)
  {
    // only update the display of power if the interval has passed
    lastRefreshTime += TimeInterval;
    UpdatePowerDisplay();
  }
}

void ReadPower()
{
  const unsigned long ReadTimeInterval = 100; // ms
  const int NumSamples = 10;
  const unsigned long CalculateTimeInterval = 1000; // ms
  static unsigned long CurrentTime = 0;
  static unsigned long LastReadTime = 0;
  static unsigned long LastCalcTime = 0;
  static int ForwardPowerArray[NumSamples];
  static int ReflectedPowerArray[NumSamples];
  static int ADCArrayIndex = 0;
  static long int ForwardAveValue = 0;
  static long int ReflectedAveValue = 0;
  static long int ForwardPeakValue = 0;
  static long int ReflectedPeakValue = 0;
  static int ForwardADCValue = 0;
  static int ReflectedADCValue = 0;
  
  CurrentTime = millis();  
  
  // take a reading from the ADC inputs every ReadTimeInterval  
  // we should do this quite often
  
  if(CurrentTime - LastReadTime >= ReadTimeInterval)
  {
    LastReadTime = LastReadTime + ReadTimeInterval;
    // read in the ADC value into the array
    ForwardADCValue = analogRead(ForwardPwr);

    ReflectedADCValue = analogRead(ReflectedPwr);
    
    ForwardPowerArray[ADCArrayIndex] = ForwardADCValue;
    ReflectedPowerArray[ADCArrayIndex] = ReflectedADCValue;
    

    // update the array indexes ready for next time and wrap round if necessary
    if (ADCArrayIndex < NumSamples - 1)
      ADCArrayIndex = ADCArrayIndex + 1;
    else
      ADCArrayIndex = 0;
  }

  // calculate the average reading every CalculateTimeInterval
  // we should do this much less often
  if(CurrentTime - LastCalcTime >= CalculateTimeInterval)
  {
    LastCalcTime = LastCalcTime + CalculateTimeInterval;

    // reset the average value so we determine it each time
    // from the values in the array right now
    ForwardAveValue = 0;
    ReflectedAveValue = 0;
    ForwardPeakValue = 0;
    ReflectedPeakValue = 0;
    
    // loop round the array finding the average & Peak value
    for (int ArrayIndex = 0; ArrayIndex <= NumSamples-1; ArrayIndex++)
    {
      ForwardAveValue = ForwardAveValue + ForwardPowerArray[ArrayIndex];
      if (ForwardPowerArray[ArrayIndex] > ForwardPeakValue)
      {
        ForwardPeakValue = ForwardPowerArray[ArrayIndex];
      }
      ReflectedAveValue = ReflectedAveValue + ReflectedPowerArray[ArrayIndex];
      if (ReflectedPowerArray[ArrayIndex] > ReflectedPeakValue)
      {
        ReflectedPeakValue = ReflectedPowerArray[ArrayIndex];
      }
    }
    // now calcualte the average
    ForwardAveValue = ForwardAveValue / NumSamples;
    ReflectedAveValue = ReflectedAveValue / NumSamples;


    if (PowerMeterMode == PEAK)
    {
      CalculatePowerValues(ForwardPeakValue,ReflectedPeakValue);
    }
    else
    {
      CalculatePowerValues(ForwardAveValue,ReflectedAveValue);
    }
  }
}

void loop() 
{
  // loop forever reading the power, checking the touch screen and
  // acting accordingly 
  CheckTouchScreen();
  ReadPower();
  MaintainPowerDisplay();
}
