;**********************************************************************
;                                                                     *
;   \pic\serialAD\ser_main.asm                                        * 
;                                                                     *
;                                                                     *
;   Firmware for PIC12F675 :  A/D converter for serial PC port .      *
;   Author: Wolfgang Buescher ( DL4YHF@qsl.net )                      *
;   Revision date:  10/2002                                           *
;   Compile with:  MPLAB v5.70.40 or later (must support PIC12F675)   *
;                                                                     *
;   Short description:                                                *
;    - Connects to the async serial interface of a PC ("COM1","COM2") *
;    - performs two-channel A/D conversions with the PIC's internal   *
;      10 bit ADC                                                     *
;    - does a  4-fold 'oversampling' to increase the A/D sample's     *
;      resolution and accuracy .                                      *
;    - sends 2500 packets * 4 byte  per second,                       *
;      packet format described in file "SerPicAD.txt" .               *
;    - settings for serial interface:                                 *
;       115200 bits/second, one startbits, 8 databits, 1 stopbit,     *
;       no parity, no handshake (neither hard nor soft)               *
;    - crystal clock: 10.0000 MHz, allows  to use the wonderful       *
;      ovenned crystal oscillator for a VERY precise sampling rate.   *
;                                                                     *
;                                                                     *
;   Revision history :                                                *
;     10/2002:                                                        *
;         - First attempt to send "high-speed" serial data            *
;           from PIC (without async interface)  to PC                 *
;           with a timer interrupt routine .                          *
;           (but it turned out context saving/restoring was too slow, *
;            so now it works without a timer. "Timebase" = main loop) *
;                                                                     *
;**********************************************************************

   PROCESSOR 12F675           ; auch unter Project..Edit Project einstellen !
   RADIX     DEC
   EXPAND

   list      p=12f675         ; list directive to define processor
#include "p12f675.inc"        ; processor specific variable definitions


; '__CONFIG' directive is used to embed configuration data within .asm file.
; The lables following the directive are located in the respective .inc file.
; See respective data sheet for additional information on configuration word.

   __CONFIG   _CPD_OFF & _CP_OFF & _BODEN_ON & _MCLRE_OFF & _PWRTE_ON & _WDT_OFF & _HS_OSC


;*****************************************************************************
;  ASSEMBLY OPTIONS (for testing, etc)
;*****************************************************************************
#define SWI_TEST_INIT_STEPS   0   ; 0=normal assembly, 1=assembly with additional debug funcs
#define SWI_SIMULATOR         0   ; 0=assembly for real target, 1=assembly for simulator


;*****************************************************************************
;  CONSTANT DEFINITIONS
;*****************************************************************************
#define C_SER_STATE_IDLE         0
#define C_SER_STATE_TX_START_BIT 1


;*****************************************************************************
;  VARIABLE DEFINITIONS
;*****************************************************************************
; PIC12F675:  Adresse 0x20..0x5F sind "General Purpose Register" (=RAM),
;            die in beiden Speicherbaenken identisch sind (RP0=0 oder 1).
V_w_temp      EQU     0x20   ; variable used for context saving
V_status_temp EQU     0x21   ; variable used for context saving

V_ser_tx_reg  EQU     0x22   ; serial transmit shift register
V_ser_rx_reg  EQU     0x23   ; serial receive shift register
V_ser_state   EQU     0x24   ; state of serial RX/TX state machine
V_ser_index   EQU     0x25   ; byte index for serial transmission. MUST NOT exceed 3.

V_SumI_l      EQU     0x26   ; Sum of 4 values from "I" channel, low byte
V_SumI_h      EQU     0x27   ; Sum of 4 values from "I" channel, upper 4 bits
V_SumQ_l      EQU     0x28   ; Sum of 4 values from "Q" channel, low byte
V_SumQ_h      EQU     0x29   ; Sum of 4 values from "Q" channel, upper 4 bits

V_AvrgI_l     EQU     0x2A   ; Average value for "I" channel, low byte
V_AvrgI_h     EQU     0x2B   ; Average value for "I" channel, upper 4 bits
V_AvrgQ_l     EQU     0x2C   ; Average value for "Q" channel, low byte
V_AvrgQ_h     EQU     0x2D   ; Average value for "Q" channel, upper 4 bits



;*****************************************************************************
;  Pin Assignments for PIC12F675 in the "A/D converter with async interface"
;*****************************************************************************

  ; "GPIO" pins (general purpose input/output)
  ; Another nasty pitfall (and there are many in a PIC): GP3 is always an INPUT(!)
#define PIN_AIN_I   GPIO,0
#define PIN_AIN_Q   GPIO,1
#define PIN_SER_TXD GPIO,2   ; see also: initialisation of TRISIO register
#define PIN_SER_RXD GPIO,3   ; GPIO.3 is always an input port, never an output

  ; Macros to set the LOGIC state of the "TXD" pin.
  ;  Note: a LOGIC "1" is -12V, a LOGIC "0" is +12V, 
  ;        and the idle state of the TXD line is a LOGIC "1" = -12V !
#if(1)  ; if an INVERTING line driver is used (simple transistor or MAX...):
  #define M_TXD_PIN_LOW  bcf PIN_SER_TXD
  #define M_TXD_PIN_HIGH bsf PIN_SER_TXD
#else   ; if a NON-INVERTING driver or no driver at all is used:
  #define M_TXD_PIN_LOW  bsf PIN_SER_TXD
  #define M_TXD_PIN_HIGH bcf PIN_SER_TXD
#endif



;**********************************************************************
   ORG     0x000             ; PIC reset vector
   goto    main              ; go to beginning of program



;**********************************************************************
;  INTERRUPT SERVICE ROUTINE
   ORG     0x004             ; interrupt vector location

   ; The PIC only has one single interrupt vector.
   ; This software has to find out what caused the interrupt
   ;  (see datasheet. For PIC12F675, INTCON and PIR contain the IR flags)
   ;   Interrupt latency time is 3 instruction cycles.
   ;   A timer interrupt is generated when timer0 overflows from 0xFF to 0x00. 
   ; Save context of main task in RAM:
        movwf   V_w_temp      ; [tl+0] save off current W register contents
        movf    STATUS,w      ; [tl+1] move status register into W register
        movwf   V_status_temp ; [tl+2] save off contents of STATUS register

   ; Check if the interrupt was caused by a timer0 - overflow.
   ;  (not required here, because TIMER0 is the only enabled interrupt source)
   ;    btfss   INTCON, T0IF  ; skip next instruction if Timer0 Interrupt Flag set
   ;    goto    NoTimerIRQ    ; other interrupt, not from Timer0


NoTimerIRQ:
   ; Check if the interrupt was caused by a change on port B.
   ; if yes, perform some smart operations.. (but not yet)
   ;    btfss   INTCON, RBIF  ; skip next instruction if Port B interrupt Flag set
   ;    goto    NoPortbIRQ    ; other interrupt, not from Port B
   ;    bcf     INTCON, RBIF  ; Port B change interrupt, clear interrupt flag.
NoPortbIRQ:



SerInt_End:
        movwf   TMR0            ; [n+0] write reload value into Timer0 register
EndTimerISR:
        bcf     INTCON, T0IF    ; [n+1] clear timer0 interrupt flag.
EndISR: ; Return from Interrupt routine: restore context, etc
        movf    V_status_temp,w ; [n+2] retrieve copy of STATUS register
        movwf   STATUS          ; [n+3] restore pre-isr STATUS register
        swapf   V_w_temp,f      ; [n+4] tricky: swapf doesn't affect flags
        swapf   V_w_temp,w      ; [n+5] restore pre-isr W register contents
        retfie                  ; [n+6] return from interrupt




;******************************************************************************
MainLoop:
  ; Timing considerations for SERIAL INTERFACE
  ;  At 115.2 kBit/sec, a single bit time (T_ser_bit) is 1 / 115200 = 8.6805555 us. 
  ;  The PIC's instruction cycle (with 10MHz crystal)
  ;  is 4 / 10MHz = 400 ns. 
  ;  To achieve an "average" bit time of 8.68 us, the number of instruction cycles
  ;     between two transmitted bits must be 21.7, so it will be mostly 22 instructions 
  ;     but somtimes only 21 instructions for every MAIN LOOP .
  ;  After the transmission of 40 data bits in a "packet" (4 * (1+8+1) bits ),
  ;     an additional "pause" is inserted for fine tuning of the 'packet TX cycle'
  ;     of exactly 400 usec = 1000 instruction cycles .
  ;     (this is required because we want to have excactly 2500.0000 packets per second).


   ;*****************************************************************************************
   ; Serial transmission and A/D conversion.
   ;  This was once done in a TIMER INTERRUPT 
   ;       but the PIC-specific ugly "save W and save FLAGS and restore and blabla"
   ;       made it too slow. 
   ;  So it was changed to an "endless loop with constant execution time".
   ;
   ;  The remaining time (after the serial TX handler) will be used to acquire as many analog
   ;  samples as possible, to increase the ADC's resolution a little :
   ;  
   ;    During transmission of a 4-byte-frame, 
   ;       four conversions of the "I"-channel and the "Q"-channel
   ;       are performed, 4 "I"-conversions summed up, and 4 "Q"-conversions summed up.
   ;       The result are 12-bit-values which is exactly the max data size which can be 
   ;       transmitted. 
   ;  
   ; Timing relationship between serial BYTE transmission and A/D conversion...
   ;
   ;         _      ____ ____ ____ ____ ____ ____ ____ ____ ___________
   ; TXD      |_st_|_D0_|_D1_| D2 | D3 | D4 | D5 | D6 | D7 |stop|  gap |__(next start bit)
   ; signal
   ;           <Tb> <Tb> <Tb> <Tb> <Tb> <Tb> <Tb> <Tb> <Tb> <Tb+Tgap>    Tb=8.6805us = 21.7 cycles
   ;          .    .    .    .    .    .    .    .    .    .    .      .  for 115.2kBit/sec
   ;          .    .    .    .    .    .    .    .    .    .    .      .
   ;          .    .    .    .    .    .    .    .    .    .    .      .
   ; ADC      | Select| Convert|Add |Select| Convert   |Add| Calculate
   ; action   |  "I"  |  "I"   | "I"| "Q"  |   "Q"     |"Q"| Averages 
   ;          .    .    .    .    .    .    .    .    .    .    .      .
   ;          .    .    .    .    .    .    .    .    .    .    .      .
   ;          .    .    .    .    .    .    .    .    .    .    .      .
   ; Count of |sta.|(D0)|(D1)|(D2)|(D3)|(D4)|(D5)|(D6)|(D7)|stop| gap  |
   ;   cycles |----+----+----+----+----+----+----+----+----+----+------|
   ; -per bit | 22 | 22 | 21 | 22 | 22 | 21 | 22 | 21 | 22 | 22 |  33  | 
   ; -total   | 22 | 44 | 65 | 87 |109 |130 |152 |173 |195 |217 | 250  |  
   ; -ideal   |21.7|43.4|65.1|86.8|108.|130.|152 |174 |195.|217 | 250.0|
   ;
   ; 
   ;*****************************************************************************************
Serial_Dispatcher:
        ; branch into the serial transmission routine, depending on its 'state'...
        ; For debugging: set breakpoint on next instruction and click "ZERO STOPWATCH" there
        movf    V_ser_state, w       ; [t=0] move state value for serial input/output into w
  ; ex: andlw   0x0F                 ; [t+1] mask off all bits except 4 LSBs
        addwf   PCL, F               ; [t+1] a "computed jump": add (w) to program counter(low)
        goto    SerState_Idle        ; [t+2] 0x00 = idle state
        goto    SerState_TxStartBit  ; [t+3] 0x01 = send start bit
        goto    SerState_TxD0        ; [t+3] 0x02 = send databit #0 (LSB)
        goto    SerState_TxD1        ; [t+3] 0x03 = send databit #1
        goto    SerState_TxD2        ; [t+3] 0x04 = send databit #2
        goto    SerState_TxD3        ; [t+3] 0x05 = send databit #3
        goto    SerState_TxD4        ; [t+3] 0x06 = send databit #4
        goto    SerState_TxD5        ; [t+3] 0x07 = send databit #5
        goto    SerState_TxD6        ; [t+3] 0x08 = send databit #6 
        goto    SerState_TxD7        ; [t+3] 0x09 = send databit #7 (MSB)
        goto    SerState_TxStopBit   ; [t+3] 0x0A = send stopbit 
        goto    SerState_TxGap       ; [t+3] 0x0B = send gap between BYTEs
        goto    SerState_WaitRx      ; [t+3] 0x0C =
        goto    SerState_Rx          ; [t+3] 0x0D =
        goto    SerState_RxStop      ; [t+3] 0x0E =
SerState_DE: goto SerState_RxDone    ; [t+3] 0x0F =

   if( (Serial_Dispatcher & 0xff) >= (SerState_DE & 0xff))
        MESSG   "Warning : Crossing Page Boundary in Computed Jump, Make Sure PCLATH is Loaded Correctly"
   endif
;---- end of jump table for serial interrupt dispatcher ------

SerState_Idle:   ; State 0x00 = Serial I/O is "idle", nothing to do
        M_TXD_PIN_HIGH          ;  set serial TX output HIGH (=stopbit, =idle state of line)
        movlw   #247            ; reload timer0 register for next timer interval
        goto    SerState_End

SerState_TxStartBit  ; 0x01 = send start bit
               ; Just before the start bit, switch the analog input multiplexer to the 
               ; "I" channel. See datasheet for PIC12F675, "A/D Acquisition requirements".
        bcf    ADCON0, CHS0     ; [t+5] connect AN0 (=GP0, "I"-chnl) to PIC's 10bit A/D converter
               ; 10 us later (=8.86us + 3 CPU cycles a 400ns) we may start the A/D conversion.
        nop                     ; [t+6] compensate 'decision' time for data outputting states..
        M_TXD_PIN_LOW           ; [t+7] set serial TX output low (=startbit on RS232 "TXD" line)
        incf    V_ser_state, F  ; [t+8] next state for serial "TX" state machine
               ; Load the "TX register" with the serial TX data..
        movf    V_ser_index, w  ; [t+9]  move "byte index" for serial transmission into w
        addwf   V_ser_index, w  ; [t+10] multiply by two for structure of "case list"
        andlw   B'00000110'     ; [t+11] limit offset for computed jump..
        addwf   PCL, F          ; [t+12] computed jump: add (w) to program counter(low)
SerTxByte0:
        movlw   0xFF            ; [t+14(!)] Byte[0] = Sync/Statusbyte, always 0xFF at the moment
        goto    SerTxSetData    ; [t+15,16]
SerTxByte1:
        movf    V_AvrgI_l , w   ; [t+14] Byte[1] = Least significant bits of I-channel (I7..I0).
        goto    SerTxSetData    ; [t+15,16]   (V_Avrg := V_Sum in state "Send Stopbit" of last byte in frame)
SerTxByte2:
        movf    V_AvrgQ_l , w   ; [t+14] Byte[2] = Least significant bits of Q-channel (Q7..Q0).
        goto    SerTxSetData    ; [t+15,16]
SerTxByte3:
        swapf   V_AvrgQ_h , w   ; [t+14] Byte[3] : Bits 7..4 = Most significant bits of Q-channel (Q11..Q8),
        iorwf   V_AvrgI_h , w   ; [t+15]           Bits 3..0 = Most significant bits of I-channel (I11..I8),
SerTxSetData2:
        nop                     ; [t+16] compensate for constant loop time
SerTxSetData:
        movwf   V_ser_tx_reg    ; [t+17] move <w> into serial TX register (realized by s/w)
        goto    SerState_End    ; [t+18] wait for end of serial bit time, here: 22 cycles total

SerState_TxD0        ; 0x02 = send databit #0 (LSB)
               ; This bit-time is also PARTLY used as ADC settling time. 
               ; Here, 8.68us passed since selecting the "I"-channel. At least 3 more cycles for settling.
        btfss   V_ser_tx_reg,0  ; [t+5] skip next instruction if data bit #0 set
        M_TXD_PIN_LOW           ; [t+6] serial TX output low
        btfsc   V_ser_tx_reg,0  ; [t+7] skip next instruction if data bit #0 clear
        M_TXD_PIN_HIGH          ; [t+8] serial TX output high ... note 800ns jitter between L- & H-bits
        incf    V_ser_state,F   ; [t+9] next state for serial "TX" state machine
               ; Meanwhile, the analog "I" input voltage is stable at the ADC. Time to convert:
        bsf     ADCON0, GO      ; [t+10] start another A/D conversion of the "I"-channel... 
               ; 11*1.6us = 17.6us = 44 instr_cycles later, the A/D conversion should be ready
               ; Debugging: No idea why ADCON0."GO" cannot be set in the simulator !
        goto    SerState_End7   ; [t+11] wait for end of serial bit time, here: 22 cycles total


SerState_TxD1        ; 0x03 = send databit #1
                ; During this bit-time, the A/D converter is busy for the "I"-channel
                ; For future extensions: enough time here to do "something else" !
        btfss   V_ser_tx_reg,1  ; [t+5] skip next instruction if data bit #0 set
        M_TXD_PIN_LOW           ; [t+6] serial TX output low
        btfsc   V_ser_tx_reg,1  ; [t+7] skip next instruction if data bit #0 clear
        M_TXD_PIN_HIGH          ; [t+8] serial TX output high
        incf    V_ser_state,F   ; [t+9] next state for serial "TX" state machine
        goto    SerState_End7   ; [t+10] wait for end of serial bit time, here: 21 cycles / bit

SerState_TxD2        ; 0x04 = send databit #2
                ; During this bit-time, the A/D converter is busy for the "I"-channel
                ; For future extensions: enough time here to do "something else" !
        btfss   V_ser_tx_reg,2  ; [t+5] skip next instruction if data bit #0 set
        M_TXD_PIN_LOW           ; [t+6] serial TX output low
        btfsc   V_ser_tx_reg,2  ; [t+7] skip next instruction if data bit #0 clear
        M_TXD_PIN_HIGH          ; [t+8] serial TX output high
        incf    V_ser_state,F   ; [t+9] next state for serial "TX" state machine
        goto    SerState_End8   ; [t+10] wait for end of serial bit time, here: 22 cycles / bit

SerState_TxD3        ; 0x05 = send databit #3
                ; During this bit-time, A/D conversion of the "I"-channel gets complete
                ; and the value can be taken from the ADC registers.
        btfss   V_ser_tx_reg,3  ; [t+5] skip next instruction if data bit #0 set
        M_TXD_PIN_LOW           ; [t+6] serial TX output low
        btfsc   V_ser_tx_reg,3  ; [t+7] skip next instruction if data bit #0 clear
        M_TXD_PIN_HIGH          ; [t+8] serial TX output high
        incf    V_ser_state,F   ; [t+9] next state for serial "TX" state machine
               ; Conversion of "I"-channel complete, switch the analog input multiplexer to the
               ; "Q" channel. See datasheet for PIC12F675, "A/D Acquisition requirements".
        bsf     ADCON0, CHS0    ; [t+10] connect AN1 (=GP1, "Q"-chnl) to the PIC's 10bit A/D converter.
                                ; 10 us later (=8.86us + 3 CPU cycles a 400ns) we may start the A/D conversion.
               ; Add the 10-bit-value from the ADC to the sum of 4 "I"-channel values
        bsf     STATUS,RP0      ;![t+11] select RAM-bank 1 because some blockhead placed ADRESL there (!)
#if(SWI_SIMULATOR)
        movlw   #1              ;![t+12] SIMULATOR ONLY: ADC-Result(low) always "1"
#else
        movf    ADRESL,    w    ;![t+12] load lower byte of ADC result (from previous conversion)
#endif
        bcf     STATUS,RP0      ;![t+13] select RAM-bank 0 as "standard" value
        addwf   V_SumI_l,  F    ; [t+14] add to sum of 4 "I"-channel values
        btfsc   STATUS, C       ; [t+15] carry from LOW to HIGH byte ?
        incf    V_SumI_h,  F    ; [t+16]  yes, increment higher byte of SUM
        movf    ADRESH,    w    ; [t+17] load upper byte of ADC result (from previous conversion)
        addwf   V_SumI_h,  F    ; [t+18] add to sum of 4 "I"-channel values (max. result is 12 bit)
  ;ex : goto    SerState_End    ; [t+19] wait for end of serial bit time, here: 22 cycles / bit
  ;but: not enough time to jump to "SerState_End", must go directly to "MainLoop" :-(
        nop                     ; [t+10]
        goto    MainLoop        ; [t+20] next bit please, here: after 22 cycles


SerState_TxD4        ; 0x06 = send databit #4
                ; During this bit-time, the voltage from the "Q" input settles
                ;  and we may start another A/D conversion for the "Q"-channel.
        btfss   V_ser_tx_reg,4  ; [t+5] skip next instruction if data bit #0 set
        M_TXD_PIN_LOW           ; [t+6] serial TX output low
        btfsc   V_ser_tx_reg,4  ; [t+7] skip next instruction if data bit #0 clear
        M_TXD_PIN_HIGH          ; [t+8] serial TX output high
        incf    V_ser_state,F   ; [t+9] next state for serial "TX" state machine
        nop                     ; [t+10] some more cycles until we start the A/D conversion..
        nop                     ; [t+11]
        nop                     ; [t+12]
        nop                     ; [t+13]
               ; Now the analog "Q" input voltage is stable at the ADC. Time to start conversion:
        bsf     ADCON0, GO      ; [t+14] start another A/D conversion of the "Q"-channel... 
               ; 11*1.6us = 17.6us = 44 instr_cycles later, the A/D conversion should be ready
        goto    SerState_End2   ; [t+15] wait for end of serial bit time, here: 21 cycles / bit


SerState_TxD5        ; 0x07 = send databit #5
                ; During this bit-time, the A/D converter is busy for the "Q"-channel.
                ; For future extensions: enough time here to do "something else" !
        btfss   V_ser_tx_reg,5  ; [t+5] skip next instruction if data bit #0 set
        M_TXD_PIN_LOW           ; [t+6] serial TX output low
        btfsc   V_ser_tx_reg,5  ; [t+7] skip next instruction if data bit #0 clear
        M_TXD_PIN_HIGH          ; [t+8] serial TX output high
        incf    V_ser_state,F   ; [t+9] next state for serial "TX" state machine
        goto    SerState_End8   ; [t+10] wait for end of serial bit time, here: 22 cycles / bit

SerState_TxD6        ; 0x08 = send databit #6 
                ; During this bit-time, the A/D converter is busy for the "Q"-channel.
                ; For future extensions: enough time here to do "something else" !
        btfss   V_ser_tx_reg,6  ; [t+5] skip next instruction if data bit #0 set
        M_TXD_PIN_LOW           ; [t+6] serial TX output low
        btfsc   V_ser_tx_reg,6  ; [t+7] skip next instruction if data bit #0 clear
        M_TXD_PIN_HIGH          ; [t+8] serial TX output high
        incf    V_ser_state,F   ; [t+9] next state for serial "TX" state machine
        goto    SerState_End7   ; [t+10] wait for end of serial bit time, here: 21 cycles / bit

SerState_TxD7        ; 0x09 = send databit #7 (MSB)
                ; During this bit-time, A/D conversion of the "Q"-channel gets complete
                ; and the value can be taken from the ADC registers.
        btfss   V_ser_tx_reg,7  ; [t+5] skip next instruction if data bit #0 set
        M_TXD_PIN_LOW           ; [t+6] serial TX output low
        btfsc   V_ser_tx_reg,7  ; [t+7] skip next instruction if data bit #0 clear
        M_TXD_PIN_HIGH          ; [t+8] serial TX output high
        incf    V_ser_state,F   ; [t+9] next state for serial "TX" state machine (->TxStopBit)
               ; Add the 10-bit-value from the ADC to the sum of 4 "Q"-channel values
        bsf     STATUS,RP0      ;![t+10] select RAM-bank 1 because some blockhead placed ADRESL there (!)
        movf    ADRESL,    w    ;![t+11] load lower byte of ADC result (from previous conversion)
        bcf     STATUS,RP0      ;![t+12] select RAM-bank 0 as "standard" value
        addwf   V_SumQ_l,  F    ; [t+13] add A/D-Result(low) to sum of 4 "Q"-channel values
        btfsc   STATUS, C       ; [t+14] carry from LOW to HIGH byte ?
        incf    V_SumQ_h,  F    ; [t+15]  yes, increment higher byte of SUM
        movf    ADRESH,    w    ; [t+16] load upper byte of ADC result (from previous conversion)
        addwf   V_SumQ_h,  F    ; [t+17] add to sum of 4 "Q"-channel values (max. result is 12 bit)
        goto    SerState_End    ; [t+18] wait for the end of serial bit time, here: 22 cycles / bit


SerState_TxStopBit   ; 0x0A = send stopbit 
                ; During this bit-time,  prepare the next 400us-cycle.
        M_TXD_PIN_HIGH          ; [t+5] serial TX output high
               ; Conversion of "Q"-channel complete, switch the analog input multiplexer to the
               ; "I" channel. See datasheet for PIC12F675, "A/D Acquisition requirements".
        bcf     ADCON0, CHS0    ; [t+6] connect AN0 (=GP0, "I"-chnl) to the PIC's 10bit A/D converter.
                                ; 10 us later (=8.86us + 3 CPU cycles a 400ns) we may start the A/D conversion.
        incf    V_ser_state, F  ; [t+7] next state for serial "TX" state machine ("Tx Gap")
        incf    V_ser_index, F  ; [t+8] increment "byte index" for serial transmission
        btfss   V_ser_index, 2  ; [t+9] if(byte index>=4) then end of 4-byte-frame ...
        goto    SerState_NoEOF  ; [t+10] "No EOF" means "no end-of-frame" here ;-)
                ; If this was the last byte in the 4-byte-frame,
                ;   we have added 4 "I" and 4 "Q" values. They will be transferred to the "Averages".
                ;  (no time here to do the same for the Q-channel, see state "TxGap")
        movf    V_SumI_l, w     ; [t+11] V_AvrgI = V_SumI
        movwf   V_AvrgI_l       ; [t+12]
        movf    V_SumI_h, w     ; [t+13]
        movwf   V_AvrgI_h       ; [t+14]
        clrf    V_SumI_l        ; [t+15] V_SumI = 0    
        clrf    V_SumI_h        ; [t+16]
        goto    SerState_End1   ; [t+17] wait for end of serial bit time, here: 22 cycles / bit
SerState_NoEOF:                 ; [t+12]
        goto    SerState_End6   ; [t+12] wait for end of serial bit time, here: 22 cycles / bit

SerState_TxGap       ; 0x0B = send 'adjustment gap' after stopbit
                ; During this bit-time,  some more preparation for next 400us-cycle is done.
        M_TXD_PIN_HIGH          ; [t+5] set serial TX output HIGH (=stopbit, =idle state of line)
        movlw   C_SER_STATE_TX_START_BIT ; [t+6] next value for state machine: send start bit
        movwf   V_ser_state     ; [t+7]
        btfss   V_ser_index, 2  ; [t+8] if(byte index>=4) then end of 4-byte-frame ...
        goto    SerState_NoEOF2 ; [t+9] "No EOF" means "no end-of-frame" here ;-)
        clrf    V_ser_index     ; [t+10] set "packet byte index" to zero for the next packet
                ; If this was the last byte in the 4-byte-frame,
                ;   we have added 4 "I" and 4 "Q" values. They will be transferred to the "Averages".
                ;  (here only time to do that for the Q-channel, see state "TxStopBit")
        movf    V_SumQ_l, w     ; [t+11] V_AvrgQ = V_SumQ
        movwf   V_AvrgQ_l       ; [t+12]
        movf    V_SumQ_h, w     ; [t+13]
        movwf   V_AvrgQ_h       ; [t+14]
        clrf    V_SumQ_l        ; [t+15] V_SumQ = 0
        clrf    V_SumQ_h        ; [t+16]
        call    Delay10         ; [t+17] wait another 10 instruction cycles
        goto    SerState_End2   ; [t+27] wait for end of the "GAP", here: total = 33 cycles / gap

SerState_NoEOF2:                ; [t+11]
        call    Delay10         ; [t+11] wait another 10 instruction cycles
        call    Delay8          ; [t+21] wait another 8  instruction cycles
        goto    SerState_End    ; [t+29] wait for end of the "GAP", here: total = 33 cycles / gap



SerState_WaitRx      ; 0x0C = ?
        M_TXD_PIN_HIGH        ; set serial TX output HIGH (=stopbit, =idle state of line)
        goto    SerState_End

SerState_Rx          ; 0x0D = ?
        M_TXD_PIN_HIGH        ; set serial TX output HIGH (=stopbit, =idle state of line)
        goto    SerState_End

SerState_RxStop      ; 0x0E = ?
        M_TXD_PIN_HIGH        ; set serial TX output HIGH (=stopbit, =idle state of line)
        goto    SerState_End

SerState_RxDone      ; 0x0F = ?
        M_TXD_PIN_HIGH        ; set serial TX output HIGH (=stopbit, =idle state of line)
        goto    SerState_End


      ; End of main loop, with some optional NOPS to keep the loop time constant...
SerState_End8:  nop    ; [t+12], sometimes [t+11]: wait 8  additional cycles before end of serial bit
SerState_End7:  nop    ; [t+13], sometimes [t+12]: wait 7  additional cycles before end of serial bit
SerState_End6:  nop    ; [t+14], sometimes [t+13]: wait 6  additional cycles before end of serial bit
SerState_End5:  nop    ; [t+15], sometimes [t+14]: wait 5  additional cycles before end of serial bit
SerState_End4:  nop    ; [t+16], sometimes [t+15]: wait 4  additional cycles before end of serial bit
SerState_End3:  nop    ; [t+17], sometimes [t+16]: wait 3  additional cycles before end of serial bit
SerState_End2:  nop    ; [t+18], sometimes [t+17]: wait TWO additional cycles before end of serial bit
SerState_End1:  nop    ; [t+19], sometimes [t+18]: wait ONE additional cycle before end of serial bit
SerState_End:          ; [t+20], sometimes [t+19] 
        goto    MainLoop      ;  [t+20] ,sometimes [t+19]
                              ; Total time (for a serial data bit) = 21 or 22 instruction cycles per loop
                              ;  (the "goto" command requires two cycles)


;******************************************************************************
;  Miscellanous subroutines

Delay10:        nop           ; delay of 10 instruction cycles (incl. "call"+"return") ...
Delay9:         nop
Delay8:         nop
Delay7:         nop
Delay6:         nop
Delay5:         nop
Delay4:         return        ; additional delay for "call" + "return": 4 cycles




;******************************************************************************
;  Program entry point after hardware reset:   Initialisation, endless loop.
main:

  ; Setup port directions for PIC12F675
        movlw   0x07                   ; Turn all "comparators" on GP2..0 OFF
        movwf   CMCON                  ; (for lowest power consumtion, and for dig.I/O)

        M_TXD_PIN_HIGH                 ; no effect yet, because ANSEL + GPIO not set 
                                       ; but GPIO.x ("TXD") will go to H level soon

        bsf     STATUS,RP0             ;! select bank 1 to access ANSEL and TRISIO registers

        ; Initialize the PIC12F675's internal 10-bit ADC, Part 1... 
        ;  Parameters: conversion clock = 1.6us, Crystal = 10MHz -> divide by 16
        movlw   B'01010011'            ;! ANSEL-Register, seriously affects behaviour of GPIO..
                ; ||||||||
                ; ||||||||_______ ANS3..ANS0: Analog Select Bits (0=digital I/O on GPx, 1=analog input)
                ; ||||                                                        
                ; ||||___________ ADCS2..ADCS0: A/D Conversion Clock Select bits. 101=Fosc/16
                ; |                                                               
                ; |______________ bit7: unimplemented in PIC12F675.
        movwf   ANSEL                  ;! write "w" register into ANSEL register
                                       ;  (affects the contents of GPIO in the simulator ?!?!?)

        ; Set TRISIO: data direction register for GPIO,   1=INPUT(!),  0=OUTPUT(!!)
        movlw   B'00111011'            ;! for TRISIO register:
                ; ||||||||
                ; ||||||||_______ GP0: 1 = high impedance, used as analog input  
                ; |||||||________ GP1: 1 = high impedance, used as analog input
                ; ||||||_________ GP2: 0 = used as digital output for "TXD"
                ; |||||__________ GP3: 1 = IS ALWAYS an input, here   "RXD"
                ; ||||___________ GP4: 1 = high impedance, don't affect xtal oscillator
                ; |||____________ GP5: 1 = high impedance, don't affect xtal oscillator
                ; ||_____________ two unused bits in TRISIO
        movwf   TRISIO                 ;! write "w" register into TRISIO register


        ; Set "OPTION"-Register (Timer-Control, internal pullups etc)
        ; The crystal clock frequency is exactly 10.000 MHz, 
        ;  the timer is driven with the highest possible frequency.
        movlw   B'10001000'            ; for OPTION register:
                ; ||||||||
                ; ||||||||_______ PS2..PS0: prescaler rate (hr: for WDT, 000 = 1:1 )
                ; |||||__________ PSA: prescaler assignment: 0=timer0, 1=WDT
                ; ||||___________ T0SE: TMR0 Source Edge Select Bit: 0= h->l
                ; |||____________ T0CS: Clock Source Bit: 0=internal Tcycle
                ; ||_____________ INTEDG: Interrupt Edge (for GP2/INT) 0=falling
                ; |______________ GPPU: GPIO pullup enable: 1=GPIO pull-ups are disabled.
        movwf   OPTION_REG             ;! write "w" register into OPTION register


        bcf     STATUS,RP0             ;! select bank 0 as "standard" value

#if(SWI_TEST_INIT_STEPS)
        M_TXD_PIN_LOW                  ; test... where's the nasty bug ?
        M_TXD_PIN_HIGH
#endif ; (SWI_TEST_INIT_STEPS)


  ; Set initial state of important output ports
        bsf     PIN_SER_TXD            ; set TXD output HIGH (=idle state)
                ; "TXD" is GP2. If there is no effect on "GPIO" in the simulator,
                ; "ANSEL" is not set properly (setting GPIO has no effect if a pin
                ;       is defined as "analog input", even if TRISIO.x=0 !! )



  ; Initialize the PIC12F675's internal 10-bit ADC, Part 2... (ANSEL already done)
        movlw   B'10000001'            ; ADCON0-Register:
                ; ||||||||
                ; ||||||||_______ ADON:       1 =  A/D converter module is operating
                ; |||||||________ GO/!DONE:   0 =  A/D conversion not in progress (must not be turned on NOW)
                ; ||||||_________ CHS1..CHS0: Analog Channel Select bits, will be set later (00= select AN0)
                ; ||||___________ bit5..4:    unimplemented.
                ; ||_____________ VCFG:       Reference Voltage control. 0=use Vdd as reference voltage.
                ; |______________ ADFM:       A/D Result Format Select bit.  1=right justified (bits 11..0 used)
        movwf   ADCON0                 ; write "w" register into ADCON0 register

#if(SWI_TEST_INIT_STEPS)
        M_TXD_PIN_LOW                  ; test... where's the nasty bug ?
        M_TXD_PIN_HIGH
        M_TXD_PIN_LOW                  
        M_TXD_PIN_HIGH
#endif ; (SWI_TEST_INIT_STEPS)

       

  ; Initialize "software" UART (which does NO LONGER use an interrupt)
        clrf    V_ser_state            ; no invalid values for state machine, using computed goto
        clrf    TMR0                   ; load Timer0 counter register
        movlw   B'00000000'            ; INTCON-Register:
                ; ||||||||_______ GPIF: General Purpose I/O interrupt flag
                ; |||||||________ INTF:
                ; ||||||_________ T0IF: Timer0 interrupt flag
                ; |||||__________ GPIE: 0 = disable interrupt on PortB change
                ; ||||___________ INTE:
                ; |||____________ T0IE: 0 = disable timer0 interrupt
                ; ||_____________ PEIE:
                ; |______________ GIE:  0 = keep global interrupts disabled
        movwf   INTCON                 ; write "w" into INTCON register (available in both banks)


  ; application-specific initialisation...
        movlw   'A'                    ; test pattern for serial transmission
        movwf   V_ser_tx_reg
        clrf    V_ser_index            ; start at byte[0] in serial TX frame

        movlw   C_SER_STATE_TX_START_BIT ; start serial transmission cycle a.s.a.p.
        movwf   V_ser_state

        clrf    V_SumI_l                 ; clear Sum of 4 values for both channels
        clrf    V_SumI_h
        clrf    V_SumQ_l
        clrf    V_SumQ_h

        clrf    V_AvrgI_l                ; clear Average values for both channel
        clrf    V_AvrgI_h
        clrf    V_AvrgQ_l
        clrf    V_AvrgQ_h


#if(0 && SWI_TEST_INIT_STEPS)
        M_TXD_PIN_LOW                  ; test... where's the nasty bug ?
        M_TXD_PIN_HIGH
        M_TXD_PIN_LOW                  
        M_TXD_PIN_HIGH
        M_TXD_PIN_LOW                  
        M_TXD_PIN_HIGH
#endif ; (SWI_TEST_INIT_STEPS)


        goto    MainLoop 




 END
